/**CFile***********************************************************************

  FileName    [cuddCof.c]

  PackageName [cudd]

  Synopsis    [Cofactoring functions.]

  Description [External procedures included in this module:
		<ul>
		<li> Cudd_Cofactor()
		</ul>
	       Internal procedures included in this module:
		<ul>
		<li> cuddGetBranches()
		<li> cuddCheckCube()
		<li> cuddCofactorRecur()
		</ul>
	      ]

  SeeAlso     []

  Author      [Fabio Somenzi]

  Copyright [ This file was created at the University of Colorado at
  Boulder.  The University of Colorado at Boulder makes no warranty
  about the suitability of this software for any purpose.  It is
  presented on an AS IS basis.]

******************************************************************************/

#include    "util.h"
#include    "cuddInt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

#ifndef lint
static char rcsid[] DD_UNUSED = "$Id: cuddCof.c,v 1.6 1997/11/05 10:55:09 fabio Exp $";
#endif

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Computes the cofactor of f with respect to g.]

  Description [Computes the cofactor of f with respect to g; g must be
  the BDD or the ADD of a cube. Returns a pointer to the cofactor if
  successful; NULL otherwise.]

  SideEffects [None]

  SeeAlso     [Cudd_bddConstrain Cudd_bddRestrict]

******************************************************************************/
DdNode *
Cudd_Cofactor(
  DdManager * dd,
  DdNode * f,
  DdNode * g)
{
    DdNode *res,*zero;

    zero = Cudd_Not(DD_ONE(dd));
    if (g == zero || g == DD_ZERO(dd)) {
	(void) fprintf(stdout,"Cudd_Cofactor: Invalid restriction 1\n");
	return(NULL);
    }
    do {
	dd->reordered = 0;
	res = cuddCofactorRecur(dd,f,g);
    } while (dd->reordered == 1);
    return(res);

} /* end of Cudd_Cofactor */


/**Function********************************************************************

  Synopsis    [Check is the two cofactors of f is the same.]

  Description [Check if the two cofactors of f is the same; the cubes should
  have the same index. The cubes could be either BDDs or ADDs. If the cofactors
  are the same, return 1; otherwise, return 0.]

  SideEffects [None]

  SeeAlso     [Yuan Lu]

******************************************************************************/
int
Cudd_CofactorEqual(
  DdManager *dd,
  DdNode *f,
  DdNode *cube1,
  DdNode *cube2)
{
  return cuddCofactorEqualRecur(dd, f, cube1, cube2);
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Computes the children of g.]

  Description []

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
void
cuddGetBranches(
  DdNode * g,
  DdNode ** g1,
  DdNode ** g0)
{
    DdNode	*G = Cudd_Regular(g);

    *g1 = cuddT(G);
    *g0 = cuddE(G);
    if (Cudd_IsComplement(g)) {
	*g1 = Cudd_Not(*g1);
	*g0 = Cudd_Not(*g0);
    }

} /* end of cuddGetBranches */


/**Function********************************************************************

  Synopsis    [Checks whether g is the BDD of a cube.]

  Description [Checks whether g is the BDD of a cube. Returns 1 in case
  of success; 0 otherwise. The constant 1 is a valid cube, but all other
  constant functions cause cuddCheckCube to return 0.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
int
cuddCheckCube(
  DdManager * dd,
  DdNode * g)
{
    DdNode *g1,*g0,*one,*zero;
    
    one = DD_ONE(dd);
    if (g == one) return(1);
    if (Cudd_IsConstant(g)) return(0);

    zero = Cudd_Not(one);
    cuddGetBranches(g,&g1,&g0);

    if (g0 == zero) {
        return(cuddCheckCube(dd, g1));
    }
    if (g1 == zero) {
        return(cuddCheckCube(dd, g0));
    }
    return(0);

} /* end of cuddCheckCube */


/**Function********************************************************************

  Synopsis    [Performs the recursive step of Cudd_Cofactor.]

  Description [Performs the recursive step of Cudd_Cofactor. Returns a
  pointer to the cofactor if successful; NULL otherwise.]

  SideEffects [None]

  SeeAlso     [Cudd_Cofactor]

******************************************************************************/
DdNode *
cuddCofactorRecur(
  DdManager * dd,
  DdNode * f,
  DdNode * g)
{
    DdNode *one,*zero,*F,*G,*g1,*g0,*f1,*f0,*t,*e,*r;
    unsigned int topf,topg;
    int comple;

    F = Cudd_Regular(f);
    if (cuddIsConstant(F)) return(f);

    one = DD_ONE(dd);

    /* The invariant g != 0 is true on entry to this procedure and is
    ** recursively maintained by it. Therefore it suffices to test g
    ** against one to make sure it is not constant.
    */
    if (g == one) return(f);
    /* From now on, f and g are known not to be constants. */

    comple = f != F;
    r = cuddCacheLookup2(dd,Cudd_Cofactor,F,g);
    if (r != NULL) {
	return(Cudd_NotCond(r,comple));
    }

    topf = dd->perm[F->index];
    G = Cudd_Regular(g);
    topg = dd->perm[G->index];

    /* We take the cofactors of F because we are going to rely on
    ** the fact that the cofactors of the complement are the complements
    ** of the cofactors to better utilize the cache. Variable comple
    ** remembers whether we have to complement the result or not.
    */
    if (topf <= topg) {
	f1 = cuddT(F); f0 = cuddE(F);
    } else {
	f1 = f0 = F;
    }
    if (topg <= topf) {
	g1 = cuddT(G); g0 = cuddE(G);
	if (g != G) { g1 = Cudd_Not(g1); g0 = Cudd_Not(g0); }
    } else {
	g1 = g0 = g;
    }

    zero = Cudd_Not(one);
    if (topf >= topg) {
	if (g0 == zero || g0 == DD_ZERO(dd)) {
	    r = cuddCofactorRecur(dd, f1, g1);
	} else if (g1 == zero || g1 == DD_ZERO(dd)) {
	    r = cuddCofactorRecur(dd, f0, g0);
	} else {
	    (void) fprintf(stdout,"Cudd_Cofactor: Invalid restriction 2\n");
	    return(NULL);
	}
	if (r == NULL) return(NULL);
    } else /* if (topf < topg) */ {
	t = cuddCofactorRecur(dd, f1, g);
	if (t == NULL) return(NULL);
    	cuddRef(t);
    	e = cuddCofactorRecur(dd, f0, g);
	if (e == NULL) {
	    Cudd_RecursiveDeref(dd, t);
	    return(NULL);
	}
	cuddRef(e);

	if (t == e) {
	    r = t;
	} else if (Cudd_IsComplement(t)) {
	    r = cuddUniqueInter(dd,(int)F->index,Cudd_Not(t),Cudd_Not(e));
	    if (r != NULL)
		r = Cudd_Not(r);
	} else {
	    r = cuddUniqueInter(dd,(int)F->index,t,e);
	}
	if (r == NULL) {
	    Cudd_RecursiveDeref(dd ,e);
	    Cudd_RecursiveDeref(dd ,t);
	    return(NULL);
	}
	cuddDeref(t);
	cuddDeref(e);
    }

    cuddCacheInsert2(dd,Cudd_Cofactor,F,g,r);

    return(Cudd_NotCond(r,comple));

} /* end of cuddCofactorRecur */


int
cuddCofactorEqualRecur(
  DdManager *dd, 
  DdNode *f, 
  DdNode *cube1, 
  DdNode *cube2)
{
  DdNode *F, *G;
  DdNode *f1, *f2;
  int res, topf, topg;

  F = Cudd_Regular(f);
  if(cuddIsConstant(F)) return 1;

  topf = dd->perm[F->index];
  G = Cudd_Regular(cube1);
  topg = dd->perm[G->index];
  if(topf < topg) {
    if(!cuddCofactorEqualRecur(dd, cuddE(F), cube1, cube2)) return 0;
    if(!cuddCofactorEqualRecur(dd, cuddT(F), cube1, cube2)) return 0;
    return 1;
  } else {
    f1 = cuddCofactorNoRefRecur(dd, f, cube1);
    f2 = cuddCofactorNoRefRecur(dd, f, cube2);
    return (f1==f2);
  }
}

DdNode *
cuddCofactorNoRefRecur(DdManager *dd, DdNode *f, DdNode *c)
{
  DdNode *F, *C;
  DdNode *f1, *f0, *c0, *c1;
  DdNode *one, *t, *e;
  int topf, topc;

  one = DD_ONE(dd);

  C=Cudd_Regular(c);
  if(cuddIsConstant(C))
    return (c==one) ? f : NULL;
  F = Cudd_Regular(f);
  if(cuddIsConstant(F)) return f;

  topf = dd->perm[F->index];
  topc = dd->perm[C->index];
  c1 = cuddT(c); c0 = cuddE(c);
  if(topf == topc) {
    f1 = cuddT(F); f0 = cuddE(F);
    if(Cudd_IsComplement(f)) {
      f1 = Cudd_Not(f1); f0 = Cudd_Not(f0);
    }
    t = cuddCofactorNoRefRecur(dd, f1, c1);
    e = cuddCofactorNoRefRecur(dd, f0, c0);
  } else { /* topf > topc */
    t = cuddCofactorNoRefRecur(dd, f, c1);
    e = cuddCofactorNoRefRecur(dd, f, c0);
  }
  return ((!e) ? t : e);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

