/**CFile***********************************************************************

  FileName    [cuddBddOp.c]

  PackageName [cudd]

  Synopsis    [Some useful functions for manipulating BDD.]

  Description []

  Author      [Marco Roveri]

  Copyright   []

******************************************************************************/

#include    "util.h"
#include    "st.h"
#include    "cuddInt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

#ifndef lint
static char rcsid[] DD_UNUSED = "$Id: $";
#endif


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/
/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int bddCheckPositiveCube ARGS((DdManager *manager, DdNode *cube));
static void ddClearFlag ARGS((DdNode *f));
static CUDD_VALUE_TYPE Cudd_fatal_error ARGS((const char * message));

/**AutomaticEnd***************************************************************/

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis    [Computes the difference between two BDD cubes.]

  Description [Computes the set difference between two BDD cubes.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
DdNode *
Cudd_bddCubeDiff(
DdManager * dd,
DdNode * a,
DdNode * b)
{
  DdNode * res;

  if (bddCheckPositiveCube(dd, a) == 0) {
    (void) fprintf(stdout,"Error: (arg_1) Can only abstract positive cubes\n");
    return(NULL);
  }
  if (bddCheckPositiveCube(dd, b) == 0) {
    (void) fprintf(stdout,"Error: (arg_2) Can only abstract positive cubes\n");
    return(NULL);
  }
  do {
    dd->reordered = 0;
    res = cudd_bddCubeDiffRecur(dd, a, b);
  } while (dd->reordered == 1);
  return(res);
}

/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
int 
Cudd_BddGetLowestVar(
DdManager *dd,
DdNode * N)
{
  int res = Cudd_BddGetLowestVarRecur(dd, N, 0);
  ddClearFlag(N);
  return(res);
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Performs the recursive step of Cudd_bddCubeDiff.]

  Description [Performs the recursive step of Cudd_bddCubeDiff.
  Returns a pointer to the result if successful; NULL otherwise.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
DdNode *
cudd_bddCubeDiffRecur(
DdManager * dd,
DdNode    * f,
DdNode    * g)
{
  DdNode       * t,  * e , * res;
  unsigned int topf, topg;
  DdNode       * one  = DD_ONE(dd);
  DdNode       * zero = Cudd_Not(one);
  
  if ((f == zero) || (g == zero))
    Cudd_fatal_error("cudd_bddCubeDiff: f == ZERO || g == ZERO");
  if (f == one) return(f);

  topf = cuddI(dd,f->index);
  topg = cuddI(dd,g->index);

  if (topf < topg) {
    e = zero;
    cuddRef(e);
    t = cudd_bddCubeDiffRecur(dd, cuddT(f), g);
    if (t == NULL) {
      cuddDeref(e);
      return(NULL);
    }
    cuddRef(t);
    res = (t == e) ? t : cuddUniqueInter(dd, f->index, t, e);
    if (res == NULL) {
      cuddDeref(t);
      Cudd_RecursiveDeref(dd, e);
      return(NULL);
    }
    cuddDeref(t);
    cuddDeref(e);
    return(res);
  }
  else if (topf == topg) {
    cuddRef(cuddT(f));
    cuddRef(cuddT(g));
    res = cudd_bddCubeDiffRecur(dd, cuddT(f), cuddT(g));
    if (res == NULL) {
      cuddDeref(cuddT(f));
      cuddDeref(cuddT(g));
      return(NULL);
    }
    cuddDeref(cuddT(f));
    cuddDeref(cuddT(g));
    return(res);
  }
  else {
    cuddRef(f);
    cuddRef(cuddT(g));
    res = cudd_bddCubeDiffRecur(dd,f,cuddT(g));
    if (res == NULL) {
      cuddDeref(f);
      cuddDeref(cuddT(g));
      return(NULL);
    }
    cuddDeref(f);
    cuddDeref(cuddT(g));
    return(res);
  }
}

/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
int 
Cudd_BddGetLowestVarRecur(
DdManager *dd,
DdNode * N,
int index)
{
  int i; 
  DdNode * RN = Cudd_Regular(N);
  
  if (Cudd_IsComplement(RN->next) || cuddIsConstant(RN)) return(index);
  RN->next = Cudd_Not(RN->next);
  i = RN->index;
  if (i > index) index = i;
  return(Cudd_BddGetLowestVarRecur(dd, cuddT(RN),
                              Cudd_BddGetLowestVarRecur(dd, cuddE(RN), index)));
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Performs a DFS from f, clearing the LSB of the next
  pointers.]

  Description []

  SideEffects [None]

  SeeAlso     [ddSupportStep ddDagInt]

******************************************************************************/
static void
ddClearFlag(
DdNode *f)
{
    if (!Cudd_IsComplement(f->next)) {
	return;
    }
    /* Clear visited flag. */
    f->next = Cudd_Regular(f->next);
    if (cuddIsConstant(f)) {
	return;
    }
    ddClearFlag(cuddT(f));
    ddClearFlag(Cudd_Regular(cuddE(f)));
    return;

} /* end of ddClearFlag */

/**Function********************************************************************

  Synopsis [Checks whether cube is an BDD representing the product of
  positive literals.]

  Description [Returns 1 in case of success; 0 otherwise.]

  SideEffects [None]

******************************************************************************/
static int
bddCheckPositiveCube(
DdManager *manager,
DdNode	  *cube)
{
  if (Cudd_IsComplement(cube)) return(0);
  if (cube == DD_ONE(manager)) return(1);
  if (cuddIsConstant(cube)) return(0);
  if (cuddE(cube) == Cudd_Not(DD_ONE(manager))) {
    return(bddCheckPositiveCube(manager, cuddT(cube)));
  }
  return(0);
} /* end of bddCheckPositiveCube */

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
static CUDD_VALUE_TYPE Cudd_fatal_error(const char * message)
{
  /* put here some useful information such as line number */
  fprintf(stderr,"\nFatal error: %s\n", message);
  exit(1);
  return(NULL);
}
