/**CFile***********************************************************************

  FileName    [error.c]

  PackageName [utils]

  Synopsis    [Error routines]

  Description [This file conatins the error routines. This file is
  partitioned in two parts. The first contains general routines, the
  second contains specific error routines.]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <setjmp.h>

#include "utils.h"
#include "node.h"
#include "opt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

extern options_ptr options;
extern FILE * nusmv_stderr;

/*
  PART 1: generic error routines.
*/

/**Variable********************************************************************

  Synopsis    [Control the behavior of <code>util_setjmp()</code>.]

  Description [Modifies the behavior of <code>util_setjmp()</code> and
  <code>util_longjmp()</code>. If it is 1, then non-local goto are
  allowed, otherwise no.]

  SeeAlso     [util_setjmp util_longjmp]

******************************************************************************/
static int longjmp_on_err = 0;
void set_longjmp_on_err(void) { longjmp_on_err = 1; }
void cancel_util_setjmp(void) { longjmp_on_err = 0; }  

/**Variable********************************************************************

  Synopsis    [The name of the currently evaluated expression.]

  Description [This variable is used to store the currently evalauted
  expression to perform it's printing.]

  SeeAlso     []

******************************************************************************/
static node_ptr the_node;
void init_the_node() { the_node = Nil;}
void set_the_node(node_ptr n) { the_node = n;}
node_ptr get_the_node() { return(the_node);}

/**Variable********************************************************************

  Synopsis    [Stack used to print out multiple or circular defined symbols.]

  Description [Stack used to print out where multiple or circular
  defined symbols occurs in the input file.] 

  SeeAlso     []

******************************************************************************/
static node_ptr io_atom_stack = Nil;
int io_atom_isempty(void) {return(io_atom_stack == Nil);}
void io_atom_push(node_ptr s) { io_atom_stack = cons(s,io_atom_stack);}
void io_atom_pop() {
  node_ptr temp;

  if (io_atom_stack == Nil) internal_error("io_atom_pop: stack empty");
  temp = cdr(io_atom_stack);
  free_node(io_atom_stack);
  io_atom_stack = temp;
}
void print_io_atom_stack(FILE * fd){
  while(!io_atom_isempty()){
    node_ptr s = car(io_atom_stack);

    io_atom_stack = cdr(io_atom_stack);
    fprintf(fd, "in definition of ");
    print_node(fd, s);
    if (s->lineno) fprintf(fd," at line %d", s->lineno);
    fprintf(fd, "\n");
  }
}

/**Variable********************************************************************

  Synopsis    [Stack context for non-local goto]

  Description [This variable is used to save the stack environment for
  later use.]

  SeeAlso     [util_setjmp util_longjmp]

******************************************************************************/
#if SOLARIS
sigjmp_buf siglongjmp_buf;
#else
jmp_buf siglongjmp_buf;
#endif

/**Function********************************************************************

  Synopsis           [Save stack context for non-local goto]

  Description        [Saves the stack environment in the global
  variable <code>siglongjmp_buf</code> for later use by <code>util_longjmp</code>.]

  SideEffects        []

  SeeAlso            [util_longjmp]

******************************************************************************/
int util_setjmp(void)
{
  int v;

  longjmp_on_err = 1;
#if SOLARIS
  v = sigsetjmp(siglongjmp_buf, 1);
#else
  v = setjmp(siglongjmp_buf);
#endif
  if (v) longjmp_on_err = 0;
  return(v);
}

/**Function********************************************************************

  Synopsis           [Restore the environment saved by <code>util_setjmp</code>.]

  Description        [Restores the environment saved by the last call of
  <code>util_setjmp()<code>. After <code>util_longjmp()</code> is
  completed, program execution continues as if the corresponding
  call of <code>util_setjmp()</code> had just returned a value
  different from <code>0</code> (zero).]

  SideEffects        []

  SeeAlso            [util_setjmp]

******************************************************************************/
void util_longjmp(void) {
  if (longjmp_on_err == 1) {
#if SOLARIS
    siglongjmp(siglongjmp_buf, 2);
#else
    longjmp(siglongjmp_buf, 2);
#endif
  }
}

/**Function********************************************************************

  Synopsis           [General routine to start error reporting.]

  Description        [This is a genral routine to be called by error
  reporting routines as first call. The file name and corresponding
  line number of the token that has generated the error (which is
  retrieved by <code>get_the_node()</code> are printed out.]

  SideEffects        []

  SeeAlso            [finish_parsing_err]

******************************************************************************/
void start_parsing_err()
{
  extern int yylineno;
  
  fprintf(nusmv_stderr,"\n");
  if ( opt_batch(options)) {
    if (get_input_file(options))
      fprintf(nusmv_stderr, "file %s: ", get_input_file(options));
    else
      fprintf(nusmv_stderr, "file stdin: ");
    if (yylineno) fprintf(nusmv_stderr, "line %d: ", yylineno);
  }
  else {
    fprintf(nusmv_stderr, "Error: ");
  }
}

/**Function********************************************************************

  Synopsis           [General routine to terminate error reporting.]

  Description        [This is the genral routine to be called as last
  routine in specific error reporting routines. A call to
  <code>nusmv_exit()</code> is performed.]

  SideEffects        []

  SeeAlso            [start_parsing_err]

******************************************************************************/
void finish_parsing_err()
{
  fprintf(nusmv_stderr,"\n");
  print_io_atom_stack(nusmv_stderr);
  nusmv_exit(1);
}

/**Function********************************************************************

  Synopsis           [General exit routine.]

  Description        [If non local goto are anebaled, instead of
  exiting from the program, then the non local goto is executed.]

  SideEffects        []

  SeeAlso            [util_setjmp util_longjmp]

******************************************************************************/
void nusmv_exit(int n)
{
  util_longjmp();
  if (opt_verbose_level_gt(options, 0) )
    fprintf(nusmv_stderr, "%s: exit(%d)\n", get_pgm_name(options), n);
  exit(n);
}

/**Function********************************************************************

  Synopsis           [General error reporting routine.]

  Description        [Produces a message on the
  <code>nusmv_stderr</code>. The arguments are similar to those of the
  <code>printf</code>.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void rpterr(const char * fmt, ...)
{
  va_list args;
  
  start_parsing_err();
  va_start(args, fmt);
  (void) vfprintf(nusmv_stderr, fmt, args);
  va_end(args);
  finish_parsing_err();
}

/**Function********************************************************************

  Synopsis           [Prints out an internal error.]

  Description        [Produces a message on the <code>nusmv_stderr</code>.
  The message is considered an internal error. The arguments are
  similar to those of the <code>printf</code>.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void internal_error(const char * fmt, ...)
{
  va_list args;
  
  va_start(args, fmt);
  fprintf(nusmv_stderr, "\n\n*** internal error *** \n");
  (void) vfprintf(nusmv_stderr, fmt, args);
  va_end(args);
  fprintf(nusmv_stderr, "\nPlease report this error to nusmv@irst.itc.it\n");
  fprintf(nusmv_stderr, "Send a copy of this output and your input.\n");
  nusmv_exit(1);
}

/*
  PART 2: Specific error routines.
*/
void error_multiple_substitution(node_ptr nodep)
{
  start_parsing_err();
  fprintf(nusmv_stderr, "Multiple substitution for ");
  print_node(nusmv_stderr, nodep);
  finish_parsing_err();
}

void division_by_zero()
{
  start_parsing_err();
  if (get_the_node() != NULL) print_node(nusmv_stderr, get_the_node());
  fprintf(nusmv_stderr, "\nError: division by zero.\n");
  finish_parsing_err();
}

void type_error(node_ptr n)
{
  start_parsing_err();
  indent_node(nusmv_stderr, "type error: value = ", n, "");
  finish_parsing_err();
}  

void range_error(node_ptr n, node_ptr var)
{
  start_parsing_err();
  indent_node(nusmv_stderr, "cannot assign value ", n, " to variable ");
  print_node(nusmv_stderr, var);
  finish_parsing_err();
}

void error_multiple_assignment(node_ptr t1)
{
  start_parsing_err();
  fprintf(nusmv_stderr, "multiply assigned: ");
  print_node(nusmv_stderr, t1);
  finish_parsing_err();
}

void error_empty_range(node_ptr name, int dim1, int dim2)
{
  start_parsing_err();
  fprintf(nusmv_stderr, "empty range type %d..%d for ", dim1, dim2);
  print_node(nusmv_stderr, name);
  finish_parsing_err();
}

void error_assign_both(node_ptr v, node_ptr v1, int lineno, int lineno2)
{
  extern int yylineno;

  yylineno = lineno;
  start_parsing_err();
  fprintf(nusmv_stderr, "assigned ");
  print_node(nusmv_stderr,v);
  fprintf(nusmv_stderr,", line %d: assigned ", lineno2);
  print_node(nusmv_stderr, v1);
  finish_parsing_err();
}

void error_unknown_var_in_order_file(node_ptr n)
{
  start_parsing_err();
  indent_node(nusmv_stderr, "unknown variable in order file :", n, "");
  finish_parsing_err();
}

void warning_variable_not_declared(node_ptr vname) {
  (void) fprintf(nusmv_stderr, "\n********   WARNING   ********\nThe variable: ");
  (void) print_node(nusmv_stderr, vname);
  (void) fprintf(nusmv_stderr, "\nhas not been declared in the source file,\n");
  (void) fprintf(nusmv_stderr, "but it appear in the input ordering file.\n");
  (void) fprintf(nusmv_stderr, "Ignoring it.\n");
  (void) fprintf(nusmv_stderr, "******** END WARNING ********\n");
}

void warning_missing_variable(node_ptr vname) {
  (void) fprintf(nusmv_stderr, "\n********   WARNING   ********\nThe variable: ");
  (void) print_node(nusmv_stderr, vname);
  (void) fprintf(nusmv_stderr, "\nhas not been specified in the ordering file.\n");
  (void) fprintf(nusmv_stderr, "It has been positioned at the end of the ordering.\n");
  (void) fprintf(nusmv_stderr, "******** END WARNING ********\n");
}

void error_var_appear_twice_in_order_file(node_ptr n)
{
  start_parsing_err();
  indent_node(nusmv_stderr, "variable appears twice in order file :", n, "");
  finish_parsing_err();
}

void warning_var_appear_twice_in_order_file(node_ptr n)
{
  (void) fprintf(nusmv_stderr, "\n********   WARNING   ********\n");
  (void) indent_node(nusmv_stderr, "variable appears twice in order file :", n, "\n");
}

void error_var_not_in_order_file(node_ptr n)
{
  start_parsing_err();
  indent_node(nusmv_stderr, "not in order file: ", n, "");
  finish_parsing_err();
}

void error_not_a_number(node_ptr n)
{
  start_parsing_err();
  fprintf(nusmv_stderr, "not a number: ");
  print_node(nusmv_stderr, n);
  finish_parsing_err();
}

void error_undefined(node_ptr s)
{
  start_parsing_err();
  print_node(nusmv_stderr, s);
  fprintf(nusmv_stderr, " undefined");
  finish_parsing_err();
}

void error_redefining(node_ptr s)
{
  start_parsing_err();
  fprintf(nusmv_stderr, "redefining ");
  print_node(nusmv_stderr, s);
  finish_parsing_err();
}

void error_circular(node_ptr s)
{
  start_parsing_err();
  fprintf(nusmv_stderr, "recursively defined: ");
  print_node(nusmv_stderr, s);
  finish_parsing_err();
}

void error_too_many_vars()
{
  start_parsing_err();
  fprintf(nusmv_stderr, "too many variables");
  finish_parsing_err();
}
