/**CFile***********************************************************************

  FileName    [assoc.c]

  PackageName [util]

  Synopsis    [A simple associative list]

  Description [This file provides the user with a data structure that
  implemnts an associative list. If there is already an entry with
  the same ky in the table, than the value associated is replaced with
  the new one.]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include <stdlib.h>
#include "util.h"
#include "node.h"
#include "assoc.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/
/*
  Initial size of the associative table.
*/
#define ASSOC_HASH_SIZE 127
/* 
   Largest everage number of entries per hash element before the table
  is grown.
*/
#define ASSOC_MAX_DENSITY  ST_DEFAULT_MAX_DENSITY

/*
  The factor the table is grown when it becames too full. 
*/
#define ASSOC_GROW_FACTOR  ST_DEFAULT_GROW_FACTOR

/*
  If is non-zero, then every time an entry is found, it is moved on 
  top of the chain. 
*/
#define ASSOC_REORDER_FLAG ST_DEFAULT_REORDER_FLAG

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int assoc_hash_fun ARGS((node_ptr, int));
static int assoc_eq_fun ARGS((node_ptr, node_ptr));
static int assoc_neq_fun ARGS((node_ptr, node_ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

hash_ptr new_assoc()
{
  st_table * new_table = st_init_table_with_params((ST_PFICPCP)assoc_neq_fun,
                                                   (ST_PFICPI)assoc_hash_fun,
                                                   ASSOC_HASH_SIZE,
                                                   ASSOC_MAX_DENSITY,
                                                   ASSOC_GROW_FACTOR,
                                                   ASSOC_REORDER_FLAG);
  if (new_table == (st_table *)NULL) {
    fprintf(stderr, "new_assoc: Out of Memory\n");
    exit(1);
  }
  return((hash_ptr)new_table);
}

node_ptr find_assoc(hash_ptr hash, node_ptr key)
{
  node_ptr data;

  if (st_lookup((st_table *)hash, (char *)key, (char **)&data)) return(data);
  else return(Nil);
}

void insert_assoc(hash_ptr hash, node_ptr key, node_ptr data)
{
  (void)st_insert((st_table *)hash, (char *)key, (char *)data);
}

void remove_assoc(hash_ptr hash, node_ptr key, node_ptr data)
{
  (void)st_delete((st_table *)hash, (char **)&key, (char **)&data);
}

/*
  Frees any internal storage associated with the hash table.
  It's user responsibility to free any storage associated with the
  pointers in the table.
*/
void free_assoc(hash_ptr hash)
{
  (void)st_free_table((st_table *)hash);
}

static enum st_retval delete_entry(char *key, char *data, char * arg)
{
  return(ST_DELETE);
}

void clear_assoc(hash_ptr hash){
  st_foreach(hash, delete_entry, NULL);
}

void clear_assoc_and_free_entries(hash_ptr hash, ST_PFSR fn){
  st_foreach(hash, fn, NULL);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

static int assoc_hash_fun(node_ptr key, int size)
{ return((int)(key) % size); }

static int assoc_eq_fun(node_ptr a1, node_ptr a2)
{ return((a1) == (a2)); }

static int assoc_neq_fun(node_ptr a1, node_ptr a2)
{ return((a1) != (a2)); }
