/**CFile***********************************************************************

  FileName    [smMisc.c]

  PackageName [sm]

  Synopsis    [This file contain the main routine for the batch use of NuSMV.]

  Description [This file contain the main routine for the batch use of
  NuSMV. The batch main executes the various model checking steps in a
  predefined order. After the processing of the input file than it
  return to the calling shell.]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "smInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: smMisc.c,v$";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
unsigned int trans_size;
int num_refine;
int num_ini_abs, num_ini_abs_var, num_ini_abs_bool;
int num_fin_abs, num_fin_abs_var, num_fin_abs_bool;
unsigned int run_dyn_size[50];

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int count_abs ARGS((node_ptr));
static int count_abs_var ARGS((node_ptr));
static int count_abs_bool ARGS((node_ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [The batch main.]

  Description [The batch main. It first read the input file, than
  flatten the hierarchy. Aftre this preliminar phase it creates the
  boolean variables necessary for the encoding and then start
  compiling the read model into BDD. Now computes the reachable states
  depending if the flag has been set. before starting verifying if the
  properties specified in the model hold or not it computes the
  fairness constraints. You can also activate the reordering and
  also choose to avoid the verification of the properties.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void smBatchMain()
{
  node_ptr init_expr         = Nil;
  node_ptr invar_expr        = Nil;
  node_ptr trans_expr        = Nil;
  node_ptr assign_expr       = Nil;
  node_ptr procs_expr        = Nil;
  node_ptr fair_expr         = Nil;
  node_ptr spec_expr         = Nil;
  node_ptr ltlspec_expr      = Nil;
  node_ptr invar_spec_expr   = Nil;
  node_ptr invar_fb_expr     = Nil;
  node_ptr invar_strong_expr = Nil;
  node_ptr after_expr        = Nil;

  /* Necessary to have standard behavior in the batch mode */
  cancel_util_setjmp();

  /* Parse the input file */
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Parsing file %s ..... ", get_input_file(options));
    fflush(nusmv_stderr);
  }

  Parser_OpenInput(get_input_file(options));

  if (yyparse()) nusmv_exit(1);

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "done.\n");
    fflush(nusmv_stderr);
  }
  
  Parser_CloseInput();
  parse_tree = reverse(parse_tree);

  /* Initializes the module hash */
  init_module_hash();
  /* check_all_implements(parse_tree); */

  /*
    Processing of the parse tree and constructions of all the
    expressions for the state machine(s). Here the expansions are
    performed so that modules and processes are created. The expansion
    of modules is such that the formal parameters (if any) are
    replaced by the actual ones and the machine is replicated.
  */
  Compile_FlattenHierarchy(sym_intern("main"), Nil, &trans_expr, &init_expr, &invar_expr,
                  &spec_expr, &ltlspec_expr, &invar_spec_expr, &invar_fb_expr,
                  &invar_strong_expr, &after_expr, &fair_expr, &abs_expr, &assign_expr,
                  &procs_expr, Nil);

  all_symbols = reverse(all_symbols);
  state_variables   = reverse(state_variables);
  input_variables = reverse(input_variables);
  /* We want input variables at the top of the ordering as default */
  all_variables = append_ns(input_variables, state_variables);
  fair_expr     = reverse(fair_expr);
  spec_expr     = reverse(spec_expr);
  ltlspec_expr  = reverse(ltlspec_expr);
  invar_spec_expr    = reverse(invar_spec_expr);
  invar_fb_expr = reverse(invar_fb_expr);
  invar_strong_expr = reverse(invar_strong_expr);
  after_expr    = reverse(after_expr);

  err_out = fopen("err_out", "w");

  /* Yuan Lu : generating variable graph*/
  Abs_AbsExtractVariableGraph(procs_expr);
  Abs_AbsBuildInterestList(spec_expr, fair_expr, get_abs_level_threshold(options));

#ifdef ABS_DEBUG
  Abs_AbsPrintVarOnLevel(spec_expr, fair_expr);
#endif

  /* Yuan Lu : automatic abstraction generation */
  /* Abs_AbsAutoGenAbsPre(procs_expr, spec_expr, fair_expr); */
  Abs_AbsAutoGenAbs(procs_expr, spec_expr, fair_expr);
  /* Initializes the build_vars routines */
  Compile_EncodeVarsInit();
  /*
    Builds the ADD containing the list of state variables
    (i.e. vars).
    Side effect on symbol_hash.
  */
  create_process_symbolic_variables(procs_expr);
  Compile_BuildVarsBdd();

  /* Yuan Lu */
  Abs_AbsEncodePost();
  if(opt_reorder(options))
    dd_autodyn_enable(dd_manager, get_reorder_method(options));

  /* Builds the ADD list representing real_state_variables */
  build_real_state_variables();

  /*
    Builds the ADD containing the list of state variables.
    Side effect on:
    - process_selector_add
    - symbol_hash (adds all <process>.running ADD leaves)
  */
  build_proc_selector(procs_expr);

  /* Builds the ADD representing frame-axioms */
  Compile_CompileFrame();

  /*
    Checking for the correctness of the state machines which have
    been declared. It should not mention DD manipulation.
  */
  Compile_CheckProgram(procs_expr, spec_expr, ltlspec_expr, invar_spec_expr, fair_expr);

  /*
    Builds the ADD representing the Initial States. This is obtained
    by conjoining the init in different modules and processes.
  */
  Compile_CompileInit(init_expr, procs_expr);

  /* Builds the ADD representing the Transition Relation */
  {
    add_ptr one = add_one(dd_manager);

    Compile_CompileModel(trans_expr, invar_expr, procs_expr, one);
    add_free(dd_manager, one);
  }

  /* Yuan Lu : after compiling, variable orderings should be written */
  if(opt_reorder(options))
    Compile_WriteOrder(get_output_order_file(options), 0);

  /* Yuan Lu : simulation for given traces
  if(get_simulate_file(options)) {
    char *filename = get_simulate_file(options);
    node_ptr trace = Abs_AbsReadTrace(filename);
    if(Abs_AbsSimulateTrace(trace)) rpterr("SOMETHING WRONG");
  }
  */

  /* Yuan Lu : applying abstraction */
  Abs_AbsAbstractInitState();

  /* Yuan Lu : Statistics for building transition relation */
  trans_size = get_dyn_node_allocated(dd_manager);
  set_dyn_node_allocated(dd_manager);

  num_refine = 0;
  num_ini_abs = count_abs(abs_expr_pre);
  num_ini_abs += count_abs(abs_expr);
  num_ini_abs_var = count_abs_var(abs_expr_pre);
  num_ini_abs_var += count_abs_var(abs_expr);
  num_ini_abs_bool = count_abs_bool(abs_expr_pre);
  num_ini_abs_bool+= count_abs_bool(abs_expr);

  while(1) {
    int sign;

    /* Builds the OBDD representing Reachable States Set */
    if (opt_forward_search(options)) {
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"Computing reachable states ...\n");
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"...done\n");
      compute_reachable_states();
    }

    /* Builds the OBDD representing the Fairness constraints */
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr,"evaluating fairness constraints...\n");
    compute_fairness_constraints(fair_expr);
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr,"...done\n");

    if (opt_check_trans(options)) {
      check_transition_relation();
    }

    /* Evaluates the Specifications */
    if (!opt_ignore_spec(options)) {
      node_ptr l = spec_expr;

      while (l) { /* Loop over specifications */
        node_ptr spec = car(l);

	/* dd_check_manager(dd_manager); */

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_spec(nusmv_stderr, spec);
          fprintf(nusmv_stderr, "\n");
        }
	/* Yuan Lu */
        if(!opt_ag_only(options)) {
          node_ptr actual_spec = cdr(spec);

          /* decide if it is a SPEC or a COMPUTE */
          switch (node_get_type(actual_spec)) {
          case MINU:
          case MAXU: check_compute(spec); break;
          default:   sign = check_spec(spec);    break;
          };
        }
        else if(opt_forward_search(options))
          check_AG_only(spec, Nil);
	else
	  sign = check_AG_step(spec, Nil);
      }
    }

    /* Evaluates the LTL specifications */
    if (!opt_ignore_ltlspec(options)) {
      node_ptr l = ltlspec_expr;

      while(l) { /* Loop over specifications */
        node_ptr spec = car(l);

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_ltlspec(nusmv_stderr, spec);
          fprintf(nusmv_stderr, "\n");
        }
        check_ltlspec(spec);
      }
    }
    /* Evaluates CHECKINVARIANTS */
    if (!opt_ignore_invar(options)) {
      node_ptr l = invar_spec_expr;

      while ( l ) {
        node_ptr invar = car(l);

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_invar(nusmv_stderr, invar);
          fprintf(nusmv_stderr, "\n");
        }
        check_invar(invar);
      }
    }
    /* Evaluates CHECKINVARIANTSFB */
    if (!opt_ignore_invar_fb(options)) {
      node_ptr l = invar_fb_expr;

      while ( l ) {
        node_ptr invar = car(l);

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_invar(nusmv_stderr, invar);
          fprintf(nusmv_stderr, "\n");
        }
        check_invar_fb(invar);
      }
    }
    /* Evaluates CHECKINVARIANTSTRONG */
    if (!opt_ignore_invar_strong(options)) {
      node_ptr l = invar_strong_expr;

      while ( l ) {
        node_ptr invar = car(l);

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_invar(nusmv_stderr, invar);
          fprintf(nusmv_stderr, "\n");
        }
        check_invar_strong(invar);
      }
    }
    /* Evaluates AFTER */
    if (!opt_ignore_after(options)) {
      int j = 0;
      node_ptr l = after_expr;

      while ( l ) {
        node_ptr after = car(l);

        l = cdr(l); j++;
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "****************************************\n");
          fprintf(nusmv_stderr, "evaluating temporal projection #%d\n", j);
          fprintf(nusmv_stderr, "****************************************\n");
        }
        check_after(after);
      }
    }
    if(sign) break;
    Abs_AbsRebuild();
    run_dyn_size[num_refine++] = get_dyn_node_allocated(dd_manager);
    set_dyn_node_allocated(dd_manager);
  }
  num_fin_abs = count_abs(abs_expr_pre);
  num_fin_abs += count_abs(abs_expr);
  num_fin_abs_var = count_abs_var(abs_expr_pre);
  num_fin_abs_var += count_abs_var(abs_expr);
  num_fin_abs_bool = count_abs_bool(abs_expr_pre);
  num_fin_abs_bool+= count_abs_bool(abs_expr);

  /* Reporting of statistical information. */
  print_usage();

  /* Computing and Reporting of the Effect of Reordering */
  if (opt_reorder(options)) {
    fprintf(nusmv_stdout, "\n========= starting reordering ============\n");
    dd_reorder(dd_manager, get_reorder_method(options), DEFAULT_MINSIZE);
    Compile_WriteOrder(get_output_order_file(options), 0);
    fprintf(nusmv_stdout, "\n========= after reordering ============\n");
    print_usage();
  }
  /* Reporting of Reachable States */
  print_reachable_states();
}

/**Function********************************************************************

  Synopsis           [Prints usage statistic.]

  Description        [Prints on <code>nusmv_stdout</code> usage
  statistics, i.e. the amount of memory used, the amount of time
  spent, the number of BDD nodes allocated and the size of the model
  in BDD.]

  SideEffects        []

  SeeAlso            [print_model_statistic]
******************************************************************************/
void print_usage(void)
{
  int i;
  fprintf(nusmv_stdout, "######################################################################\n");
  util_print_cpu_stats(nusmv_stdout);
  fprintf(nusmv_stdout, "######################################################################\n");
  fprintf(nusmv_stdout, "Initial abstraction (identity abstraction does not count)\n");
  fprintf(nusmv_stdout, "\tNum of interesting formula clusters : %d\n", num_ini_abs);
  fprintf(nusmv_stdout, "\tNum of symbolic variables involved in these formula clusters: %d\n", num_ini_abs_var);
  fprintf(nusmv_stdout, "\tNum of corresponding boolean variables : %d\n", num_ini_abs_bool);
  fprintf(nusmv_stdout, "Final abstraction after refinement\n");
  fprintf(nusmv_stdout, "\tNum of interesting formula clusters : %d\n", num_fin_abs);
  fprintf(nusmv_stdout, "\tNum of variables involved in these formula clusters: %d\n", num_fin_abs_var);
  fprintf(nusmv_stdout, "\tNum of corresponding boolean variables : %d\n", num_fin_abs_bool);
  fprintf(nusmv_stdout, "Num of Refinement Steps: %d\n", num_refine);
  for(i=0;i<num_refine;i++)
    fprintf(nusmv_stdout, "\tBDD size for %dth refinement:%d\n", i, run_dyn_size[i]);
  fprintf(nusmv_stdout, "\nBDD statistics\n");
  fprintf(nusmv_stdout, "--------------------\n");
/*  fprintf(nusmv_stdout, "BDD nodes allocated: %d\n", get_dd_nodes_allocated(dd_manager)); */
  fprintf(nusmv_stdout, "BDD nodes dynamically allocated for transition relation: %d\n", trans_size);
  fprintf(nusmv_stdout, "BDD nodes dynamically allocated for model checking: %d\n", get_dyn_node_allocated(dd_manager));
  print_model_statistic();
}

static int count_abs(node_ptr list)
{
  int cnt = 0;
  for(;list;list=cdr(list),cnt++);
  return cnt;
}

static int count_abs_var(node_ptr list)
{
  int cnt = 0;
  for(;list;list=cdr(list)) {
    node_ptr varl = car(car(car(list)));
    for(;varl;varl=cdr(varl),cnt++);
  }
  return cnt;
}

static int count_abs_bool(node_ptr list)
{
  int cnt = 0;
  for(;list;list=cdr(list)) {
    node_ptr varl = car(car(car(list)));
    for(;varl;varl=cdr(varl)) {
      int range = (int) lookup_range_hash(car(varl));
      if(range <3) cnt += 1;
      else if(range <3) cnt += 1;
      else if(range <5) cnt += 2;
      else if(range <9) cnt += 3;
      else if(range <17) cnt += 4;
      else if(range <33) cnt += 5;
      else if(range <65) cnt += 6;
      else if(range <129) cnt += 7;
      else if(range <257) cnt += 8;
      else if(range <513) cnt += 9;
      else if(range <1025) cnt += 10;
      else if(range <2049) cnt += 11;
      else if(range <4097) cnt += 12;
      else if(range <8193) cnt += 13;
      else cnt += 5000;
    }
  }
  return cnt;
}

