/**CFile***********************************************************************

  FileName    [smMain.c]

  PackageName [sm]

  Synopsis    [Main NuSMV routine. Parses command line at invocation of NuSMV.]

  Author      [Originated from SIS]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

  Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "smInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: smMain.c,v 1.2 1997/02/12 23:04:22 hsv Exp $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
FILE *nusmv_stderr;
FILE *nusmv_stdout;
FILE *nusmv_historyFile;
FILE *nusmv_stdpipe;
DdManager * dd_manager = (DdManager *)NULL;

options_ptr options = (options_ptr)NULL;
cmp_struct_ptr cmps = (cmp_struct_ptr)NULL;
/* USED TO RETURN A VALUEE FROM sm_parselineoptions */
static char * NuSMV_CMD_LINE = (char *)NULL;
/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int NusmvrcSource ARGS((void));
static void UsagePrint ARGS((char * program, char * unknown_option));
static void sm_ParseLineOptions ARGS((int argc, char ** argv, options_ptr options));

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [required]

  Description [optional]

  SideEffects [required]

  SeeAlso     [optional]

******************************************************************************/
int
main(
  int  argc,
  char ** argv)
{
  int  status, quit_flag;
  char readcmd[20], writecmd[20];
  char *cmdline;
  quit_flag = -1;     /* Quick quit */


  SmInit();
  cmdline = util_strsav("");
  (void) strcpy(readcmd, "");
  (void) strcpy(writecmd, "");

  sm_ParseLineOptions(argc, argv, options);
  if (! opt_batch(options)) { /* interactive mode */
    /* initiliazes the commands to handle with options */
    (void) init_options_cmd();
    (void) fprintf(nusmv_stdout, "%s\n", Sm_NuSMVReadVersion());
    if (!opt_ignore_init_file(options)) {
      (void) NusmvrcSource();
    }
    if (NuSMV_CMD_LINE != NULL) {
      char * command = ALLOC(char, strlen(NuSMV_CMD_LINE) + strlen("source ") + 1);

      sprintf(command,"source %s", NuSMV_CMD_LINE);
      quit_flag = Cmd_CommandExecute(command);
      FREE(command);
      FREE(NuSMV_CMD_LINE);
    }
    while ((quit_flag = Cmd_CommandExecute("source -ip -")) >= 0);
    status = 0;
  }
  else { /* batch mode */
    /* In the batch mode we dont want to read the ~/.nusmvrc file. */
    /* The system has to behave as the original NuSMV */
    /*   if (!opt_ignore_init_file(options)) { */
    /*       (void) NusmvrcSource(); */
    /*     } */
    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stdout, "Starting the batch interaction.\n");
    smBatchMain();
  }

  FREE(cmdline);
  /* Value of "quit_flag" is determined by the "quit" command */
  if (quit_flag == -1 || quit_flag == -2) {
    status = 0;
  }
  if (quit_flag == -2) {
    /*    Hrc_ManagerFree(globalHmgr); */
    SmEnd();
  } else {
    SmEnd();
  }
  
  exit(status);
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Sources the .nusmvrc file.]

  Description [Sources the .nusmvrc file.  Always sources the .nusmvrc from
  library.  Then source the .nusmvrc from the home directory.  If there is none
  in the home directory, then execute the one in the current directory if one
  is present.  Returns 1 if scripts were successfully executed, else return 0.]

  SideEffects []

  SeeAlso     [optional]

******************************************************************************/
static int
NusmvrcSource()
{
  char *commandLine;
  char *libraryName;
  char *homeFile;
  struct stat home;
  struct stat cur;
  int s1;
  int s2;			/* flags for checking the stat() call */
  int status0;
  int status1 = TRUE;
  int status2 = TRUE;

  /*
   * First execute the standard .nusmvrc.
   */
  libraryName = Sm_NuSMVObtainLibrary();
  commandLine = ALLOC(char, strlen(libraryName) + 30);
  (void) sprintf(commandLine, "source -s %s/master.nusmvrc", libraryName);
  FREE(libraryName);
  status0 = Cmd_CommandExecute(commandLine);
  FREE(commandLine);

  /*
   * Look in home directory and current directory for .nusmvrc.
   */
  homeFile = util_tilde_expand("~/.nusmvrc");
  s1 = stat(homeFile, &home);
  FREE(homeFile);
  s2 = stat(".nusmvrc", &cur);

  /*
   * If .nusmvrc is present in both the home and current directories, then read
   * it from the home directory.  Otherwise, read it from wherever it's
   * located.
   */
  if ((s1 == 0) && (s2 == 0) && (home.st_ino == cur.st_ino)){
    /* ~/.nusmvrc == .nusmvrc : Source the file only once */
    status1 = Cmd_CommandExecute("source -s ~/.nusmvrc");
  }
  else {
    if (s1 == 0) {
      status1 = Cmd_CommandExecute("source -s ~/.nusmvrc");
    }
    if (s2 == 0) {
      status2 = Cmd_CommandExecute("source -s .nusmvrc");
    }
  }

  return (status0 && status1 && status2);
}


/**Function********************************************************************

  Synopsis    [Prints the usage of the NuSMV shell interface.]

  SideEffects []

  SeeAlso     [optional]

******************************************************************************/
static void
UsagePrint(
  char * program, char * unknown_option)
{
  char *libraryName;
  
  (void) fprintf(nusmv_stderr, "%s\n\n", Sm_NuSMVReadVersion());
  if (unknown_option != NULL) {
    (void) fprintf(nusmv_stderr,
                   "The command line option \"%s\" is unknown.\n", unknown_option);
  }
  (void) fprintf(nusmv_stderr,
                 "Usage:\t%s [-s] [-ctt] [-v vl] [-is] [-ils] [-ii] [-iifb] [-iis] [-ia] [-r] [-f] \\\n", program);
  (void) fprintf(nusmv_stderr, "\t[-int] [-h|-help] [-i iv_file] [-o ov_file] [-AG] [-GUI] [-load script_file]\\\n");
  (void) fprintf(nusmv_stderr, "\t[-abs level] [-reorder] [-reordersize rs] [[-cp cp_t]|[-dp]] [input-file]\n");
  (void) fprintf(nusmv_stderr, "Where:\n");
  (void) fprintf(nusmv_stderr, "\t -s\t\t do not read any initialization file\n");
  libraryName = Sm_NuSMVObtainLibrary();
  (void) fprintf(nusmv_stderr, "\t   \t\t (%s/master.nusmvrc, ~/.nusmvrc) default in batch mode.\n", libraryName);
  FREE(libraryName);
  (void) fprintf(nusmv_stderr, "\t -ctt\t\t enables checking for the totality of the transition relation\n");
  (void) fprintf(nusmv_stderr, "\t -v vl\t\t set verbose level to \"vl\"\n");
#ifdef OPT_QUANTIFY_INPUT
  (void) fprintf(nusmv_stderr, "\t -qi\t\t quantify out input variables from Transition Relation.\n");
#endif
  (void) fprintf(nusmv_stderr, "\t -is\t\t do not check SPEC and COMPUTE\n");
  (void) fprintf(nusmv_stderr, "\t -ils\t\t do not check LTLSPEC\n");
  (void) fprintf(nusmv_stderr, "\t -ii\t\t do not check INVARSPEC\n");
  (void) fprintf(nusmv_stderr, "\t -iifb\t\t do not check CHECKINVARIANTFB\n");
  (void) fprintf(nusmv_stderr, "\t -iis\t\t do not check CHECKINVARIANTSTRONG\n");
  (void) fprintf(nusmv_stderr, "\t -ia\t\t do not check AFTER\n");
  (void) fprintf(nusmv_stderr, "\t -r\t\t enables printing of reachable states\n");
  (void) fprintf(nusmv_stderr, "\t -f\t\t enables forward search\n");
  (void) fprintf(nusmv_stderr, "\t -int\t\t enables interactive mode\n");
  (void) fprintf(nusmv_stderr, "\t -h|-help\t prints out current message\n");
  (void) fprintf(nusmv_stderr, "\t -i iv_file\t reads order of variables from file \"iv_file\"\n");
  (void) fprintf(nusmv_stderr, "\t -o ov_file\t prints order of variables to file \"ov_file\"\n");
  (void) fprintf(nusmv_stderr, "\t -AG\t\t enables AG only search\n");
  (void) fprintf(nusmv_stderr, "\t -GUI\t\t enables GUI behavior\n");
  (void) fprintf(nusmv_stderr, "\t -load sc_file\t loads NuSMV commands from file \"sc_file\"\n");
  (void) fprintf(nusmv_stderr, "\t -abs level \t enable abstraction model and specify how \n\t\t\t many levels of variables are considered in \n\t\t\t the variable dependency graph\n");
  (void) fprintf(nusmv_stderr, "\t -reorder\t enables dynamic reordering of variables\n");
  (void) fprintf(nusmv_stderr, "\t -m method\t sets the variable ordering method to \"method\"\n");
  (void) fprintf(nusmv_stderr, "\t\t\t reordering will be activated\n");
  (void) fprintf(nusmv_stderr, "\t -cp cp_t\t enables conjunctive partitioning and set\n");
  (void) fprintf(nusmv_stderr, "\t\t\t the threshold of each partition to \"cp_t\"\n");
  (void) fprintf(nusmv_stderr, "\t -dp\t\t enables disjunctive partitioning\n");
  (void) fprintf(nusmv_stderr, "\t -iwls95 cp_t\t enables Iwls95 conjunctive partitioning and set\n");
  (void) fprintf(nusmv_stderr, "\t\t\t the threshold of each partition to \"cp_t\"\n");
  (void) fprintf(nusmv_stderr, "\t input-file\t\t the file from which both the model and \n");
  (void) fprintf(nusmv_stderr, "\t\t\t the spec were read\n");
  
  exit(2);
}

/**Function********************************************************************

  Synopsis    [Parses the command line options.]

  Description []

  SideEffects []

******************************************************************************/
static void sm_ParseLineOptions(int argc, char ** argv, options_ptr options)
{
  /* parses the Program Name */
  argc--;
  set_pgm_name(options, *(argv++));

  while (argc > 0) {
    if (strcmp(*argv,"-load") == 0) {
      argv++; argc--;
      if (argc == 0) {
        fprintf(nusmv_stderr, "The \"-load\" command line options requires an argument.\n");
        exit(1);
      }
      NuSMV_CMD_LINE = ALLOC(char, strlen(*argv)+1);
      strcpy(NuSMV_CMD_LINE, *argv);
      argv++; argc--;
      continue;
    }
    if (strcmp(*argv,"-is") == 0){
      argv++; argc--;
      set_ignore_spec(options);
      continue;
    }
    if (strcmp(*argv,"-ils") == 0){
      argv++; argc--;
      set_ignore_ltlspec(options);
      continue;
    }
    else if (strcmp(*argv,"-iis") == 0){
      argv++; argc--;
      set_ignore_invar_strong(options);
      continue;
    }
    else if (strcmp(*argv,"-iifb") == 0){
      argv++; argc--;
      set_ignore_invar_fb(options);
      continue;
    }
    else if (strcmp(*argv,"-ii") == 0){
      argv++; argc--;
      set_ignore_invar(options);
      continue;
    }
#ifdef OPT_QUANTIFY_INPUT
    else if (strcmp(*argv,"-qi") == 0){
      argv++; argc--;
      set_quantify_input(options);
      continue;
    }
#endif
    else if (strcmp(*argv,"-ia") == 0){
      argv++; argc--;
      set_ignore_after(options);
      continue;
    }
    else if (strcmp(*argv,"-r") == 0){
      argv++; argc--;
      set_print_reachable(options);
      continue;
    }
    else if (strcmp(*argv,"-f") == 0){
      argv++; argc--;
      set_forward_search(options);
      continue;
    }
    else if (strcmp(*argv, "-help") == 0) {
      argv++; argc--;
      UsagePrint(get_pgm_name(options), NULL);
    }
    else if (strcmp(*argv, "-h") == 0) {
      argv++; argc--;
      UsagePrint(get_pgm_name(options), NULL);
    }
    else if (strcmp(*argv,"-int") == 0){
      argv++; argc--;
      unset_batch(options);
#if HAVE_SETVBUF
      setvbuf(nusmv_stdout, (char *)NULL, _IOLBF, 0);
#endif      
      continue;
    }
    else if (strcmp(*argv,"-AG") == 0){
      argv++; argc--;
      set_ag_only(options);
      continue;
    }
    else if (strcmp(*argv,"-GUI") == 0){
      argv++; argc --;
      set_gui_behavior(options);
      continue;
    }
    else if (strcmp(*argv, "-reorder") == 0) {
      argv++; argc--;
      set_reorder(options);
      continue;
    }
    else if (strcmp(*argv, "-ctt") == 0) {
      argv++; argc--;
      set_check_trans(options);
      continue;
    }
    else if (strcmp(*argv, "-m") == 0){
      if (argc < 2) {
        (void) fprintf(nusmv_stderr, "The \"-m\" command line option requires an argument.\n");
        exit(1);
      }
      argv++; argc -= 2;
      {
        unsigned int reorder_method = StringConvertToDynOrderType(*(argv++));
        if ( reorder_method == REORDER_NONE) {
          (void) fprintf(nusmv_stderr, "The method \"%s\" is not a valid reorder method.\n",
                         *argv);
          exit(1);
        }
        set_reorder_method(options, reorder_method);
      }
      continue;
    }
    else if (strcmp(*argv,"-v") == 0){
      if (argc < 2) {
        (void) fprintf(nusmv_stderr, "The \"-v\" command line option requires an argument.\n");
        exit(1);
      }
      {
        char *err_occ[1];
        int cur_verbose;
        
        err_occ[0] = "";
        argv++; argc -= 2;
        cur_verbose = strtol(*(argv++),err_occ, 10);
        if (strcmp(err_occ[0], "") != 0) {
          (void) fprintf(nusmv_stderr, "Error: \"%s\" is not a valid value for the \"-v\" command line option.\n", err_occ[0]);
          exit(1);
        }
        set_verbose_level(options, cur_verbose);
      }
#if HAVE_SETVBUF
      setvbuf(nusmv_stdout, (char *)NULL, _IOLBF, 0);
#endif
      continue;
    }
    else if (strcmp(*argv, "-i") == 0){
      argv++; argc -= 2;
      set_input_order_file(options, *(argv++));
      continue;
    }
    else if (strcmp(*argv, "-o") == 0){
      argv++;  argc -= 2;
      set_output_order_file(options, *(argv++));
      continue;
    }
    else if(strcmp(*argv, "-cp") == 0){
      if (argc < 2) {
        (void) fprintf(nusmv_stderr, "The \"-cp\" command line option requires an argument.\n");
        exit(1);
      } 
      {
        char *err_occ[1];
        int cur_cpl;
        
        err_occ[0] = "";
        argv++; argc -= 2;
        cur_cpl = strtol(*(argv++), err_occ, 10);
        if (strcmp(err_occ[0], "") != 0) {
          fprintf(stderr, "Error: \"%s\" is not a valid value for the \"-cp\" line option.\n", err_occ[0]);
          exit(1);
        }
        set_conj_partitioning(options);
        set_conj_part_threshold(options, cur_cpl);
      }
      continue;
    }
    /* Yuan Lu */
    else if(strcmp(*argv, "-abs") == 0) {
      if(argc < 2) {
	(void) fprintf(nusmv_stderr, "The \"-al\" command line option requires an argument.\n");
	exit(1);
      }
      {
	char *err_occ[1];
	int cur_al;
	err_occ[0] = "";
	argv++; argc -= 2;
	cur_al = strtol(*(argv++), err_occ, 10);
	if(strcmp(err_occ[0], "") != 0) {
	  fprintf(stderr, "Error: \"%s\" is not a valid value for the \"-al\" line option.\n", err_occ[0]);
	  exit(1);
	}
	set_abs_level_threshold(options, cur_al);
      }
      continue;
    }
    /* Yuan Lu */
    else if (strcmp(*argv, "-sim") == 0){
      argv++;  argc -= 2;
      set_simulate_file(options, *(argv++));
      continue;
    }
    /* Yuan Lu */
    else if (strcmp(*argv, "-coi") == 0) {
      argv++; argc -= 1;
      set_coi(options);
    }
    else if(strcmp(*argv,"-dp") == 0){
      argv++;
      argc--;
      set_disj_partitioning(options);
      continue;
    }
    else if(strcmp(*argv, "-iwls95") == 0){
      if (argc < 2) {
        (void) fprintf(nusmv_stderr, "The \"-iwls95\" command line option requires an argument.\n");
        exit(1);
      } 
      {
        char *err_occ[1];
        int cur_cpl;
        
        err_occ[0] = "";
        argv++; argc -= 2;
        cur_cpl = strtol(*(argv++), err_occ, 10);
        if (strcmp(err_occ[0], "") != 0) {
          fprintf(stderr, "Error: \"%s\" is not a valid value for the \"-iwls95\" line option.\n", err_occ[0]);
          exit(1);
        }
        set_iwls95cp_partitioning(options);
        set_image_cluster_size(options, cur_cpl);
      }
      continue;
    }
    else if(strcmp(*argv,"-s") == 0){
      argv++;
      argc--;
      set_ignore_init_file(options);
      continue;
    }
    else if (argc == 1 && (**argv) != '-'){
      set_input_file(options, *(argv++));
      argc--;
      continue;
    }
    else  {
      UsagePrint(get_pgm_name(options), *argv);
    }
  }
}


