%{
/**CFile***********************************************************************

  FileName    [grammar.y]

  PackageName [parser]

  Synopsis    [Yacc for NuSMV input language parser parser]

  SeeAlso     [input.lex]

  Author      [Marco Roveri]

  Copyright   [Copyright (c) 1998 by ITC-IRST and Carnegie Mellon University.
  All Rights Reserved.  This software is for educational purposes only.
  Permission is given to use, copy, modify, and distribute this software
  and its documentation provided that this introductory message is not
  removed and no monies are exchanged. No guarantee is expressed or
  implied by the distribution of this code. 
  Send bug-reports and/or questions to: nusmv@irst.itc.it]

******************************************************************************/

#include <setjmp.h>
#include <malloc.h>
#include "util.h"
#include "utils.h"
#include "node.h"
#include "opt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

node_ptr parse_tree;
node_ptr parse_tree_int;

void yyerror(char *s);

%}
/*
  The number of conflicts: shift/reduce expected.
  If more, than a warning message is printed out.
*/
%expect 1

%union {
  node_ptr node;
}

/*
  All of the terminal grammar symbols (tokens recognized by the lexical analyzer) 
  Note: all binary operators associate from left to right,  operators are
        listed from lowest to highest priority 

  Note: The following token are not used inside the grammar, but are
  used by other modules inside the system (i.e. the compiler, mc).
  STEP RESET ASYNC MODTYPE LAMBDA CONTEXT EU AU EBU ABU MINU MAXU
  FORMAT CONSTANT SCALAR CONS OVER BDD ATLINE APROPOS IFTHENELSE
  QUOTE DL_ATOM APATH EPATH
*/

%left GOTO LET STEP EVAL RESET
%left ASYNC MODULE PROCESS MODTYPE LAMBDA CONTEXT EU AU EBU ABU MINU MAXU
%left VAR IVAR DEFINE INIT TRANS INVAR FORMAT SPEC LTLSPEC COMPUTE 
%left ABSTRACT GROUP
%left INVARSPEC CHECKINVARIANTFB CHECKINVARIANTSTRONG AFTER 
%left FAIRNESS ISA CONSTANT ASSIGN INPUT OUTPUT IMPLEMENTS
%left BOOLEAN ARRAY OF SCALAR CONS OVER BDD
%left SEMI LP RP LB RB LCB RCB LLCB RRCB
%left EQDEF TWODOTS ATLINE
%left <node> FALSEEXP TRUEEXP
%left APROPOS SELF SIGMA
%left CASE ESAC COLON
%left IF THEN ELSE IFTHENELSE
%left STAR

%left <node> ATOM
%left <node> NUMBER
%left <node> QUOTE
%left <node> DL_ATOM

%left  COMMA
%right IMPLIES
%left  IFF
%left  OR
%left  AND
%left  NOT
%left  EX AX EF AF EG AG EE AA UNTIL EBF EBG ABF ABG BUNTIL MMIN MMAX
%left  OP_NEXT OP_GLOBAL OP_FUTURE
%left  APATH EPATH
%left  EQUAL LT GT LE GE
%left  UNION
%left  SETIN
%left  MOD
%left  PLUS MINUS
%left  TIMES DIVIDE
%left  UMINUS		/* supplies precedence for unary minus */
%left  NEXT SMALLINIT
%left  DOT
%left  UU


/* all nonterminals return a parse tree node */
%type <node> module_list module declarations declaration var input_var var_list
%type <node> type isa init trans invar define define_list spec ltlspec compute fairness
%type <node> abst abstform abstpart grps
%type <node> check_invar check_invar_fb check_invar_strong
%type <node> after_check dl_act_expr_list cond_expr dl_expr dl_atomic_expr goal_expr
%type <node> atom_list decl_var_id var_id subrange invar_expr_list
%type <node> number simple_expr assign_expr init_expr next_expr fair_expr invar_expr 
%type <node> trans_expr ctl_expr compute_expr s_case_list n_case_list constant
%type <node> ltl_expr ltl_orexpr ltl_andexpr ltl_untilexpr ltl_atomexpr ltl_simple_expr
%type <node> module_sign module_type simple_expr_list assign assign_list assign_type atom_set
%type <node> input output implements term_list constant_list actual_params
%type <node> command command_case trace state
/* %type <node> trace state */

%start begin
%%
begin         : module_list {parse_tree = $1;}
              | command {parse_tree_int = $1;}
              ;

/*
 An NuSMV program is a repetition of modules.
 Each module has a signature and a body.
*/
module_list   : module {$$ = cons($1, Nil);}
              | module_list module {$$ = cons($2, $1);}
              ;
module        : MODULE module_sign declarations {$$ = new_node(MODULE, $2, $3);}
              ;
module_sign   : ATOM {$$ = new_node(MODTYPE, $1, Nil);}
              | ATOM LP atom_list RP {$$ = new_node(MODTYPE, $1, $3);}
              ;
atom_list     : ATOM {$$ = cons(find_atom($1), Nil);}
              | atom_list COMMA ATOM {$$ = cons(find_atom($3), $1);}
              ;


/* The body of a module */
declarations  : {$$ = Nil;}
              | declarations declaration {$$ = cons($2, $1);}
              ;
declaration   : isa
              | var
              | input_var
              | assign 
              | init
              | invar
              | trans
              | define
              | fairness
              | check_invar
              | check_invar_fb
              | check_invar_strong
              | after_check
              | spec
              | ltlspec
              | abst
              | grps
              | compute
              | implements
              | input
              | output
              ;

/* Module macro-expansion */
isa           : ISA ATOM {$$ = new_node(ISA, $2, Nil);}
              ;


/*
 Variable declarations:
 This includes also the instantiation of module
 (in synchronous and asynchronous product).
*/
var           : VAR var_list {$$ = new_node(VAR, $2, Nil);}
              ;
input_var     : IVAR var_list {$$ = new_node(IVAR, $2, Nil);}
              ;
var_list      : {$$ = Nil;}
              | var_list decl_var_id COLON type SEMI {$$ = cons(new_node(COLON, $2, $4), $1);}
              ;
type          : BOOLEAN {$$ = new_node(BOOLEAN, Nil, Nil);}
              | subrange
              | LCB constant_list RCB {$$ = new_node(SCALAR, $2, Nil);}
              | ARRAY subrange OF type {$$ = new_node(ARRAY, $2, $4);}
              | module_type
              | PROCESS module_type {$$ = new_node(PROCESS, $2, Nil);}
              ;
constant_list : constant {$$ = cons(find_atom($1), Nil);}
              | constant_list COMMA constant {$$ = cons(find_atom($3), $1);}
              ;
constant      : ATOM
              | number
	      | FALSEEXP
	      | TRUEEXP
              ;
subrange      : number TWODOTS number {$$ = new_node(TWODOTS, $1, $3);}
              ;
number        : NUMBER
	      | PLUS NUMBER %prec UMINUS { $$ = $2; }
	      | MINUS NUMBER %prec UMINUS {$2->left.inttype = -($2->left.inttype); $$ = $2;}
              ;
module_type   : ATOM {$$ = new_node(MODTYPE, $1, Nil);}
              | ATOM LP actual_params RP {$$ = new_node(MODTYPE, $1, $3);}
              ;
actual_params : simple_expr_list


/* Assignments of initial, current or next value of variables */
assign        : ASSIGN assign_list {$$ = new_node(ASSIGN, $2, Nil);}
              ;
assign_list   : {$$ = Nil;}
              | assign_list assign_type {$$ = new_node(AND, $1, $2);}
              ;
assign_type   : var_id EQDEF simple_expr SEMI
                 {$$ = new_node(EQDEF, $1, $3);} 
              | SMALLINIT LP var_id RP EQDEF simple_expr SEMI
                { $$ = new_node(EQDEF, new_node(SMALLINIT, $3, Nil), $6);}
              | NEXT LP var_id RP EQDEF assign_expr SEMI
                { $$ = new_node(EQDEF, new_node(NEXT, $3, Nil), $6);}
              ;
assign_expr   : next_expr
              ;

/* Direct finite state machine definition (init, invar, trans) */
init          : INIT init_expr {$$ = new_node(INIT, $2, Nil);}
              ;
init_expr     : simple_expr
              ;
invar         : INVAR invar_expr {$$ = new_node(INVAR, $2, Nil);}
              ;
invar_expr    : simple_expr
              ;
trans         : TRANS trans_expr {$$ = new_node(TRANS, $2, Nil);}
              ;
trans_expr    : next_expr
              ;

/* Definitions */
define        : DEFINE define_list {$$ = new_node(DEFINE, $2, Nil);}
              ;
define_list   : {$$ = Nil;}
              | define_list decl_var_id EQDEF assign_expr SEMI {$$ = cons(new_node(EQDEF, $2, $4), $1);}
              ;


/* Specifications and computation of min and max distance */
fairness      : FAIRNESS fair_expr {$$ = new_node(FAIRNESS, $2, Nil);}
              ;
fair_expr     : ctl_expr
              ;

spec          : SPEC ctl_expr {$$ = new_node(SPEC, $2, Nil);}
              ;

ltlspec       : LTLSPEC ltl_expr {$$ = new_node(LTLSPEC, $2, Nil);}
              ;

check_invar        : INVARSPEC invar_expr_list SEMI
                      {$$ = new_node(INVARSPEC, $2, Nil);};
check_invar_fb     : CHECKINVARIANTFB invar_expr_list SEMI
                      {$$ = new_node(CHECKINVARIANTFB, $2, Nil);};
check_invar_strong : CHECKINVARIANTSTRONG invar_expr_list SEMI
                      {$$ = new_node(CHECKINVARIANTSTRONG, $2, Nil);};
invar_expr_list : invar_expr {$$ = cons($1, Nil);}
                | invar_expr_list COMMA invar_expr {$$ = cons($3, $1);}
                ;
 
after_check     : AFTER dl_expr COLON goal_expr SEMI {$$ = new_node(AFTER, $2, $4);}
                ;
dl_expr         : dl_atomic_expr {$$ = new_node(DL_ATOM, $1, Nil);}
                | LP dl_expr RP {$$ = $2;}
                | dl_expr SEMI dl_expr { $$ = new_node(SEMI, $1, $3);}
                | dl_expr UU dl_expr { $$ = new_node(UU, $1, $3);}
                | IF cond_expr THEN dl_expr ELSE dl_expr
                   { $$ = new_node(IFTHENELSE, $2, cons($4, $6));}
                | LP dl_expr RP STAR { $$ = new_node(STAR, $2, Nil);}
                ;
dl_atomic_expr  : LLCB dl_act_expr_list RRCB { $$ = $2;}
                ;
dl_act_expr_list : simple_expr {$$ = cons($1, Nil);}
	         | dl_act_expr_list COMMA simple_expr {$$ = cons($3, $1);}
                 ;
cond_expr       : simple_expr
                ;
goal_expr       : simple_expr
                ;

abst          : ABSTRACT abstform {$$ = new_node(ABSTRACT, $2, Nil);}
              ;
abstform      : LB term_list RB IMPLIES ATOM number
                {$$ = new_node(IMPLIES, $2, new_node(EQDEF,
                               find_atom($5), find_atom($6)));}
              | var_id IMPLIES LCB constant_list RCB
                {$$ = new_node(IMPLIES, find_atom($1), $4);}
              | var_id IMPLIES abstpart {$$ = new_node(IMPLIES, find_atom($1), $3);}
              ;
abstpart      : number TWODOTS number {$$ = new_node(TWODOTS, find_atom($1), find_atom($3));}
              | number TWODOTS abstpart {$$ = new_node(TWODOTS, find_atom($1), $3);}
              ;
grps          : GROUP LB term_list RB {$$ = new_node(GROUP, $3, Nil);}
              ;

compute       : COMPUTE compute_expr {$$ = new_node(COMPUTE, $2, Nil);}
              ;
compute_expr  : MMIN LB ctl_expr COMMA ctl_expr RB { $$ = new_node(MINU, $3, $5); }
              | MMAX LB ctl_expr COMMA ctl_expr RB { $$ = new_node(MAXU, $3, $5); }
              ;

/* Variable identifiers.
   decl_var_id is used for declarations, array-like syntax not allowed.
   var_id is used to reference variables, includes array-like syntax.
 */

decl_var_id   : ATOM
              | SELF {$$ = new_node(SELF, Nil, Nil);}
              | decl_var_id DOT ATOM {$$ = new_node(DOT, $1, $3);}
              ;
var_id        : ATOM
              | SELF {$$ = new_node(SELF,Nil,Nil);}
              | var_id DOT ATOM {$$ = new_node(DOT, $1, $3);}
              | var_id LB simple_expr RB {$$ = new_node(ARRAY, $1, $3);}
              ;


/* Simple expressions. Do not involve next state variables or CLT. */

simple_expr_list : simple_expr {$$ = cons($1,Nil);}
	         | simple_expr_list COMMA simple_expr {$$ = cons($3, $1);}
                 ;
simple_expr   : var_id
              | number
              | subrange
              | FALSEEXP
              | TRUEEXP
              | LP simple_expr RP { $$ = $2; }
	      | simple_expr IMPLIES simple_expr { $$ = new_node(IMPLIES, $1, $3); }
	      | simple_expr IFF simple_expr { $$ = new_node(IFF, $1, $3); }
	      | simple_expr OR simple_expr { $$ = new_node(OR,$1, $3); }
	      | simple_expr AND simple_expr { $$ = new_node(AND, $1, $3); }
	      | NOT simple_expr { $$ = new_node(NOT, $2, Nil); }
              | simple_expr PLUS simple_expr { $$ = new_node(PLUS, $1, $3); }
              | simple_expr MINUS simple_expr { $$ = new_node(MINUS, $1, $3); }
              | simple_expr TIMES simple_expr { $$ = new_node(TIMES, $1, $3); }
              | simple_expr DIVIDE simple_expr { $$ = new_node(DIVIDE, $1, $3); }
              | simple_expr MOD simple_expr { $$ = new_node(MOD, $1, $3); }
              | simple_expr EQUAL simple_expr { $$ = new_node(EQUAL, $1, $3); }
              | simple_expr LT simple_expr { $$ = new_node(LT, $1, $3); }
              | simple_expr GT simple_expr { $$ = new_node(GT, $1, $3); }
              | simple_expr LE simple_expr { $$ = new_node(LE, $1, $3); }
              | simple_expr GE simple_expr { $$ = new_node(GE, $1, $3); }
              | LCB atom_set RCB { $$ = $2; } 
              | simple_expr UNION simple_expr { $$ = new_node(UNION, $1, $3); }
              | simple_expr SETIN simple_expr { $$ = new_node(SETIN, $1, $3); }
              | CASE s_case_list ESAC { $$ = $2; }
              | SIGMA LB ATOM EQUAL subrange RB simple_expr
                { $$ = new_node(SIGMA, new_node(EQUAL, $3, $5), $7); }
	      ;
atom_set      : constant
              | atom_set COMMA constant {$$ = new_node(UNION, $1, $3);}
              ; 

s_case_list   : {$$=new_node(TRUEEXP, Nil, Nil);}
              | simple_expr COLON simple_expr SEMI s_case_list
                {
	          $$ = new_node(CASE, new_node(COLON, $1, $3), $5);
	        }
              ;



/* Next expressions. Contain next stae variables, but no CTL. */
next_expr     : var_id
              | number
              | subrange
              | FALSEEXP
              | TRUEEXP
/*              | NEXT LP var_id RP { $$ = new_node(NEXT, $3, Nil); } */
/* We dont want expression of this kind: next(alpha & next(beta))   */
              | NEXT LP simple_expr RP { $$ = new_node(NEXT, $3, Nil); } 
              | LP next_expr RP { $$ = $2; }
	      | next_expr IMPLIES next_expr { $$ = new_node(IMPLIES, $1, $3); }
	      | next_expr IFF next_expr { $$ = new_node(IFF, $1, $3); }
	      | next_expr OR next_expr { $$ = new_node(OR, $1, $3); }
	      | next_expr AND next_expr { $$ = new_node(AND, $1, $3); }
	      | NOT next_expr { $$ = new_node(NOT, $2, Nil); }
              | next_expr PLUS next_expr { $$ = new_node(PLUS, $1, $3); }
              | next_expr MINUS next_expr { $$ = new_node(MINUS, $1, $3); }
              | next_expr TIMES next_expr { $$ = new_node(TIMES, $1, $3); }
              | next_expr DIVIDE next_expr { $$ = new_node(DIVIDE, $1, $3); }
              | next_expr MOD next_expr { $$ = new_node(MOD, $1, $3); }
              | next_expr EQUAL next_expr { $$ = new_node(EQUAL, $1, $3); }
              | next_expr LT next_expr { $$ = new_node(LT, $1, $3); }
              | next_expr GT next_expr { $$ = new_node(GT, $1, $3); }
              | next_expr LE next_expr { $$ = new_node(LE, $1, $3); }
              | next_expr GE next_expr { $$ = new_node(GE, $1, $3); }
              | LCB atom_set RCB { $$ = $2; } 
              | next_expr UNION next_expr { $$ = new_node(UNION, $1, $3); }
              | next_expr SETIN next_expr { $$ = new_node(SETIN, $1, $3); }
              | CASE n_case_list ESAC { $$ = $2; }
              | SIGMA LB ATOM EQUAL subrange RB next_expr
                { $$ = new_node(SIGMA, new_node(EQUAL, $3, $5), $7); }
              ;

n_case_list   : {$$=new_node(TRUEEXP, Nil, Nil);}
              | next_expr COLON next_expr SEMI n_case_list
                {
	          $$ = new_node(CASE, new_node(COLON, $1, $3), $5);
	        }
              ;



/* (Bounded) CTL formulas. Contain CTL operators, but no next variables. */

ctl_expr      : var_id
              | number
              | subrange
              | FALSEEXP
              | TRUEEXP
              | LP ctl_expr RP { $$ = $2; }
	      | ctl_expr IMPLIES ctl_expr { $$ = new_node(IMPLIES, $1, $3); }
	      | ctl_expr IFF ctl_expr { $$ = new_node(IFF, $1, $3); }
	      | ctl_expr OR ctl_expr { $$ = new_node(OR, $1, $3); }
	      | ctl_expr AND ctl_expr { $$ = new_node(AND, $1, $3); }
	      | NOT ctl_expr { $$ = new_node(NOT, $2, Nil); }
              | ctl_expr PLUS ctl_expr { $$ = new_node(PLUS, $1, $3); }
              | ctl_expr MINUS ctl_expr { $$ = new_node(MINUS, $1, $3); }
              | ctl_expr TIMES ctl_expr { $$ = new_node(TIMES, $1, $3); }
              | ctl_expr DIVIDE ctl_expr { $$ = new_node(DIVIDE, $1, $3); }
              | ctl_expr MOD ctl_expr { $$ = new_node(MOD, $1, $3); }
              | ctl_expr EQUAL ctl_expr { $$ = new_node(EQUAL, $1, $3); }
              | ctl_expr LT ctl_expr { $$ = new_node(LT, $1, $3); }
              | ctl_expr GT ctl_expr { $$ = new_node(GT, $1, $3); }
              | ctl_expr LE ctl_expr { $$ = new_node(LE, $1, $3); }
              | ctl_expr GE ctl_expr { $$ = new_node(GE, $1, $3); }
              | LCB atom_set RCB { $$ = $2; } 
              | ctl_expr UNION ctl_expr { $$ = new_node(UNION, $1, $3); }
              | ctl_expr SETIN ctl_expr { $$ = new_node(SETIN, $1, $3); }
/*            | CASE ctl_case_list ESAC { $$ = $2; } */
/*            | SIGMA LB ATOM EQUAL subrange RB ctl_expr */
/*                { $$ = new_node(SIGMA, new_node(EQUAL, $3, $5), $7); } */
              | EX ctl_expr { $$ = new_node(EX, $2, Nil); }
              | AX ctl_expr { $$ = new_node(AX, $2, Nil); }
              | EF ctl_expr { $$ = new_node(EF, $2, Nil); }
              | AF ctl_expr { $$ = new_node(AF, $2, Nil); }
              | EG ctl_expr { $$ = new_node(EG, $2, Nil); }
              | AG ctl_expr { $$ = new_node(AG, $2, Nil); }
	      | AA LB ctl_expr UNTIL ctl_expr RB { $$ = new_node(AU, $3, $5); }
	      | EE LB ctl_expr UNTIL ctl_expr RB { $$ = new_node(EU, $3, $5); }
	      | AA LB ctl_expr BUNTIL subrange ctl_expr RB
                       { $$ = new_node(ABU, new_node(AU, $3, $6), $5); }
	      | EE LB ctl_expr BUNTIL subrange ctl_expr RB
                       { $$ = new_node(EBU, new_node(EU, $3, $6), $5); }
              | EBF subrange ctl_expr { $$ = new_node(EBF, $3, $2); }
              | ABF subrange ctl_expr { $$ = new_node(ABF, $3, $2); }
              | EBG subrange ctl_expr { $$ = new_node(EBG, $3, $2); }
              | ABG subrange ctl_expr { $$ = new_node(ABG, $3, $2); }
	      ;

/* LTL grammar */
ltl_expr      : ltl_orexpr {$$ = $1;}
              | ltl_expr IMPLIES ltl_orexpr {$$ = new_node(IMPLIES, $1, $3);}
              | ltl_orexpr IFF ltl_orexpr {$$ = new_node(IFF, $1, $3);}
              ;
ltl_orexpr    : ltl_andexpr {$$ = $1;}
              | ltl_orexpr OR ltl_andexpr {$$ = new_node(OR, $1, $3);}
              ;
ltl_andexpr   : ltl_untilexpr {$$ = $1;}
              | ltl_andexpr AND ltl_untilexpr {$$ = new_node(AND, $1, $3);}
              ;
ltl_untilexpr : ltl_atomexpr {$$ = $1;}
              | ltl_untilexpr UNTIL ltl_atomexpr {$$ = new_node(UNTIL, $1, $3);}
              ;
ltl_atomexpr  : NOT ltl_atomexpr {$$ = new_node(NOT, $2, Nil);}
              | OP_NEXT ltl_atomexpr {$$ = new_node(OP_NEXT, $2, Nil);} 
              | OP_GLOBAL ltl_atomexpr {$$ = new_node(OP_GLOBAL, $2, Nil);}
              | OP_FUTURE ltl_atomexpr {$$ = new_node(OP_FUTURE, $2, Nil);}
              | LP ltl_expr RP {$$ = $2;}
              | ltl_simple_expr
              ;
ltl_simple_expr : var_id
                | number
                | subrange
                | FALSEEXP
                | TRUEEXP
                | LP ltl_simple_expr RP {$$ = $2;}
                | ltl_simple_expr PLUS ltl_simple_expr {$$ = new_node(PLUS, $1, $3);}
                | ltl_simple_expr MINUS ltl_simple_expr {$$ = new_node(MINUS, $1, $3);}
                | ltl_simple_expr TIMES ltl_simple_expr {$$ = new_node(TIMES, $1, $3);}
                | ltl_simple_expr DIVIDE ltl_simple_expr {$$ = new_node(DIVIDE, $1, $3);}
                | ltl_simple_expr MOD ltl_simple_expr {$$ = new_node(MOD, $1, $3);}
                | ltl_simple_expr EQUAL ltl_simple_expr {$$ = new_node(EQUAL, $1, $3);}
                | ltl_simple_expr LT ltl_simple_expr {$$ = new_node(LT, $1, $3);}
                | ltl_simple_expr GT ltl_simple_expr {$$ = new_node(GT, $1, $3);}
                | ltl_simple_expr LE ltl_simple_expr {$$ = new_node(LE, $1, $3);}
                | ltl_simple_expr GE ltl_simple_expr {$$ = new_node(GE, $1, $3);}
                | LCB atom_set RCB { $$ = $2; } 
                | ltl_simple_expr UNION ltl_simple_expr { $$ = new_node(UNION, $1, $3);}
                | ltl_simple_expr SETIN ltl_simple_expr { $$ = new_node(SETIN, $1, $3);}
                ;

/*
   Only for homomorphism checking (currently disabled).
 */
implements    : IMPLEMENTS ATOM {$$ = new_node(IMPLEMENTS, $2, Nil);}
              ;
input         : INPUT var_list {$$ = new_node(INPUT, $2, Nil);}
              ;
output        : OUTPUT term_list SEMI {$$ = new_node(OUTPUT, $2, Nil);}
              ;
term_list     : var_id {$$ = cons($1, Nil);}
              | term_list COMMA var_id {$$ = cons($3, $1);}
              ;

command       : command_case {$$ = $1;}
              | error SEMI {fprintf(stderr, "Not recognized command\n"); return(1);}
              | error {fprintf(stderr, "Not recognized command\n"); return(1);}
              ;

command_case  : SPEC ctl_expr SEMI
                 {parse_tree_int = new_node(SPEC, $2, Nil); return(0);}
              | LTLSPEC ltl_expr SEMI
                 {parse_tree_int = new_node(LTLSPEC, $2, Nil); return(0);}
              | COMPUTE compute_expr SEMI
                 {parse_tree_int = new_node(COMPUTE, $2, Nil); return(0);}
              | INVARSPEC invar_expr SEMI
                 {parse_tree_int = new_node(INVARSPEC, $2, Nil); return(0);}
              | CHECKINVARIANTFB invar_expr SEMI
                 {parse_tree_int = new_node(CHECKINVARIANTFB, $2, Nil); return(0);}
              | CHECKINVARIANTSTRONG invar_expr SEMI
                 {parse_tree_int = new_node(CHECKINVARIANTSTRONG, $2, Nil); return(0);}
              | AFTER dl_expr COLON goal_expr SEMI
                 {parse_tree_int = new_node(AFTER, $2, $4); return(0);}
              | GOTO state SEMI
                 {parse_tree_int = new_node(GOTO, $2, Nil); return(0);}
              | LET var_id EQDEF simple_expr SEMI
                 {parse_tree_int = new_node(LET, new_node(EQDEF, $2, $4), Nil); return(0);}
              | EVAL ctl_expr SEMI
                 {parse_tree_int = new_node(EVAL, $2, Nil); return(0);}
              | INIT init_expr SEMI
                 {parse_tree_int = new_node(INIT, $2, Nil); return(0);}
              | FAIRNESS fair_expr SEMI
                 {parse_tree_int = new_node(FAIRNESS, $2, Nil); return(0);}
              | TRANS trans_expr SEMI
                 {parse_tree_int = new_node(TRANS, $2, Nil); return(0);}
              ;

trace         : NUMBER {$$ = (node_ptr)find_atom($1);}
              | state DOT NUMBER {$$ = find_node(DOT, $1, (node_ptr)find_atom($3));}
              ;

state         : trace DOT NUMBER {$$ = find_node(DOT, $1, (node_ptr)find_atom($3));}
              ;


%%

/* Additional source code */
void yyerror(char *s)
{
    extern FILE * nusmv_stderr;
    extern options_ptr options;
    extern char yytext[];
    
    start_parsing_err();
    fprintf(nusmv_stderr, "at token \"%s\": %s\n", yytext, s);
    if (opt_batch(options)) finish_parsing_err();
}

int yywrap()
{
  return(1);
}
