/**CFile***********************************************************************

  FileName    [optCmd.c]

  PackageName [opt]

  Synopsis    [The option command file.]

  Description [optional]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "optInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

int CommandSetVariable ARGS((int  argc, char ** argv));
int CommandUnsetVariable ARGS((int  argc, char ** argv));

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void print_options ARGS((options_ptr opt, FILE * fn));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initialize the NuSMV options.]

  Description        [The NuSMV options are initialized. A pointer to
  a structure containing the NuSMV options is allocated, its fields
  are initialized and the pointer is returned.]

  SideEffects        []

******************************************************************************/
options_ptr init_options()
{
  options_ptr opt;

  opt = ALLOC(options_rec, 1);
  opt->pgm_name = DEFAULT_PGM_NAME;
  opt->input_file = DEFAULT_INPUT_FILE;
  opt->input_order_file = DEFAULT_INPUT_ORDER_FILE;
  opt->output_order_file = DEFAULT_OUTPUT_ORDER_FILE;
  opt->verbose = 0;
#ifdef OPT_QUANTIFY_INPUT
  opt->quantify_input = false;
#endif
  opt->check_trans = false;
  opt->ignore_spec = false;
  opt->ignore_ltlspec = false;
  opt->ignore_invar = false;
  opt->ignore_invar_fb = false;
  opt->ignore_invar_strong = false;
  opt->ignore_after = false;
  opt->forward_search = false;
  opt->print_reachable = false;
  opt->reorder = false;
  opt->reorder_method  = DEFAULT_REORDER;
  opt->dynamic_reorder = false;
  opt->batch = true;
  opt->partition_method = Monolithic;
  opt->conj_part_threshold = DEFAULT_CONJ_PART_THRESHOLD;
  opt->image_cluster_size = DEFAULT_IMAGE_CLUSTER_SIZE;
  opt->ignore_init_file = false;
  opt->ag_only = false;
  opt->gui_behavior = false;

  /* Yuan Lu */
  opt->coi = 0;

  return(opt);
}

/**Function********************************************************************

  Synopsis           [Initialize the NuSMV options command]

  Description        [This function declares the interactive shell
  commands necessary to manipulate NuSMV options.]

  SideEffects        []

******************************************************************************/
void init_options_cmd()
{
  char *path;
  char *lib_name;

  Cmd_CommandAdd("set", CommandSetVariable, 0);
  Cmd_CommandAdd("unset", CommandUnsetVariable, 0);
  lib_name = Sm_NuSMVObtainLibrary();
  path = ALLOC(char, strlen(lib_name) + 20);
  sprintf(path, "set open_path .:%s", lib_name);
  Cmd_CommandExecute(path);
  FREE(lib_name);
  FREE(path);
}

void set_pgm_name(options_ptr opt, char * str)
{
  assert(opt != NULL);
  opt->pgm_name = util_strsav(str);
}
void reset_pgm_name(options_ptr opt)
{
  assert(opt != NULL);
  opt->pgm_name = DEFAULT_PGM_NAME;
}
char * get_pgm_name(options_ptr opt)
{
  assert(opt != NULL);
  return(opt->pgm_name);
}


void set_input_file(options_ptr opt, char * str)
{
  assert(opt != NULL);
  opt->input_file = util_strsav(str);
}
void reset_input_file(options_ptr opt)
{
  assert(opt != NULL);
  opt->input_file = DEFAULT_INPUT_FILE;
}
char * get_input_file(options_ptr opt)
{
  assert(opt != NULL);
  return(opt->input_file);
}

void set_input_order_file(options_ptr opt, char * str)
{
  assert(opt != NULL);
  opt->input_order_file = util_strsav(str);
}
void reset_input_order_file(options_ptr opt)
{
  assert(opt != NULL);
  opt->input_order_file = DEFAULT_INPUT_ORDER_FILE;
}
char * get_input_order_file(options_ptr opt)
{
  assert(opt != NULL);
  return(opt->input_order_file);
}

void set_output_order_file(options_ptr opt, char * str)
{
  assert(opt != NULL);
  opt->output_order_file = util_strsav(str);
}
void reset_output_order_file(options_ptr opt)
{
  assert(opt != NULL);
  opt->output_order_file = DEFAULT_OUTPUT_ORDER_FILE;
}
char * get_output_order_file(options_ptr opt)
{
  assert(opt != NULL);
  return(opt->output_order_file);
}

boolean is_default_order_file(options_ptr opt)
{
  assert(opt != NULL);
  return((strcmp(opt->output_order_file, DEFAULT_OUTPUT_ORDER_FILE) == 0));
}

/* Yuan Lu : simulate file */
void set_simulate_file(options_ptr opt, char * str)
{
  assert(opt != NULL);
  opt->simulate_file = util_strsav(str);
}
void reset_simulate_file(options_ptr opt)
{
  assert(opt != NULL);
  opt->simulate_file = DEFAULT_OUTPUT_ORDER_FILE;
}
char * get_simulate_file(options_ptr opt)
{
  assert(opt != NULL);
  return(opt->simulate_file);
}

void set_verbose_level(options_ptr opt, int level)
{
  assert(opt != NULL);
  opt->verbose = level;
}
boolean opt_verbose_level_eq(options_ptr opt, int level)
{
  assert(opt != NULL);
  return((opt->verbose == level));
}

boolean opt_verbose_level_gt(options_ptr opt, int level)
{
  assert(opt != NULL);
  return((opt->verbose > level));
}
boolean opt_verbose_level_ge(options_ptr opt, int level)
{
  assert(opt != NULL);
  return((opt->verbose >= level));
}
boolean opt_verbose_level_lt(options_ptr opt, int level)
{
  assert(opt != NULL);
  return((opt->verbose < level));
}
boolean opt_verbose_level_le(options_ptr opt, int level)
{
  assert(opt != NULL);
  return((opt->verbose <= level));
}

#ifdef OPT_QUANTIFY_INPUT
void set_quantify_input(options_ptr opt){
  assert(opt != NULL);
  opt->quantify_input = true;
}
void unset_quantify_input(options_ptr opt){
  assert(opt != NULL);
  opt->quantify_input = false;
}
void reset_quantify_input(options_ptr opt){
  assert(opt != NULL);
  opt->quantify_input = false;
}
boolean opt_quantify_input(options_ptr opt){
  assert(opt != NULL);
  return(opt->quantify_input == true);
}
#endif

void set_ignore_spec(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_spec = true;
}
void unset_ignore_spec(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_spec = false;
}
boolean opt_ignore_spec(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ignore_spec == true));
}

void set_ignore_ltlspec(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_ltlspec = true;
}
void unset_ignore_ltlspec(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_ltlspec = false;
}
boolean opt_ignore_ltlspec(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ignore_ltlspec == true));
}

void set_check_trans(options_ptr opt)
{
  assert(opt != NULL);
  opt->check_trans = true;
}
void unset_check_trans(options_ptr opt)
{
  assert(opt != NULL);
  opt->check_trans = false;
}
boolean opt_check_trans(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->check_trans == true));
}

void set_ignore_invar(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_invar = true;
}
void unset_ignore_invar(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_invar = false;
}
boolean opt_ignore_invar(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ignore_invar == true));
}

void set_ignore_invar_fb(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_invar_fb = true;
}
void unset_ignore_invar_fb(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_invar_fb = false;
}
boolean opt_ignore_invar_fb(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ignore_invar_fb == true));
}

void set_ignore_invar_strong(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_invar_strong = true;
}
void unset_ignore_invar_strong(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_invar_strong = false;
}
boolean opt_ignore_invar_strong(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ignore_invar_strong == true));
}

void set_ignore_after(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_after = true;
}
void unset_ignore_after(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_after = false;
}
boolean opt_ignore_after(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ignore_after == true));
}

void set_forward_search(options_ptr opt)
{
  assert(opt != NULL);
  opt->forward_search = true;
}
void unset_forward_search(options_ptr opt)
{
  assert(opt != NULL);
  opt->forward_search = false;
}
boolean opt_forward_search(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->forward_search == true));
}

void set_print_reachable(options_ptr opt)
{
  assert(opt != NULL);
  opt->print_reachable = true;
}
void unset_print_reachable(options_ptr opt)
{
  assert(opt != NULL);
  opt->print_reachable = false;
}
boolean opt_print_reachable(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->print_reachable == true));
}

void set_reorder(options_ptr opt)
{
  assert(opt != NULL);
  opt->reorder = true;
}
void unset_reorder(options_ptr opt)
{
  assert(opt != NULL);
  opt->reorder = false;
}
boolean opt_reorder(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->reorder == true));
}

void set_reorder_method(options_ptr opt, unsigned int method)
{
  assert(opt != NULL);
  opt->reorder_method = method;
}
void unset_reorder_method(options_ptr opt)
{
  assert(opt != NULL);
  opt->reorder_method = DEFAULT_REORDER;
}
unsigned int get_reorder_method(options_ptr opt)
{
  assert(opt != NULL);
  return(opt->reorder_method);
}

void set_dynamic_reorder(options_ptr opt)
{
  assert(opt != NULL);
  opt->dynamic_reorder = true;
}
void unset_dynamic_reorder(options_ptr opt)
{
  assert(opt != NULL);
  opt->dynamic_reorder = false;
}
boolean opt_dynamic_reorder(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->dynamic_reorder == true));
}

void set_batch(options_ptr opt)
{
  assert(opt != NULL);
  opt->batch = true;
}
void unset_batch(options_ptr opt)
{
  assert(opt != NULL);
  opt->batch = false;
}
boolean opt_batch(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->batch == true));
}

void set_partition_method(options_ptr opt, Partition_Method method)
{
  assert(opt != NULL);
  opt->partition_method = method;
}
Partition_Method get_partition_method(options_ptr opt)
{
  assert(opt != NULL);
  return(opt->partition_method);
}
void set_monolithic(options_ptr opt)
{
  set_partition_method(opt, Monolithic);
}
void set_conj_partitioning(options_ptr opt)
{
  set_partition_method(opt, Conjunctive);
}
void reset_partitioning_method(options_ptr opt)
{
  set_partition_method(opt, Monolithic);
}
void set_disj_partitioning(options_ptr opt)
{
  set_partition_method(opt, Disjunctive);
}
void set_iwls95cp_partitioning(options_ptr opt)
{
  set_partition_method(opt, Iwls95CP);
}
boolean opt_monolithic(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->partition_method == Monolithic));
}
boolean opt_conj_partitioning(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->partition_method == Conjunctive));
}
boolean opt_disj_partitioning(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->partition_method == Disjunctive));
}
boolean opt_iwls95cp_partitioning(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->partition_method == Iwls95CP));
}

void set_conj_part_threshold(options_ptr opt, int threshold)
{
  assert(opt != NULL);
  opt->conj_part_threshold = threshold;
}
void reset_conj_part_threshold(options_ptr opt)
{
  assert(opt != NULL);
  opt->conj_part_threshold = DEFAULT_CONJ_PART_THRESHOLD;
}
int get_conj_part_threshold(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->conj_part_threshold));
}

void set_image_cluster_size(options_ptr opt, int threshold)
{
  assert(opt != NULL);
  opt->image_cluster_size = threshold;
}
void reset_image_cluster_size(options_ptr opt)
{
  assert(opt != NULL);
  opt->image_cluster_size = DEFAULT_IMAGE_CLUSTER_SIZE;
}
int get_image_cluster_size(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->image_cluster_size));
}

void set_ignore_init_file(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_init_file = true;
}
void unset_ignore_init_file(options_ptr opt)
{
  assert(opt != NULL);
  opt->ignore_init_file = false;
}
boolean opt_ignore_init_file(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ignore_init_file == true));
}

void set_ag_only(options_ptr opt)
{
  assert(opt != NULL);
  opt->ag_only = true;
}
void unset_ag_only(options_ptr opt)
{
  assert(opt != NULL);
  opt->ag_only = false;
}
boolean opt_ag_only(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->ag_only == true));
}

void set_gui_behavior(options_ptr opt)
{
  assert(opt != NULL);
  opt->gui_behavior = true;
}
void unset_gui_behavior(options_ptr opt)
{
  assert(opt != NULL);
  opt->gui_behavior = false;
}
boolean opt_gui_behavior(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->gui_behavior == true));
}

/* Yuan Lu */
void set_abs_level_threshold(options_ptr opt, int threshold)
{
  assert(opt != NULL);
  opt->abs_level = threshold;
}
void reset_abs_level_threshold(options_ptr opt)
{
  assert(opt != NULL);
  opt->abs_level = DEFAULT_ABS_LEVEL_THRESHOLD;
}
int get_abs_level_threshold(options_ptr opt)
{
  assert(opt != NULL);
  return((opt->abs_level));
}

/* Yuan Lu */
void set_coi(options_ptr opt)
{
  assert(opt != NULL);
  opt->coi = 1;
}
void reset_coi(options_ptr opt)
{
  assert(opt != NULL);
  opt->coi=0;
}
int get_coi(options_ptr opt)
{
  assert(opt != NULL);
  return opt->coi;
}

Partition_Method str2method (char *m)
{
  if (strcmp(m, "Monolithic") == 0) return(Monolithic);
  if (strcmp(m, "Conjunctive") == 0) return(Conjunctive);
  if (strcmp(m, "Disjunctive") == 0) return(Disjunctive);
  if (strcmp(m, "Iwls95CP") == 0) return(Iwls95CP);
  return -1;
}

char * method2str(Partition_Method m){
  if (m == Monolithic) return("Monolithic");
  if (m == Conjunctive) return("Conjunctive");
  if (m == Disjunctive) return("Disjunctive");
  if (m == Iwls95CP) return("Iwls95CP");
  return("Unknown");
}

void print_partition_method (FILE * output_stream)
{
  (void) fprintf(output_stream, "Monolithic, Conjunctive, Disjunctive, Iwls95CP");
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis          [Set an environment variable]

  CommandName       [set]

  CommandSynopsis   [Set an environment variable]

  CommandArguments  [\[-h\] \[&lt;name&gt;\] \[&lt;value&gt;\]] 

  CommandDescription [ A variable environment is maintained by the command 
  interpreter.
  The "set" command sets a variable to a particular value, and the
  "unset" command removes the definition of a variable.
  If "set" is given no arguments, it prints the current value of all variables.<p>

  Command options:<p>
  <dl> <dt> -h
  <dd> Print the command usage.
  </dl>
  <dl> <dt> &lt;name&gt;
  <dd> Variable name
  </dl>
  <dl> <dt> &lt;value&gt;
  <dd> Value to be assigned to the variable.
  </dl>

 <p>
  Interpolation of variables is allowed when using the set command. The
  variables are  referred to with the prefix of '$'. So for example, the following
  can be done to check the value of a set variable:<br>
  <code>
  NuSMV> set foo bar <br>
  NuSMV> echo $foo <br>
  bar <br>
  </code>
  
  The last line "bar" will the output produced by NuSMV.<p>

  Variables can be extended by using the character ':' to concatenate
  values. For example : <br>
  <code>
  NuSMV> set foo bar <br>
  NuSMV> set foo $foo:foobar <br>
  NuSMV> echo $foo <br>
  bar:foobar <br>
  </code>
  The variable <code> foo </code> is extended with the value <code>
  foobar </code>. <p>
  
  Whitespace characters may be present within quotes. However, variable
  interpolation lays the restriction that the characters ':' and '/' may
  not be used within quotes. This is to allow for recursive interpolation.
  So for example, the following is allowed<br>
    <code>
  NuSMV> set "foo bar" this <br>
  NuSMV> echo $"foo bar"<br>
  this <br>
  </code> 
  The last line will be the output produced by NuSMV. <br>
  But in the following, the  value of the variable <code> foo/bar </code>
  will not be interpreted correctly:<p>
  <code>
  NuSMV> set "foo/bar" this <br>
  NuSMV> echo $"foo/bar" <br>
  foo/bar <br>
  </code>
  If a variable is not set by the "set" command, then the variable is returned
  unchanged.
  <p>

  Different commands use environment information for different purposes.
  The command interpreter makes use of the following parameters:<p>

 <dl>
 <dt><b>autoexec</b>
 <dd>     Defines a command string to be automatically executed after every 
           command processed by the command interpreter.
           This is useful for things like timing commands, or tracing the 
           progress of optimization.
</dl>


 <dl><dt><b>open_path</b>
 <dd>      "open_path" (in analogy to the shell-variable PATH) is a list of
           colon-separated strings giving directories to be searched whenever
           a file is opened for read.  Typically the current directory (.) is
           first in this list.  The standard system library (typically 
           $NuSMV_LIBRARY_PATH) is always implicitly appended to the current path.
           This provides a convenient short-hand mechanism for reaching 
           standard library files.
 </dl>
 <dl><dt> <b>nusmv_stderr </b>
 <dd>   Standard error (normally stderr) can be re-directed to a file
           by setting the variable nusmv_stderr.
 </dl>           

 <dl><dt>  <b>nusmv_stdout</b>
 <dd>           Standard output (normally stdout) can be re-directed to a file
           by setting the variable nusmv_stdout.
 </dl>           
  ]

  SideEffects        []

  SeeAlso            [unset]

******************************************************************************/
int CommandSetVariable(int  argc, char ** argv)
{
  char *flag_value, *key, *value;
  avl_generator *gen;
  int c;
  
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch(c) {
      case 'h':
        goto usage;
        break;
      default:
        goto usage;
    }
  }
  if (argc == 0 || argc > 3) {
    goto usage ;
  }
  else if (argc == 1) {
    (void) fprintf(nusmv_stdout, "The user variables are:\n");
    avl_foreach_item(cmdFlagTable, gen, AVL_FORWARD, &key, &value) {
      (void) fprintf(nusmv_stdout, "%s\t\t\"%s\"\n", key, value);
    }
    /* (void) fprintf(nusmv_stdout, "The internal variables are:\n"); */
    print_options(options, nusmv_stdout);
    return 0;
  }
  else {
    key = argv[1];
    if (avl_delete(cmdFlagTable, &key, &value)) {
      FREE(key);
      FREE(value);
    }
    flag_value = argc == 2 ? util_strsav("") : util_strsav(argv[2]);
    if (strcmp(argv[1], "nusmv_stdout") == 0) {
      if (nusmv_stdout != stdout) {
        (void) fclose(nusmv_stdout);
      }
      if (strcmp(flag_value, "") == 0) {
        flag_value = "-";
      }
      nusmv_stdout = Cmd_FileOpen(flag_value, "w", NIL(char *), 0);
      if (nusmv_stdout == NULL) {
        nusmv_stdout = stdout;
      }
#if HAVE_SETVBUF
      setvbuf(nusmv_stdout, (char *)NULL, _IOLBF, 0);
#endif      
    }
    if (strcmp(argv[1], "nusmv_stderr") == 0) {
      if (nusmv_stderr != stderr) {
        (void) fclose(nusmv_stderr);
      }
      if (strcmp(flag_value, "") == 0) {
        flag_value = "-";
      }
      nusmv_stderr = Cmd_FileOpen(flag_value, "w", NIL(char *), 0);
      if (nusmv_stderr == NULL) {
        nusmv_stderr = stderr;
      }
#if HAVE_SETVBUF
      setvbuf(nusmv_stderr, (char *)NULL, _IOLBF, 0);
#endif      
    }
    if (strcmp(argv[1], "history") == 0) {
      if (nusmv_historyFile != NIL(FILE)) {
        (void) fclose(nusmv_historyFile);
      }
      if (strcmp(flag_value, "") == 0) {
        nusmv_historyFile = NIL(FILE);
      }
      else {
        nusmv_historyFile = Cmd_FileOpen(flag_value, "w", NIL(char *), 0);
        if (nusmv_historyFile == NULL) {
          nusmv_historyFile = NIL(FILE);
        }
      }
    }
    if (strcmp(argv[1], PROGRAM_NAME) == 0) {
      set_pgm_name(options, flag_value);
      return 0;
    }
    if (strcmp(argv[1], INPUT_FILE) == 0) {
      if (strcmp(flag_value, "") == 0)
        reset_input_file(options);
      else 
        set_input_file(options, util_strsav(flag_value));
      return 0;
    }
    if (strcmp(argv[1], INPUT_ORDER_FILE) == 0) {
      if (strcmp(flag_value, "") == 0)
        reset_input_order_file(options);
      else 
        set_input_order_file(options, util_strsav(flag_value));
      return 0;
    }
    if (strcmp(argv[1], OUTPUT_ORDER_FILE) == 0) {
      if (strcmp(flag_value, "") == 0)
        reset_output_order_file(options);
      else 
        set_output_order_file(options, util_strsav(flag_value));
      return 0;
    }
    if (strcmp(argv[1], VERBOSE_LEVEL) == 0) {
      char *err_occ[1];
      int cur_verbose;

      err_occ[0] = "";
      cur_verbose = strtol(flag_value,err_occ, 10);
      if (strcmp(err_occ[0], "") != 0) {
        (void) fprintf(nusmv_stderr, "Error: \"%s\" is not a valid value for the \"%s\" option.\n", err_occ[0], VERBOSE_LEVEL);
        return 1;
      }
      set_verbose_level(options, cur_verbose);
      return 0;
    }
#ifdef OPT_QUANTIFY_INPUT
    if (strcmp(argv[1], QUANTIFY_INPUT) == 0) {
      set_quantify_input(options);
      return 0;
    }
#endif
    if (strcmp(argv[1], IGNORE_SPEC) == 0) {
      set_ignore_spec(options);
      return 0;
    }
    if (strcmp(argv[1], IGNORE_LTLSPEC) == 0) {
      set_ignore_ltlspec(options);
      return 0;
    }
    if (strcmp(argv[1], OPT_CHECK_TRANS) == 0) {
      set_check_trans(options);
      return 0;
    }
    if (strcmp(argv[1], IGNORE_INVAR) == 0) {
      set_ignore_invar(options);
      return 0;
    }
    if (strcmp(argv[1], IGNORE_INVAR_FB) == 0) {
      set_ignore_invar_fb(options);
      return 0;
    }
    if (strcmp(argv[1], IGNORE_INVAR_STRONG) == 0) {
      set_ignore_invar_strong(options);
      return 0;
    }
    if (strcmp(argv[1], IGNORE_AFTER) == 0) {
      set_ignore_after(options);
      return 0;
    }
    if (strcmp(argv[1], FORWARD_SEARCH) == 0) {
      set_forward_search(options);
      return 0;
    }
    if (strcmp(argv[1], PRINT_REACHABLE) == 0) {
      set_print_reachable(options);
      return 0;
    }
    if (strcmp(argv[1], ENABLE_REORDER) == 0) {
      set_reorder(options);
      return 0;
    }
    if (strcmp(argv[1], REORDER_METHOD) == 0) {
      unsigned int method = StringConvertToDynOrderType(flag_value);

      if (method == REORDER_NONE) {
        (void) fprintf(nusmv_stderr, "Error: \"%s\" is not a valid ordering method.\n", flag_value);
        return 1;
      }
      set_reorder_method(options, method);
      return 0;
    }
    if (strcmp(argv[1], DYNAMIC_REORDER) == 0) {
      int current_ordering_method = dd_get_ordering_method(dd_manager);

      set_dynamic_reorder(options);
      dd_autodyn_enable(dd_manager, current_ordering_method);
      return 0;
    }
    if (strcmp(argv[1], PARTITION_METHOD) == 0) {
      if (flag_value == NULL) {
        (void) fprintf(nusmv_stderr, "The option \"%s\" requires an argument.\n", PARTITION_METHOD);
        return 1;
      }
      else {
        if (str2method(flag_value) != -1) {
          set_partition_method(options, str2method(flag_value));
          return 0;
        } else {
          (void) fprintf(nusmv_stderr, "Error: \"%s\" is not a valid value for the \"%s\" option.\n", flag_value, PARTITION_METHOD);
        return 1;
        }
      }
    }
    if (strcmp(argv[1], CONJ_PART_THRESHOLD) == 0) {
      char *err_occ[1];
      int cur_rt;
  
      err_occ[0] = "";
      cur_rt = strtol(flag_value, err_occ, 10);
      if (strcmp(err_occ[0], "") != 0) {
        (void) fprintf(nusmv_stderr, "Error: \"%s\" is not a valid value for the \"%s\" option.\n", err_occ[0], CONJ_PART_THRESHOLD);
        return 1;
      }
      set_conj_part_threshold(options, cur_rt);
      return 0;
    }
    if (strcmp(argv[1], IMAGE_CLUSTER_SIZE) == 0) {
      char *err_occ[1];
      int cur_rt;
  
      err_occ[0] = "";
      cur_rt = strtol(flag_value, err_occ, 10);
      if (strcmp(err_occ[0], "") != 0) {
        (void) fprintf(nusmv_stderr, "Error: \"%s\" is not a valid value for the \"%s\" option.\n", err_occ[0], IMAGE_CLUSTER_SIZE);
        return 1;
      }
      set_image_cluster_size(options, cur_rt);
      return 0;
    }
    if (strcmp(argv[1], IGNORE_INIT_FILE) == 0) {
      set_ignore_init_file(options);
      return 0;
    }
    if (strcmp(argv[1], AG_ONLY_SEARCH) == 0) {
      set_ag_only(options);
      return 0;
    }

    /* None of the internal options have been recognized */
    (void) avl_insert(cmdFlagTable, util_strsav(argv[1]), flag_value);
    return 0;
  }

  usage:
      (void) printf("usage: set [-h] [name [value]]\n");
      (void) fprintf(nusmv_stderr, "   -h \t\tprint the command usage\n");
      return 1;
}


/**Function********************************************************************

  Synopsis          [Unset an environment variable]

  CommandName       [unset]

  CommandSynopsis   [Unset an environment variable]

  CommandArguments  [\[-h\] &lt;variables&gt;] 

  CommandDescription [A variable environment is maintained by the command 
  interpreter. 
  The "set" command sets a variable to a particular value, and the
  "unset" command removes the definition of a variable. <p>
  Command options:<p>
  </dl>
  <dl><dt> -h
  <dd> Print the command usage.
  </dl>
  <dl><dt> &lt;variables&gt;
  <dd> Variables to be unset
  </dl>
  ]

  SideEffects        []

  SeeAlso            [set]

******************************************************************************/
int CommandUnsetVariable(int  argc, char ** argv)
{
  int i;
  char *key, *value;
  int c;
  
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch(c) {
      case 'h':
        goto usage;
        break;
      default:
        goto usage;
    }
  }
  
  if (argc < 2) {
    goto usage;
  }
  if (strcmp(argv[1], PROGRAM_NAME) == 0) {
    reset_pgm_name(options);
    return 0;
  }
  if (strcmp(argv[1], INPUT_FILE) == 0) {
    reset_input_file(options);
    return 0;
  }
  if (strcmp(argv[1], INPUT_ORDER_FILE) == 0) {
    reset_input_order_file(options);
    return 0;
  }
  if (strcmp(argv[1], OUTPUT_ORDER_FILE) == 0) {
    reset_output_order_file(options);
    return 0;
  }
  if (strcmp(argv[1], VERBOSE_LEVEL) == 0) {
    set_verbose_level(options, 0);
    return 0;
  }
#ifdef OPT_QUANTIFY_INPUT
  if (strcmp(argv[1], QUANTIFY_INPUT) == 0) {
    reset_quantify_input(options);
    return 0;
  }
#endif
  if (strcmp(argv[1], IGNORE_SPEC) == 0) {
    unset_ignore_spec(options);
    return 0;
  }
  if (strcmp(argv[1], IGNORE_LTLSPEC) == 0) {
    unset_ignore_ltlspec(options);
    return 0;
  }
  if (strcmp(argv[1], OPT_CHECK_TRANS) == 0) {
    unset_check_trans(options);
    return 0;
  }
  if (strcmp(argv[1], IGNORE_INVAR) == 0) {
    unset_ignore_invar(options);
    return 0;
  }
  if (strcmp(argv[1], IGNORE_INVAR_FB) == 0) {
    unset_ignore_invar_fb(options);
    return 0;
  }
  if (strcmp(argv[1], IGNORE_INVAR_STRONG) == 0) {
    unset_ignore_invar_strong(options);
    return 0;
  }
  if (strcmp(argv[1], IGNORE_AFTER) == 0) {
    unset_ignore_after(options);
    return 0;
  }
  if (strcmp(argv[1], FORWARD_SEARCH) == 0) {
    unset_forward_search(options);
    return 0;
  }
  if (strcmp(argv[1], PRINT_REACHABLE) == 0) {
    unset_print_reachable(options);
    return 0;
  }
  if (strcmp(argv[1], ENABLE_REORDER) == 0) {
    unset_reorder(options);
    return 0;
  }
  if (strcmp(argv[1], REORDER_METHOD) == 0) {
    unset_reorder_method(options);
    return 0;
  }
  if (strcmp(argv[1], DYNAMIC_REORDER) == 0) {
    unset_dynamic_reorder(options);
    return 0;
  }
  if (strcmp(argv[1], PARTITION_METHOD) == 0) {
    set_monolithic(options);
    return(0);
  }
  if (strcmp(argv[1], CONJ_PART_THRESHOLD) == 0) {
    reset_conj_part_threshold(options);
    return 0;
  }
  if (strcmp(argv[1], IMAGE_CLUSTER_SIZE) == 0) {
    reset_image_cluster_size(options);
    return 0;
  }
  if (strcmp(argv[1], IGNORE_INIT_FILE) == 0) {
    unset_ignore_init_file(options);
    return 0;
  }
  if (strcmp(argv[1], AG_ONLY_SEARCH) == 0) {
    unset_ag_only(options);
    return 0;
  }
  /* End of internal variables */
  for(i = 1; i < argc; i++) {
    key = argv[i];
    if (avl_delete(cmdFlagTable, &key, &value)) {
      FREE(key);
      FREE(value);
    }
  }
  return 0;


  usage:
    (void) fprintf(nusmv_stderr, "usage: unset [-h] variables \n");
    (void) fprintf(nusmv_stderr, "   -h \t\tprint the command usage\n");
    return 1;  
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

static void print_options(options_ptr opt, FILE * fn)
{
  /*  fprintf(fn, "%s\t\t\"%s\"\n", PROGRAM_NAME, opt->pgm_name); */
  fprintf(fn, "%s\t\t\"%s\"\n", PROGRAM_NAME,
          (opt->pgm_name != (char *)NULL) ? opt->pgm_name : "(null)" );
  fprintf(fn, "%s\t\t\"%s\"\n", INPUT_FILE,
          (opt->input_file != (char *)NULL) ? opt->input_file : "(null)" );
  fprintf(fn, "%s\t\"%s\"\n",   INPUT_ORDER_FILE,
          (opt->input_order_file != (char *)NULL) ? opt->input_order_file : "(null)");
  fprintf(fn, "%s\t\"%s\"\n",   OUTPUT_ORDER_FILE, opt->output_order_file);
  fprintf(fn, "%s\t\t\"%d\"\n", VERBOSE_LEVEL, opt->verbose);
#ifdef OPT_QUANTIFY_INPUT
  fprintf(fn, "%s\t\t\"%d\"\n", QUANTIFY_INPUT, opt->quantify_input);
#endif
  fprintf(fn, "%s\t\t\"%d\"\n", OPT_CHECK_TRANS, opt->check_trans);
  fprintf(fn, "%s\t\t\"%d\"\n", IGNORE_SPEC, opt->ignore_spec);
  fprintf(fn, "%s\t\t\"%d\"\n", IGNORE_LTLSPEC, opt->ignore_ltlspec);
  fprintf(fn, "%s\t\t\"%d\"\n", IGNORE_INVAR, opt->ignore_invar);
  fprintf(fn, "%s\t\t\"%d\"\n", IGNORE_INVAR_FB, opt->ignore_invar_fb);
  fprintf(fn, "%s\t\"%d\"\n",   IGNORE_INVAR_STRONG, opt->ignore_invar_strong);
  fprintf(fn, "%s\t\t\"%d\"\n", IGNORE_AFTER, opt->ignore_after);
  fprintf(fn, "%s\t\t\"%d\"\n", FORWARD_SEARCH, opt->forward_search);
  fprintf(fn, "%s\t\t\"%d\"\n", PRINT_REACHABLE, opt->print_reachable);
  fprintf(fn, "%s\t\t\"%d\"\n", ENABLE_REORDER, opt->reorder);
  fprintf(fn, "%s\t\t\"%s\"\n", REORDER_METHOD, DynOrderTypeConvertToString(opt->reorder_method));
  fprintf(fn, "%s\t\t\"%d\"\n", DYNAMIC_REORDER, opt->dynamic_reorder);
  fprintf(fn, "%s\t\"%s\"\n",   PARTITION_METHOD, method2str(opt->partition_method));
  fprintf(fn, "%s\t\"%d\"\n",   CONJ_PART_THRESHOLD, opt->conj_part_threshold);
  fprintf(fn, "%s\t\"%d\"\n",   IMAGE_CLUSTER_SIZE, opt->image_cluster_size);
  fprintf(fn, "%s\t\"%d\"\n",   IGNORE_INIT_FILE, opt->ignore_init_file);
  fprintf(fn, "%s\t\t\"%d\"\n",   AG_ONLY_SEARCH, opt->ag_only);
}
