/**CFile***********************************************************************

  FileName    [nodeWffPrint.c]

  PackageName [node]

  Synopsis    [Pretty printing of formulas represented using node struct.]

  Description [This file conatins the code to perform pretty printing
  of a formula represented with a node struct.]

  SeeAlso     [node.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "nodeInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

#define LINE_LENGTH 1000

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int util_strncat ARGS((char *s1, char *s2, int size));
static int sprint_node_recur ARGS((char *str, int size, node_ptr n, int p));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Pretty print a formula in a string.]

  Description        [Pretty print a formula in a string.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int sprint_node(char *str, int size, node_ptr n)
{ return(sprint_node_recur(str,size,n,0)); }


/**Function********************************************************************

  Synopsis           [Pretty print a formula on a file]

  Description        [Pretty print a formula on a file]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_node(FILE *stream, node_ptr n)
{ print_node_atcol(stream,n,0); }

/**Function********************************************************************

  Synopsis           [Pretty print a formula in a given file at given column]

  Description        [Pretty print a formula in a given file at given column]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int print_node_atcol(FILE *stream, node_ptr n, int col)
{
  char buf[LINE_LENGTH + 1];
  int c,p;

  buf[0] = 0;
  c = sprint_node(buf, LINE_LENGTH, n);
  p = strlen(buf);
  if (!c) p += 3;
  if (col + p >= LINE_LENGTH) {
    fprintf(stream, "\n");
    col = 0;
    while((col++) < get_indent_size() + 1) fprintf(stream, "  ");
  }
  fprintf(stream, "%s", buf);
  if (!c) fprintf(stream, "...");
  return(col + p);
}

/**Function********************************************************************

  Synopsis           [Concatenate two strings.]

  Description        [This function appends only the first <tt>size</tt>
  characters of the <tt>s2</tt> string to the <tt>s1</tt> string
  overwriting the `\0' character at the end of <tt>s1</tt>. The strings
  may not overlap, and the  <tt>s1</tt>  string  must  have  enough
  space for the result.]

  SideEffects        [The string <tt>s1</tt> is modified.]

******************************************************************************/
static int util_strncat(char *s1, char *s2, int size)
{
  while(size && *s2){
    if(*s1) s1++;
    else{
      *(s1++) = *(s2++);
      *s1 = 0;
    }
    size--;
  }
  return(!*s2);
}

/**Function********************************************************************

  Synopsis           [Recursive step of sprint_node]

  Description        [Recursive step of sprint_node]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int sprint_node_recur(char *str, int size, node_ptr n, int p)
{
  char *op;
  int priority, arity;     /* 0: unary, 1: binary, 2: terciary, 3:quad */
  if(!n)return(1);
  if(n == (node_ptr)(-1)) return(util_strncat(str, "*no value*", size));
  switch(n->type){
  case TRUEEXP:
    return(util_strncat(str, "TRUE", size));
  case FALSEEXP:
    return(util_strncat(str, "FALSE", size));
  case ATOM:
    if(!util_strncat(str, n->left.strtype->text, size))return(0);
    if(cdr(n)){
      char buf[20];

      sprintf(buf, "_%d", (int)cdr(n));
      return(util_strncat(str, buf, size));
    }
    return(1);
  case NUMBER:
    {
      char buf[20];

      sprintf(buf,"%d", (int)car(n));
      return(util_strncat(str, buf, size));
    }
  case DOT:
    if(!n->left.nodetype) return(sprint_node(str, size, n->right.nodetype));
    return(sprint_node(str, size, n->left.nodetype)
	   && util_strncat(str, ".", size)
	   && sprint_node(str, size, n->right.nodetype));
  case CONS:
    return(sprint_node(str,size, n->left.nodetype)
	   && (!n->right.nodetype
	       || (util_strncat(str, " , ", size)
		   && sprint_node(str, size, n->right.nodetype))));
  case ARRAY:
    return(sprint_node(str, size, n->left.nodetype)
	   && util_strncat(str, "[", size)
	   && sprint_node(str, size, n->right.nodetype)
	   && util_strncat(str, "]", size));
  case TWODOTS: op = ".."; priority = 3; arity = 1; break;
  case IMPLIES: op = "->"; priority = 4; arity = 1; break;
  case IFF: op = "<->"; priority = 4; arity = 1; break;
  case OR: op = "|"; priority = 5; arity = 1; break;
  case AND: op = "&"; priority = 6; arity = 1; break;
  case NOT: op = "!"; priority = 7; arity = 0; break;
  case EX: op = "EX "; priority = 8; arity = 0; break;
  case AX: op = "AX "; priority = 8; arity = 0; break;
  case EF: op = "EF "; priority = 8; arity = 0; break;
  case AF: op = "AF "; priority = 8; arity = 0; break;
  case EG: op = "EG "; priority = 8; arity = 0; break;
  case AG: op = "AG "; priority = 8; arity = 0; break;
  case OP_NEXT: op = " X "; priority = 8; arity = 0; break;
  case OP_GLOBAL: op = " G "; priority = 8; arity = 0; break;
  case OP_FUTURE: op = " F "; priority = 8; arity = 0; break;
  case UNTIL: op = "U"; priority = 8; p = 9; arity = 1; break;
  case EU:
    if(!util_strncat(str, "E", size)) return(0);
    op = "U"; priority = 8; p = 9; arity = 1; break;
  case AU:
    if(!util_strncat(str, "A", size)) return(0);
    op = "U"; priority = 8; p = 9; arity = 1; break;
  case EBU:
    if(!util_strncat(str, "E", size)) return(0);
    op = "BU"; priority = 8; p = 9; arity = 3; break;
  case ABU:
    if(!util_strncat(str, "A", size)) return(0);
    op = "BU"; priority = 8; p = 9; arity = 3; break;
  case EBF: op = "EBF "; priority = 8; arity = 2; break;
  case ABF: op = "ABF "; priority = 8; arity = 2; break;
  case EBG: op = "EBG "; priority = 8; arity = 2; break;
  case ABG: op = "ABG "; priority = 8; arity = 2; break;
  case MINU:
    if(!util_strncat(str, "MIN", size)) return(0);
    op = ","; priority = 8; p = 9; arity = 1; break;
  case MAXU:
    if(!util_strncat(str, "MAX", size)) return(0);
    op = ","; priority = 8; p = 9; arity = 1; break;
  case EQUAL: op = "="; priority = 9; arity = 1; break;
  case LT:    op = "<"; priority = 9; arity = 1; break;
  case GT:    op = ">"; priority = 9; arity = 1; break;
  case LE:    op = "<="; priority = 9; arity = 1; break;
  case GE:    op = ">="; priority = 9; arity = 1; break;
  case UNION: op = "union"; priority = 10; arity = 1; break;
  case SETIN: op = "in"; priority = 11; arity = 1; break;
  case MOD:   op = "mod"; priority = 12; arity = 1; break;
  case PLUS:  op = "+"; priority = 13; arity = 1; break;
  case MINUS: op = "-"; priority = 13; arity = 1; break;
  case TIMES: op = "*"; priority = 14; arity = 1; break;
  case DIVIDE: op = "/"; priority = 14; arity = 1; break;
  case NEXT:
    if (!util_strncat(str, "next", size)) return(0);
    op = ""; priority = 0; p = 1; arity = 0; break;
  default:
    internal_error("sprint_node_recur: type = %d", n->type);
  }
  if(priority < p && !util_strncat(str, "(", size)) return(0);
  switch(arity){
  case 0:
    if (!util_strncat(str, op, size))return(0);
    if (!sprint_node_recur(str, size, n->left.nodetype, priority)) return(0);
    break;
  case 1:
    if (!sprint_node_recur(str, size, n->left.nodetype,priority)) return(0);
    if (!util_strncat(str, " ", size)) return(0);
    if (!util_strncat(str, op, size)) return(0);
    if (!util_strncat(str, " ", size)) return(0);
    if (!sprint_node_recur(str, size, n->right.nodetype, priority))return(0);
    break;
  case 2:
    /* EF a..b f */
    if (!util_strncat(str, op, size)) return(0);                /* EF */
    if (!sprint_node_recur(str, size, (n->right.nodetype)->left.nodetype, priority))
                                       return(0);         /* a */
    if (!util_strncat(str, ".." ,size)) return(0);
    if (!sprint_node_recur(str, size, (n->right.nodetype)->right.nodetype, priority))
                                       return(0);         /* b */
    if (!util_strncat(str, " ", size)) return(0);
    if (!sprint_node_recur(str, size, n->left.nodetype, priority)) return(0); /* f */
    break;
  case 3:
    /* E[f BU a..b g] */
    if (!sprint_node_recur(str, size, (n->left.nodetype)->left.nodetype, priority))
                                       return(0);         /* f */
    if (!util_strncat(str, " ", size)) return(0);
    if (!util_strncat(str, op, size)) return(0);                /* BU */
    if (!util_strncat(str, " ", size)) return(0);
    if (!sprint_node_recur(str, size, (n->right.nodetype)->left.nodetype, priority))
                                       return(0);         /* a */
    if (!util_strncat(str, "..", size)) return(0);
    if (!sprint_node_recur(str, size, (n->right.nodetype)->right.nodetype, priority))
                                       return(0);         /* b */
    if (!util_strncat(str, " ", size)) return(0);
    if (!sprint_node_recur(str, size, (n->left.nodetype)->right.nodetype, priority))
                                       return(0);         /* g */
    break;
  }
  if (priority < p && !util_strncat(str, ")", size)) return(0);
  return(1);
}

