/**CHeaderFile*****************************************************************

  FileName    [node.h]

  PackageName [node]

  Synopsis    [The header file of the <tt>node</tt> package.]

  Description [The <tt>node</tt> package implements a data structure
  which offers constructs with e syntax and semantic lisp like.]

  Author      [Marco Roveri; Modified by Yuan Lu]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

  Revision    [$Id: $]

******************************************************************************/

#ifndef _node_h
#define _node_h

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#define Nil ((node_ptr)0)
#define FAILURE_NODE ((node_ptr)(-1))
#define CLOSED_NODE (node_ptr)(-2)

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/
typedef union value_ node_val;
typedef struct node node_rec;
typedef struct node * node_ptr;
typedef node_ptr (*NPFN)(node_ptr);
typedef node_ptr (*VPFN)(node_ptr);
/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Possible value that a node can assume.]

  Description [The value of a node is generic. It may be an integer, a
  pointer to a node, a pointer to a string_ structure or a pointer to
  an ADD or BDD. This in order to have a behavior lisp like, in which
  a variable may be a n integer, an atom or a list.]

  SeeAlso     [string_]

******************************************************************************/
union value_ {
  int inttype;
  struct node *nodetype;
  struct string_ * strtype;
  void * bddtype;
};

/**Struct**********************************************************************

  Synopsis    [The <tt>node</tt> data structure.]

  Description [This data structure allows the implementation of a lisp
  like s-expression.
  <ul>
  <li><b>lineno</b> It is used to store the line number of the input
      file to which the stored data refers to.
  <li><b>type</b> It indicates the kind of node we are dealing
      with. I.e. if the node is <em>CONS</em> like, or a leaf, and in
      this case which kind of leaf (<em>NUMBER</em>, <em>ATOM</em>, ...).
  <li><b>left</b> It's the left branch of the s-expression.
  <li><b>right</b> It's the left branch of the s-expression.
  <li><b>link</b> It's a pointer used in the internal hash.
  ]

******************************************************************************/
struct node {
  struct node *link;
  short int type;
  short int lineno;
  node_val left;
  node_val right;
};

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/
#define caar(_n_) car(car(_n_))
#define cadr(_n_) car(cdr(_n_))
#define cdar(_n_) cdr(car(_n_))
#define cddr(_n_) cdr(cdr(_n_))
#define node_get_type(_n_) _n_->type
#define node_get_lineno(_n_) _n_->lineno
#define node_get_lstring(_n_) _n_->left.strtype

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

void     node_init ARGS((void));
void     node_quit ARGS((void));
void     free_node ARGS((node_ptr));
void     swap_nodes ARGS((node_ptr *, node_ptr *));
void     walk ARGS((VPFN fun, node_ptr));
node_ptr new_node ARGS((int, node_ptr, node_ptr));
node_ptr cons ARGS((node_ptr, node_ptr));
node_ptr car ARGS((node_ptr));
node_ptr cdr ARGS((node_ptr));
node_ptr append ARGS((node_ptr, node_ptr));
node_ptr append_ns ARGS((node_ptr, node_ptr));
int      memberp ARGS((node_ptr, node_ptr));
node_ptr reverse ARGS((node_ptr));
node_ptr last ARGS((node_ptr));
node_ptr map ARGS((NPFN fun, node_ptr));
node_ptr even_elements ARGS((node_ptr));
node_ptr odd_elements ARGS((node_ptr));
node_ptr node_subtract ARGS((node_ptr, node_ptr));
node_ptr find_node ARGS((int, node_ptr, node_ptr));
node_ptr node_setin ARGS((node_ptr, node_ptr));
node_ptr node_equal ARGS((node_ptr, node_ptr));
int      in_list ARGS((node_ptr, node_ptr));
node_ptr find_atom ARGS((node_ptr));
void     print_sexp ARGS((FILE *, node_ptr));
void     print_node ARGS((FILE *, node_ptr));
int      sprint_node ARGS((char *, int, node_ptr));
int      print_node_atcol ARGS((FILE *, node_ptr, int));

/* Yuan Lu */
node_ptr copy_tree ARGS((node_ptr));
int log_2 ARGS((int));
int power2 ARGS((int));
int num_in_list ARGS((node_ptr));
void split_power_half ARGS((node_ptr, node_ptr*, node_ptr*));
node_ptr gen_list_integer ARGS((int, int));
node_ptr gen_list_until ARGS((node_ptr, node_ptr));
node_ptr node_cp ARGS((node_ptr));
node_ptr list_union ARGS((node_ptr, node_ptr));
int is_list_intersect ARGS((node_ptr, node_ptr));

#endif /* _node_h */
