/**CFile***********************************************************************

  FileName    [mcPlanning.c]

  PackageName [mc]

  Synopsis    [Dedicated algorithms for the verification of
  invariants on-the-fly wrt reachability analysis.]

  Description [Dedicated algorithms for the verification of
  invariants on-the-fly wrt reachability analysis.]

  SeeAlso     [mcMc.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "mcInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

#ifdef MBP
#include "Input.h"
#include "mbp_exec.h"
#include "mbp_io.h"
#endif

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

extern bdd_ptr state_variables_bdd;
extern bdd_ptr actions_bdd;
extern node_ptr all_symbols;

#ifdef MBP
extern int real_nstvars;
extern int act_nvars;
#endif

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/
#define FB_HEURISTIC
/* #define DEBUG_PLANNING */
#define FORWARD_BACKWARD_HEURISTICS 0
#define FORWARD_BACKWARD_NO_OPT 1

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_invar(node_ptr invar_expr_lst)
{
  node_ptr l;
  /* node_ptr invar_cxt = car(invar_expr_lst); */
  node_ptr invar_expr = cdr(invar_expr_lst);

  l = invar_expr;
  while (l) {
    node_ptr cur_inv = car(l);

    l = cdr(l);
    fprintf(nusmv_stdout, "Checking invariant: ");
    print_node(nusmv_stdout, cur_inv);
    fprintf(nusmv_stdout, "\n");
    check_invariant_forward(cur_inv);
  }
}

/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_invar_fb(node_ptr invar_expr_lst)
{
  node_ptr l;
  /* node_ptr invar_cxt = car(invar_expr_lst); */
  node_ptr invar_expr = cdr(invar_expr_lst);

  l = invar_expr;
  while (l) {
    node_ptr cur_inv = car(l);

    l = cdr(l);
    fprintf(nusmv_stdout, "Checking invariant: ");
    print_node(stdout, cur_inv);
#if FORWARD_BACKWARD_NO_OPT
    fprintf(nusmv_stdout, ".\n Using Forward and Backward Search.\n");
    check_invariant_forward_backward(cur_inv);
#else
    fprintf(nusmv_stdout, ".\n Using Forward and Backward OPT Search.\n");
    check_invariant_forward_backward_op(cur_inv);
#endif
  }
}

/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_invar_strong(node_ptr invar_expr_lst)
{
  node_ptr l;
  /* node_ptr invar_cxt = car(invar_expr_lst); */
  node_ptr invar_expr = cdr(invar_expr_lst);

  l = invar_expr;
  while (l) {
    node_ptr cur_inv = car(l);

    l = cdr(l);
    fprintf(nusmv_stdout, "Checking sub invariant: ");
    print_node(nusmv_stdout, cur_inv);
    fprintf(nusmv_stdout, "\n Using Strong Invariant Check\n");
    check_invariant_strong(cur_inv);
  }
}


/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_after(node_ptr after_expr_lst)
{
  node_ptr after_expr = car(cdr(after_expr_lst));
  node_ptr goal_expr  = cdr(cdr(after_expr_lst));

  temporal_projection(after_expr, goal_expr);
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void compute_plan(bdd_ptr reached_goal_bdd, node_ptr plan_list, int nstep, int flag)
{
  int i;
  node_ptr plan;
  bdd_ptr state;
  bdd_ptr zero = bdd_zero(dd_manager);

  nstep = nstep - 1;
#ifdef MBP
  fprintf(nusmv_stderr,"Size of reached goal: %d bdd nodes, %g states\n",
          bdd_size(dd_manager, reached_goal_bdd),
          bdd_count_states(dd_manager, reached_goal_bdd));
#else
  fprintf(nusmv_stderr,"Size of reached invar: %d bdd nodes, %g states\n",
          bdd_size(dd_manager, reached_goal_bdd),
          bdd_count_states(dd_manager, reached_goal_bdd));
#endif
  state = bdd_pick_one_state_rand(dd_manager, reached_goal_bdd);
  plan = cons((node_ptr)state, Nil);
  /* ignore first element of plan_list, which is the current frontier of reachable states */
  if (plan_list != Nil) plan_list = cdr(plan_list);
  for(i = nstep; (i >= 0) ; i--) {
    bdd_ptr intersection;
    bdd_ptr image = Img_ImageBwd(state);
    

    if (car(plan_list) == Nil) internal_error("plan_list == Nil: %d", i);
    intersection = bdd_and(dd_manager, image, (bdd_ptr)car(plan_list));
    if (intersection == zero) internal_error("Intersection == 0: Step %d",i);
    state = bdd_pick_one_state(dd_manager, intersection);
    plan = cons((node_ptr)state, plan);
    bdd_free(dd_manager, image);
    bdd_free(dd_manager, intersection);
    plan_list = cdr(plan_list);
  }
  {
    node_ptr l = (flag == 0) ? plan : reverse(plan);

    clear_print_hash();
    i = 0;
    while( l ){
      fprintf(nusmv_stdout, "\n********************\nState %d\n********************\n", i++);
      print_state((bdd_ptr)car(l), 1);
      l = cdr(l);
    }
    /* free the list of plan */
    l = plan;
    while( l ) {
      node_ptr m = l;

      bdd_free(dd_manager,(bdd_ptr)car(l));
      l = cdr(l);
      free_node(m);
    }
  }
  bdd_free(dd_manager, zero);
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void check_invariant_forward(node_ptr inv_expr)
{ 
  int nstep = 0;
  node_ptr plan_reach_list = Nil;
  bdd_ptr reachable_states;
  bdd_ptr target_states = bdd_zero(dd_manager);
  bdd_ptr zero = bdd_zero(dd_manager);
  
  /* f_frontier is the frontier */
  bdd_ptr f_frontier = bdd_dup(init_bdd);
  bdd_ptr goal_states = eval_spec(inv_expr,Nil);

  if (opt_verbose_level_gt(options, 0)) {
#ifdef MBP
    fprintf(nusmv_stderr,"Size of goal states: %d bdd nodes, %g states\n",
            bdd_size(dd_manager, goal_states), bdd_count_states(dd_manager, goal_states));
#else
    fprintf(nusmv_stderr,"Size of invariant states: %d bdd nodes, %g states\n",
            bdd_size(dd_manager, goal_states), bdd_count_states(dd_manager, goal_states));
#endif
    fprintf(nusmv_stderr,"Size of initial states: %d bdd nodes\n", bdd_size(dd_manager, init_bdd));
    fprintf(nusmv_stderr,"Starting to slice the space of reachable states ...\n");
  }

  plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), plan_reach_list);
  reachable_states = bdd_dup(init_bdd);
  while ((f_frontier != zero) && (target_states == zero)) {
    bdd_ptr f_prev_reachable = bdd_dup(reachable_states);
    
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "iteration %d: reachable state BDD size: %d bdd nodes, %g states; frontier: %d bdd nodes.\n",
              nstep, bdd_size(dd_manager, reachable_states),
              bdd_count_states(dd_manager, reachable_states),
              bdd_size(dd_manager, f_frontier));
    
    /* Intersect the frontier with normal assignments */
    bdd_and_accumulate(dd_manager, &f_frontier, invar_bdd);
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr,"invariant combined: %d bdd nodes\n",
              bdd_size(dd_manager, f_frontier));
    {
      bdd_ptr image = Img_ImageFwd(f_frontier);

      bdd_or_accumulate(dd_manager, &reachable_states, image);
      bdd_free(dd_manager, image);
    }
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "forward step done: %d bdd nodes\n",
              bdd_size(dd_manager, reachable_states));

    {
      bdd_ptr not_f_prev_reachable = bdd_not(dd_manager, f_prev_reachable);
      
      bdd_free(dd_manager, f_frontier);
      f_frontier = bdd_and(dd_manager, reachable_states, not_f_prev_reachable);
      bdd_free(dd_manager, not_f_prev_reachable);
    }
    plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), plan_reach_list);
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr,"new frontier computed: %d bdd nodes\n",
              bdd_size(dd_manager, f_frontier));
    bdd_free(dd_manager, target_states);
    target_states = bdd_and(dd_manager, goal_states, f_frontier);
    if (opt_verbose_level_gt(options, 0)) {
      if (target_states != zero) {
#ifdef MBP
        fprintf(nusmv_stderr, "Goal set reached at step %d: %d bdd nodes, %g states\n",
                nstep, bdd_size(dd_manager, target_states), bdd_count_states(dd_manager, target_states));
#else
        fprintf(nusmv_stderr, "States not satisfying invariant reached at step %d: %d bdd nodes, %g states\n",
                nstep, bdd_size(dd_manager, target_states), bdd_count_states(dd_manager, target_states));

#endif
      }
      else {
#ifdef MBP
        fprintf(nusmv_stderr, "Goal not reached at step %d\n", nstep);
#else
        fprintf(nusmv_stderr, "States satisfying invariants not reached at step %d\n", nstep);
#endif
      }
    }
    bdd_free(dd_manager, f_prev_reachable);
#ifdef DEBUG_PLANNING
    assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
    nstep++;
  }
  if (target_states != zero)
    compute_plan(target_states, plan_reach_list, nstep, 0);
  else {
#ifdef MBP
    fprintf(nusmv_stdout, "Fix point found. Goal unreachable\n");
#else
    fprintf(nusmv_stdout, "The invariant: ");
    print_node(nusmv_stdout, inv_expr);
    fprintf(nusmv_stdout, " is satisfied.\n");
#endif
  }
  bdd_free(dd_manager, target_states);
  bdd_free(dd_manager, f_frontier);
  walk_dd(dd_manager, (void (*)())bdd_free, plan_reach_list);
  bdd_free(dd_manager, zero);
  bdd_free(dd_manager, reachable_states);
  {
    node_ptr l = plan_reach_list;

    while (l) {
      node_ptr m = l;

      l = cdr(l);
      free_node(m);
    }
  }
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void check_invariant_forward_opt(node_ptr inv_expr)
{ 
  int nstep = 0;
  node_ptr plan_reach_list = Nil;
  bdd_ptr reachable_states;
  bdd_ptr target_states = bdd_zero(dd_manager);
  bdd_ptr zero = bdd_zero(dd_manager);
  /* f_frontier is the frontier */
  bdd_ptr f_frontier = bdd_dup(init_bdd);
  bdd_ptr goal_states = eval_spec(inv_expr,Nil);

  if (opt_verbose_level_gt(options, 0)) {
#ifdef MBP
    fprintf(nusmv_stderr,"Size of goal states: %d bdd nodes, %g states.\n",
            bdd_size(dd_manager, goal_states), bdd_count_states(dd_manager, goal_states));
#else
    fprintf(nusmv_stderr,"Size of invariant states: %d bdd nodes, %g states\n",
            bdd_size(dd_manager, goal_states), bdd_count_states(dd_manager, goal_states));
#endif
    fprintf(nusmv_stderr,"Size of initial states: %d bdd nodes.\n", bdd_size(dd_manager, init_bdd));
    fprintf(nusmv_stderr,"Starting to slice the space of reachable states ...\n");
  }

  plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), plan_reach_list);
  reachable_states = bdd_dup(init_bdd);
  while ((f_frontier != zero) && (target_states == zero)) {
    bdd_ptr f_prev_reachable = bdd_dup(reachable_states);
    
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr,"iteration %d: BDD size: %d bdd nodes, %g states; frontier size = %d bdd nodes.\n",
              nstep, bdd_size(dd_manager, reachable_states),
              bdd_count_states(dd_manager, reachable_states),
              bdd_size(dd_manager, f_frontier));
    
    /* Intersect the frontier with normal assignments */
    bdd_and_accumulate(dd_manager, &f_frontier, invar_bdd);
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr,"invariant combined: %d bdd nodes\n",
              bdd_size(dd_manager, f_frontier));
    {
      bdd_ptr not_f_prev_reachable = bdd_not(dd_manager, f_prev_reachable);
      bdd_ptr image = Img_ImageFwdMonolithicOpt(f_frontier, f_prev_reachable, not_f_prev_reachable);

      bdd_or_accumulate(dd_manager, &reachable_states, image);
      bdd_free(dd_manager, f_frontier);
      f_frontier = bdd_and(dd_manager, reachable_states, not_f_prev_reachable);
      bdd_free(dd_manager, not_f_prev_reachable);
      bdd_free(dd_manager, image);
    }
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "forward step done: %d bdd nodes.\n",
              bdd_size(dd_manager, reachable_states));

    plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), plan_reach_list);
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr,"new frontier computed: %d bdd nodes.\n",
              bdd_size(dd_manager, f_frontier));
    bdd_free(dd_manager, target_states);
    target_states = bdd_and(dd_manager, goal_states, f_frontier);
    if (opt_verbose_level_gt(options, 0)) {
      if (target_states != zero) {
#ifdef MBP
        fprintf(nusmv_stderr, "Goal set reached at step %d: %d bdd nodes, %g states.\n",
                nstep, bdd_size(dd_manager, target_states), bdd_count_states(dd_manager, target_states));
#else
        fprintf(nusmv_stderr, "States not satisfying invariant reached at step %d: %d bdd nodes, %g states\n",
                nstep, bdd_size(dd_manager, target_states), bdd_count_states(dd_manager, target_states));

#endif
       }
      else {
#ifdef MBP
        fprintf(nusmv_stderr, "Goal not reached at step %d\n", nstep);
#else
        fprintf(nusmv_stderr, "States satisfying invariants not reached at step %d\n", nstep);
#endif
      }
    }
    bdd_free(dd_manager, f_prev_reachable);
#ifdef DEBUG_PLANNING
    assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
    nstep++;
  }
  if (target_states != zero)
    compute_plan(target_states, plan_reach_list, nstep, 0);
  else { 
#ifdef MBP
    fprintf(nusmv_stderr, "Fix point found. Goal unreachable\n");
#else
    fprintf(nusmv_stdout, "The invariant: ");
    print_node(nusmv_stdout, inv_expr);
    fprintf(nusmv_stdout, " is satisfied.\n");
#endif
  }
  bdd_free(dd_manager, target_states);
  bdd_free(dd_manager, f_frontier);
  walk_dd(dd_manager, (void (*)())bdd_free, plan_reach_list);
  bdd_free(dd_manager, zero);
  bdd_free(dd_manager, reachable_states);
  {
    node_ptr l = plan_reach_list;

    while (l) {
      node_ptr m = l;

      l = cdr(l);
      free_node(m);
    }
  }
}


/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void compute_plan_fb(bdd_ptr target_states,
                     bdd_ptr f_target, node_ptr f_plan_reach_list, int f_step,
                     bdd_ptr b_target, node_ptr b_plan_reach_list, int b_step)
{
  if (f_step != 0) compute_plan(target_states, f_plan_reach_list, f_step, 0);
  if (b_step != 0) compute_plan(target_states, b_plan_reach_list, b_step, 1);
}


/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void check_invariant_forward_backward(node_ptr inv_expr)
{ 
  int f_step = 0;
  int b_step = 0;
  int turn = 1;
  node_ptr f_plan_reach_list = Nil;
  node_ptr b_plan_reach_list = Nil;
  bdd_ptr zero = bdd_zero(dd_manager);
  /* computes the BDD representing the invar to check */
  bdd_ptr f_target = eval_spec(inv_expr,Nil);
  bdd_ptr b_target = bdd_dup(init_bdd);
  bdd_ptr f_cur_reachable = bdd_dup(b_target);
  bdd_ptr b_cur_reachable = bdd_dup(f_target);
  bdd_ptr f_prev_reachable = bdd_dup(f_cur_reachable);
  bdd_ptr b_prev_reachable = bdd_dup(b_cur_reachable);
  bdd_ptr f_frontier = bdd_dup(f_cur_reachable);
  bdd_ptr b_frontier = bdd_dup(b_cur_reachable);
  bdd_ptr target_states = bdd_and(dd_manager, f_target, b_target);

  f_plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), f_plan_reach_list);
  b_plan_reach_list = cons((node_ptr)bdd_dup(b_frontier), b_plan_reach_list);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "size of target sets: forward = %d bdd nodes, backward = %d bdd nodes.\n",
	    bdd_size(dd_manager, f_target), bdd_size(dd_manager, b_target));
    fprintf(nusmv_stderr, "Starting search ...\n");
  }
  
  while ((f_frontier != zero) && (b_frontier != zero) && (target_states == zero)) {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Iteration %d, %d forward, %d backward.\n",
              f_step + b_step, f_step, b_step);
      fprintf(nusmv_stderr, "forward frontier size = %d bdd nodes, %g states.\n",
              bdd_size(dd_manager, f_frontier),
              bdd_count_states(dd_manager, f_frontier));
      fprintf(nusmv_stderr, "backward frontier size = %d bdd nodes, %g states.\n",
              bdd_size(dd_manager, b_frontier),
              bdd_count_states(dd_manager, b_frontier));
    }
#ifndef FB_HEURISTIC
    if (FORWARD_BACKWARD_HEURISTICS == 1) { /* Step forward */
#else
    if /* (turn == 1) */
      (bdd_size(dd_manager, f_frontier) < bdd_size(dd_manager, b_frontier)) { 
      turn = 0;
#endif
      {
        bdd_ptr image = Img_ImageFwd(f_frontier);
 
        bdd_free(dd_manager, f_prev_reachable);
        f_prev_reachable = bdd_dup(f_cur_reachable);
        bdd_or_accumulate(dd_manager, &f_cur_reachable, image);
        bdd_free(dd_manager, image);
      }

      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr, "forward step done: %d bdd nodes\n",
                bdd_size(dd_manager, f_cur_reachable));
      {
        bdd_ptr not_f_prev_reachable = bdd_not(dd_manager, f_prev_reachable);

        f_frontier = bdd_and(dd_manager, f_cur_reachable, not_f_prev_reachable);
        bdd_free(dd_manager, not_f_prev_reachable);
      }

      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"new frontier computed: %d bdd nodes.\n",
                bdd_size(dd_manager, f_frontier));
      bdd_and_accumulate(dd_manager, &f_frontier, invar_bdd);
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"frontier simplification with invariants, new size = %d bdd nodes.\n",
                bdd_size(dd_manager, f_frontier));
      bdd_free(dd_manager, b_target);
      b_target = bdd_dup(f_frontier);
      bdd_free(dd_manager, target_states);
      target_states = bdd_and(dd_manager, f_frontier, f_target);
      f_plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), f_plan_reach_list);
      if (opt_verbose_level_gt(options, 0)) {
        if (target_states == zero) {
#ifdef MBP
          fprintf(nusmv_stderr, "Goal not reached at iteration %d forward, %d backward.\n", f_step, b_step);
#else
          fprintf(nusmv_stderr, "States satisfying invariant set reached at iteration %d forward, %d backward.\n", f_step, b_step);

#endif
        } else {
#ifdef MBP
          fprintf(nusmv_stderr, "Goal set reached at step %d, size = %d bdd nodes.", f_step + b_step, bdd_size(dd_manager, target_states));
#else
          fprintf(nusmv_stderr, "States not satisfying invariant set reached at step %d, size = %d bdd nodes.", f_step + b_step, bdd_size(dd_manager, target_states));
#endif
        }
      }
      f_step++;
#ifdef DEBUG_PLANNING
      assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
    } else {/* Step backward */
#ifdef FB_HEURISTIC
      turn = 1;
#endif
      {
        bdd_ptr image = Img_ImageFwd(b_frontier);

        bdd_free(dd_manager, b_prev_reachable);
        b_prev_reachable = bdd_dup(b_cur_reachable);
        bdd_or_accumulate(dd_manager, &b_cur_reachable, image);
        bdd_free(dd_manager, image);
      }

      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr, "backward step done, size = %d bdd nodes.\n",
                bdd_size(dd_manager, b_cur_reachable));
      {
        bdd_ptr not_b_prev_reachable = bdd_not(dd_manager, b_prev_reachable);

        b_frontier = bdd_and(dd_manager, b_cur_reachable, not_b_prev_reachable);
        bdd_free(dd_manager, not_b_prev_reachable);
      }

      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"new frontier computed, size = %d bdd nodes.\n",
                bdd_size(dd_manager, b_frontier));
      bdd_and_accumulate(dd_manager, &b_frontier, invar_bdd);
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr, "frontier simplification with invariants, new size = %d bdd nodes.\n",
                bdd_size(dd_manager, b_frontier));

      bdd_free(dd_manager, f_target);
      f_target = bdd_dup(b_frontier);
      bdd_free(dd_manager, target_states);
      target_states = bdd_and(dd_manager, b_frontier, b_target);
      b_plan_reach_list = cons((node_ptr)bdd_dup(b_frontier), b_plan_reach_list);
      if (opt_verbose_level_gt(options, 0)) {
        if (target_states == zero) {
#ifdef MBP
          fprintf(nusmv_stderr, "Goal not reached at iteration %d forward, %d backward\n", f_step, b_step);
#else
          fprintf(nusmv_stderr, "States satisfying invariant set reached at iteration %d forward, %d backward\n", f_step, b_step);
#endif
        } else {
#ifdef MBP
          fprintf(nusmv_stderr, "Goal set reached at step %d, size = %d bdd nodes.\n", f_step + b_step, bdd_size(dd_manager, target_states));
#else
          fprintf(nusmv_stderr, "States not satisfying invariant set reached at step %d, size = %d bdd nodes.\n", f_step + b_step, bdd_size(dd_manager, target_states));
#endif
        }
      }
      b_step++;
#ifdef DEBUG_PLANNING
      assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
    }
  }
  if ((f_frontier == 0) || (b_frontier == zero)) {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "\n****************************************\n");
#ifdef MBP
      fprintf(nusmv_stderr, "Fix point found, goal is unreachable\n");
#else
      fprintf(nusmv_stderr, "The invariant is satisfied.\n");
#endif
      fprintf(nusmv_stderr, "****************************************\n");
    }
  } else {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "\n****************************************\n");
#ifdef MBP
      fprintf(nusmv_stderr, "The plan is:\n");
#else
      fprintf(nusmv_stderr, "The counterexample is:\n");
#endif
      fprintf(nusmv_stderr, "****************************************\n");
    }
    compute_plan_fb(target_states, f_target, f_plan_reach_list, f_step, b_target, b_plan_reach_list, b_step);
  }
  walk_dd(dd_manager, (void (*)())bdd_free, f_plan_reach_list);
  walk_dd(dd_manager, (void (*)())bdd_free, b_plan_reach_list);
  
  bdd_free(dd_manager, f_target);
  bdd_free(dd_manager, b_target);
  bdd_free(dd_manager, f_cur_reachable);
  bdd_free(dd_manager, b_cur_reachable);
  bdd_free(dd_manager, f_prev_reachable);
  bdd_free(dd_manager, b_prev_reachable);
  bdd_free(dd_manager, f_frontier);
  bdd_free(dd_manager, b_frontier);
  bdd_free(dd_manager, target_states);  
  bdd_free(dd_manager, zero);
  {
    node_ptr l = f_plan_reach_list;

    while (l) {
      node_ptr m = l;

      l = cdr(l);
      free_node(m);
    }
  }
  {
    node_ptr l = b_plan_reach_list;

    while (l) {
      node_ptr m = l;

      l = cdr(l);
      free_node(m);
    }
  }
#ifdef DEBUG_PLANNING
  assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void check_invariant_forward_backward_op(node_ptr inv_expr)
{ 
  int f_step = 0;
  int b_step = 0;
  int turn = 1;
  node_ptr f_plan_reach_list = Nil;
  node_ptr b_plan_reach_list = Nil;
  bdd_ptr zero = bdd_zero(dd_manager);
  /* computes the BDD representing the invar to check */
  bdd_ptr f_target = eval_spec(inv_expr,Nil);
  bdd_ptr b_target = bdd_dup(init_bdd);
  bdd_ptr f_cur_reachable = bdd_dup(b_target);
  bdd_ptr b_cur_reachable = bdd_dup(f_target);
  bdd_ptr f_prev_reachable = bdd_dup(f_cur_reachable);
  bdd_ptr b_prev_reachable = bdd_dup(b_cur_reachable);
  bdd_ptr f_frontier = bdd_dup(f_cur_reachable);
  bdd_ptr b_frontier = bdd_dup(b_cur_reachable);
  bdd_ptr target_states = bdd_and(dd_manager, f_target, b_target);

  f_plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), f_plan_reach_list);
  b_plan_reach_list = cons((node_ptr)bdd_dup(b_frontier), b_plan_reach_list);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Size of target sets: forward = %d bdd nodes, backward = %d bdd nodes.\n",
	    bdd_size(dd_manager, f_target), bdd_size(dd_manager, b_target));
    fprintf(nusmv_stderr, "Starting search ...\n");
  }
  
  while ((f_frontier != zero) && (b_frontier != zero) && (target_states == zero)) {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Iteration %d, %d forward, %d backward.\n",
              f_step + b_step, f_step, b_step);
      fprintf(nusmv_stderr, "forward frontier size = %d bdd nodes, %g states.\n",
              bdd_size(dd_manager, f_frontier),
              bdd_count_states(dd_manager, f_frontier));
      fprintf(nusmv_stderr, "backward frontier size = %d bdd nodes, %g states.\n",
              bdd_size(dd_manager, b_frontier),
              bdd_count_states(dd_manager, b_frontier));
    }
#ifndef FB_HEURISTIC
    if (FORWARD_BACKWARD_HEURISTICS == 1) { /* Step forward */
#else
    if /* (turn == 1) { */
      (bdd_size(dd_manager, f_frontier) < bdd_size(dd_manager, b_frontier)) { 
      turn = 0;
#endif
      {
        bdd_ptr not_f_prev_reachable, image;
 
        bdd_free(dd_manager, f_prev_reachable);
        f_prev_reachable = bdd_dup(f_cur_reachable);
        not_f_prev_reachable = bdd_not(dd_manager, f_prev_reachable);
        image = Img_ImageFwdMonolithicOpt(f_frontier, f_prev_reachable, not_f_prev_reachable); 
        bdd_or_accumulate(dd_manager, &f_cur_reachable, image);
        bdd_free(dd_manager, f_frontier);
        f_frontier = bdd_and(dd_manager, f_cur_reachable, not_f_prev_reachable);
        bdd_free(dd_manager, not_f_prev_reachable);
        bdd_free(dd_manager, image);
      }

      if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr, "forward step done, size = %d bdd nodes.\n",
                           bdd_size(dd_manager, f_cur_reachable));
      if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr,"new frontier computed, size = %d bdd nodes.\n",
                           bdd_size(dd_manager, f_frontier));
      bdd_and_accumulate(dd_manager, &f_frontier, invar_bdd);
      if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr,"frontier simplification with invariants, new size = %d bdd nodes.\n",
                           bdd_size(dd_manager, f_frontier));
      bdd_free(dd_manager, b_target);
      b_target = bdd_dup(f_frontier);
      bdd_free(dd_manager, target_states);
      target_states = bdd_and(dd_manager, f_frontier, f_target);
      f_plan_reach_list = cons((node_ptr)bdd_dup(f_frontier), f_plan_reach_list);
      if (opt_verbose_level_gt(options, 0)) {
        if (target_states == zero) {
          fprintf(nusmv_stderr, "Goal not reached at iteration %d forward, %d backward.\n", f_step, b_step);
        } else {
          fprintf(nusmv_stderr, "Goal set reached at step %d, size = %d bdd nodes.", f_step + b_step,
                  bdd_size(dd_manager, target_states));
        }
      }
      f_step++;
#ifdef DEBUG_PLANNING
      assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
    } else {/* Step backward */
#ifdef FB_HEURISTIC
      turn = 1;
#endif
      {
        bdd_ptr not_b_prev_reachable, image;

        bdd_free(dd_manager, b_prev_reachable);
        b_prev_reachable = bdd_dup(b_cur_reachable);
        not_b_prev_reachable = bdd_not(dd_manager, b_prev_reachable);
        image = Img_ImageFwdMonolithicOpt(b_frontier, b_prev_reachable, not_b_prev_reachable);
        bdd_or_accumulate(dd_manager, &b_cur_reachable, image);
        bdd_free(dd_manager, b_frontier);
        b_frontier = bdd_and(dd_manager, b_cur_reachable, not_b_prev_reachable);
        bdd_free(dd_manager, not_b_prev_reachable);
        bdd_free(dd_manager, image);
      }

      if (opt_verbose_level_gt(options, 0)) {
        fprintf(nusmv_stderr, "backward step done, size = %d bdd nodes.\n",
                bdd_size(dd_manager, b_cur_reachable));
        fprintf(nusmv_stderr,"new frontier computed, size = %d bdd nodes. \n",
                bdd_size(dd_manager, b_frontier));
      }
      bdd_and_accumulate(dd_manager, &b_frontier, invar_bdd);
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"frontier simplification with invariants, new size = %d bdd nodes.\n",
                bdd_size(dd_manager, b_frontier));

      bdd_free(dd_manager, f_target);
      f_target = bdd_dup(b_frontier);
      bdd_free(dd_manager, target_states);
      target_states = bdd_and(dd_manager, b_frontier, b_target);
      b_plan_reach_list = cons((node_ptr)bdd_dup(b_frontier), b_plan_reach_list);
      if (opt_verbose_level_gt(options, 0)) {
        if (target_states == zero) {
          fprintf(nusmv_stderr, "Goal not reached at iteration %d forward, %d backward\n", f_step, b_step);
        } else {
          fprintf(nusmv_stderr, "Goal set reached at step %d, size = %d bdd nodes.\n", f_step + b_step,
                  bdd_size(dd_manager, target_states));
        }
      }
      b_step++;
#ifdef DEBUG_PLANNING
      assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
    }
  }
  if ((f_frontier == 0) || (b_frontier == zero)) {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "\n****************************************\n");
      fprintf(nusmv_stderr, "Fix point found, goal is unreachable\n");
      fprintf(nusmv_stderr, "****************************************\n");
    }
  } else {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "\n****************************************\n");
      fprintf(nusmv_stderr, "The plan is:\n");
      fprintf(nusmv_stderr, "****************************************\n");
    }
    compute_plan_fb(target_states, f_target, f_plan_reach_list, f_step, b_target, b_plan_reach_list, b_step);
  }
  walk_dd(dd_manager, (void (*)())bdd_free, f_plan_reach_list);
  walk_dd(dd_manager, (void (*)())bdd_free, b_plan_reach_list);
  
  bdd_free(dd_manager, f_target);
  bdd_free(dd_manager, b_target);
  bdd_free(dd_manager, f_cur_reachable);
  bdd_free(dd_manager, b_cur_reachable);
  bdd_free(dd_manager, f_prev_reachable);
  bdd_free(dd_manager, b_prev_reachable);
  bdd_free(dd_manager, f_frontier);
  bdd_free(dd_manager, b_frontier);
  bdd_free(dd_manager, target_states);  
  bdd_free(dd_manager, zero);
  {
    node_ptr l = f_plan_reach_list;

    while (l) {
      node_ptr m = l;

      l = cdr(l);
      free_node(m);
    }
  }
  {
    node_ptr l = b_plan_reach_list;

    while (l) {
      node_ptr m = l;

      l = cdr(l);
      free_node(m);
    }
  }
#ifdef DEBUG_PLANNING
  assert(Cudd_DebugCheck(dd_manager) == 0);
#endif
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void temporal_projection(node_ptr dl_expr, node_ptr goal_expr)
{
  bdd_ptr current_states = bdd_dup(init_bdd);
  bdd_ptr goal_bdd = eval_spec(goal_expr, Nil);
  
  eval_after(dl_expr, &current_states);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "The number of states after the execution of the plan is: %g\n", bdd_count_states(dd_manager, current_states));
  }
  
  {
    bdd_ptr intersection_state_goal = bdd_and(dd_manager, goal_bdd, current_states);

    if (intersection_state_goal == current_states) {
      fprintf(nusmv_stdout, "****************************************\n");
      print_node(nusmv_stdout, goal_expr);
      fprintf(nusmv_stdout, " holds after plan execution!\n");
      fprintf(nusmv_stdout, "****************************************\n");
    }
    else {
      fprintf(nusmv_stdout, "****************************************\n");
      print_node(nusmv_stdout, goal_expr);
      fprintf(nusmv_stdout, " does not hold after plan execution!\n");
      fprintf(nusmv_stdout, "****************************************\n");
    }
    bdd_free(dd_manager, intersection_state_goal);
  }
  bdd_free(dd_manager, goal_bdd);
  bdd_free(dd_manager, current_states);
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void eval_after(node_ptr dl_expr, bdd_ptr * cur_states)
{
  if (dl_expr == Nil) return;
  switch(node_get_type(dl_expr)){
  case DL_ATOM:
    {
      node_ptr actions_expr = car(dl_expr);

      if (opt_verbose_level_gt(options, 0)) {
        fprintf(nusmv_stderr, "evaluating actions: ");
        print_node(nusmv_stderr, actions_expr);
        fprintf(nusmv_stderr, "\nThe number of states before actions execution is: %g\n",
                bdd_count_states(dd_manager, *cur_states));
      }
      {
        bdd_ptr actions = eval_spec(actions_expr, Nil);
        bdd_ptr cur_states_and_actions = bdd_and(dd_manager, actions, *cur_states);
        
        bdd_and_accumulate(dd_manager, &cur_states_and_actions, invar_bdd);
        bdd_free(dd_manager, *cur_states);
        *cur_states = Img_ImageFwd(cur_states_and_actions);
        if (opt_verbose_level_gt(options, 0))
          fprintf(nusmv_stderr, "The number of states after actions execution is: %g\n",
                  bdd_count_states(dd_manager, *cur_states));
        bdd_free(dd_manager, actions);
        bdd_free(dd_manager, cur_states_and_actions);
      }
    }
    break;
  case SEMI:
    eval_after(car(dl_expr), cur_states); /* Side effects on cur_states */
    eval_after(cdr(dl_expr), cur_states);
    break;
  case UU: 
    {
      bdd_ptr cur_states_1 = bdd_dup(*cur_states);
      bdd_ptr cur_states_2 = bdd_dup(*cur_states);

      eval_after(car(dl_expr), &cur_states_1);
      eval_after(cdr(dl_expr), &cur_states_2);
      bdd_free(dd_manager, *cur_states);
      *cur_states = bdd_or(dd_manager, cur_states_1, cur_states_2);
      bdd_free(dd_manager, cur_states_1);
      bdd_free(dd_manager, cur_states_2);
    }
    break;
  case IFTHENELSE: 
    {
      node_ptr condition_expr         = car(dl_expr);
      node_ptr then_expr              = car(cdr(dl_expr));
      node_ptr else_expr              = cdr(cdr(dl_expr));
      bdd_ptr  condition_bdd          = eval_spec(condition_expr, Nil);
      bdd_ptr  cur_states_and_cond     = bdd_and(dd_manager, *cur_states, condition_bdd);
      bdd_ptr  not_condition_bdd      = bdd_not(dd_manager, condition_bdd);
      bdd_ptr  cur_states_and_not_cond = bdd_and(dd_manager, *cur_states, not_condition_bdd);
      
      eval_after(then_expr, &cur_states_and_cond);
      eval_after(else_expr, &cur_states_and_not_cond);
      bdd_free(dd_manager, *cur_states);
      *cur_states = bdd_or(dd_manager, cur_states_and_not_cond, cur_states_and_cond);
      bdd_free(dd_manager, condition_bdd);
      bdd_free(dd_manager, cur_states_and_cond);
      bdd_free(dd_manager, not_condition_bdd);
      bdd_free(dd_manager, cur_states_and_not_cond);
    }
    break;
  case STAR: 
    {
      node_ptr dl_expr_body = car(dl_expr);
      bdd_ptr  frontier   = bdd_dup(*cur_states);
      bdd_ptr  zero       = bdd_zero(dd_manager);
      
      while( frontier != zero ) {
        bdd_ptr prev_states = bdd_dup(*cur_states);
        bdd_ptr new_states  = bdd_dup(*cur_states);

        eval_after(dl_expr_body, &new_states);

        bdd_or_accumulate(dd_manager, cur_states, new_states);
        bdd_free(dd_manager, new_states);
        {
          bdd_ptr not_prev_states = bdd_not(dd_manager, prev_states);

          bdd_free(dd_manager, frontier);
          frontier = bdd_and(dd_manager, not_prev_states, *cur_states);
          bdd_free(dd_manager, not_prev_states);
        }
      }
      bdd_free(dd_manager, frontier);
      bdd_free(dd_manager, zero);
    }
    break;
  default:
    internal_error("Action composer %d not defined\n", node_get_type(dl_expr));
    return;
  }
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr ax(bdd_ptr f)
{
  extern bdd_ptr ex(bdd_ptr);
  bdd_ptr result;
  bdd_ptr not_f = bdd_not(dd_manager, f);
  bdd_ptr tmp   = ex(not_f);

  result = bdd_not(dd_manager, tmp);
  bdd_free(dd_manager, tmp);
  bdd_free(dd_manager, not_f);
  return(result);
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void check_invariant_strong(node_ptr goal_expr)
{
  int i            = 0;
  bdd_ptr Y        = eval_spec(goal_expr, Nil);
  bdd_ptr old_Y    = bdd_zero(dd_manager);
  int goal_reached = 0;

  if (opt_verbose_level_gt(options, 0))
#ifdef MBP
    fprintf(nusmv_stderr, "Size of Goal = %g states, %d bdd nodes.\n",
            bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
#else
    fprintf(nusmv_stderr, "Size of invariant = %g states, %d bdd nodes.\n",
            bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
#endif
  while((old_Y != Y) && (!goal_reached)){
    i++;
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr,"iteration %d: size of Y%d = %g states, %d BDD nodes\n",
              i, i, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    bdd_free(dd_manager, old_Y);
    old_Y = bdd_dup(Y);
    {
      bdd_ptr tmp = ax(Y);

      bdd_or_accumulate(dd_manager, &Y, tmp);
      bdd_free(dd_manager, tmp);
    }
    {
      bdd_ptr intersection_with_init = bdd_and(dd_manager, init_bdd, Y);

      goal_reached = (int)(intersection_with_init == init_bdd);
      bdd_free(dd_manager, intersection_with_init);
    }
  }
  if (goal_reached) {
#ifdef MBP
    fprintf(nusmv_stdout, "Goal found.\n");
    fprintf(nusmv_stdout, "The maximum length of the plan is %d.\n",i);
#else
    fprintf(nusmv_stdout, "States not satisfying invariant found.\n");
    fprintf(nusmv_stdout, "The maximum length of the counterexample is %d.\n",i);
#endif
  }
  else {
#ifdef MBP
    fprintf(nusmv_stdout, "Fix Point Reached. Goal is unreachable.\n");
#else
    fprintf(nusmv_stdout, "The invariant is satisfied.\n");
#endif
  }
}

#ifdef MBP
/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr strong_pre_image(bdd_ptr f)
{
  extern bdd_ptr ex(bdd_ptr);
  extern bdd_ptr states_with_successor_bdd;
  bdd_ptr result;
  bdd_ptr not_f   = bdd_not(dd_manager, f);
  bdd_ptr tmp_1   = ex(not_f);
  bdd_ptr tmp_2   = bdd_not(dd_manager, tmp_1);
 
  result = bdd_and(dd_manager, tmp_2, states_with_successor_bdd);
  bdd_free(dd_manager, not_f);
  bdd_free(dd_manager, tmp_1);
  bdd_free(dd_manager, tmp_2);
  return(result);
}


/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr prune_states(bdd_ptr pimg, bdd_ptr acc)
{
  bdd_ptr result;
  bdd_ptr not_acc = bdd_not(dd_manager, acc);
  
  result = bdd_and(dd_manager, pimg, not_acc);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "PreImage: size = %d bdd nodes.\n", bdd_size(dd_manager, pimg));
    fprintf(nusmv_stderr, "Pruned PreImage: size = %d bdd nodes.\n", bdd_size(dd_manager, result));
  }
  bdd_free(dd_manager, not_acc);
  return(result);
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void check_invariant_strong1(node_ptr goal_expr)
{
  int i                = 1;
  int goal_reached     = 0;
  bdd_ptr Acc          = eval_spec(goal_expr, Nil);
  bdd_ptr old_Acc      = bdd_zero(dd_manager);
  bdd_ptr zero         = bdd_zero(dd_manager);
  bdd_ptr State_Action = bdd_zero(dd_manager);

  if (opt_verbose_level_gt(options, 5)){
    int j;
    extern int real_nstvars;
    extern node_ptr state_variables;
    bdd_ptr * InitArray;

    Cudd_ExtractMinterm(dd_manager, init_bdd, real_nstvars, &InitArray);
    fprintf(nusmv_stderr, "The set of initial states is:\n");
    for(j = 0; InitArray[j] != NULL; j++) {
      printf("%d = <",j);
      print_action_or_state(InitArray[j], state_variables);
      printf(">\n");
    }
  }
  if (opt_verbose_level_gt(options, 0)) {
    extern int real_nstvars;
    extern int act_nvars;

    fprintf(nusmv_stderr, "Size of Goal = %g states, %d bdd nodes.\n",
            bdd_count_states(dd_manager, Acc), bdd_size(dd_manager, Acc));
    indent(nusmv_stderr);
    fprintf(nusmv_stderr,"iteration %d: size of State_Action[%d] = %g states, %d BDD nodes\n",
            i, i-1, bdd_count_minterm(dd_manager, State_Action, real_nstvars + act_nvars),
            bdd_size(dd_manager, State_Action));
    fprintf(nusmv_stderr,"iteration %d: size of Acc[%d] = %g states, %d BDD nodes\n",
            i, i, bdd_count_states(dd_manager, Acc), bdd_size(dd_manager, Acc));
  }
  {
    bdd_ptr intersection_with_init = bdd_and(dd_manager, init_bdd, Acc);

    goal_reached = (int)(intersection_with_init == init_bdd);
    bdd_free(dd_manager, intersection_with_init);
  }
  while((old_Acc != Acc) && (!goal_reached)){
    bdd_free(dd_manager, old_Acc);
    old_Acc = bdd_dup(Acc);
    {
      bdd_ptr PreImage = strong_pre_image(Acc);
      bdd_ptr Pruned_PreImage = prune_states(PreImage, Acc);
      bdd_ptr Projected_PreImage = bdd_forsome(dd_manager, PreImage, actions_bdd);

      bdd_or_accumulate(dd_manager, &State_Action, Pruned_PreImage);
      bdd_or_accumulate(dd_manager, &Acc, Projected_PreImage);
      if (opt_verbose_level_gt(options, 0)) {
        extern node_ptr all_symbols;
        extern int real_nstvars;
        extern int act_nvars;

        if (opt_verbose_level_gt(options, 6)) {
          bdd_ptr * SAi;
          int j;

          Cudd_ExtractMinterm(dd_manager, Pruned_PreImage, real_nstvars + act_nvars, &SAi);
          for(j = 0; SAi[j] != NULL; j++) {
            printf("%d = <",j);
            print_action_or_state(SAi[j], all_symbols);
            printf(">\n");
          }
        }
        fprintf(nusmv_stderr, "Iteration %d: State_Action: size = %d bdd nodes, %g states\n",
                i, bdd_size(dd_manager, State_Action),
                bdd_count_minterm(dd_manager, State_Action, real_nstvars+act_nvars));
        fprintf(nusmv_stderr, "Iteration %d: Acc: size = %d bdd nodes, %g States\n",
                i, bdd_size(dd_manager, Acc), bdd_count_states(dd_manager, Acc));
      }
      bdd_free(dd_manager, PreImage);
      bdd_free(dd_manager, Pruned_PreImage);
      bdd_free(dd_manager, Projected_PreImage);
    }
    {
      bdd_ptr intersection_with_init = bdd_and(dd_manager, init_bdd, Acc);

      goal_reached = (int)(intersection_with_init == init_bdd);
      bdd_free(dd_manager, intersection_with_init);
    }
    i++;
    if (opt_verbose_level_gt(options, 0)) {
      extern int real_nstvars;
      extern int act_nvars;

      indent(nusmv_stderr);
      fprintf(nusmv_stderr,"Iteration %d: size of State_Action[%d] = %g states, %d BDD nodes\n",
              i, i-1, bdd_count_minterm(dd_manager, State_Action, real_nstvars + act_nvars),
              bdd_size(dd_manager, State_Action));
      fprintf(nusmv_stderr,"Iteration %d: size of Acc[%d] = %g states, %d BDD nodes\n",
              i, i, bdd_count_states(dd_manager, Acc), bdd_size(dd_manager, Acc));
    }
  }
  if (goal_reached) {
    fprintf(nusmv_stdout, "Goal found.\n");
    fprintf(nusmv_stdout, "The maximum length of the plan is %d.\n",i-1);
    if (opt_verbose_level_gt(options, 2)) {
      int i;
      DdNode **StateActArray;

      Cudd_ExtractMinterm(dd_manager, State_Action, real_nstvars + act_nvars, &StateActArray);
      fprintf(nusmv_stdout, "****************************************\n");
      fprintf(nusmv_stdout, "Starting Extended Plan Executor\n");
      fprintf(nusmv_stdout, "Press Carriage Return to go on:");
      getchar();
      exec_extended_plan(init_bdd, StateActArray, 0);
      fprintf(nusmv_stdout, "****************************************\n");
  }

  }
  else {
    fprintf(nusmv_stdout, "Fix Point Reached. Goal is unreachable.\n");
  }
}

#endif

/**Function********************************************************************

  Synopsis           [Print an invariant specification]

  Description        [Print an invariant specification]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_invar(FILE *file, node_ptr n)
{
  node_ptr context = Nil;
  
  if (node_get_type(n) == CONTEXT) {
    context = car(n);
    n = cdr(n);
  }
  indent_node(file, "invariant ", n, " ");
  if (context) {
    fprintf(file, "(in module ");
    print_node(file, context);
    fprintf(file, ") ");
  }
}
