/**CFile***********************************************************************

  FileName    [mcMc.c]

  PackageName [mc]

  Synopsis    [Fair CTL model checking routines.]

  Description [Fair CTL model checking routines.]

  SeeAlso     [mcExplain.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "mcInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The set of reachable states.]

  Description [The set of reachable states.]

  SeeAlso     []

******************************************************************************/
bdd_ptr reachable_states_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [Slices of reachable states.]

  Description [This is the list of the slices of reachable states
  accumulated during computation of the reachable states.]

  SeeAlso     []

******************************************************************************/
node_ptr reachable_states_layers = Nil;


/**Variable********************************************************************

  Synopsis    [The BDD representing the FAIRNESS constraints.]

  Description [This list contains the result of the evaluation of each
  fairness constraint. It is used to perform Fair CTL model checking
  of EG formulae.]

  SeeAlso     []

******************************************************************************/
node_ptr fairness_constraints_bdd = Nil;


/**Variable********************************************************************

  Synopsis    [The set of fair states.]

  Description [The set of fair states. Used to perform Fair CTL model
  checking of EX and EU formulae.]

  SeeAlso     []

******************************************************************************/
bdd_ptr fair_states_bdd = (bdd_ptr)NULL;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static bdd_ptr fair_iter ARGS((bdd_ptr g, node_ptr fc));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Verifies that M,s0 |= alpha ]

  Description [Verifies that M,s0 |= alpha using the fair CTL model checking.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
int check_spec(node_ptr spec)
{
  bdd_ptr s0, zero, tmp_1, tmp_2;
  node_ptr exp;
  /* Yuan Lu */
  int sign;

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "evaluating ");
    print_spec(nusmv_stderr, spec);
    fprintf(nusmv_stderr, "\n");
  }

  zero = bdd_zero(dd_manager);
  /* Computes the fair states, if needed */
  if (!fair_states_bdd) {
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "Computing fair states...\n");
    fair_states_bdd = compute_fair_states();
    if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr,"done\n");
  }
  printf("\n");

  /* Evaluates the spec */
  s0 = eval_spec(spec,Nil);
  /*
    Intersect with initial states (invar are normal assignments)
  */
  { /*
      We quantify out the input variables from the result of the
      formula evaluation.
    */
    bdd_ptr tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);

    bdd_free(dd_manager, s0);
    s0 = tmp;
  }
  tmp_1 = bdd_not(dd_manager, s0);
  tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
  bdd_free(dd_manager, tmp_1);
  bdd_free(dd_manager, s0);
  s0 = bdd_and(dd_manager, init_bdd, tmp_2);
  bdd_free(dd_manager, tmp_2);
  /* Prints out the result, if not true explain. */
  fprintf(nusmv_stdout, "-- ");
  print_spec(nusmv_stdout, spec);

  if (s0 == zero) {
    fprintf(nusmv_stdout, "is true\n");
    /* Yuan Lu */
    sign = 1;
  } else {
    fprintf(nusmv_stdout, "is false\n");
    tmp_1 = bdd_pick_one_state(dd_manager, s0);
    bdd_free(dd_manager, s0);
    s0 = tmp_1;
    bdd_ref(s0);
    exp = reverse(explain(cons((node_ptr)s0, Nil), spec, Nil));
    if (!exp) exp = cons((node_ptr)s0, Nil);

    /* Yuan Lu : Refine abstraction
    print_explanation(exp); */
    sign = Abs_AbsRefineAbstraction(exp);

    walk_dd(dd_manager, bdd_free, exp);
  }
  bdd_free(dd_manager, s0);
  bdd_free(dd_manager, zero);
  return sign;
}

/**Function********************************************************************

  Synopsis    [Compute quantitative characteristics on the model.]

  Description [Compute the given quantitative characteristics on the model.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_compute(node_ptr spec)
{
  int s0;

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "evaluating ");
    print_spec(nusmv_stderr, spec);
    fprintf(nusmv_stderr, "\n");
  }
  if (!reachable_states_bdd) compute_reachable_states();
  if (!fair_states_bdd) fair_states_bdd = compute_fair_states();
  s0 = eval_compute(spec, Nil);
  fprintf(nusmv_stdout, "-- ");
  print_compute(nusmv_stdout, spec);
  if (s0 == -1) {
    fprintf(nusmv_stdout, "is infinity\n");
  }
  else {
    fprintf(nusmv_stdout, "is %d\n", s0);
  };
  /*
    *** print witness (later) ***
  */
}

/**Function********************************************************************

  Synopsis    [Check that the transition relation is total]

  Description [Check that the transition relation is total. If not the
  case than a deadlock state is printed out.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_transition_relation(void)
{
  bdd_ptr deadlock_states;
  bdd_ptr zero = bdd_zero(dd_manager);
  
  if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr, "Checking for transition relation totality.\n");
  deadlock_states = trans_is_total();
  if (deadlock_states != zero) {
    bdd_ptr ds = bdd_pick_one_state(dd_manager, deadlock_states);
    fprintf(nusmv_stdout, "##########################################################\n");
    fprintf(nusmv_stdout, "The Transition relation is not total. A deadlock state is:\n");
    print_state(ds, 0);
    fprintf(nusmv_stdout, "##########################################################\n");
    bdd_free(dd_manager, ds);
  } else {
    fprintf(nusmv_stdout, "##########################################################\n");
    fprintf(nusmv_stdout, "The Transition relation is total\n");
    fprintf(nusmv_stdout, "##########################################################\n");
  }
  bdd_free(dd_manager, deadlock_states);
  bdd_free(dd_manager, zero);
}


/**Function********************************************************************

  Synopsis           [Compute reachable states]

  Description        [  This function computes reachable states.<br>
  According to the specified options, the system can compute the
  reachable states in different ways:<br>
  <ul>
  <li> <code>reachable_states</code> is the result</li>
  <li> <code>from_lower_bound</code> is the frontier</li>
  <li> <code>from_upper_bound</code> is the previous value of
       <code>reachable_states</code></li>
  <li> <code>not_from_upper_bound</code> is the complement of
       <code>from_upper_bound</code></li> 
  </ul>
]

  SideEffects        [The global variable <code>reachable_states</code> is updated.] 

  SeeAlso            []

******************************************************************************/
void compute_reachable_states()
{ 
  int i = 0;
  bdd_ptr zero = bdd_zero(dd_manager);
  /* from_lower_bound is the frontier */
  bdd_ptr from_lower_bound = bdd_dup(init_bdd);

  reachable_states_layers = Nil;
  reachable_states_bdd = bdd_dup(init_bdd);
  if (opt_verbose_level_gt(options, 0))
    fprintf(nusmv_stderr, "computing reachable state space...\n");
  while (from_lower_bound != zero) {
    bdd_ptr from_upper_bound = bdd_dup(reachable_states_bdd);
    if (opt_ag_only(options))
      reachable_states_layers = cons((node_ptr)bdd_dup(reachable_states_bdd), reachable_states_layers);

    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "iteration %d: BDD size = %d, frontier size = %d, states = %g\n",
              i, bdd_size(dd_manager, reachable_states_bdd),
              bdd_size(dd_manager, from_lower_bound),
              bdd_count_states(dd_manager, reachable_states_bdd));
    
    /* intersect the frontier with normal assignments  */
    bdd_and_accumulate(dd_manager, &from_lower_bound, invar_bdd);
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "invariant combined, size = %d\n",
              bdd_size(dd_manager, from_lower_bound));
    {
      bdd_ptr image = Img_ImageFwd(from_lower_bound);

      bdd_or_accumulate(dd_manager, &reachable_states_bdd, image);
      bdd_free(dd_manager, image);
    }
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "forward step done, size = %d\n",
              bdd_size(dd_manager, reachable_states_bdd));
    {
      bdd_ptr not_from_upper_bound = bdd_not(dd_manager, from_upper_bound);
      
      bdd_free(dd_manager, from_lower_bound);
      from_lower_bound = bdd_and(dd_manager, reachable_states_bdd, not_from_upper_bound);
      bdd_free(dd_manager, not_from_upper_bound);
    }
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "new frontier computed, size = %d\n",
              bdd_size(dd_manager, from_lower_bound));
    bdd_free(dd_manager, from_upper_bound);
    i++;
  }
  bdd_free(dd_manager, from_lower_bound);
  bdd_free(dd_manager, zero);
}

bdd_ptr compute_reachable_states_while_check(bdd_ptr invariant)
{
  int i = 0;
  bdd_ptr zero = bdd_zero(dd_manager);
  /* from_lower_bound is the frontier */
  bdd_ptr from_lower_bound = bdd_dup(init_bdd);

  walk_dd(dd_manager, (VPFDD)bdd_free, reachable_states_layers);
  reachable_states_layers = Nil;
  reachable_states_bdd = bdd_dup(init_bdd);
  while (from_lower_bound != zero) {
    bdd_ptr from_upper_bound = bdd_dup(reachable_states_bdd);
    if (opt_ag_only(options)) {
      bdd_ptr intersect = bdd_and(dd_manager, from_lower_bound, invariant);
      reachable_states_layers = cons((node_ptr)bdd_dup(reachable_states_bdd), reachable_states_layers);
      if(intersect != zero) return intersect;
      bdd_free(dd_manager, intersect);
    }

    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "iteration %d: BDD size = %d, frontier size = %d, states = %g\n",
              i, bdd_size(dd_manager, reachable_states_bdd),
              bdd_size(dd_manager, from_lower_bound),
              bdd_count_states(dd_manager, reachable_states_bdd));

    /* intersect the frontier with normal assignments  */
    bdd_and_accumulate(dd_manager, &from_lower_bound, invar_bdd);
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "invariant combined, size = %d\n",
              bdd_size(dd_manager, from_lower_bound));
    {
      bdd_ptr image = Img_ImageFwd(from_lower_bound);

      bdd_or_accumulate(dd_manager, &reachable_states_bdd, image);
      bdd_free(dd_manager, image);
    }
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "forward step done, size = %d\n",
              bdd_size(dd_manager, reachable_states_bdd));
    {
      bdd_ptr not_from_upper_bound = bdd_not(dd_manager, from_upper_bound);

      bdd_free(dd_manager, from_lower_bound);
      from_lower_bound = bdd_and(dd_manager, reachable_states_bdd, not_from_upper_bound);
      bdd_free(dd_manager, not_from_upper_bound);
    }
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "new frontier computed, size = %d\n",
              bdd_size(dd_manager, from_lower_bound));
    bdd_free(dd_manager, from_upper_bound);
    i++;
  }
  bdd_free(dd_manager, from_lower_bound);
  return zero;
}


/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EX(g)</i>.]

  Description        [Computes the set of states satisfying <i>EX(g)</i>.]

  SideEffects        []

  SeeAlso            [eu ef eg]

******************************************************************************/
bdd_ptr ex(bdd_ptr g)
{
  bdd_ptr tmp;
  bdd_ptr result = bdd_dup(g);
  
  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &result, fair_states_bdd);
  tmp = bdd_and(dd_manager, result, invar_bdd);
  bdd_free(dd_manager, result);
  result = Img_ImageBwd(tmp);
  bdd_free(dd_manager, tmp);
  if (reachable_states_bdd) bdd_and_accumulate(dd_manager, &result, reachable_states_bdd);

  /* Yuan Lu : after compiling, variable orderings should be written */
  if(opt_reorder(options))
    Compile_WriteOrder(get_output_order_file(options), 0);

  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>E\[ f U g \]</i>.]

  Description        [Computes the set of states satisfying <i>E\[ f U g \]</i>.]

  SideEffects        []

  SeeAlso            [ebu]

******************************************************************************/
bdd_ptr eu(bdd_ptr f, bdd_ptr g)
{
  bdd_ptr new, oldY;
  bdd_ptr zero = bdd_zero(dd_manager);
  bdd_ptr Y = bdd_dup(g);
  int n = 1;

  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  if (reachable_states_bdd) bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  if (opt_verbose_level_gt(options, 0))
    indent_node(nusmv_stderr, "eu: computing fixed point approximations for ",
                get_the_node()," ...\n");
  oldY = bdd_dup(Y);
  new = bdd_dup(Y);
  while(new != zero){
    bdd_ptr tmp_1, tmp_2;
    
    if (opt_verbose_level_gt(options, 0)) {
      double states = bdd_count_states(dd_manager, Y);
      int size = bdd_size(dd_manager, Y);

      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n", n++, states,size);
    }
    bdd_free(dd_manager, oldY);
    oldY = bdd_dup(Y);
    tmp_1 = ex(new);
    tmp_2 = bdd_and(dd_manager, f, tmp_1);
    bdd_free(dd_manager, tmp_1);
    bdd_or_accumulate(dd_manager, &Y, tmp_2);
    bdd_free(dd_manager, tmp_2);
    tmp_1 = bdd_not(dd_manager, oldY);
    bdd_free(dd_manager, new);
    new = bdd_and(dd_manager, Y, tmp_1);
    bdd_free(dd_manager, tmp_1);
  }
  bdd_free(dd_manager, zero);
  bdd_free(dd_manager, new);
  bdd_free(dd_manager, oldY);
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EF(g)</i>.]

  Description        [Computes the set of states satisfying <i>EF(g)</i>.]

  SideEffects        []

  SeeAlso            [eu ex]

******************************************************************************/
bdd_ptr ef(bdd_ptr g)
{
  bdd_ptr result, one;

  one = bdd_one(dd_manager);
  result = eu(one, g);
  bdd_free(dd_manager, one);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EG(g)</i>.]

  Description        [Computes the set of states satisfying <i>EG(g)</i>.]

  SideEffects        []

  SeeAlso            [eu ex ef]

******************************************************************************/
bdd_ptr eg(bdd_ptr g)
{
  int n = 1;
  bdd_ptr tmp_1;
  bdd_ptr Y = bdd_dup(g);
  bdd_ptr oldY = bdd_zero(dd_manager);

  if (opt_verbose_level_gt(options, 0))
    indent_node(nusmv_stderr, "eg: computing fixed point approximations for ",
                get_the_node(), " ...\n");
  while (Y != oldY) {
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    bdd_free(dd_manager, oldY);
    oldY = bdd_dup(Y);
    {
      bdd_ptr Z = fair_iter(Y, fairness_constraints_bdd);

      bdd_and_accumulate(dd_manager, &Y, Z);
      bdd_free(dd_manager, Z);
    }
    tmp_1 = ex(Y);
    bdd_and_accumulate(dd_manager, &Y, tmp_1);
    bdd_free(dd_manager, tmp_1);
  }
  bdd_free(dd_manager, oldY);
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>A\[f U g\]</i>.]

  Description        [Computes the set of states satisfying <i>A\[f U g\]</i>.]

  SideEffects        []

  SeeAlso            [ax af ex ef]

******************************************************************************/
bdd_ptr au(bdd_ptr f, bdd_ptr g)
{
  bdd_ptr result, tmp_1, tmp_2, tmp_3, tmp_4;
  
  tmp_1 = bdd_not(dd_manager, f);
  tmp_2 = bdd_not(dd_manager, g);
  tmp_3 = eg(tmp_2);
  tmp_4 = bdd_and(dd_manager, tmp_1, tmp_2);
  bdd_free(dd_manager, tmp_1);
  tmp_1 = eu(tmp_2, tmp_4);
  bdd_free(dd_manager, tmp_2);
  tmp_2 = bdd_or(dd_manager, tmp_1, tmp_3);
  result = bdd_not(dd_manager, tmp_2);
  
  bdd_free(dd_manager, tmp_1);
  bdd_free(dd_manager, tmp_2);
  bdd_free(dd_manager, tmp_3);
  bdd_free(dd_manager, tmp_4);

  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of <code>fair_states</code>.]

  Description        [Computes the set of <code>fair_states</code>.]

  SideEffects        []

  SeeAlso            [eg]

******************************************************************************/
bdd_ptr compute_fair_states()
{
  bdd_ptr result, one;
  extern node_ptr one_number;

  set_the_node(one_number);
  one = bdd_one(dd_manager);
  result = eg(one);
  bdd_free(dd_manager, one);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>E\[f U^{inf..sup} g\]</i>.]

  Description        [Computes the set of states satisfying
                      <i>E\[f U^{inf..sup} g\]</i></i>.]

  SideEffects        []

  SeeAlso            [eu]

******************************************************************************/
bdd_ptr ebu(bdd_ptr f, bdd_ptr g, int inf, int sup)
{
  int i;
  bdd_ptr oldY, tmp_1, tmp_2;
  bdd_ptr Y = bdd_dup(g);
  int n = 1;

  if (inf > sup || inf < 0) return(bdd_zero(dd_manager));
  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  if (reachable_states_bdd) bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  if (opt_verbose_level_gt(options, 0))
    indent_node(nusmv_stderr, "ebu: computing fixed point approximations for ",
                get_the_node()," ...\n");

  /* compute Y = g | (f & ex(Y)) for states within the bound */
  for (i = sup; i > inf; i--) {
    /* There are more states within the bounds */
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = ex(Y);
    tmp_2 = bdd_and(dd_manager, f, tmp_1);
    bdd_or_accumulate(dd_manager, &Y, tmp_2);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, tmp_2);

    if (Y == oldY) {
      /* fixpoint found. collect garbage, and goto next phase */
      i = inf + 1;
    }
  }

  /* compute Y = f & ex(Y) for states before the bound */
  for (i = inf; i > 0; i--) {
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = bdd_dup(Y);
    tmp_1 = ex(Y);
    bdd_free(dd_manager, Y);
    Y = bdd_and(dd_manager, f, tmp_1);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, oldY);
    if (Y == oldY) {
      /* fixpoint found. collect garbage, and finish */
      i = 1;
    }
  }
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EG^{inf..sup}(g)</i>.]

  Description        [Computes the set of states satisfying
                      <i>EG^{inf..sup}(g)</i>.]

  SideEffects        []

  SeeAlso            [eg]

******************************************************************************/
bdd_ptr ebg(bdd_ptr g, int inf, int  sup)
{
  int i;
  bdd_ptr oldY, tmp_1;
  bdd_ptr Y = bdd_dup(g);
  int n = 1;

  if (inf > sup || inf < 0) return(bdd_zero(dd_manager));
  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  if (reachable_states_bdd) bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  if (opt_verbose_level_gt(options, 0))
    indent_node(nusmv_stderr, "ebg: computing fixed point approximations for ",
                get_the_node()," ...\n");
  /* compute Y = g & ex(Y) for states within the bound */
  for (i = sup; i > inf; i--) {
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = ex(Y);
    bdd_and_accumulate(dd_manager, &Y, tmp_1);
    bdd_free(dd_manager, tmp_1);
    if (Y == oldY) {
      /* fixpoint found. goto next phase */
      i = inf + 1;
    }
  }
  /* compute Y = ex(Y) for states before the bound */
  for (i = inf; i > 0; i--) {
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = ex(Y);
    bdd_free(dd_manager, Y);
    Y = tmp_1;
    if (Y == oldY) {
      /* fixpoint found. */
      i = 1;
    }
  }
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>A\[f U^{inf..sup} g\]</i>.]

  Description        [Computes the set of states satisfying
                     <i>A\[f U^{inf..sup} g\]</i>.]

  SideEffects        []

  SeeAlso            [au]

******************************************************************************/
bdd_ptr abu(bdd_ptr f, bdd_ptr g, int inf, int sup)
{
  int i;
  bdd_ptr oldY, tmp_1, tmp_2;
  bdd_ptr Y = bdd_dup(g);
  int n = 1;

  if (inf > sup || inf < 0) return(bdd_zero(dd_manager));
  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  if (reachable_states_bdd) bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  if (opt_verbose_level_gt(options, 0))
    indent_node(nusmv_stderr, "abu: computing fixed point approximations for ",
                get_the_node(), " ...\n");
  /* compute Y = g | (f & ax(Y)) for states within the bound */
  for (i = sup; i > inf; i--) {
    if (opt_verbose_level_gt(options, 0)){
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = bdd_not(dd_manager, Y);
    tmp_2 = ex(tmp_1);
    bdd_free(dd_manager, tmp_1);
    tmp_1 = bdd_not(dd_manager, tmp_2);
    bdd_free(dd_manager, tmp_2);
    tmp_2 = bdd_and(dd_manager, f, tmp_1);
    bdd_or_accumulate(dd_manager, &Y, tmp_2);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, tmp_2);
    if (Y == oldY) {
      /* fixpoint found. goto next phase */
      i = inf + 1;
    }
  }
  /* compute Y = f & ax(Y) for states before the bound */
  for (i = inf; i > 0; i--) {
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = bdd_dup(Y);
    tmp_1 = bdd_not(dd_manager, Y);
    tmp_2 = ex(tmp_1);
    bdd_free(dd_manager, tmp_1);
    tmp_1 = bdd_not(dd_manager, tmp_2);
    bdd_free(dd_manager, tmp_2);
    bdd_free(dd_manager, Y);
    Y = bdd_and(dd_manager, f, tmp_1);
    bdd_free(dd_manager, oldY);
    bdd_free(dd_manager,tmp_1);
    
    if (Y == oldY) {
      /* fixpoint found. finish */
      i = 1;
    }
  }
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EF^{inf..sup}(g)</i>.]

  Description        [Computes the set of states satisfying
                     <i>EF^{inf..sup}(g)</i>.]

  SideEffects        []

  SeeAlso            [ef]

******************************************************************************/
bdd_ptr ebf(bdd_ptr g, int inf, int sup)
{
  bdd_ptr one, result;

  one = bdd_one(dd_manager);
  result = ebu(one, g, inf, sup);
  bdd_free(dd_manager, one);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the minimum length of the shortest path
  from <i>f</i> to <i>g</i>.]

  Description        [This function computes the minimum length of the
  shortest path from <i>f</i> to <i>g</i>.<br>
  Starts from <i>f</i> and proceeds forward until finds a state in <i>g</i>.
  Notice that this function works only if <code>-f</code> option is used.]

  SideEffects        []

  SeeAlso            [maxu]

******************************************************************************/
int minu(bdd_ptr f, bdd_ptr g)
{
  int i;
  int n = 1;
  bdd_ptr R, Rp, tmp_1, tmp_2;
  bdd_ptr zero = bdd_zero(dd_manager);

  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &g, fair_states_bdd);
  if (reachable_states_bdd) bdd_and_accumulate(dd_manager, &f, reachable_states_bdd);
  if (opt_verbose_level_gt(options, 0))
    indent_node(nusmv_stderr, "minu: computing fixed point approximations for ",
                get_the_node(), " ...\n");
  i = 0;
  Rp = bdd_and(dd_manager, f, invar_bdd); /* starts searching from f */
  do {
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Rp%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Rp), bdd_size(dd_manager, Rp));
    }
    if ( (tmp_1 = bdd_and(dd_manager, Rp, g)) != zero ) {
      /* If current frontier intersects g return minimum */
      bdd_free(dd_manager, tmp_1);
      bdd_free(dd_manager, zero);
      return(i);
    }
    bdd_free(dd_manager, tmp_1);
    R = Rp;
    /* go forward */
    tmp_1 = Img_ImageFwd(R);
    tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
    bdd_free(dd_manager, tmp_1);
    Rp = bdd_or(dd_manager, R, tmp_2);
    
    i++;
  } while ( Rp != R );
  /* could not find g anywhere. A fixpoint has been found. g will not be
     ever found, so return infinity. */
  bdd_free(dd_manager,zero);
  return(-1);
}

/**Function********************************************************************

  Synopsis           [This function computes the maximum length of the
  shortest path from <i>f</i> to <i>g</i>.]

  Description        [This function computes the maximum length of the
  shortest path from <i>f</i> to <i>g</i>. It starts from !g and
  proceeds backward until no states in <i>f</i> can be found. In other
  words, it looks for the maximum length of <i>f->AG!g</i>. This
  function works only if <code>-f</code> option is used.]

  SideEffects        []

  SeeAlso            [minu]

******************************************************************************/
int maxu(bdd_ptr f, bdd_ptr g)
{
  int i;
  int n = 1;
  bdd_ptr R, Rp;
  bdd_ptr notg, tmp_1;
  bdd_ptr zero = bdd_zero(dd_manager);

  if (opt_verbose_level_gt(options, 0))
    indent_node(nusmv_stderr, "maxu: computing fixed point approximations for ",
                get_the_node()," ...\n");
  tmp_1 = bdd_not(dd_manager, g);
  notg = bdd_and(dd_manager, tmp_1, invar_bdd);
  bdd_free(dd_manager, tmp_1);
  i = 0;
  R = bdd_one(dd_manager);
  Rp = notg; /* starts from !g */
  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &Rp, fair_states_bdd);
  if (reachable_states_bdd) bdd_and_accumulate(dd_manager, &Rp, reachable_states_bdd);
  do {
    if (opt_verbose_level_gt(options, 0)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Rp%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Rp), bdd_size(dd_manager, Rp));
    }
    if ((tmp_1 = bdd_and(dd_manager, Rp, f)) == zero) {
      /* !g does not intersect f anymore. The maximum length of a path
         completely in !g is i. This is the maximum. */
      bdd_free(dd_manager, tmp_1);
      bdd_free(dd_manager, zero); /* before returning zero must be freed */
      return(i);
    };
    bdd_free(dd_manager, tmp_1);
    R = Rp;
    tmp_1 = ex(R);
    Rp = bdd_and(dd_manager, tmp_1, notg);
    bdd_free(dd_manager, tmp_1);
    i++;
  } while (R != Rp);
  /* a fixpoint has been found in which !g & f holds, so return infinity */
  bdd_free(dd_manager, zero); /* before returning zero must be freed */
  return(-1);
}

/**Function********************************************************************

  Synopsis           [Checks the totality of the transition relation.]

  Description        [This function checks that the transition
  relation is total. If not, the set of deadlock states is
  returned. The checking is performed by computing <em> (not(EX(T)) and
  INVAR)</em> if the result of such a computation is the set of
  deadlock states. The reachable states must be computed before. This
  is because considering EX(true) we can found states that are not
  reachable, and thus does not affect the model checking algorithm.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr trans_is_total(void)
{
  bdd_ptr states_with_succ, deadlock_states;
  
  if (reachable_states_bdd == (bdd_ptr)NULL) compute_reachable_states();
  
  states_with_succ = ex(reachable_states_bdd);
  if (opt_verbose_level_gt(options, 0))
    fprintf(nusmv_stderr, "The # of states with a successor are: %g\n",
            bdd_count_states(dd_manager, states_with_succ));

  deadlock_states = bdd_not(dd_manager, states_with_succ);
  bdd_and_accumulate(dd_manager, &deadlock_states, invar_bdd);
  bdd_and_accumulate(dd_manager, &deadlock_states, reachable_states_bdd);
  if (opt_verbose_level_gt(options, 0))
    fprintf(nusmv_stderr, "The # of states without a successor are: %g\n",
            bdd_count_states(dd_manager, deadlock_states));
  bdd_free(dd_manager, states_with_succ);
  return(deadlock_states);
}

/**Function********************************************************************

  Synopsis           [Prints statistical information about reachable states.]

  Description        [Prints statistical information about reachable
  states, i.e. the real number of reachable states. It is computed
  taking care of the encoding and of the indifferent variables in the
  encoding.]

  SideEffects        []

  SeeAlso            [compute_reachable_states]

******************************************************************************/
void print_reachable_states()
{
  int bits_encoding_var;

  if (opt_print_reachable(options)) {
    if (!reachable_states_bdd) compute_reachable_states();
    {
      bdd_ptr r;
      bdd_ptr mask = bdd_one(dd_manager);
      int n = 0;
      node_ptr l = real_state_variables;

      while (l != Nil) {
        add_ptr tmp_1 = make_var_mask(dd_manager, (add_ptr)car(l),
                                      &bits_encoding_var, 2);
        /*
          We use "2" as last argument of the above call, because we
          encode current and next variables as successive variables in
          the order. If we choose to change this, we need to modify
          the "make_var_mask" implementation.
        */
        bdd_ptr tmp_2 = add_to_bdd(dd_manager, tmp_1);

        add_free(dd_manager, tmp_1);
        n += bits_encoding_var;
        bdd_and_accumulate(dd_manager, &mask, tmp_2);
        bdd_free(dd_manager, tmp_2);
        l = cdr(l);
      }
      {
        bdd_ptr proc_cube, iv;

        proc_cube = bdd_support(dd_manager, process_selector_add);
        iv = bdd_forall(dd_manager, invar_bdd, proc_cube);
        r = bdd_and(dd_manager, reachable_states_bdd, iv);
        bdd_free(dd_manager, proc_cube);
        bdd_free(dd_manager, iv);
      }
      {
        bdd_ptr reached_bdd = bdd_and(dd_manager, r, mask);
        double reached = bdd_count_minterm(dd_manager, reached_bdd, n);
        double space   = bdd_count_minterm(dd_manager, mask, n);

        bdd_free(dd_manager, r);
        bdd_free(dd_manager, reached_bdd);
        fprintf(nusmv_stdout, "reachable states: %g (2^%g) out of %g (2^%g)\n",
                reached, log(reached)/log(2.0),
                space, log(space)/log(2.0));
      }
    }
  }
}

/**Function********************************************************************

  Synopsis           [Print out a CTL specification]

  Description        [Print out a CTL specification]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_spec(FILE *file, node_ptr n)
{
  node_ptr context = Nil;
  
  if (node_get_type(n) == CONTEXT) {
    context = car(n);
    n = cdr(n);
  }
  indent_node(file, "specification ", n, " ");
  if(context) {
    (void) fprintf(file, "(in module ");
    (void) print_node(file, context);
    (void) fprintf(file, ") ");
  }
}


/**Function********************************************************************

  Synopsis           [Print out a COMPUTE specification]

  Description        [Print out a COMPUTE specification]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_compute(FILE *file, node_ptr n)
{
  node_ptr context = Nil;

  if (node_get_type(n) == CONTEXT){
    context = car(n);
    n = cdr(n);
  }
  indent_node(file, "the result of ", n, " ");
  if (context) {
    fprintf(file, "(in module ");
    print_node(file, context);
    fprintf(file, ") ");
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Auxiliary function to  handle <tt>CTL</tt> model
  checking under <em>FAIRNESS</em> constraints.]

  Description        [This function is used in the computation of
  <tt>EG alpha</tt> under <em>FAIRNESS</em> constraints. For each
  fairness constraint <tt>f</tt> in the list <code>fc</code>, it
  computes the set of states satisfying <tt>E \[g U f\]<tt>, then 
  intersects the corresponding partial result and the resulting set is
  returned back.]

  SideEffects        []

  SeeAlso            [eg eu]

******************************************************************************/
static bdd_ptr fair_iter(bdd_ptr g, node_ptr fc)
{
  if(fc == Nil) return(bdd_one(dd_manager));
  if(node_get_type(fc) == CONS){
    bdd_ptr left, right, res;

    left = fair_iter(g, car(fc));
    right = fair_iter(g, cdr(fc));
    res = bdd_and(dd_manager, left, right);
    bdd_free(dd_manager, left);
    bdd_free(dd_manager, right);
    return(res);
  }
  if(node_get_type(fc) == BDD){
    bdd_ptr tmp_1, tmp_2, res;
 
    tmp_1 = bdd_dup((bdd_ptr)car(fc));
    tmp_2 = bdd_and(dd_manager, g, tmp_1);
    res = eu(g, tmp_2);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, tmp_2);
    return(res);
  }
  internal_error("fair_iter: fc->type = %d", node_get_type(fc));
}

