/**CFile***********************************************************************

  FileName    [mcExplain.c]

  PackageName [mc]

  Synopsis    [Witness and Debug generator for Fair CTL models.]

  Description [This file contains the code to find counterexamples
  execution trace that shows a cause of the problem. Here are
  implemented the techniques described in the CMU-CS-94-204 Technical
  Report by E. Clarke, O. Grumberg, K. McMillan and X. Zhao.]

  SeeAlso     [mcMc.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "mcInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The trace counter]

  Description [This variable is used to number execution sequences
  during counterexamples printing. It is the first number of the label
  identifying a state in an execution path.]

  SeeAlso     [optional]

******************************************************************************/
static int trace_number = 1; 
void reset_trace_number(void) {trace_number = 1;}

/**Variable********************************************************************

  Synopsis    [The hash used to store traces.]

  Description [This is an associative list used to store traces
  generated by the counterexamples routines. Associates to
  trace_number.state_number the corresponding BDD.]

  SeeAlso     []

******************************************************************************/
static hash_ptr traces_hash;
void init_traces_hash() { traces_hash = new_assoc(); }
void insert_traces_hash(node_ptr x, node_ptr y) { insert_assoc(traces_hash, x, y);}
node_ptr lookup_traces_hash(node_ptr x) {return(find_assoc(traces_hash, x));}
static assoc_retval traces_hash_free(char *key, char *data, char * arg) {
  add_ptr element = (add_ptr)data;

  if (element != (add_ptr)NULL) {
    add_free(dd_manager, element);
  }
  return(ASSOC_DELETE);
}
void clear_traces_hash() {clear_assoc_and_free_entries(traces_hash, traces_hash_free);}

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static node_ptr fairness_explain  ARGS((node_ptr p, bdd_ptr f, node_ptr fc));
static node_ptr explain_recur     ARGS((node_ptr s, node_ptr n, node_ptr context));


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Counterexamples and witnesses generator.]

  Description        [This function takes as input a CTL formula and
  returns a witness showing how the given formula does not hold. The
  result consists of a list of states (i.e. an execution trace) that
  leads to a state in which the given formula does not hold.]

  SideEffects        []

  SeeAlso            [explain_recur ex_explain eu_explain eg_explain
  ebg_explain ebu_explain]

******************************************************************************/
node_ptr explain(node_ptr path, node_ptr spec_formula, node_ptr context)
{
  node_ptr res = explain_recur(path, spec_formula, context);

  return(res);
}
 
/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [This function computes a path that is a witness
                      for <i>EX(f)</i>.]

  Description        [This function finds a path that is an example for
  <i>EX(f)</i>. <code>p<code> is a BDD which represents the first
  state of the path. It essentially is an initial state from which the
  example can be found.  The formula <i>EX(f)</i> holds under
  fairness constraints in a state <i>s_i</i> iff there is a
  successor state <i>s_{i+1}</i> such that <i>s_{i+1}</i>
  satisfies <i>f</i> and </i>s_{i+1}</i> is the beginning of some
  fair computation path.  We look for states that can be reached from
  the state stored as first element in <code>p</code>, which are fair and
  in which <i>f</i> is satisfied. The algorithm computes more than
  one state, in order to have only one state we apply
  <code>bdd_sat</code>. The result of this application is then put in
  AND with p to form the example.]

  SideEffects        []

  SeeAlso            [explain]

******************************************************************************/
node_ptr ex_explain(node_ptr p, bdd_ptr f)
{
  bdd_ptr zero, state, next_state, tmp_1;
  bdd_ptr acc = bdd_dup(f);

  if (p == Nil) return(p);
  if (fair_states_bdd) bdd_and_accumulate(dd_manager, &acc , fair_states_bdd);
  next_state = Img_ImageFwd((bdd_ptr)car(p));
  tmp_1 = bdd_and(dd_manager, invar_bdd, next_state);
  bdd_free(dd_manager, next_state);
  bdd_and_accumulate(dd_manager, &acc, tmp_1);
  bdd_free(dd_manager, tmp_1);
  state = bdd_pick_one_state(dd_manager, acc);
  bdd_free(dd_manager, acc);
  if (state == (zero = bdd_zero(dd_manager))) {
    bdd_free(dd_manager, zero);
    return(Nil);
  }
  bdd_free(dd_manager, zero);
  bdd_ref(state);
  p = cons((node_ptr)state, p);
  return(p);
}

/**Function********************************************************************

  Synopsis           [This function finds a path that is an example
                      for <i>E\[f U g\]</i>]

  Description [This function finds a path that is an example for
  <i>E\[f U g\]</i>.  The first element of <code>path</code> is a BDD
  that represents the first state of the witness path. It is an
  initial state from which the example can be found. The procedure is
  to try to execute <code>eu(f,g)</code> again, looking for a path
  from <code>p</code> to a state where <i>g</i> is valid.  We look for
  states that can be reached from the first element of
  <code>path</code>. At each step we generate a set of states
  <i>s_i</i> that can be reached in one step from <i>s_{i-1}</i>. We
  extract one minterm form each <i>s_i</i> and we store it in a list.]

  SideEffects        []

  SeeAlso            [explain]

******************************************************************************/
node_ptr eu_explain(node_ptr path, bdd_ptr f, bdd_ptr g)
{
  bdd_ptr tmp_1, tmp_2;
  bdd_ptr acc = bdd_dup(g);
  bdd_ptr zero = bdd_zero(dd_manager);
  
  if (path == Nil) return(path);
  {
    /* Set of states reached so far - initially just one state */
    bdd_ptr Y = bdd_dup((bdd_ptr)car(path));
    /* Set of states reached so far along a path satisfying f. If we ever use
       Z, it means that car(path) does not satisfy g, therefore it satisfies f */
    bdd_ptr Z = bdd_dup(Y);
    /* States added to Y just now */
    bdd_ptr new = bdd_dup(Y);
    /* Iteration counter */
    int n = 0;
    /* initialize list with first state (list is reversed) */
    node_ptr witness_path = path;
    
    if (opt_verbose_level_gt(options, 0))
      indent_node(nusmv_stderr, "searching (counter)example for ", get_the_node(), "\n");
    if (fair_states_bdd) bdd_and_accumulate(dd_manager, &acc, fair_states_bdd);
    /* acc = g /\ fair_states */
    while (new != zero){
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr, "eu: iteration %d: states = %g, BDD nodes = %d\n",
                n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
      bdd_and_accumulate(dd_manager, &new, invar_bdd);
      { /* Block */
        bdd_ptr tmp_1, x;

        tmp_1 = bdd_and(dd_manager, new, acc);
        /* x is one state in Y & acc */
	x = bdd_pick_one_state(dd_manager, tmp_1);
        bdd_free(dd_manager, tmp_1);
        if (x != zero) { /* did we reach (g /\ fair_states_bdd) ? */
          /* Yes. Instantiate the Y's, and return a list of states */
          node_ptr m = witness_path;

          /* We use an aux variable "m" to perform side effects on "witness_path" */
	  while ( 1 ) {
            if (reachable_states_bdd) {
              bdd_ptr reachable_x;
            
              reachable_x = bdd_and(dd_manager, x, reachable_states_bdd);
              if (reachable_x == zero) {
                bdd_free(dd_manager, reachable_x);
                fprintf(nusmv_stdout, "this state is not reachable :\n");
                print_state(x, 0);
                internal_error("eu_explain: state not reachable");
              }
              else {
                bdd_free(dd_manager, reachable_x);
              }
	    }
            bdd_free(dd_manager, (bdd_ptr)car(m));
            /* substitute Y for one state of Y in the list */
	    m->left.bddtype = bdd_dup(x); 
	    if (m == path) { /* if we reached the first state, it's over */
              bdd_free(dd_manager, zero);
              bdd_free(dd_manager, x); /* just added */
              return(witness_path);
            }
	    m = cdr(m);
            /*
              instantiate the next Y. x is a state in car(m), such that there
              is a path from the current x to it.
            */
            tmp_1 = Img_ImageBwd(x);
            tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
            bdd_free(dd_manager, tmp_1);
            tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(m), tmp_2);
            bdd_free(dd_manager, tmp_2);
	    /* if l != path, car(path) may include states not satisfying f */
	    if (m == path) bdd_and_accumulate(dd_manager, &tmp_1, f);
            bdd_free(dd_manager, x);
	    x = bdd_pick_one_state(dd_manager, tmp_1);
            bdd_free(dd_manager, tmp_1);
	  } /* while (1) */
	} /* if (x |= zero) */
      } /* Block */
      /*
        generate the next Y, that is, the set of states that can be reached
        in one step from the states in Y that satisfy f.
      */
      tmp_1 = bdd_and(dd_manager, f, new);
      tmp_2 = Img_ImageFwd(tmp_1);
      bdd_free(dd_manager, tmp_1);
      bdd_free(dd_manager, Y);
      Y = bdd_or(dd_manager, Z, tmp_2);
      bdd_free(dd_manager, tmp_2);
      bdd_free(dd_manager, new);
      tmp_1 = bdd_not(dd_manager, Z);
      new = bdd_and(dd_manager, Y, tmp_1);
      bdd_free(dd_manager, tmp_1);
      /*
        In case the new Y cannot satisfy g, save its subset of states
        that satisfies f on the state list.
      */
      bdd_free(dd_manager, Z);
      Z = bdd_and(dd_manager, f, Y);
      bdd_ref(Z);
      witness_path = cons((node_ptr)Z, witness_path);
    } /* while (new != zero) */
    /* reached the fixpoint and could not find it. Release the list. */
    while (witness_path != path) {
      node_ptr m = witness_path;

      bdd_free(dd_manager, (bdd_ptr)car(witness_path));
      witness_path = cdr(witness_path);
      free_node(m);
    } /* while (witness_path != p) */
    bdd_free(dd_manager, zero);
    return(Nil);
  } /* if (p != nil) */
} /* eu_explain */

/**Function********************************************************************

  Synopsis           [This function finds a path that is an example
                      for <i>E\[f U g\]^{sup}_{inf}</i>.]

  Description        [This function finds a path that is an example
  for <i>E\[f U g\]^{sup}_{inf}</i>. The first element of
  <code>path</code> is a BDD that represents the first state of the
  path. It is an initial state from which the example can be found.
  The procedure is to try to execute <code>ebu(f, g, inf, sup)</code>, looking
  for a path, with length <code>(sup - inf)<code>, from <code>p</code>
  to a state where <i>g</i> is valid using only transitions from
  states satisfying <i>f</i>.]

  SideEffects        []

  SeeAlso            [explain]

******************************************************************************/
node_ptr ebu_explain(node_ptr path, bdd_ptr f, bdd_ptr g, int inf, int sup)
{
  if (path == Nil) return(path);
  { /* Block */
    int i;
    bdd_ptr Z, tmp_1, tmp_2;
    bdd_ptr zero = bdd_zero(dd_manager);
    bdd_ptr Y = bdd_dup((bdd_ptr)car(path));
    int n = 0;
    node_ptr witness_path = path;

    if (opt_verbose_level_gt(options, 0))
      indent_node(nusmv_stderr, "searching (counter)example for ", get_the_node(), "\n");
    /* looks for a path from the first element of "path", with length
      inf, using only transitions from states satisfying "f". The sets
      of states in the list may contain states in which "f" is false.
      This is performed later, when a complete (counter)example is found,
      to avoid the need of recovering the "old" "car(path)" */
    for (i = 0; i < inf; i++) {
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr, "ebu: iteration %d: states = %g, BDD nodes = %d\n",
                n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
      Z = bdd_dup(Y);
      bdd_and_accumulate(dd_manager, &Y, invar_bdd);
      tmp_1 = bdd_and(dd_manager, Y, f);
      bdd_free(dd_manager, Y);
      Y = Img_ImageFwd(tmp_1);
      if (Y == zero) {
        /* there is no valid path */
        bdd_free(dd_manager, Z);
	while (witness_path != path) {
	  node_ptr m = witness_path;

	  bdd_free(dd_manager, (bdd_ptr)car(witness_path));
	  witness_path = cdr(witness_path);
	  free_node(m);
	} /* (witness_path != path) */
        bdd_free(dd_manager, zero);
	return(Nil);
      } /* (Y == zero) */
      bdd_ref(Y);
      witness_path = cons((node_ptr)Y, witness_path);
      if (Z == Y) {
        /* fixpoint found - fill the list with Y to length inf. */
	while (++i < inf) {
          bdd_ref(Y);
          witness_path = cons((node_ptr)Y, witness_path);
	} /* (++i < inf) */
	/* No need for further garbage collections. */
	break;
      } /* (Z == Y) */
    } /* for (i = 0; i< inf ; i++) */
    /*
      At this point, car(witness_path) is the set of states that can be reached
      in inf steps, using transitions from states where "f" is valid.
      Now we can call eu_explain(witness_path, f, g).  eu_explain will find a
      shortest extension from car(witness_path) to a state where "g" is valid. We
      then check that the length of this path is less than or equal to (sup-inf).
    */
    { /* Block 2 */
      node_ptr new_witness_path = eu_explain(witness_path, f, g);

      if (new_witness_path != Nil) {
	node_ptr m = new_witness_path;

	for (i = 0; m != Nil && m != witness_path; i++, m = cdr(m));
	if (m == Nil) internal_error("ebu_explain: cannot get back to l");
       
	/* did we reach g in time? */
	if (i <= (sup - inf)) {
          /* Yes. Instantiate the Y's, and return a list of states */
	  bdd_ptr x = bdd_dup(witness_path->left.bddtype);

          m = witness_path;
	  while ( 1 ) {
            if (reachable_states_bdd) {
              bdd_ptr reachable_x = bdd_and(dd_manager, x, reachable_states_bdd);

              if (reachable_x == zero) {
                bdd_free(dd_manager, reachable_x);
                fprintf(nusmv_stdout, "this state is not reachable :\n");
                print_state(x, 0);
                internal_error("ebu_explain: state not reachable");
              } /* if (reachable_x == zero) */
              bdd_free(dd_manager, reachable_x);
            } /* if (reachable_states_bdd) */
            bdd_free(dd_manager, (bdd_ptr)car(m));
            bdd_ref(x);
	    m->left.bddtype = x; /* substitute Y for one state of Y in the list */
	    if (m == path) {  /* if we reached the first state, it's over */
              bdd_free(dd_manager, zero);
              return(new_witness_path);
            } /* if (m == path) */
	    m = cdr(m);
            /*
             instantiate the next Y. x is a state in car(m), such that there
             is a path from the current x to it.
            */
            tmp_1 = Img_ImageBwd(x);
            tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
            bdd_free(dd_manager, tmp_1);
            tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(m), tmp_2);
            bdd_free(dd_manager, tmp_2);
            tmp_2 = bdd_and(dd_manager, tmp_1, f);
            bdd_free(dd_manager, tmp_1);
	    /* if witness_path != path, car(m) may include states not satisfying f */
            x = bdd_pick_one_state(dd_manager, tmp_2);
            bdd_free(dd_manager, tmp_2);
	  } /* while( 1 ) */
	} /* if (i <= (sup - inf)) */
	/* path from witness_path to new_witness_path is longer than requested. */
	witness_path = new_witness_path;
      } /* if (new_witness_path != Nil) */
      /* Could not find an example - free all newly allocated nodes */
      while (witness_path != path) {
	node_ptr m = witness_path;

	bdd_free(dd_manager, (bdd_ptr)car(witness_path));
	witness_path = cdr(witness_path);
	free_node(m);
      } /* (witness_path != path) */
      bdd_free(dd_manager, zero);
      return (Nil);
    } /* Block 2 */
  } /* Block */
} /* ebu_explain */

/**Function********************************************************************

  Synopsis           [This function finds a path that is an example
                      for <i>EG(g)</i>.]

  Description [This function finds a path that is an example for
  <i>EG(g)</i>. The first element <code>p</code> is the BDD that
  represents the first state of the path. It is an initial state from
  which the example can be found.<br>

  The procedure is based on the greatest fixed point characterization
  for the CTL operator <b>EG</b>. The CTL formula <i>EG(g)</i> under
  fairness constraints means that there exists a path beginning with
  current state on which <i>g</i> holds globally (invariantly) and
  each formula in the set of fairness constraints holds infinitely
  often on the path.  If we denote with <i>EG(g)</i> the set of states
  that satisfy <i>EG(g)</i> under fairness constraints, we can
  construct the witness path incrementally by giving a sequence of
  prefixes of the path of increasing length until a cycle is found. At
  each step in the construction we must ensure that the current prefix
  can be extended to a fair path along which each state satisfies
  <i>EG(g)</i>.]

  SideEffects        []

  SeeAlso            [explain]

******************************************************************************/
node_ptr eg_explain(node_ptr witness_path, bdd_ptr g)
{
  node_ptr old_witness_path;
  bdd_ptr f, zero, tmp_1;

  if (witness_path == Nil) return(witness_path);
  zero = bdd_zero(dd_manager);
  f = eg(g);
  bdd_ref(f);
  tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(witness_path), f);
  if (tmp_1 == zero) {
    bdd_free(dd_manager, f);
    bdd_free(dd_manager, tmp_1); 
    bdd_free(dd_manager, zero);
    return(Nil);
  } /* if (tmp_1 == zero) */
  bdd_free(dd_manager, tmp_1);
  while ( 1 ) {
    bdd_ptr start = bdd_dup(witness_path->left.bddtype);

    witness_path = fairness_explain(witness_path, f, fairness_constraints_bdd);
    tmp_1 = ex(start);
    g = bdd_and(dd_manager, f, tmp_1);
    bdd_free(dd_manager, tmp_1);
    old_witness_path = witness_path;
    witness_path = ex_explain(eu_explain(witness_path, f, g), start);
    if (witness_path) break;
    witness_path = ex_explain(old_witness_path, f);
    if (witness_path == Nil) internal_error("eg_explain: witness_path == Nil");
  } /* while ( 1 ) */
  bdd_free(dd_manager, zero);
  bdd_free(dd_manager, f);
  return(witness_path);
} /* eg_explain */

/**Function********************************************************************

  Synopsis           [This function finds a path of length
                     <tt>(sup-inf)</tt> that is an example for
                     <i>EG(g)^{sup}_{inf}</i>.]

  Description        [This function finds a path of length
  <tt>(sup-inf)</tt> that is an example for <i>EG(g)^{sup}_{inf}</i>. 
  The first element of <code>p</code> is the BDD that represents the
  first state of the path. It is an initial state from which the
  example has to be found.]

  SideEffects        []

  SeeAlso            [explain]

******************************************************************************/
node_ptr ebg_explain(node_ptr path, bdd_ptr g, int inf, int sup)
{
  if (path == Nil) return(path);
  { /* Block */
    int i;
    bdd_ptr Z, tmp_1, tmp_2, tmp_3;
    bdd_ptr zero = bdd_zero(dd_manager);
    bdd_ptr Y = bdd_dup((bdd_ptr)car(path));
    int n = 0;
    node_ptr witness_path = path;

    if (opt_verbose_level_gt(options, 0))
      indent_node(nusmv_stderr, "searching (counter)example for ", get_the_node(), "\n");

    /* look for a path of length inf from car(path) */
    for (i=0; i < inf; i++) {
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr, "ebg: iteration %d: states = %g, BDD nodes = %d\n",
                n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
      Z = bdd_dup(Y);
      tmp_1 = Img_ImageFwd(Y);
      bdd_free(dd_manager, Y);
      Y = bdd_and(dd_manager, invar_bdd, tmp_1);
      bdd_free(dd_manager, tmp_1);
      if (Y == zero) {
        /* there is no valid path */
        bdd_free(dd_manager, Z);
	while (witness_path != path) {
	  node_ptr m=witness_path;

	  bdd_free(dd_manager, (bdd_ptr)car(witness_path));
	  witness_path = cdr(witness_path);
	  free_node(m);
	} /* while (witness_path != path) */
        bdd_free(dd_manager, zero);
	return(Nil);
      } /* if (Y == zero) */
      bdd_ref(Y);
      witness_path = cons((node_ptr)Y, witness_path);
      if (Z == Y) {
        /* fixpoint found - fill the list with Y to length inf. */
	while (++i < inf) {
          bdd_ref(Y);
          witness_path = cons((node_ptr)Y, witness_path);
        } /* while (++i < inf) */
	break;
      } /* if (Z == Y) */
    } /* for (i=0; i < inf; i++) */
    /*
      At this point, car(witness_path) is the set of states that can be reached
      in inf steps.  Look for a continuation path of length sup-inf
      using transitions from states that satisfy g
    */
    { /* Block 2 */
      bdd_ptr fair_g = bdd_dup(g);
      
      if (fair_states_bdd) bdd_and_accumulate(dd_manager, &fair_g, fair_states_bdd);
      { /* Block 3 */
        node_ptr old_witness_path = witness_path;

        Y = (bdd_ptr) car(witness_path);
        for (i = inf; i < sup; i++) {
          if (opt_verbose_level_gt(options, 0))
            fprintf(nusmv_stderr, "ebg: iteration %d: states = %g, BDD nodes = %d\n",
                    n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
          Z = bdd_dup(Y);
          tmp_1 = bdd_and(dd_manager, fair_g, Y);
          tmp_2 = Img_ImageFwd(tmp_1);
          bdd_free(dd_manager, tmp_1);
          Y = bdd_and(dd_manager, invar_bdd, tmp_2);
          bdd_free(dd_manager, tmp_2);
          if (Y == zero) {
            /* there is no valid path */
            bdd_free(dd_manager, Z);
            while (witness_path != path) {
              node_ptr m=witness_path;

              bdd_free(dd_manager, (bdd_ptr)car(witness_path));
              witness_path = cdr(witness_path);
              free_node(m);
            } /* while (witness_path != path) */
            bdd_free(dd_manager, zero);
            return(Nil);
          } /* if (Y == zero) */
          bdd_ref(Y);
          witness_path = cons((node_ptr)Y, witness_path);
          if (Y == Z) {
            /* fixpoint found - fill the list with Y to length sup. */
            while (++i < sup) {
              bdd_ref(Y);
              witness_path = cons((node_ptr)Y, witness_path);
            } /* while (++i < sup) */
            /* No need for further garbage collections. */
            break;
          } /* if (Y == Z) */
        } /* for (i = inf; i < sup; i++) */
        /*
          transform witness_path from a list of sets of states into a list of states.
        */
        { /* Block 4 */
          node_ptr m = witness_path;
          bdd_ptr tmp_1 = bdd_and(dd_manager, fair_g, (bdd_ptr)car(m));
          bdd_ptr x = bdd_pick_one_state(dd_manager, tmp_1);

          /* g should holds in all states up to car(old_witness_path), inclusive */
          while ( 1 ) {
            bdd_free(dd_manager, (bdd_ptr)car(m));
            bdd_ref(x);
            m->left.bddtype = x;
            if (m == old_witness_path) break;
            m = cdr(m);
            /* instantiate the next Y */
            tmp_1 = Img_ImageBwd(x);
            tmp_2 = bdd_and(dd_manager, fair_g, invar_bdd);
            tmp_3 = bdd_and(dd_manager, tmp_2, tmp_1);
            bdd_free(dd_manager, tmp_1);
            bdd_free(dd_manager, tmp_2);
            tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(m), tmp_3);
            bdd_free(dd_manager, tmp_3);
            x = bdd_pick_one_state(dd_manager, tmp_1);
          } /* while ( 1 ) */
          bdd_free(dd_manager, fair_g);
          /* Continue instantiating up to car(path), inclusive */
          while ( 1 ) {
            bdd_free(dd_manager, (bdd_ptr)car(m));
            bdd_ref(x);
            m->left.bddtype = x;
            if (m == path) break;
            m = cdr(m);
            /* instantiate the next Y */
            tmp_1 = Img_ImageBwd(x);
            tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
            bdd_free(dd_manager, tmp_1);
            tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(m), tmp_2);
            bdd_free(dd_manager, tmp_2);
            x = bdd_pick_one_state(dd_manager, tmp_1);
            bdd_free(dd_manager, tmp_1);
          } /* while ( 1 ) */
          bdd_free(dd_manager, zero);
          return(witness_path);
        } /* Block 4 */
      } /* Block 3 */
    } /* Block 2 */
  } /* Block */
} /* ebg_explain */


/**Function********************************************************************

  Synopsis           [Prints out a counterexample trace.]

  Description        [Takes as input a trace and prints out it.]

  SideEffects        []

  SeeAlso            [explain]

******************************************************************************/
void print_explanation(node_ptr p)
{
  int state_number = 1;
  node_ptr last_state = last(p);
  
  fprintf(nusmv_stdout, "-- as demonstrated by the following execution sequence\n");
  clear_print_hash();
  while (p){
    if (cdr(p) && (car(p) == last_state))
      fprintf(nusmv_stdout, "-- loop starts here --\n");
    fprintf(nusmv_stdout, "state %d.%d:\n", trace_number, state_number);
    print_state((bdd_ptr)car(p),1);
    fprintf(nusmv_stdout, "\n");
    bdd_ref((bdd_ptr)car(p));
    insert_traces_hash(find_node(DOT,find_node(NUMBER,(node_ptr)trace_number,Nil),
                                find_node(NUMBER,(node_ptr)state_number,Nil)),car(p));
    state_number++;
    p = cdr(p);
  }
  trace_number++;
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Recursively traverse the formula CTL and rewrite
                      it in order to use the base witnesses generator functions.]

  Description        [Recursively traverse the formula CTL and rewrite
  it in order to use the base witnesses generator functions.<br>
  The rewritings performed use the equivalence between CTL formulas,
  i.e. <i>A\[f U g\]</i> is equivalent to <i>!(E\[!g U (!g & !f)\] | EG !g)</i>.]

  SideEffects        []

  SeeAlso            [explain]

******************************************************************************/
static node_ptr explain_recur(node_ptr path, node_ptr formula_expr, node_ptr context)
{
  bdd_ptr a1, a2;
  node_ptr new_path;

  if (formula_expr == Nil) return(Nil);
  yylineno = formula_expr->lineno;
  switch (node_get_type(formula_expr)) {
  case CONTEXT: 
    return(explain(path, cdr(formula_expr), car(formula_expr)));
  case AND:
  case OR:
  case NOT:
  case IMPLIES:
  case IFF:
    new_path = explain(path, car(formula_expr), context);
    if (new_path) return(new_path);
    return(explain(path, cdr(formula_expr), context));
  case EX:
    a1 = eval_spec(car(formula_expr), context);
    set_the_node(formula_expr);
    new_path = ex_explain(path, a1);
    bdd_free(dd_manager, a1);
    if (new_path) {
      node_ptr q = explain(new_path, car(formula_expr), context);

      if (q) return(q);
    }
    return(new_path);
  case AX:
    return(explain(path, find_node(NOT, find_node(EX, find_node(NOT, car(formula_expr), Nil), Nil),
                                     Nil), context));
  case EF:
    return(explain(path, find_node(EU, one_number, car(formula_expr)), context));
  case AG:
    return(explain(path, find_node(NOT, find_node(EU, one_number,
                                                   find_node(NOT, car(formula_expr), Nil)),
                                     Nil), context));
  case EG:
    a1 = eval_spec(car(formula_expr), context);
    set_the_node(formula_expr);
    new_path = eg_explain(path, a1);
    bdd_free(dd_manager, a1);
    return(new_path);
  case AF:
    /* AF g and !EG !g are equivalent. */
    return (explain(path, find_node(NOT, find_node(EG, find_node(NOT, car(formula_expr), Nil), Nil),
                                      Nil), context));
  case EU:
    a1 = eval_spec(car(formula_expr), context);
    a2 = eval_spec(cdr(formula_expr), context);
    set_the_node(formula_expr);
    new_path = eu_explain(path, a1, a2);
    bdd_free(dd_manager, a2);
    bdd_free(dd_manager, a1);
    if (new_path) {
      node_ptr q = explain(new_path, cdr(formula_expr), context);

      if (q) return(q);
    }
    return(new_path);
  case AU:
    /* A[f U g] and !(E[!g U (!g & !f)] | EG !g) are equivalent. */
    return (explain(path, find_node
		    (NOT, find_node
		     (OR, find_node
		      (EU, find_node
		       (NOT, cdr(formula_expr), Nil), find_node
		       (AND, find_node
			(NOT, car(formula_expr), Nil), find_node
			(NOT, cdr(formula_expr), Nil))), find_node
		      (EG, find_node
		       (NOT, cdr(formula_expr), Nil), Nil)), Nil),
		    context));
  case EBU:
    a1 = eval_spec(car(car(formula_expr)), context);
    a2 = eval_spec(cdr(car(formula_expr)), context);
    {
      int inf = eval_num(car(cdr(formula_expr)), context);
      int sup = eval_num(cdr(cdr(formula_expr)), context);

      set_the_node(formula_expr);
      new_path = ebu_explain(path, a1, a2, inf, sup);
    }
    bdd_free(dd_manager, a2);
    bdd_free(dd_manager, a1);
    if (new_path) {
      node_ptr q = explain(new_path, cdr(car(formula_expr)), context);

      if (q) return(q);
    }
    return(new_path);
  case ABU:
    /*
      A[f BU l..h g] is equivalent to
      ! ((EBF 0..(l - 1) !f)
        | EBG l..l ((EBG 0..(h - l) !g)
      	      | E[!g BU 0..(h - l) (!g & !f)]))
     
      f:car(car(formula_expr)) g:cdr(car(formula_expr)) l:car(cdr(l)) h:cdr(car(formula_expr))
    */
    return (explain(path, find_node
		    (NOT, find_node
		     (OR, find_node
		      (EBF, find_node
		       (NOT, car(car(formula_expr)), Nil), find_node
		       (TWODOTS,
			zero_number, find_node
			(MINUS, car(cdr(formula_expr)), one_number))), find_node
		      (EBG, find_node
		       (OR, find_node
			(EBG, find_node
			 (NOT, cdr(car(formula_expr)), Nil), find_node
			 (TWODOTS,
			  zero_number, find_node
			  (MINUS,
			   cdr(cdr(formula_expr)),
			   car(cdr(formula_expr))))), find_node
			(EBU, find_node
			 (EU, find_node
			  (NOT, cdr(car(formula_expr)), Nil), find_node
			  (AND, find_node
			   (NOT, car(car(formula_expr)), Nil), find_node
			   (NOT, cdr(car(formula_expr)), Nil))), find_node
			 (TWODOTS,
			  zero_number, find_node
			  (MINUS,
			   cdr(cdr(formula_expr)),
			   car(cdr(formula_expr)))))), find_node
		       (TWODOTS,
			car(cdr(formula_expr)),
			car(cdr(formula_expr))))), Nil),
		    context));
						   
  case EBF:
    /* EBF range g and E[1 BU range g] are equivalent.  */
    return (explain(path, find_node
		    (EBU, find_node
		     (EU, one_number, car(formula_expr)),
		     cdr(formula_expr)),
		    context));
  case ABG:
    /* ABG range g and !EBF range !g are equivalent. */
    return (explain(path, find_node
		    (NOT, find_node
		     (EBF, find_node
                      (NOT, car(formula_expr), Nil),
		      cdr(formula_expr)), Nil),
		    context));
  case EBG:
    a1 = eval_spec(car(formula_expr), context);
    {
      int inf = eval_num(car(cdr(formula_expr)), context);
      int sup = eval_num(cdr(cdr(formula_expr)), context);

      set_the_node(formula_expr);
      new_path = ebg_explain(path, a1, inf, sup);
    }
    bdd_free(dd_manager, a1);
    return(new_path);
  case ABF:
    /* ABF range g and !EBG range !g are equivalent. */
    return (explain(path, find_node
		    (NOT, find_node
		     (EBG, find_node
                      (NOT, car(formula_expr), Nil),
		      cdr(formula_expr)), Nil),
		    context));
  case ATOM:
    {
      node_ptr name  = find_node(DOT, context, find_atom(formula_expr));
      node_ptr temp1 = lookup_param_hash(name);
      node_ptr temp2 = lookup_symbol_hash(name);
      bdd_ptr  temp3 = (bdd_ptr)lookup_constant_hash(find_atom(formula_expr));

      if((temp1 && temp2) || (temp2 && temp3) || (temp3 && temp1))
	rpterr("atom \"%s\" is ambiguous", formula_expr->left.strtype->text);
      if (temp1) return(explain(path, temp1, context));
      if (temp3) return(Nil);
    } /* fall through on purpose here */
  case DOT:
  case ARRAY:
    {
      node_ptr t = eval_struct(formula_expr, context);
      node_ptr v = lookup_symbol_hash(t);

      if (v) return(explain(path, v, context));
      return(Nil);
    }
  default:
    return(Nil);
  }
}

/**Function********************************************************************

  Synopsis           [Auxiliary function to the computation of a
  witness of the formula <i>EG f</i>.]

  Description        [In the computation of the witness for the
  formula <i>EG f</i>, at each step we must ensure that the current
  prefix can be extended to a fair path along which each state
  satisfies <i>f</i>. This function performs the inner fixpoint
  computation for each fairness constraints in the fix point
  computation of the formula <i>EG(f)<i>. For every constraints
  <i>h</i>, we obtain an increasing sequence of approximations Q_0^h,
  Q_1^h, ..., where each Q_i^h is the set of states from which a state
  in the accumulated set can be reached in <i>i</i> or fewer steps,
  while satisfying <i>f</i>.]

  SideEffects        []

  SeeAlso            [explain, eg_explain, fair_iter, eg]

******************************************************************************/
static node_ptr fairness_explain(node_ptr p, bdd_ptr f, node_ptr fc)
{
  if (fc == Nil) return(p);
  if (node_get_type(fc) == CONS) {
    p = fairness_explain(p, f, car(fc));
    p = fairness_explain(p, f, cdr(fc));
    return(p);
  }
  if (node_get_type(fc) == BDD) {
    bdd_ptr Y = bdd_and(dd_manager, f, (bdd_ptr)car(fc));
    node_ptr res = eu_explain(p, f, Y);

    bdd_free(dd_manager, Y);
    return(res);
  }
  internal_error("fairness_explain: fc->type == %d", node_get_type(fc));
}
