/**CFile***********************************************************************

  FileName    [mcEval.c]

  PackageName [mc]

  Synopsis    [CTL to BDD compiler]

  Description [In this file there is the code to compile CTL formulas
  into BDD and the code to call the model checking algorithms.]

  SeeAlso     [mcMc.c mcExplain.c mcACTL.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "mcInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/
typedef bdd_ptr (*BDDPFDB)(DdManager *, bdd_ptr);
typedef bdd_ptr (*BDDPFB)(bdd_ptr);
typedef bdd_ptr (*BDDPFDBB)(DdManager *, bdd_ptr, bdd_ptr);
typedef bdd_ptr (*BDDPFBB)(bdd_ptr, bdd_ptr);
typedef bdd_ptr (*BDDPFDBII)(DdManager *, bdd_ptr, int, int);
typedef bdd_ptr (*BDDPFBII)(bdd_ptr, int, int);
typedef bdd_ptr (*BDDPFDBBII)(DdManager *, bdd_ptr, bdd_ptr, int, int);
typedef bdd_ptr (*BDDPFBBII)(bdd_ptr, bdd_ptr, int, int);

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static bdd_ptr eval_spec_recur ARGS((node_ptr n, node_ptr context));
static int eval_compute_recur ARGS((node_ptr n, node_ptr context));
static bdd_ptr unary_bdd_op ARGS((DdManager * dd, BDDPFDB op, node_ptr n, int resflag, int argflag, node_ptr context));
static bdd_ptr binary_bdd_op ARGS((DdManager * dd, BDDPFDBB op, node_ptr n, int resflag, int argflag1, int argflag2, node_ptr context));
static bdd_ptr unary_mod_bdd_op ARGS((DdManager * dd, BDDPFB op, node_ptr n, int resflag, int argflag, node_ptr context));
static bdd_ptr binary_mod_bdd_op ARGS((DdManager * dd, BDDPFBB op, node_ptr n, int resflag, int argflag1, int argflag2, node_ptr context));
static bdd_ptr binary_mod_bdd_op_ns ARGS((DdManager * dd, BDDPFBB op, node_ptr n, int resflag, int argflag1, int argflag2, node_ptr context));
static bdd_ptr ternary_mod_bdd_op ARGS((DdManager * dd, BDDPFBII op, node_ptr n, int resflag, int argflag, node_ptr context));
static bdd_ptr quad_mod_bdd_op ARGS((DdManager * dd, BDDPFBBII op, node_ptr n, int resflag, int argflag1, int argflag2, node_ptr context));
static bdd_ptr eval_sign ARGS((bdd_ptr a, int flag));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Compile a CTL formula into BDD and performs
  Model Checking.]

  Description        [Compile a CTL formula into BDD and performs
  Model Checking.]

  SideEffects        []

  SeeAlso            [eval_compute]

******************************************************************************/
bdd_ptr eval_spec(node_ptr n, node_ptr context){
  bdd_ptr res;
  int temp = yylineno;

  if (n == Nil) return(bdd_one(dd_manager));
  yylineno = n->lineno;
  res = eval_spec_recur(n, context);
  yylineno = temp;
  return(res);
}

/**Function********************************************************************

  Synopsis           [This function takes a list of formulas, and
  returns the list of their BDDs.]

  Description        [This function takes as input a list of formulae,
  and return as output the list of the corresponding BDDs, obtained by
  evaluating each formula in the given context.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr eval_formula_list(node_ptr nodes, node_ptr context)
{
  if (nodes == Nil) return(Nil);
  if (node_get_type(nodes) == CONS)
    return(find_node(CONS, eval_formula_list(car(nodes), context),
		           eval_formula_list(cdr(nodes), context)));
  return(find_node(BDD,(node_ptr)eval_spec(nodes,context),Nil));
}

/**Function********************************************************************

  Synopsis           [Computes shortest and longest length of the path
  between two set of states.]

  Description        [This function performs the invocation of the
  routines to compute the length of the shortest and longest execution
  path between two set of states s_1 and s_2.]

  SideEffects        []

  SeeAlso            [eval_spec]

******************************************************************************/
int eval_compute(node_ptr n, node_ptr context){
  int res;
  int temp = yylineno;

  if (n == Nil) internal_error("eval_compute: n = NIL\n");
  yylineno = n->lineno;
  res = eval_compute_recur(n,context);
  yylineno = temp;
  return(res);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Recursive step of <code>eval_spec</code>.]

  Description        [Performs the recursive step of <code>eval_spec</code>.]

  SideEffects        []

  SeeAlso            [eval_spec]

******************************************************************************/
static bdd_ptr eval_spec_recur(node_ptr n, node_ptr context)
{
  if (n == Nil) return(bdd_one(dd_manager));
  switch (node_get_type(n)){
  case CONTEXT: return(eval_spec(cdr(n),car(n)));
  case AND:     return(binary_bdd_op(dd_manager, bdd_and, n, 1, 1, 1, context));
  case OR:      return(binary_bdd_op(dd_manager, bdd_or, n, 1, 1, 1, context));
  case NOT:     return(unary_bdd_op(dd_manager, bdd_not, n, 1, 1, context));
  case IMPLIES: return(binary_bdd_op(dd_manager, bdd_or, n, 1, -1, 1, context));
  case IFF:     return(binary_bdd_op(dd_manager, bdd_xor, n, -1, 1, 1, context));
  case EX:      return(unary_mod_bdd_op(dd_manager, ex, n,  1,  1, context));
  case AX:      return(unary_mod_bdd_op(dd_manager, ex, n, -1, -1, context));
  case EF:      return(unary_mod_bdd_op(dd_manager, ef, n,  1,  1, context));
  case AG:      return(unary_mod_bdd_op(dd_manager, ef, n, -1, -1, context));
  case AF:      return(unary_mod_bdd_op(dd_manager, eg, n, -1, -1, context));
  case EG:      return(unary_mod_bdd_op(dd_manager, eg, n,  1,  1, context));
  case EU:      return(binary_mod_bdd_op(dd_manager, eu, n, 1, 1, 1, context));
  case AU:      return(binary_mod_bdd_op(dd_manager, au, n, 1, 1, 1, context));
  case EBU:     return(quad_mod_bdd_op(dd_manager, ebu, n, 1, 1, 1, context));
  case ABU:     return(quad_mod_bdd_op(dd_manager, abu, n, 1, 1, 1, context));
  case EBF:     return(ternary_mod_bdd_op(dd_manager, ebf, n, 1, 1, context));
  case ABF:     return(ternary_mod_bdd_op(dd_manager, ebg, n, -1, -1, context));
  case EBG:     return(ternary_mod_bdd_op(dd_manager, ebg, n, 1, 1, context));
  case ABG:     return(ternary_mod_bdd_op(dd_manager, ebf, n, -1, -1, context));
  default:      
    { /* for all the other we call the eval, and convert the result from ADD to BDD. */
      bdd_ptr tmp_bdd, res_bdd;
      add_ptr res_add = eval(n, context);
      
      if (res_add == NULL) {
        rpterr("eval_spec: res = NULL after a call to \"eval\".");
        nusmv_exit(1);
      }
      /* Yuan Lu
      res_bdd = add_to_bdd(dd_manager, res_add);
      */
      res_bdd = add_to_bdd(dd_manager, res_add);
      add_free(dd_manager, res_add);
      tmp_bdd = Abs_AbsAbstractCur(res_bdd, abs_expr_pre);
      bdd_free(dd_manager, res_bdd);
      res_bdd = Abs_AbsAbstractCur(tmp_bdd, abs_expr);
      bdd_free(dd_manager, tmp_bdd);
      /* if(dd_check_manager(dd_manager)) exit(0); */
      return(res_bdd);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Recursive step of <code>eval_compute</code>.]

  Description        [Performs the recursive step of <code>eval_compute</code>.]

  SideEffects        []

  SeeAlso            [eval_compute]

******************************************************************************/
static int eval_compute_recur(node_ptr n, node_ptr context)
{
  if (n == Nil) internal_error("eval_compute_recur: n = NIL\n");
  switch(node_get_type(n)){
  case CONTEXT: return(eval_compute_recur(cdr(n),car(n)));
  case MINU: return((int)binary_mod_bdd_op_ns(dd_manager, (BDDPFBB)minu, n, 1, 1, 1, context));
  case MAXU: return((int)binary_mod_bdd_op_ns(dd_manager, (BDDPFBB)maxu, n, 1, 1, 1, context));
  default:   internal_error("eval_compute: type = %d\n", node_get_type(n));
  }
}

/**Function********************************************************************

  Synopsis           [Applies unary operation.]

  Description        [Takes in input the expression <code>n</code> and a
  unary operation <code>op</code>. Evaluates <code>n</n> and applies to this
  partial result the unary operator <code>op</code>. The sign of the
  partial result and of the result depends respectively from the flag
  <code>argflag</code> and <code>resflag</code>.] 

  SideEffects        []

  SeeAlso            [binary_bdd_op, ternary_bdd_op, quaternary_bdd_op]

******************************************************************************/
static bdd_ptr unary_bdd_op(DdManager * dd, BDDPFDB op, node_ptr n,
                            int resflag, int argflag, node_ptr context)
{
  bdd_ptr tmp_1, tmp_2, res;
  bdd_ptr arg = eval_spec(car(n), context);

  set_the_node(n);

  /* compute and ref argument of operation according its sign */
  tmp_1 = eval_sign(arg, argflag);

  /* apply and ref the result of the application of "op" to previous arg. */
  tmp_2 = op(dd, tmp_1);

  /* compute and ref the result according to sign of the result */
  res = eval_sign(tmp_2, resflag);

  /* free temporary results */
  bdd_free(dd, tmp_1);
  bdd_free(dd, tmp_2);
  bdd_free(dd, arg);

  return(res);
}

/**Function********************************************************************

  Synopsis           [Applies binary operation.]

  Description        [Takes in input the expression <code>n</code> and a
  binary operation <code>op</code>. Extracts from <code>n</n> the operands
  and evaluates them. The binary operator <code>op</code> is then applied
  to these partial results. The sign of the partial results and of the
  result depends respectively from the flags <code>argflag1</code>,
  <code>argflag2</code> and <code>resflag</code>.]

  SideEffects        []

  SeeAlso            [unary_bdd_op, ternary_bdd_op, quaternary_bdd_op]

******************************************************************************/
static bdd_ptr binary_bdd_op(DdManager * dd, BDDPFDBB op, node_ptr n,
                             int resflag, int argflag1, int argflag2, node_ptr context)
{
  bdd_ptr tmp_1, tmp_2, tmp_3, res;
  bdd_ptr arg1 = eval_spec(car(n), context);
  bdd_ptr arg2 = eval_spec(cdr(n), context);

  set_the_node(n);

  tmp_1 = eval_sign(arg1, argflag1);
  tmp_2 = eval_sign(arg2, argflag2);
  tmp_3 = op(dd, tmp_1, tmp_2);
  res = eval_sign(tmp_3, resflag);

  bdd_free(dd, tmp_1);
  bdd_free(dd, tmp_2);
  bdd_free(dd, tmp_3);
  bdd_free(dd, arg1);
  bdd_free(dd, arg2);

  return(res);
}


/**Function********************************************************************

  Synopsis           [Applies unary operation.]

  Description        [Takes in input the expression <code>n</code> and a
  unary operation <code>op</code>. Evaluates <code>n</n> and applies to this
  partial result the unary operator <code>op</code>. The sign of the
  partial result and of the result depends respectively from the flag
  <code>argflag</code> and <code>resflag</code>.] 

  SideEffects        []

  SeeAlso            [binary_bdd_op, ternary_bdd_op, quaternary_bdd_op]

******************************************************************************/
static bdd_ptr unary_mod_bdd_op(DdManager * dd, BDDPFB op, node_ptr n,
                                int resflag, int argflag, node_ptr context)
{
  bdd_ptr tmp_1, tmp_2, res;
  bdd_ptr arg = eval_spec(car(n), context);

  set_the_node(n);

  /* compute and ref argument of operation according its sign */
  tmp_1 = eval_sign(arg, argflag);

  /* apply and ref the result of the application of "op" to previous arg. */
  tmp_2 = op(tmp_1);

  /* compute and ref the result according to sign of the result */
  res = eval_sign(tmp_2, resflag);

  /* free temporary results */
  bdd_free(dd, tmp_1);
  bdd_free(dd, tmp_2);
  bdd_free(dd, arg);

  return(res);
}

/**Function********************************************************************

  Synopsis           [Applies binary operation.]

  Description        [Takes in input the expression <code>n</code> and a
  binary operation <code>op</code>. Extracts from <code>n</n> the operands
  and evaluates them. The binary operator <code>op</code> is then applied
  to these partial results. The sign of the partial results and of the
  result depends respectively from the flags <code>argflag1</code>,
  <code>argflag2</code> and <code>resflag</code>.]

  SideEffects        []

  SeeAlso            [unary_bdd_op, ternary_bdd_op, quaternary_bdd_op]

******************************************************************************/
static bdd_ptr binary_mod_bdd_op(DdManager * dd, BDDPFBB op, node_ptr n, int resflag,
                                 int argflag1, int argflag2, node_ptr context)
{
  bdd_ptr tmp_1, tmp_2, tmp_3, res;
  bdd_ptr arg1 = eval_spec(car(n), context);
  bdd_ptr arg2 = eval_spec(cdr(n), context);

  set_the_node(n);

  tmp_1 = eval_sign(arg1, argflag1);
  tmp_2 = eval_sign(arg2, argflag2);
  tmp_3 = op(tmp_1, tmp_2);
  res = eval_sign(tmp_3, resflag);

  bdd_free(dd, tmp_1);
  bdd_free(dd, tmp_2);
  bdd_free(dd, tmp_3);
  bdd_free(dd, arg1);
  bdd_free(dd, arg2);

  return(res);
}

/**Function********************************************************************

  Synopsis           [Applies binary operation.]

  Description        [Takes in input the expression <code>n</code> and a
  binary operation <code>op</code>. Extracts from <code>n</n> the operands
  and evaluates them. The binary operator <code>op</code> is then applied
  to these partial results. The sign of the partial results and of the
  result depends respectively from the flags <code>argflag1</code>,
  <code>argflag2</code> and <code>resflag</code>.<br>
  The only difference between this and "binary_mod_bdd_op" is that the
  result of the application of the operation passed as argument is not
  referenced. This is used for example in the "minu" and "maxu" operations.]

  SideEffects        []

  SeeAlso            [unary_bdd_op, ternary_bdd_op, quaternary_bdd_op]

******************************************************************************/
static bdd_ptr binary_mod_bdd_op_ns(DdManager * dd, BDDPFBB op, node_ptr n, int resflag,
                                    int argflag1, int argflag2, node_ptr context)
{
  bdd_ptr tmp_1, tmp_2, res;
  bdd_ptr arg1 = eval_spec(car(n), context);
  bdd_ptr arg2 = eval_spec(cdr(n), context);

  set_the_node(n);

  tmp_1 = eval_sign(arg1, argflag1);
  tmp_2 = eval_sign(arg2, argflag2);
  res = op(tmp_1, tmp_2);

  /* Yuan Lu
  bdd_free(dd, tmp_1);
  bdd_free(dd, tmp_2);
  */
  bdd_free(dd, arg1);
  bdd_free(dd, arg2);

  return(res);
}

/**Function********************************************************************

  Synopsis           [Applies ternary operation.]

  Description        [Takes in input the expression <code>n</code> and a
  ternary operation <code>op</code>. Extracts from <code>n</n> the operands
  and evaluates them.<br>
  The second and third arguments have to evaluate to numbers. And
  <code>op</code> is a function that takes as input an BDD an two integers.
  The ternary operator <code>op</code> is then applied to these partial
  results. The sign of the partial result and of the result depends
  respectively from the flags <code>argflag</code> and <code>resflag</code>.]

  SideEffects        []

  SeeAlso            [unary_bdd_op, binary_bdd_op, quaternary_bdd_op]

******************************************************************************/
static bdd_ptr ternary_mod_bdd_op(DdManager * dd, BDDPFBII op, node_ptr n,
                                  int resflag, int argflag, node_ptr context)
{
  bdd_ptr tmp_1, tmp_2, res;
  bdd_ptr arg1 = eval_spec(car(n), context);
  int arg2 = eval_num(car(cdr(n)), context);
  int arg3 = eval_num(cdr(cdr(n)), context);

  set_the_node(n);

  tmp_1 = eval_sign(arg1, argflag);
  tmp_2 = op(tmp_1, arg2, arg3);
  res = eval_sign(tmp_2, resflag);

  bdd_free(dd, tmp_1);
  bdd_free(dd, tmp_2);
  bdd_free(dd, arg1);

  return(res);
}

/**Function********************************************************************

  Synopsis           [Applies quaternary operation.]

  Description        [Takes in input the expression <code>n</code> and a
  quaternary operation <code>op</code>. Extracts from <code>n</n> the operands
  and evaluates them.<br>
  The third and fourth arguments have to evaluate to numbers. And
  <code>op</code> is a function that takes as input two BDD and two integers.
  The quaternary operator <code>op</code> is then applied to these partial
  results. The sign of the partial result and of the result depends
  respectively from the flags <code>argflag1</code>, <code>argflag2</code> and
  <code>resflag</code>.]

  SideEffects        []

  SeeAlso            [unary_bdd_op, binary_bdd_op, ternary_bdd_op]

******************************************************************************/
static bdd_ptr quad_mod_bdd_op(DdManager * dd, BDDPFBBII op, node_ptr n, int resflag,
                               int argflag1, int argflag2, node_ptr context)
{
  bdd_ptr tmp_1, tmp_2, tmp_3, res;
  bdd_ptr arg1 = eval_spec(car(car(n)), context);
  bdd_ptr arg2 = eval_spec(cdr(car(n)), context);
  int arg3 = eval_num(car(cdr(n)), context);
  int arg4 = eval_num(cdr(cdr(n)), context);

  set_the_node(n);

  tmp_1 = eval_sign(arg1, argflag1);
  tmp_2 = eval_sign(arg2, argflag1);
  tmp_3 = op(tmp_1, tmp_2, arg3, arg4);
  res = eval_sign(res, resflag);

  bdd_free(dd, tmp_1);
  bdd_free(dd, tmp_2);
  bdd_free(dd, tmp_3);
  bdd_free(dd, arg1);
  bdd_free(dd, arg2);

  return(res);
}


/**Function********************************************************************

  Synopsis           [Complements a BDD according to a flag.]

  Description        [Given the BDD <code>a</code>, this function returns
  the negation of BDD <code>a</code> or <code>a</code> itself according the
  value of <code>flag</code>. If <code>flag = -1</code> then returns <code>not
  a</code>, else returns <code>a</code>. It is important that the BDD is a
  zero/one BDD (i.e. it has only zero or one as leaf).]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static bdd_ptr eval_sign(bdd_ptr a, int flag)
{
  if (flag == -1 ) return(bdd_not(dd_manager, a));
  else {
    bdd_ref(a);
    return(a);
  }
}
