/**CFile***********************************************************************

  FileName    [mcCmd.c]

  PackageName [mc]

  Synopsis    [Model checking commands.]

  Description [This file contains all the shell command to deal with
  model checking and for counterexample navigation.]

  SeeAlso     [cmdCmd.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "mcInt.h" 

#define NEWCMD 1
static char rcsid[] UTIL_UNUSED = "$Id: $";

/* prototypes of the command functions */
int CommandCheckTrans ARGS((int argc, char **argv));
int CommandCheckSpec ARGS((int argc, char **argv));
int CommandCheckInvar ARGS((int argc, char **argv));
int CommandCheckInvarFB ARGS((int argc, char **argv));
int CommandCheckInvarStrong ARGS((int argc, char **argv));
int CommandCheckAfter ARGS((int argc, char **argv));
int CommandCheckCompute ARGS((int argc, char **argv));
int CommandComputeReachable ARGS((int argc, char **argv));
int CommandPrintReachableStates ARGS((int argc, char **argv));
int CommandGotoState ARGS((int argc, char **argv));
int CommandPrintCurrentState ARGS((int argc, char **argv));
int CommandAssign ARGS((int argc, char **argv));
int CommandStep ARGS((int argc, char **argv));
int CommandEval ARGS((int argc, char **argv));

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [It's used to store the current state when the interactive
  model inspection feature is enabled.]

  Description [It's used to store the current state when the interactive
  model inspection feature is enabled.]

******************************************************************************/
static bdd_ptr current_state_bdd = (bdd_ptr)NULL;


/**Variable********************************************************************

  Synopsis    [It's used to store the current state label when the interactive
  model inspection feature is enabled.]

  Description [It's used to store the current state label when the interactive
  model inspection feature is enabled.]

******************************************************************************/
static node_ptr current_state_label = Nil;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int UsageCheckTrans ARGS((void));
static int UsageCheckSpec ARGS((void));
static int UsageCheckInvar ARGS((void));
static int UsageCheckInvarFB ARGS((void));
static int UsageCheckInvarStrong ARGS((void));
static int UsageCheckAfter ARGS((void));
static int UsageCheckCompute ARGS((void));
static int UsageComputeReachable ARGS((void));
static int UsagePrintReachableStates ARGS((void));
static int UsageGotoState  ARGS((void));
static int UsagePrintCurrentState ARGS((void));
static int UsageAssign ARGS((void));
static int UsageStep ARGS((void));
static int UsageEval ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the mc package.]

  Description        [Initializes the mc package.]

  SideEffects        []

******************************************************************************/
void Mc_Init(void){
  Cmd_CommandAdd("check_trans", CommandCheckTrans, 0);
  Cmd_CommandAdd("check_spec", CommandCheckSpec, 0);
  Cmd_CommandAdd("compute", CommandCheckCompute, 0);
  Cmd_CommandAdd("check_invar", CommandCheckInvar, 0);
  Cmd_CommandAdd("_check_invar_fb", CommandCheckInvarFB, 0);
  Cmd_CommandAdd("_check_invar_strong", CommandCheckInvarStrong, 0);
  Cmd_CommandAdd("_check_after", CommandCheckAfter, 0);
  Cmd_CommandAdd("compute_reachable", CommandComputeReachable, 0);
  Cmd_CommandAdd("print_reachable_states", CommandPrintReachableStates, 0);
  Cmd_CommandAdd("goto_state", CommandGotoState, 0);
  Cmd_CommandAdd("print_current_state", CommandPrintCurrentState, 0);
  Cmd_CommandAdd("assign", CommandAssign, 0);
  Cmd_CommandAdd("step", CommandStep, 0);
  Cmd_CommandAdd("eval", CommandEval, 0);
}

/**Function********************************************************************

  Synopsis           [Quit the mc package]

  Description        [Quit the mc package]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Mc_End(void)
{
  if (fair_states_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, fair_states_bdd);
  if (fairness_constraints_bdd != Nil) walk_dd(dd_manager, bdd_free, fairness_constraints_bdd);
  if (reachable_states_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, reachable_states_bdd);
  if (reachable_states_layers != Nil)  walk_dd(dd_manager, bdd_free, reachable_states_layers);
}

/**Function********************************************************************

  Synopsis           [Checks the transition relation for totality.]

  CommandName        [check_trans] 	   

  CommandSynopsis    [Checks the transition relation for totality.]  

  CommandArguments   [\[-h\] \[-m \| -o output-file\]]

  CommandDescription [
  Checks if the transition relation is total. If the transition
  relation is not total then a potential deadlock state is shown out.
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command to the program
            specified by the <tt>PAGER</tt> shell variable if
            defined, else through the UNIX "more" command.
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command to the file
       <tt>output-file</tt>.
  </dl>
  The reachable states are computed before in order to guarantee that
  deadlock states are actually reachable.]

  SideEffects        []

******************************************************************************/
int CommandCheckTrans(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hmo:")) != EOF){
    switch(c){
    case 'h': return(UsageCheckTrans());
    case 'o':
      if (useMore == 1) return(UsageCheckTrans());
      dbgFileName = util_strsav(util_optarg);
      offSet +=2 ;
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckTrans());
      useMore = 1;
      offSet++;
      break;
    default:  return(UsageCheckTrans());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }

  
  if (argc != util_optind) return(UsageCheckTrans());

  check_transition_relation();
  set_forward_search(options);

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  return(0);
}

static int UsageCheckTrans()
{
  (void) fprintf(nusmv_stderr, "usage: check_trans [-h] [-m| -o file]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" environment variable if defined,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\tWrites the generated output to \"file\".\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Perform fair CTL model checking.]

  CommandName        [check_spec] 	   

  CommandSynopsis    [Perform fair CTL model checking.]  

  CommandArguments   [\[-h\] \[-m | -o output-file\] \[ctl-expression\]]  

  CommandDescription [Evaluates <tt>SPEC</tt> statements. If no
  <tt>ctl-expression</tt> is given at command line, then the command evaluates
  the previously parsed <tt>SPEC</tt> statements, if any. If a formula
  is false, a path witnessing its falsity is printed. Infinite paths
  are represented by a finite sequence of states leading to a cycle.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>SPEC</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the <tt>UNIX</tt> "more" command.
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>SPEC</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>ctl-expression</tt>
       <dd> A CTL formula to be checked.
  </dl><p>

  If the <tt>ag_only_search</tt> environment variable has been set, and
  the set of reachable states has been already computed, then a
  specialized algorithm to check AG formulas is used instead of the
  standard model checking algorithms.]  

  SideEffects        []

******************************************************************************/
int CommandCheckSpec(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hmo:")) != EOF){
    switch(c){
    case 'h': return(UsageCheckSpec());
    case 'o':
      if (useMore == 1) return(UsageCheckSpec());
      dbgFileName = util_strsav(util_optarg);
      offSet +=2 ;
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckSpec());
      useMore = 1;
      offSet++;
      break;
    default:  return(UsageCheckSpec());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
    
  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  
  if ((argc - util_optind) == 0) {
    if (!opt_ignore_spec(options)) {
      node_ptr l = cmp_struct_get_spec(cmps);

      if (opt_verbose_level_gt(options, 1))
        fprintf(nusmv_stdout, "READING SPEC FROM MODEL FILE.\n");
      while (l) { /* Loop over specifications */
        node_ptr spec = car(l);

        l = cdr(l);
        if(!(opt_ag_only(options) && opt_forward_search(options))) {
          node_ptr actual_spec = cdr(spec);

          /* decide if it is a SPEC or a COMPUTE */
          switch (node_get_type(actual_spec)) {
          case MINU:
          case MAXU: break;
          default:
            check_spec(spec);
            break;
          };
        }
        else {
          if (reachable_states_bdd == (bdd_ptr)NULL) {
            fprintf(nusmv_stderr, "The forward search and AG only search have been enabled, but the reachable\n");
            fprintf(nusmv_stderr, "states have not yet been computed. Use \"compute_reachable\" to do it.\n");
            return(1);
          }
          check_AG_only(spec, Nil);
        }
      }
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
  }
  else {
    int i;
    node_ptr parsed_command = Nil;
    int status = 0;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "SPEC ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == SPEC)) {
        if(!(opt_ag_only(options) && opt_forward_search(options))) {
          check_spec(car(parsed_command));
        } else {
          check_AG_only(car(parsed_command), Nil);
        }
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"check_spec ctl_expr\"\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"check_spec ctl_expr\"\n");
      status = 1;
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    return(status);
  }  
  return(0);
}

static int UsageCheckSpec()
{
  (void) fprintf(nusmv_stderr, "usage: check_spec [-h] [-m| -o file] [\"ctl-expr\"]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" environment variable if defined,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\tWrites the generated output to \"file\".\n");
  (void) fprintf(nusmv_stderr, "   ctl-expr \tif ctl-expr, then it is model checked, else\n");
  (void) fprintf(nusmv_stderr, "    \t\tif not specified the specs from model file are used.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Performs model checking of invariants]

  CommandName        [check_invar] 	   

  CommandSynopsis    [Performs model checking of invariants]  

  CommandArguments   [\[-h\] \[-m \| -o output-file\] \[invar-expression\]]  

  CommandDescription [Performs invariant checking on the given
  model. An invariant is a set of states. Checking the invariant is
  the process of determining that all states reachable from the
  initial states lie in the invariant.

  Invariants to be verified should be provided as simple formulas
  (without temporal operators) in the input file via the
  <tt>INVARSPEC</tt> keyword or directly at command line.<p>

  All the fairness conditions associated with the model are
  ignored.<p>

  If an invariant does not hold, a proof of failure is demonstrated.
  This consists of a path starting from an initial state to a state
  lying outside the invariant. This path has the property that it is the
  shortest path leading to a state outside the invariant.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the program in processing
           <tt>INVARSPEC</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the UNIX "more" command.
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>INVARSPEC</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>invar-expression</tt>
       <dd> An invariant to be verified.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckInvar(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hmo:")) != EOF){
    switch(c){
    case 'h': return(UsageCheckInvar());
    case 'o':
      if (useMore == 1) return(UsageCheckInvar());
      dbgFileName = util_strsav(util_optarg);
      offSet +=2 ;
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckInvar());
      useMore = 1;
      offSet++;
      break;
    default:  return(UsageCheckInvar());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
    
  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  if ((argc - util_optind) == 0) {
    if (!opt_ignore_invar(options)) {
      node_ptr l = cmp_struct_get_invar_spec(cmps);

      if (opt_verbose_level_gt(options, 1))
        fprintf(nusmv_stdout, "READING INVAR FROM MODEL FILE.\n");
      while (l) { /* Loop over specifications */
        node_ptr invar = car(l);

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_invar(nusmv_stderr, invar);
          fprintf(nusmv_stderr, "\n");
        }
        check_invar(invar);
      }
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
  }
  else {
    int i;
    node_ptr parsed_command = Nil;
    int status = 0;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "INVARSPEC ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == INVARSPEC)) {
        node_ptr invar = car(parsed_command);

        fprintf(nusmv_stdout, "Checking invariant:");
        print_node(nusmv_stdout, invar);
        fprintf(nusmv_stdout, "\n");
        check_invariant_forward(invar);
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"check_invar invar_expr\"\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"check_invar invar_expr\"\n");
      status = 1;
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    return(status);
  }  
  return 0;
}

static int UsageCheckInvar()
{
  (void) fprintf(nusmv_stderr, "usage: check_invar [-h] [-m| -o file] [\"simple-expr\"]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" shell variable if defined,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\tWrites the generated output to \"file\".\n");
  (void) fprintf(nusmv_stderr, "   simple-expr \tif simple-expr, then it is checked, else\n");
  (void) fprintf(nusmv_stderr, "    \t\tif not specified the invar from model file are used.\n");
  return 1;
}

/*x*Function********************************************************************

  Synopsis           [Implements the check_invar_fb command.]

  CommandName        [_check_invar_fb] 	   

  CommandSynopsis    [Implements the check_invar_fb command.]  

  CommandArguments   [\[-h\] \[-m \| -o output-file\] \[invar-expression\]]  

  CommandDescription [
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the program in processing
           <tt>CHECKINVARIANTFB</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else through the
           UNIX "more" command.
    <dt> <tt>-o output_file</tt>
       <dd> Writes the output generated by the program in processing
           <tt>CHECKINVARIANTFB</em>s to the file <tt>output-file</tt>.
    <dt> <tt>invar-expression</em>
       <dd> An invariant to be checked.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckInvarFB(int argc, char ** argv)
{
  int c;
  int offSet = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hmo:")) != EOF){
    switch(c){
    case 'h': return(UsageCheckInvarFB());
    case 'o':
      if (useMore == 1) return(UsageCheckInvarFB());
      dbgFileName = util_strsav(util_optarg);
      offSet +=2 ;
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckInvarFB());
      useMore = 1;
      offSet++;
      break;
    default:  return(UsageCheckInvarFB());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  if ((argc - util_optind) == 0) {
    if (!opt_ignore_invar_fb(options)) {
      node_ptr l = cmp_struct_get_invar_fb(cmps);

      if (opt_verbose_level_gt(options, 1))
        fprintf(nusmv_stdout, "READING INVAR_FB FROM MODEL FILE.\n");
      while (l) { /* Loop over specifications */
        node_ptr invar = car(l);

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_invar(nusmv_stderr, invar);
          fprintf(nusmv_stderr, "\n");
        }
        check_invariant_forward_backward(invar);
      }
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
  }
  else {
    int i;
    node_ptr parsed_command = Nil;
    int status = 0;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "CHECKINVARIANTFB ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == CHECKINVARIANTFB)) {
        node_ptr invar = car(parsed_command);

        fprintf(nusmv_stdout, "Checking invariant:");
        print_node(nusmv_stdout, invar);
        fprintf(nusmv_stdout, "\n");
        check_invariant_forward_backward(invar);
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"check_invar_fb invar_expr\"\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"check_invar_fb invar_expr\"\n");
      status = 1;
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    return(status);
  }  
  return 0;
}

static int UsageCheckInvarFB()
{
  (void) fprintf(nusmv_stderr, "usage: check_invar_fb [-h] [-m| -o file] [\"simple-expr\"]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" shell variable if defined,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\tWrites the generated output to \"file\".\n");
  (void) fprintf(nusmv_stderr, "   simple-expr \tif simple-expr, then it is checked, else\n");
  (void) fprintf(nusmv_stderr, "    \t\tif not specified the fb invar from model file are used.\n");
  return 1;
}

/*x*Function********************************************************************

  Synopsis           [Implements the check_invar_strong command.]

  CommandName        [_check_invar_strong] 	   

  CommandSynopsis    [Implements the check_invar_strong command.]  

  CommandArguments   [\[-h\] \[-m \| -o file\] \[invar-expression\]]  

  CommandDescription [
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>CHECKINVARIANTSTRONG</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else through the
           UNIX "more" command.
    <dt> <tt>-o output_file</tt>
       <dd> Writes the output generated by the command in processing
           <em>CHECKINVARIANTSTRONG</em>s to the file <tt>output-file</tt>.
    <dt> <tt>invar-expression</em>
       <dd> if a <tt>invar-expression</tt> is specified, then it is model
            checked, else if not specified the strong invar from model file
            are used.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckInvarStrong(int argc, char ** argv)
{
  int c;
  int offSet = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hmo:")) != EOF){
    switch(c){
    case 'h': return(UsageCheckInvarStrong());
    case 'o':
      if (useMore == 1) return(UsageCheckInvarStrong());
      dbgFileName = util_strsav(util_optarg);
      offSet +=2 ;
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckInvarStrong());
      useMore = 1;
      offSet++;
      break;
    default:  return(UsageCheckInvarStrong());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
    
  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  if ((argc - util_optind) == 0) {
    if (!opt_ignore_invar_strong(options)) {
      node_ptr l = cmp_struct_get_invar_strong(cmps);

      if (opt_verbose_level_gt(options, 1))
        fprintf(nusmv_stdout, "READING INVAR_STRONG FROM MODEL FILE.\n");
      while (l) { /* Loop over specifications */
        node_ptr invar = car(l);

        l = cdr(l);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "evaluating ");
          print_invar(nusmv_stderr, invar);
          fprintf(nusmv_stderr, "\n");
        }
        check_invariant_strong(invar);
      }
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
  }
  else {
    int i;
    node_ptr parsed_command = Nil;
    int status = 0;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "CHECKINVARIANTSTRONG ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == CHECKINVARIANTSTRONG)) {
        node_ptr invar = car(parsed_command);

        fprintf(nusmv_stdout, "Checking strong invariant:");
        print_node(nusmv_stdout, invar);
        fprintf(nusmv_stdout, "\n");
        check_invariant_strong(invar);
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"check_invar_strong invar_expr\"\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"check_invar_strong invar_expr\"\n");
      status = 1;
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    return(status);
  }  
  return 0;
}

static int UsageCheckInvarStrong()
{
  (void) fprintf(nusmv_stderr, "usage: check_invar_strong [-h] [-m| -o file] [\"simple-expr\"]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" shell variable if defined,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\tWrites the generated output to \"file\".\n");
  (void) fprintf(nusmv_stderr, "   simple-expr \tif simple-expr, then it is checked, else\n");
  (void) fprintf(nusmv_stderr, "    \t\tif not specified the strong invar from model file are used.\n");
  return 1;
}

/*x*Function********************************************************************

  Synopsis           [Implements the check_after command.]

  CommandName        [_check_after] 	   

  CommandSynopsis    [Implements the check_after command.]  

  CommandArguments   [\[-h \| dl_expr : goal_expr\]]  

  CommandDescription [
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>dl_expr : goal_expr</tt>
       <dd> if a similar term is given, then it is checked, else the
            <tt>AFTER</tt> from the model file is used if any.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckAfter(int argc, char ** argv)
{
  int c;
  int offSet = 0;
  int status = 0;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageCheckAfter());
    default:  return(UsageCheckAfter());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
    
  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (argc == 1) {
    if (!opt_ignore_after(options)){
      node_ptr l = cmp_struct_get_after(cmps);
      int j = 0;

      if (opt_verbose_level_gt(options, 1))
        fprintf(nusmv_stdout, "READING AFTER EXPR FROM MODEL FILE.\n");
      while (l) { /* Loop over specifications */
        node_ptr after = car(l);

        l = cdr(l); j++;
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "****************************************\n");
          fprintf(nusmv_stderr, "evaluating temporal projection #%d\n", j);
          fprintf(nusmv_stderr, "****************************************\n");
        }
        check_after(after);
      }
    }
  }
  else {
    int i;
    node_ptr parsed_command = Nil;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "AFTER ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == AFTER)) {
        node_ptr after = car(parsed_command);

        check_after(after);
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"check_after dl_expr : goal_expr\"\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"check_after dl_expr : goal_expr\"\n");
      status = 1;
    }
  }
  return(status);
}  

static int UsageCheckAfter()
{
  (void) fprintf(nusmv_stderr, "usage: check_after [-h | dl_expr : goal_expr]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrintsthe command usage\n");
  (void) fprintf(nusmv_stderr, "    \t\tif the a term of the form \"dl_expr : goal_expr\" is given, then it is checked, else\n");
  (void) fprintf(nusmv_stderr, "    \t\tif not specified the after from model file are used.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Performs computation of quantitative characteristics]

  CommandName        [compute] 	   

  CommandSynopsis    [Performs computation of quantitative characteristics]

  CommandArguments   [\[-h\] \[-m \| -o output-file\] \[compute-expression\]]  

  CommandDescription [This command deals with the computation of
  quantitative characteristics of real time systems. It is able to
  compute the length of the shortest (longest) path from two given set
  of states.<p>
  
  <center><code>MAX \[ alpha , beta \]</code></center><br>
  <center><code>MIN \[ alpha , beta \]</code></center><p>
  
  Properties of the above form can be specified in the input file via
  the keyword <code>COMPUTE</code> or directly at command line.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>COMPUTE</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the UNIX "more" command.
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>COMPUTE</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>compute-expression</tt>
       <dd> A quantitative characteristic to be computed.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckCompute(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hmo:")) != EOF){
    switch(c){
    case 'h': return(UsageCheckCompute());
    case 'o':
      if (useMore == 1) return(UsageCheckCompute());
      dbgFileName = util_strsav(util_optarg);
      offSet +=2 ;
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckCompute());
      useMore = 1;
      offSet++;
      break;
    default:  return(UsageCheckCompute());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
    
  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  if ((argc - util_optind) == 0) {
    if (!opt_ignore_spec(options)) {
      node_ptr l = cmp_struct_get_spec(cmps);

      if (opt_verbose_level_gt(options, 1))
        fprintf(nusmv_stdout, "READING COMPUTE FROM MODEL FILE.\n");
      while (l) { /* Loop over specifications */
        node_ptr spec = car(l);

        l = cdr(l);
        {
          node_ptr actual_spec = cdr(spec);

          /* decide if it is a SPEC or a COMPUTE */
          switch (node_get_type(actual_spec)) {
          case MINU:
          case MAXU: check_compute(spec); break;
          default: break;
          };
        }
      }
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
  }
  else {
    int i;
    node_ptr parsed_command = Nil;
    int status = 0;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "COMPUTE ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == COMPUTE)) {
        check_spec(car(parsed_command));
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"compute compute_expr\"\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"compute compute_expr\"\n");
      status = 1;
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    return(status);
  }  
  return 0;
}

static int UsageCheckCompute()
{
  (void) fprintf(nusmv_stderr, "usage: compute [-h] [-m| -o file] [compute-expr]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" shell variable if defined,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\tWrites the generated output to \"file\".\n");
  (void) fprintf(nusmv_stderr, "   compute-expr \tif compute-expr, then it is model checked, else");
  (void) fprintf(nusmv_stderr, "    \t\tif not specified the specs from model file are used.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Computes the set of reachable states]

  CommandName        [compute_reachable] 	   

  CommandSynopsis    [Computes the set of reachable states]  

  CommandArguments   [\[-h\]]  

  CommandDescription [Computes the set of reachable states. The result
  is then used to simplify image and preimage computations. This can
  result in improved performances for models with sparse state
  spaces. Sometimes this option may slow down the performances because
  the computation of reachable states may be very expensive. The
  environment variable <tt>forward_search</tt> is set.]

  SideEffects        []

******************************************************************************/
int CommandComputeReachable(int argc, char **argv)
{
  int c;
  
  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageComputeReachable());
    default:  return(UsageComputeReachable());
    }
  }
  if (argc != util_optind) return(UsageComputeReachable());

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }
  
  if (reachable_states_bdd == (bdd_ptr)NULL) {
    if (opt_verbose_level_gt(options, 0)){
      fprintf(nusmv_stderr, "Starting computation of reachable states....\n");
    }
    compute_reachable_states();
    set_forward_search(options);
    if (opt_verbose_level_gt(options, 0)){
      fprintf(nusmv_stderr, "done.\n");
    }
  }
  else {
    fprintf(nusmv_stderr, "Reachable States already computed.\n");
  }
  return 0;
}

static int UsageComputeReachable()
{
  (void) fprintf(nusmv_stderr, "usage: compute_reachable [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Prints the number of reachable states.]

  CommandName        [print_reachable_states] 	   

  CommandSynopsis    [Prints the number of reachable states.]  

  CommandArguments   [\[-h\]]  

  CommandDescription [Prints the number of reachable states of the
  given model. The reachable states are coputed if needed.]

  SideEffects        []

******************************************************************************/
int CommandPrintReachableStates(int argc, char **argv)
{
  int c;
  /*
   * Parse the command line.
  */
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch (c) {
    case 'h': return(UsagePrintReachableStates());
    default:
      return(UsagePrintReachableStates());
    }
  }
  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  set_forward_search(options);
  set_print_reachable(options);
  (void) fprintf(nusmv_stdout, "######################################################################\n");
  (void) print_reachable_states();
  (void) fprintf(nusmv_stdout, "######################################################################\n");
  return(0);
}

static int UsagePrintReachableStates()
{
  (void) fprintf(nusmv_stderr, "usage: print_reachable_states [-h]]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Goes to a given state of a trace]

  CommandName        [goto_state] 	   

  CommandSynopsis    [Goes to a given state of a trace]  

  CommandArguments   [\[-h\] state]  

  CommandDescription [Makes <tt>state</tt> the <em>current
  state</em>. This command is used to navigate alongs traces
  produced by NuSMV. During the navigation, there is a <em>current
  state</em>, and the <em>current trace</em> is the trace the
  <em>current state</em> belongs to.]  

  SideEffects        [<em>state</em> became the current state.]

******************************************************************************/
int CommandGotoState(int argc, char **argv)
{
  int c;
  int status = 0;
  int offSet = 0;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageGotoState());
    default:  return(UsageGotoState());
    }
  }
  if (argc == 1) return(UsageGotoState());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  {
    int i;
    node_ptr parsed_command = Nil;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "GOTO ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == GOTO)) {
        node_ptr label = car(parsed_command);
        bdd_ptr state   = (bdd_ptr)lookup_traces_hash(label);
          
        if (state) {
          if (current_state_bdd != (bdd_ptr)NULL)
            bdd_free(dd_manager, current_state_bdd);
          current_state_bdd = bdd_dup(state);
          current_state_label = label;
          clear_print_hash();
          print_state(state, 1);
        }
        else {
          indent_node(nusmv_stderr, "The label ", label, " is undefined.\n");
        }
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"goto_state state\".\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"goto_state state\".\n");
      status = 1;
    }
  }
  return(status);
}

static int UsageGotoState()
{
  (void) fprintf(nusmv_stderr, "usage: goto_state [-h] state\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  (void) fprintf(nusmv_stderr, "   state \tSets current state to \"state\".\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Prints the current state]

  CommandName        [print_current_state] 	   

  CommandSynopsis    [Prints the current state]  

  CommandArguments   [\[-h\] \[-v\]]  

  CommandDescription [Prints the name of the <em>current state</em> if
  defined.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-v</tt>
       <dd> Prints the value of all the state variables of the <em>current state</em>.
  </dl>
  ]  

  SideEffects        []

******************************************************************************/
int CommandPrintCurrentState(int argc, char ** argv)
{
  int c;
  int Verbosely = 1;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hv")) != EOF){
    switch(c){
    case 'h': return(UsagePrintCurrentState());
    case 'v': {
      Verbosely = 0;
      break;
    }
    default:  return(UsagePrintCurrentState());
    }
  }

  if ((current_state_label != Nil) && (current_state_bdd != (bdd_ptr)NULL)) {
    (void) fprintf(nusmv_stdout, "Current state is ");
    (void) print_node(nusmv_stdout, current_state_label);
    (void) fprintf(nusmv_stdout, "\n");
    if (Verbosely == 0) print_state(current_state_bdd, Verbosely);
  }
  else {
    (void) fprintf(nusmv_stdout, "The current state has not yet been defined.\n");
    (void) fprintf(nusmv_stdout, "Use \"goto_state\" to define the current state.\n");
    return(1);
  }
  return(0);
}

static int UsagePrintCurrentState()
{
  (void) fprintf(nusmv_stderr, "usage: print_current_state [-h] [-v]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  (void) fprintf(nusmv_stderr, "   -v \t\tPrints the value of each state variable in the current state.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Assigns the evaluation of an expression to a
  given variable]

  CommandName        [assign] 	   

  CommandSynopsis    [Assigns the evaluation of an expression to a
  given variable]  

  CommandArguments   [\[-h\] var_id := simple-expression]  

  CommandDescription [Assign the value of <tt>simple-expression</tt>, as
  evaluated in the <em>current state</em> to variable <tt>var_id</tt>.<p>

  Command options:<p>
  <dl>
    <dt> <tt>var_id := simple-expression</tt>
       <dd> Assigns the value of <tt>simple-expression</tt>, as evaluated in
       the <em>current state</em>, to <tt>var_id</tt>. This command
       changes the <em>current state</em>. The value of all other
       variables in the new <em>current state</em> remains the same as
       it was in the old <em>current state</em>.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandAssign(int argc, char **argv)
{
  int c;
  int status = 0;
  int offSet = 0;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageAssign());
    default:  return(UsageAssign());
    }
  }
  if (argc == 1) return(UsageAssign());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
  {
    int i;
    node_ptr parsed_command = Nil;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "LET ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == LET)) {
        node_ptr var = car(car(parsed_command));
        node_ptr val = cdr(car(parsed_command));
        node_ptr x;

        if (current_state_bdd == (bdd_ptr)NULL) {
          fprintf(nusmv_stderr, "No current state.\n");
          status = 1;
        }
        else {
          x = lookup_symbol_hash(eval_struct(var, Nil));
          if (!x || (node_get_type(x) != VAR)) {
            indent_node(stderr,"",var," is not a variable\n");
            status = 1;
          }
          else {
            add_ptr v = eval(var, Nil);
            add_ptr r = eval(val, Nil);
            add_ptr interactive_state_add = bdd_to_add(dd_manager, current_state_bdd);
            add_ptr res = add_if_then(dd_manager, interactive_state_add, r);
            add_ptr w = add_leaf(dd_manager, add_value(dd_manager, res));
            add_ptr q = add_setin(dd_manager, v, w);
            add_ptr zero = add_zero(dd_manager);

            if (q == zero) {
              indent_node(nusmv_stderr, "cannot assign ",
                          add_get_leaf(dd_manager, w), " to variable ");
              print_node(nusmv_stderr, var);
              fprintf(nusmv_stderr, "\n");
              status = 1;
            }
            else {
              bdd_ptr q_bdd = add_to_bdd(dd_manager, q);
              bdd_ptr support_v = bdd_support(dd_manager, v);
              bdd_ptr tmp_1 = bdd_forsome(dd_manager, current_state_bdd, support_v);
              bdd_ptr tmp_2 = bdd_and(dd_manager, tmp_1, q_bdd);

              bdd_free(dd_manager, current_state_bdd);
              current_state_bdd = bdd_pick_one_state(dd_manager, tmp_2);
              current_state_label = Nil;
              print_state(current_state_bdd, 1);
              bdd_free(dd_manager, q_bdd);
              bdd_free(dd_manager, support_v);
              bdd_free(dd_manager, tmp_1);
              bdd_free(dd_manager, tmp_2);
            }
            add_free(dd_manager, v);
            add_free(dd_manager, r);
            add_free(dd_manager, interactive_state_add);
            add_free(dd_manager, res);
            add_free(dd_manager, w);
            add_free(dd_manager, q);
            add_free(dd_manager, zero);
          }
        }
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"assign var_id := simple_expr\".\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"assign var_id := simple_expr\".\n");
      status = 1;
    }
  }
  return(status);
}
  
static int UsageAssign()
{
  (void) fprintf(nusmv_stderr, "usage: assign [-h] \"var_id := simple_expr\"\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tprint the command usage.\n");
  (void) fprintf(nusmv_stderr, "      \t\tEvaluates the assignment.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Moves to next state in the current trace.]

  CommandName        [step] 	   

  CommandSynopsis    [Moves to next state in the current trace.]  

  CommandArguments   [\[-h\] \[-v\]]  

  CommandDescription [Moves to next state in the <em>current
  trace</em>. If the <em>current state</em> is the last state in the
  trace, then a successor state is generated and stored.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-v</tt>
       <dd> Prints the value of all state variables, otherwise it
            prints only the values of the state variables that have
            changed from previous state.

  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandStep(int argc, char **argv)
{
  int c;
  int Verbosely = 1;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hv")) != EOF){
    switch(c){
    case 'v': {
      Verbosely = 0;
      break;
    }
    case 'h': return(UsageStep());
    default:  return(UsageStep());
    }
  }
  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (current_state_bdd == (bdd_ptr)NULL) {
    fprintf(nusmv_stderr, "no current state defined\n");
    return 1;
  }
  if (current_state_label) {
    node_ptr new_label = find_node(DOT, car(current_state_label),
                                   node_plus1(cdr(current_state_label)));
    bdd_ptr z = (bdd_ptr)lookup_traces_hash(new_label);

    if (z) {
      indent_node(nusmv_stdout,"state ", new_label, ":\n");
      bdd_free(dd_manager, current_state_bdd);
      current_state_bdd = bdd_dup(z);
      current_state_label = new_label;
      print_state(z, 1);
      return 0;
    }
    else {
      indent_node(nusmv_stderr, "No state ", new_label, "\n");
    }
  }
  
  {
    bdd_ptr one = bdd_one(dd_manager);
    node_ptr q = ex_explain(cons((node_ptr)current_state_bdd, Nil), one);

    if (!q ) {
      (void) fprintf(nusmv_stderr, "Current state has no successor\n");
    }
    else {
      (void) fprintf(nusmv_stderr, "The successor state is: \n"); 
      bdd_free(dd_manager, current_state_bdd);
      current_state_bdd = bdd_dup((bdd_ptr)car(q));
      print_state(current_state_bdd, Verbosely);
    }
    bdd_free(dd_manager, one);
  }
  return 0;
}

 
static int UsageStep()
{
  (void) fprintf(nusmv_stderr, "usage: step [-h] [-v]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tprint the command usage.\n");
  (void) fprintf(nusmv_stderr, "   -v \t\tforce the printing of the state,\n");
  (void) fprintf(nusmv_stderr, "      \t\talso if equal to the previous one.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Evaluates an expression in the current state.]

  CommandName        [eval] 	   

  CommandSynopsis    [Evaluates an expression in the current state.]  

  CommandArguments   [\[-h\] ctl-expression]

  CommandDescription [Evaluates <tt>ctl-expression</tt> in the <em>current
  state</em>. Since we deal with CTL formulae, the evaluation activates
  the model checking algorithms, and can produce a new trace starting
  from <em>current state</em>, to be used by later commands. If
  additional traces are created from state <em>n.i</em>, they are
  numbered <em>t+1, t+2, t+3, ...</em>, where <em>t</em> is the
  number corresponding to the last trace generated. Within these traces,
  the states are numbered <em>(t+i).1, (t+i).2, (t+i).3, ...</em>.<p>

  Command options:<p>
  <dl>
    <dt> <tt>ctl-expression</tt>
       <dd> The CTL formula to be evaluated.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandEval(int argc, char **argv)
{
  int c;
  int status = 0;
  int offSet = 0;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageEval());
    default:  return(UsageEval());
    }
  }
  if (argc == 1) return(UsageEval());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  {
    int i;
    node_ptr parsed_command = Nil;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "EVAL ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == EVAL)) {
        if (current_state_bdd == (bdd_ptr)NULL) {
          fprintf(nusmv_stderr,"no current state defined\n");
          status = 1;
        }
        else {
          node_ptr expression = car(parsed_command);
          bdd_ptr r_bdd       = eval_spec(expression, Nil);
          add_ptr r           = bdd_to_add(dd_manager, r_bdd);
          add_ptr i_s_add     = bdd_to_add(dd_manager, current_state_bdd);
          add_ptr tmp_1       = add_if_then(dd_manager, i_s_add, r);
          node_ptr w          = add_value(dd_manager, tmp_1);

          fprintf(nusmv_stdout, "######################################################################\n");
          indent_node(nusmv_stdout, "", expression, " ");
          indent_node(nusmv_stdout, "= ", w, "\n");
          fprintf(nusmv_stdout, "######################################################################\n");
          {
              node_ptr expl = reverse(explain(cons((node_ptr)current_state_bdd,
                                                   Nil), expression, Nil));
              if (expl) {
                print_explanation(expl);
              }
          }
          bdd_free(dd_manager, r_bdd);
          add_free(dd_manager, r);
          add_free(dd_manager, i_s_add);
          add_free(dd_manager, tmp_1);
        }
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"eval ctl_expr\".\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"eval ctl_expr\".\n");
      status = 1;
    }
  }
  return(status);
}

  
static int UsageEval()
{
  (void) fprintf(nusmv_stderr, "usage: eval [-h] ctl_expr\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tprint the command usage.\n");
  (void) fprintf(nusmv_stderr, "      \t\tEvaluates the ctl_expr.\n");
  return 1;
}
