/**CFile***********************************************************************

  FileName    [mcAGonly.c]

  PackageName [mc]

  Synopsis    [This file contains the code to deal with AG formulas in a special way.]

  Description [This file contains the code to deal with AG formulas
  only, using special purpose algorithms. This functionality is invoked
  with the -AG option and works only in conjunction with the -f
  (forward search) option.]

  Comments [None]

  SeeAlso     [mcMc.c mcEval.c mcExplain.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "mcInt.h" 
#include "mc.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static node_ptr cont_AG_counterexample(node_ptr l, node_ptr p);

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [This function checks for SPEC of the form AG alpha in "context" .]

  Description [The implicit assumption is that "spec" must be an AG
  formula (i.e. it must contain only conjunctions and AG's).  No attempt
  is done to normalize the formula (e.g. push negations). The AG mode
  relies on the previous computation and storage of the reachable
  state space (<tt>reachable_states_layers</tt>), used in
  counterexample computation.] 

  SideEffects        []

  SeeAlso            [check_spec]

******************************************************************************/
int check_AG_only(node_ptr spec, node_ptr context)
{
  if (spec == Nil) return(1);
  switch(node_get_type(spec)) {
  case CONTEXT:
    return(check_AG_only(cdr(spec),car(spec)));
  case AND:
    return(check_AG_only(spec->left.nodetype, context) &
           check_AG_only(spec->right.nodetype, context));
  case AG:
    {
      bdd_ptr tmp_1, tmp_2, acc;
      bdd_ptr zero = bdd_zero(dd_manager);
      bdd_ptr s0 = eval_spec(car(spec),context);

      { /*
          We quantify out the input variables from the result of the
          formula evaluation.
        */
        bdd_ptr tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);
        
        bdd_free(dd_manager, s0);
        s0 = tmp;
      }
      tmp_1 = bdd_not(dd_manager, s0);
      tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
      bdd_free(dd_manager, tmp_1);
      acc = bdd_and(dd_manager, reachable_states_bdd, tmp_2);
      bdd_free(dd_manager, tmp_2);
      fprintf(nusmv_stdout, "-- ");
      print_spec(nusmv_stdout, spec);
      if(acc == zero){
        fprintf(nusmv_stdout, "is true\n");
        bdd_free(dd_manager, s0);
        bdd_free(dd_manager, acc);
        bdd_free(dd_manager, zero);
        return(1);
      }
      {
	node_ptr l;

        fprintf(nusmv_stdout, "is false\n");        
        l = make_AG_counterexample(reachable_states_layers, acc);
        bdd_free(dd_manager, acc);
        bdd_free(dd_manager, s0);
        bdd_free(dd_manager, zero);
	print_explanation(l);
	return(0);
      }
    }
  default:
    return(0);
  }
}

/**Function********************************************************************

  Synopsis           [This function checks for SPEC of the form AG alpha in "context" .]

  Description [The implicit assumption is that "spec" must be an AG
  formula (i.e. it must contain only conjunctions and AG's).  No attempt
  is done to normalize the formula (e.g. push negations). The AG mode
  relies on the previous computation and storage of the reachable
  state space (<tt>reachable_states_layers</tt>), used in
  counterexample computation.]

  SideEffects        [Yuan Lu]

  SeeAlso            [check_spec]

******************************************************************************/
int check_AG_step(node_ptr spec, node_ptr context)
{
  if (spec == Nil) return(1);
  switch(node_get_type(spec)) {
  case CONTEXT:
    return(check_AG_step(cdr(spec),car(spec)));
  case AND:
    return(check_AG_step(spec->left.nodetype, context) &
           check_AG_step(spec->right.nodetype, context));
  case AG:
    {
      bdd_ptr tmp_1, tmp_2, acc;
      bdd_ptr zero = bdd_zero(dd_manager);
      bdd_ptr s0 = eval_spec(car(spec),context);
      int sign;

      { /*
          We quantify out the input variables from the result of the
          formula evaluation.
        */
        bdd_ptr tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);

        bdd_free(dd_manager, s0);
        s0 = tmp;
      }
      tmp_1 = bdd_not(dd_manager, s0);
      tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
      bdd_free(dd_manager, tmp_1);
      acc = compute_reachable_states_while_check(tmp_2);
      bdd_free(dd_manager, tmp_2);
      fprintf(nusmv_stdout, "-- ");
      print_spec(nusmv_stdout, spec);
      if(acc == zero){
        fprintf(nusmv_stdout, "is true\n");
        bdd_free(dd_manager, s0);
        bdd_free(dd_manager, acc);
        bdd_free(dd_manager, zero);
        return(1);
      }
      {
        node_ptr l;

        fprintf(nusmv_stdout, "is false\n");
        l = make_AG_counterexample(reachable_states_layers, acc);
        bdd_free(dd_manager, acc);
        bdd_free(dd_manager, s0);
        bdd_free(dd_manager, zero);
        sign = Abs_AbsRefineAbstraction(l);
        walk_dd(dd_manager, bdd_free, l);
      }
      return sign;
    }
  default:
    return 1;
  }
}


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [This function constructs a counterexample
  starting from state s0.]

  Description        [Compute a counterexample starting from a given state.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr make_AG_counterexample(node_ptr l, bdd_ptr s0)
{
  node_ptr p;
  bdd_ptr s, tmp_1, zero;

  if (l == Nil) return(Nil);
  p = make_AG_counterexample(cdr(l), s0);
  if (p != Nil) return(p);
  tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(l), s0);
  s = bdd_pick_one_state(dd_manager, tmp_1);
  bdd_free(dd_manager, tmp_1);
  if (s == (zero = bdd_zero(dd_manager))) {
    bdd_free(dd_manager, zero);
    return(Nil);
  }
  bdd_free(dd_manager, zero);
  return(cont_AG_counterexample(cdr(l), cons((node_ptr)s, Nil)));
}

/**Function********************************************************************

  Synopsis           [Auxiliary function to compute a counterexample.]

  Description        [Auxiliary function to compute a counterexample.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr cont_AG_counterexample(node_ptr l, node_ptr p)
{
  bdd_ptr s, tmp_1, tmp_2;
  bdd_ptr zero = bdd_zero(dd_manager);
  
  if (l == Nil) return(p);
  tmp_1 = Img_ImageBwd((bdd_ptr)car(p));
  tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
  bdd_free(dd_manager, tmp_1);
  tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(l), tmp_2);
  bdd_free(dd_manager, tmp_2);
  s = bdd_pick_one_state(dd_manager, tmp_1);
  if (s == zero) internal_error("make_AG_counterexample: s == ZERO");
  bdd_free(dd_manager, tmp_1);
  bdd_free(dd_manager, zero);
  return(cont_AG_counterexample(cdr(l),cons((node_ptr)s,p)));
}

