/**CFile***********************************************************************

  FileName    [ltlCmd.c]

  PackageName [ltl]

  Synopsis    [Shell commands to deal with the LTL model checking.]

  Description [Shell commands to deal with the LTL model checking.]

  SeeAlso     [mc]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "ltlInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

int CommandCheckLtlSpec(int argc, char ** argv);

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int PrintCheckLtlSpec ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the ltl package.]

  Description        [Initializes the ltl package.]

  SideEffects        [None]

******************************************************************************/
void Ltl_Init(void)
{
  Cmd_CommandAdd("check_ltlspec", CommandCheckLtlSpec, 0);
}

/**Function********************************************************************

  Synopsis           [Performs LTL model checking]

  CommandName        [check_ltlspec] 	   

  CommandSynopsis    [Performs LTL model checking]  

  CommandArguments   [\[-h\] \[-m \| -o output-file\] \[ltl-expression\]]  

  CommandDescription [
  Performs model checking of LTL formulas. If no <tt>ltl-expression</tt> is
  given at command line, then the previously parsed <tt>LTLSPEC</tt>,
  if any, are considered. LTL model checking is reduced to CTL model
  checking as described in the paper by \[CGH97\]<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>LTLSPEC</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the Unix "more" command.
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>LTLSPEC</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>ltl-expression</tt>
       <dd> A LTL formula to be checked.
  </dl>]

  SideEffects        [None]

******************************************************************************/
int CommandCheckLtlSpec(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hmo:")) != EOF){
    switch(c){
    case 'h': return(PrintCheckLtlSpec());
    case 'o':
      if (useMore == 1) return(PrintCheckLtlSpec());
      dbgFileName = util_strsav(util_optarg);
      offSet +=2 ;
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(PrintCheckLtlSpec());
      useMore = 1;
      offSet++;
      break;
    default:  return(PrintCheckLtlSpec());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
    
  if (!opt_monolithic(options)) {
    (void) fprintf(nusmv_stderr, "The \"check_ltlspec\" can be used only with \"Monolithic\" partition method.\n");
    (void) fprintf(nusmv_stderr, "Use \"build_model -f -m Monolithic\" to build the monolithic transition relation.\n");
    return(1);
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  
  if ((argc - util_optind) == 0) {
    node_ptr l = cmp_struct_get_ltlspec(cmps);

    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stdout, "READING LTLSPEC FROM MODEL FILE.\n");
    while (l) { /* Loop over LTL specifications */
      node_ptr spec = car(l);
  
      check_ltlspec(spec);
      l = cdr(l);
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
  }
  else {
    int i;
    node_ptr parsed_command = Nil;
    int status = 0;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "LTLSPEC ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == LTLSPEC)) {
        /* check_ltlspec wants <<cxt> <ltlformula>> as input. */
        check_ltlspec(cons(Nil, car(parsed_command)));
      } else {
        fprintf(nusmv_stderr, "Parsing error: expected \"check_ltlspec ltl_expr\"\n");
        status = 1;
      }
    } else {
      fprintf(nusmv_stderr, "Parsing error: expected \"check_ltlspec ltl_expr\"\n");
      status = 1;
    }
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    return(status);
  }  
  return(0);
}


static int PrintCheckLtlSpec()
{
  (void) fprintf(nusmv_stderr, "usage: check_ltlspec [-h] [-m| -o file] [\"ltl_expr\"]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tprint the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tpipe output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" environment variable if any,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\twrites the debugger output to \"file\".\n");
  (void) fprintf(nusmv_stderr, "   ltl_expr \tif ltl_expr, then it is model checked, else\n");
  (void) fprintf(nusmv_stderr, "    \t\tif not specified the ltl specs from model file are used.\n");
  return 1;
}
