/**CFile***********************************************************************

  FileName    [ltl.c]

  PackageName [ltl]

  Synopsis    [Routines to perform reduction of LTL model checking to
  CTL model checking.]

  Description [Here we perform the reduction of LTL model checking to
  CTL model checking. The technique adopted has been taken from [1].
  <ol>
    <li>O. Grumberg E. Clarke and K. Hamaguchi. "Another Look at LTL Model Checking".
       <em>Formal Methods in System Design</em>, 10(1):57--71, February 1997.</li>
  </ol>]

  SeeAlso     [mc]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "ltlInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/
/**Macro***********************************************************************

  Synopsis     [The LTL to NuSMV translator.]

  Description  [The name of the external program that performs
  translation of a LTL formula to the NuSMV code that represents the
  corresponding tableau.]

  SideEffects []

******************************************************************************/
#define LTL_TRANSLATOR "ltl2smv"

/**Macro***********************************************************************

  Synopsis     [The base module name declared by the <tt>LTL_TRANSLATOR</tt>.]

  Description  [The base module name declared by the <tt>LTL_TRANSLATOR</tt>.]

******************************************************************************/
#define LTL_MODULE_BASE_NAME "ltl_spec_"

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The counter of the LTL SPEC.]

  Description [The counter of LTL SPEC. It is used as seed to generate
  variables and model name by the <tt>LTL_TRANSLATOR</tt>.]

******************************************************************************/
static int ltl_spec_counter = 0;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void check_LtlSpecInt(node_ptr spec, node_ptr ltl_expr);

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis    [The main routine to perform LTL model checking.]

  Description [The main routine to perform LTL model checking. It
  first take the LTL formula, print it in a file. Call the LTL2SMV
  translator on it an read in the generated tableau. The tableau is
  instantiated, compiled and then conjoined with the original model
  (both the set of fairness conditions and the transition relation are
  affected by this operation, for this reason we save the current
  model, and after the verification of the property we restore the
  original one).]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_ltlspec(node_ptr ltl_spec)
{

  char buffer[512];
  FILE * tmp_fp;
  FILE * old_yyin;
  extern FILE * yyin;
  char * Tmp_File_Name_in;
  char * Tmp_File_Name_out;
  char * Cmd_To_Execute;
  char * Ltl_ModuleName;
  
  node_ptr context = car(ltl_spec);
  node_ptr ltl_formula = cdr(ltl_spec);
  

  if (!opt_monolithic(options)) {
    (void) fprintf(nusmv_stderr, "The model checking of LTL formulas can be performed only with the \"Monolithic\"\n");
    (void) fprintf(nusmv_stderr, "transition relation.\n");
    nusmv_exit(1);
  }

  ltl_spec_counter++; /* Increment the seed */

  Tmp_File_Name_in = util_strsav(tmpnam(buffer));
  if (Tmp_File_Name_in == NIL(char)) {
    (void)fprintf(nusmv_stderr, "Could not create input temporary file. ");
    return;
  }
  tmp_fp = fopen(Tmp_File_Name_in, "w");
  if (tmp_fp == NIL(FILE)) {
    (void) fprintf(nusmv_stderr, "Could not create input temporary file. ");
    (void) fprintf(nusmv_stderr, "Clean up /tmp an try again.\n");
    return;
  }
  Tmp_File_Name_out = util_strsav(tmpnam(buffer));
  if (Tmp_File_Name_out == NIL(char)) {
    (void)fprintf(nusmv_stderr, "Could not create output temporary file. ");
    return;
  }
  {
    char ltl_spec_counter_str[6]; 
    int j = 4; /* Number of spaces */

    sprintf(ltl_spec_counter_str, "%d", ltl_spec_counter);
    j += strlen(LTL_TRANSLATOR);
    j += strlen(ltl_spec_counter_str);
    j += strlen(Tmp_File_Name_in);
    j += strlen(Tmp_File_Name_out) + 4;
    Cmd_To_Execute = ALLOC(char, j);
  }
  if (Cmd_To_Execute == NIL(char)) {
    fprintf(nusmv_stderr, "Unable to allocate Cmd_To_Execute\n");
    nusmv_exit(1);
  }
  /* Write the LTL formula into the temporary file. */
  if (opt_verbose_level_gt(options, 0)) 
    (void) fprintf(nusmv_stderr, "Computing the corresponding tableau....");
  print_node(tmp_fp, ltl_formula);
  fclose(tmp_fp);
  sprintf(Cmd_To_Execute, "%s %d %s %s", LTL_TRANSLATOR, ltl_spec_counter, Tmp_File_Name_in, Tmp_File_Name_out);
  if (system(Cmd_To_Execute) != 0) {
    (void) fprintf(nusmv_stderr, "Unable to execute the command:\n%s\n", Cmd_To_Execute);
    nusmv_exit(1);
  }
  if (opt_verbose_level_gt(options, 0)) 
    (void) fprintf(nusmv_stderr, ".... done\n");

  /* Save the old input file descriptor, and open the new one. */
  old_yyin = yyin;
  Parser_OpenInput(Tmp_File_Name_out);

  /* Parse the new input file and store the result in parse_tree */
  if (opt_verbose_level_gt(options, 0)) 
      (void) fprintf(nusmv_stderr, "Parsing the generated tableau....");
  if (yyparse()) nusmv_exit(1);
  if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "....done\n");

  { /* We insert the definition of the current module in the module_hash
       in order to make it available for the Compile_FlattenHierarchy routines. */
    extern node_ptr parse_tree;
    node_ptr cur_module = car(reverse(parse_tree));
    node_ptr name = find_atom(car(car(cur_module)));
    node_ptr params = cdr(car(cur_module));
    node_ptr def = cdr(cur_module);

    if (lookup_module_hash(name)) error_redefining(name);
    insert_module_hash(name, new_node(LAMBDA, params, reverse(def)));
  }


  {
    node_ptr ltl_init_expr         = Nil;
    node_ptr ltl_invar_expr        = Nil;
    node_ptr ltl_trans_expr        = Nil;
    node_ptr ltl_assign_expr       = Nil;
    node_ptr ltl_procs_expr        = Nil;
    node_ptr ltl_fair_expr         = Nil;
    node_ptr ltl_abst_expr         = Nil; /* Yuan Lu */
    node_ptr ltl_spec_expr         = Nil;
    node_ptr ltl_ltlspec_expr      = Nil;
    node_ptr ltl_invar_spec_expr   = Nil;
    node_ptr ltl_invar_fb_expr     = Nil;
    node_ptr ltl_invar_strong_expr = Nil;
    node_ptr ltl_after_expr        = Nil;
    node_ptr old_state_variables   = state_variables;
    node_ptr old_all_variables     = all_variables;
    node_ptr old_all_symbols       = all_symbols;
    node_ptr old_input_variables   = input_variables;
    node_ptr old_real_state_variables = real_state_variables;
    bdd_ptr  old_trans_bdd             = bdd_dup(trans_bdd);
    node_ptr old_fairness_constraints_bdd = fairness_constraints_bdd;
    bdd_ptr  tab_trans_bdd, old_fair_states_bdd;

    state_variables = Nil;
    input_variables = Nil;
    all_symbols = Nil;
    fair_states_bdd = (bdd_ptr)NULL;

    old_fair_states_bdd = (fair_states_bdd) ? bdd_dup(fair_states_bdd) : (bdd_ptr)NULL;
    

    {
      char ltl_spec_counter_str[6];
      int j = 0;
      
      sprintf(ltl_spec_counter_str, "%d", ltl_spec_counter);
      j += strlen(LTL_MODULE_BASE_NAME);
      j += strlen(ltl_spec_counter_str);
      Ltl_ModuleName = ALLOC(char, j);
      if (Ltl_ModuleName == NIL(char)) {
        (void) fprintf(nusmv_stderr, "Unable to allocate \"Ltl_ModuleName\".\n");
        nusmv_exit(1);
      }
    }

    sprintf(Ltl_ModuleName, "%s%d", LTL_MODULE_BASE_NAME, ltl_spec_counter);
    
    /*
      We call Compile_FlattenHierarchy with the name of the generated tableau,
      and as root name the actual context. This in order to make local
      to the corresponding module the LTL specification, as happen in
      the checking of CTL specifications.
    */
    /* Yuan Lu */
    Compile_FlattenHierarchy(sym_intern(Ltl_ModuleName), context, &ltl_trans_expr, 
                    &ltl_init_expr, &ltl_invar_expr, &ltl_spec_expr, &ltl_ltlspec_expr,
                    &ltl_invar_spec_expr, &ltl_invar_fb_expr, &ltl_invar_strong_expr,
                    &ltl_after_expr, &ltl_fair_expr, &ltl_abst_expr, &ltl_assign_expr,
                    &ltl_procs_expr, Nil);

    all_symbols     = reverse(all_symbols);
    all_variables   = reverse(all_variables);
    state_variables       = reverse(state_variables);
    input_variables = reverse(input_variables);
    all_variables   = append_ns(input_variables, state_variables);
    real_state_variables = append_ns(old_real_state_variables, real_state_variables);
    ltl_fair_expr   = reverse(ltl_fair_expr);
    ltl_spec_expr   = reverse(ltl_spec_expr);

    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "Creating LTL tableau variables...\n");
    Compile_BuildVarsBdd();
    build_real_state_variables();

    all_symbols = append_ns(old_all_symbols, all_symbols);
    all_variables = append_ns(old_all_variables, all_variables);
    state_variables = append_ns(old_state_variables, state_variables);
    input_variables = append_ns(old_input_variables, input_variables);
    
    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "...done\n");

    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "Compiling the LTL tableau automata...\n");

    { /*
        Since process variables don't affect the TRANS statements, we
        can compute the Transition Relation of the LTL Tableau (which
        is defined only using the TRANS construct), simply by calling
        the evaluator and conjoining the result with the original
        transition relation. 
        ((P1 -> (A & B)) & (P2 -> (C & B))) iff (((P1 -> A) & (P2 -> C)) & B)
        using the hypothesis that (P1 | P2) iff True.
        Where B is the part relative to the TRANS statements, and P1
        and P2 are the running conditions of process 1 and 2
        respectively. For this reason if we want to add a D which comes
        form a TRANS statement, we can simply conjoin D with the
        original transition relation.
      */
      add_ptr tab_trans_add;
      add_ptr one = add_one(dd_manager);
      
      tab_trans_add = eval_simplify(ltl_trans_expr, Nil, one);
      tab_trans_bdd = add_to_bdd(dd_manager, tab_trans_add);
      add_free(dd_manager, tab_trans_add);
      add_free(dd_manager, one);
    }
    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "... done\n");

    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "Conjoining the LTL tableau with the model transition relation...\n");
    if (opt_verbose_level_gt(options, 1))
      (void) fprintf(nusmv_stderr, "Size of the original transition relation: %d BDD nodes\n", bdd_size(dd_manager, trans_bdd));  

    bdd_and_accumulate(dd_manager, &trans_bdd, tab_trans_bdd);
    {
      bdd_ptr csv = add_to_bdd(dd_manager, state_variables_add);
      bdd_ptr nsv = add_to_bdd(dd_manager, next_state_variables_add);

      bdd_and_accumulate(dd_manager, &state_variables_bdd, csv);
      bdd_and_accumulate(dd_manager, &next_state_variables_bdd, nsv);
      bdd_free(dd_manager, csv);
      bdd_free(dd_manager, nsv);
    }
    if (opt_verbose_level_gt(options, 1))
      (void) fprintf(nusmv_stderr, "Size of the new transition relation: %d BDD nodes\n", bdd_size(dd_manager, trans_bdd));  
    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "...done\n");  
    
    if (opt_verbose_level_gt(options, 1))
      (void) fprintf(nusmv_stderr, "Computing the LTL tableau fairness constraints....\n");
    compute_fairness_constraints(ltl_fair_expr);
    fairness_constraints_bdd = cons(old_fairness_constraints_bdd, fairness_constraints_bdd);
    if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "...done\n");  
    { /* We have only one SPEC */
      node_ptr spec = car(ltl_spec_expr);
      
      if (opt_verbose_level_gt(options, 0)) {
        (void) fprintf(nusmv_stderr, "Evaluating the CTL formula ");
        (void) print_spec(nusmv_stderr, spec);
        (void) fprintf(nusmv_stderr, "\n");
      }
      check_LtlSpecInt(spec, ltl_spec);
    }

    state_variables       = old_state_variables;
    input_variables = old_input_variables;
    all_symbols     = old_all_symbols;
    all_variables   = old_all_variables;
    real_state_variables = old_real_state_variables;

    bdd_free(dd_manager, trans_bdd);
    trans_bdd = old_trans_bdd;
    fairness_constraints_bdd = old_fairness_constraints_bdd;
    if (fair_states_bdd) bdd_free(dd_manager, fair_states_bdd);
    fair_states_bdd = old_fair_states_bdd;
  }
  unlink(Tmp_File_Name_in);
  unlink(Tmp_File_Name_out);
  FREE(Cmd_To_Execute);
  FREE(Ltl_ModuleName);
}

/**Function********************************************************************

  Synopsis           [Print the LTL specification.]

  Description        [Print the LTL specification.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_ltlspec(FILE *file, node_ptr n)
{
  node_ptr context = Nil;
  
  if (node_get_type(n) == CONTEXT) {
    context = car(n);
    n = cdr(n);
  }
  indent_node(file, "LTL specification ", n, " ");
  if(context) {
    fprintf(file, "(in module ");
    print_node(file, context);
    fprintf(file, ") ");
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Verifies if M,s_0 |= LTL2CTL]

  Description [Verifies if M,s_0 |= LTL2CTL, where LTL2CTL is the
  formula obtained from the translation of the LTL formula to CTL.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
static void check_LtlSpecInt(node_ptr spec, node_ptr ltl_expr)
{
  bdd_ptr s0, zero, tmp_1, tmp_2;
  node_ptr exp;

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "evaluating ");
    print_spec(nusmv_stderr, ltl_expr);
    fprintf(nusmv_stderr, "\n");
  }

  zero = bdd_zero(dd_manager);
  /* Computes the fair states, if needed */
  if (!fair_states_bdd) {
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "Computing fair states...\n");
    fair_states_bdd = compute_fair_states();
    if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr,"done\n");
  }
  /* Evaluates the spec */
  s0 = eval_spec(spec,Nil);
  /*
    Intersect with initial states (invar are normal assignments)
  */
  { /*
      We quantify out the input variables from the result of the
      formula evaluation.
    */
    bdd_ptr tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);

    bdd_free(dd_manager, s0);
    s0 = tmp;
  }
  tmp_1 = bdd_not(dd_manager, s0);
  tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
  bdd_free(dd_manager, tmp_1);
  bdd_free(dd_manager, s0);
  s0 = bdd_and(dd_manager, init_bdd, tmp_2);
  bdd_free(dd_manager, tmp_2);
  /* Prints out the result, if not true explain. */
  fprintf(nusmv_stdout, "-- ");
  print_ltlspec(nusmv_stdout, cdr(ltl_expr));

  if (s0 == zero)
    fprintf(nusmv_stdout, "is true\n");
  else {
    fprintf(nusmv_stdout, "is false\n");
    tmp_1 = bdd_pick_one_state(dd_manager, s0);
    bdd_free(dd_manager, s0);
    s0 = tmp_1;
    exp = reverse(explain(cons((node_ptr)s0, Nil), spec, Nil));
    if (!exp) exp = cons((node_ptr)s0, Nil);
    print_explanation(exp);
  }
  bdd_free(dd_manager, s0);
  bdd_free(dd_manager, zero);
}
