/**CFile***********************************************************************

  FileName    [imgMono.c]

  PackageName [img]

  Synopsis    [Routines for image computation using a monolithic transition
  relation.]

  Description []

  SeeAlso     [imgIwls95.c, imgConj.c imgDisj.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "imgInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Computes the backward image of a set of states.]

  Description        [Computes the backward image of a set of
  states. The resulting set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            [ImgImageFwdMonolithic, Img_ImageFwdMonolithicOpt]

******************************************************************************/
bdd_ptr ImgImageBwdMonolithic(bdd_ptr g)
{
  bdd_ptr result, next_g, vars;

  next_g = bdd_shift_forward(dd_manager, g);
#ifdef OPT_QUANTIFY_INPUT
  if (opt_quantify_input(options)) {
    vars = bdd_dup(next_state_variables_bdd);
  } else {
    vars = bdd_and(dd_manager, next_input_variables_bdd, next_state_variables_bdd);
  }
#else
  vars = bdd_and(dd_manager, next_input_variables_bdd, next_state_variables_bdd);
#endif
  result = bdd_and_abstract(dd_manager, trans_bdd, next_g, vars);
  bdd_free(dd_manager, next_g);
  bdd_free(dd_manager, vars);
  return(result);
}

/**Function********************************************************************

  Synopsis    [Computes the backward image of a set of states between
  fromLowerBound and fromUpperBound.]

  Description [Computes the backward image of a set of states between
  fromLowerBound and fromUpperBound.  First a set of states between
  fromLowerBound and fromUpperBound is computed.  Then, the transition
  relation is simplified by cofactoring it wrt to the set of states found in
  the first step, and wrt the toCareSet. The resulting set is in term
  of current state variables.] 

  SideEffects [None]

  SeeAlso     [ImgImageFwdMonolithic, Img_ImageFwdMonolithicOpt]
******************************************************************************/
bdd_ptr Img_ImageBwdMonolithicOpt(bdd_ptr FromLowerBound, bdd_ptr FromUpperBound,
                             bdd_ptr toCareSet)
{
  bdd_ptr result, vars, next_FromUpperBound, next_FromLowerBound;
  bdd_ptr OptimizedRelation, subOptimizedRelation, domainSubset;
  
  vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
  next_FromUpperBound = bdd_shift_forward(dd_manager, FromUpperBound);
  next_FromLowerBound = bdd_shift_forward(dd_manager, FromLowerBound);
  domainSubset = bdd_between(dd_manager, next_FromLowerBound, next_FromUpperBound);
  subOptimizedRelation = bdd_cofactor(dd_manager, trans_bdd, toCareSet);
  OptimizedRelation = bdd_cofactor(dd_manager, subOptimizedRelation, domainSubset);
  bdd_free(dd_manager, next_FromUpperBound);
  bdd_free(dd_manager, next_FromLowerBound);
  bdd_free(dd_manager, domainSubset);
  bdd_free(dd_manager, subOptimizedRelation);
  result = bdd_forsome(dd_manager, OptimizedRelation, vars);
  bdd_free(dd_manager, OptimizedRelation);
  bdd_free(dd_manager, vars);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the forward image of a set of states.]

  Description        [Computes the forward image of a set of
  states. The resulting set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            [ImgImageBwdMonolithic, ImgImageBwdMonolithicOpt]

******************************************************************************/
bdd_ptr ImgImageFwdMonolithic(bdd_ptr g)
{
  bdd_ptr result, next_result, vars;

#ifdef OPT_QUANTIFY_INPUT
  if (opt_quantify_input(options)) {
    vars = bdd_dup(state_variables_bdd);
  } else {
    vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
  }
#else
  vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
#endif
  next_result = bdd_and_abstract(dd_manager, trans_bdd, g, vars);
  result = bdd_shift_backward(dd_manager, next_result);
  bdd_free(dd_manager, next_result);
  bdd_free(dd_manager, vars);
  return(result);
}

/**Function********************************************************************

  Synopsis    [Computes the forward image of a set of states between
  fromLowerBound and fromUpperBound.]

  Description [Computes the forward image of a set of states between
  fromLowerBound and fromUpperBound.  First a set of states between
  fromLowerBound and fromUpperBound is computed.  Then, the transition
  relation is simplified by cofactoring it wrt to the set of states found in
  the first step, and wrt the toCareSet. The resulting set is in term
  of current state variables.] 

  SideEffects [None]

  SeeAlso     [ImgImageBwdMonolithic, Img_ImageBwdMonolithicOpt]
******************************************************************************/
bdd_ptr Img_ImageFwdMonolithicOpt(bdd_ptr FromLowerBound, bdd_ptr FromUpperBound,
                               bdd_ptr toCareSet)
{
  bdd_ptr result, next_result, vars, next_toCareSet, domainSubset;
  bdd_ptr OptimizedRelation, subOptimizedRelation;
  
  vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
  next_toCareSet = bdd_shift_forward(dd_manager, toCareSet);
  domainSubset = bdd_between(dd_manager, FromLowerBound, FromUpperBound);
  subOptimizedRelation = bdd_cofactor(dd_manager, trans_bdd, next_toCareSet);
  OptimizedRelation = bdd_cofactor(dd_manager, subOptimizedRelation, domainSubset);
  bdd_free(dd_manager, next_toCareSet);
  bdd_free(dd_manager, domainSubset);
  bdd_free(dd_manager, subOptimizedRelation);
  next_result = bdd_forsome(dd_manager, OptimizedRelation, vars);
  result = bdd_shift_backward(dd_manager, next_result);
  bdd_free(dd_manager, OptimizedRelation);
  bdd_free(dd_manager, next_result);
  bdd_free(dd_manager, vars);
  return(result);
}
