/**CFile***********************************************************************

  FileName    [imgIwls95.c]

  PackageName [img]

  Synopsis    [Image computation for IWLS95 technique.]

  Description [Routines for image computation using component transition
               relation approach described in the proceedings of
               IWLS'95 \[1\], (henceforth referred to IWLS95 technique).]

  SeeAlso     [compileIwls95.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "imgInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Computes the backward image of a set of
  states.]

  Description        [Computes the backward image of a set of states.
  It uses an implicitly conjoined transition relation. The resulting
  set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            [ImgImageFwdIwls95]

******************************************************************************/
bdd_ptr ImgImageBwdIwls95(bdd_ptr g)
{
  bdd_ptr vars = bdd_and(dd_manager, input_variables_bdd, next_state_variables_bdd);
  bdd_ptr result = Iwls95ImageBwd(dd_manager, Iwls95_Bwd_trans, g, vars);

  bdd_free(dd_manager, vars);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the forward image of a set of
  states.]

  Description        [Computes the forward image of a set of states.
  It uses an implicitly conjoined transition relation. The resulting
  set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            [ImgImageBwdIwls95]

******************************************************************************/
bdd_ptr ImgImageFwdIwls95(bdd_ptr g)
{
  bdd_ptr vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
  bdd_ptr result = Iwls95ImageFwd(dd_manager, Iwls95_Fwd_trans, g, vars);

  bdd_free(dd_manager, vars);
  return(result);
}

/**Function********************************************************************

  Synopsis [Computes the forward image of a set S of states in term of
  current state variables.]

  Description [First the forward image computation function is called,
  this returns an image in next variables. Later variable substitution
  is performed to obtain image on current state variables.]

  SideEffects        []

  SeeAlso            [iWls95ImageBwd, IwlsImageBoth]

******************************************************************************/
bdd_ptr Iwls95ImageFwd(DdManager *dd, node_ptr Clist, bdd_ptr S, bdd_ptr vars)
{
  bdd_ptr result, tmp;

  tmp = Iwls95ImageBoth(dd, Clist, S, vars);
  result = bdd_shift_backward(dd, tmp);
  bdd_free(dd, tmp);
  return(result);
}

/**Function********************************************************************

  Synopsis [Computes the backward image of a set S of states in term of
  current state variables.]

  Description [First the current state variables of state S are
  substituted with the corresponding next one. Later the backward
  image computation is performed.]

  SideEffects        []

  SeeAlso            [Iwls95ImageBoth, Iwls95ImageFwd]

******************************************************************************/
bdd_ptr Iwls95ImageBwd(DdManager *dd, node_ptr Clist, bdd_ptr S, bdd_ptr vars)
{
  bdd_ptr result, tmp;

  tmp = bdd_shift_forward(dd, S);
  result = Iwls95ImageBoth(dd, Clist, tmp, vars);
  bdd_free(dd, tmp);
  return(result);
}

/**Function********************************************************************

  Synopsis [Performs both forward or backward image computation.]

  Description [Depending the Cluster list passed as argument, this
  function computes respectively the forward or backward image of a
  given state.]

  SideEffects        []

  SeeAlso            [Iwls95ImageBwd, Iwls95ImageFwd]

******************************************************************************/
bdd_ptr Iwls95ImageBoth(DdManager *dd, node_ptr Clist, bdd_ptr S, bdd_ptr vars)
{
  bdd_ptr result;
  int it = 0;
  Iwls95OptionStruct_t * option = Iwls95GetOptions();
  bdd_ptr cur_prod = bdd_dup(S);
  long maxSize = 0;
  long intermediateSize = 0;
  
  while(Clist != Nil) {
    Iwls95Cluster_Info_t * Ci = (Iwls95Cluster_Info_t *)car(Clist);
    bdd_ptr new_p = bdd_and_abstract(dd, Ci->Ti, cur_prod, Ci->Ei);

    it++;
    if (option->verbosity >= 1) {
      intermediateSize = bdd_size(dd, new_p);
      (void) fprintf(nusmv_stdout, "Size of intermediate product = %10ld (BDD nodes).\n", intermediateSize);
      if (maxSize < intermediateSize) {
        maxSize = intermediateSize;
      }
    }
    bdd_free(dd, cur_prod);
    cur_prod = new_p;
    Clist = cdr(Clist);
  }
  result = bdd_forsome(dd, cur_prod, vars);
  bdd_free(dd, cur_prod);
  if (option->verbosity >= 1){
    (void) fprintf(nusmv_stdout, "Max. BDD size for intermediate product = %10ld (BDD nodes)\n", maxSize);
  }
  return(result);
}
