/**CFile***********************************************************************

  FileName    [imgDisj.c]

  PackageName [img]

  Synopsis    [Routines for image computation using a disjunctively
  decomposed transition relation.]

  Description [Routines for image computation using a disjunctively
  decomposed transition relation.]

  SeeAlso     [imgIwls95.c, imgConj.c, imgMono.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "imgInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Computes the backward image of a set of
  states.]

  Description        [Computes the backward image of a set of states.
  It uses an implicitly disjoined transition relation. The resulting
  set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            [ImgImageFwdDisjunctive]

******************************************************************************/
bdd_ptr ImgImageBwdDisjunctive(bdd_ptr g)
{
  bdd_ptr next_vars;
  bdd_ptr next_g = bdd_shift_forward(dd_manager, g);
  bdd_ptr acc = bdd_zero(dd_manager);
  node_ptr t_r_d = dp_trans_bdd;

#ifdef OPT_QUANTIFY_INPUT
  if (opt_quantify_input(options)) {
    next_vars = bdd_dup(next_state_variables_bdd);
  } else {
    next_vars = bdd_and(dd_manager, next_input_variables_bdd, next_state_variables_bdd);
  }
#else
  next_vars = bdd_and(dd_manager, next_input_variables_bdd, next_state_variables_bdd);
#endif
  while(cdr(t_r_d)){
    bdd_ptr tmp_1;
    
    if (opt_verbose_level_gt(options, 0))
      fprintf(stderr,"relational product: size of acc = %d\n",
              bdd_size(dd_manager, acc));
    tmp_1 = bdd_and_abstract(dd_manager, (bdd_ptr)car(t_r_d), next_g, next_vars);
    bdd_or_accumulate(dd_manager, &acc, tmp_1);
    bdd_free(dd_manager, tmp_1);
    t_r_d = cdr(t_r_d);
  }
  bdd_free(dd_manager, next_vars);
  bdd_free(dd_manager, next_g);
  return(acc);
}

/**Function********************************************************************

  Synopsis           [Computes the forward image of a set of
  states.]

  Description        [Computes the forward image of a set of states.
  It uses an implicitly disjoined transition relation. The resulting
  set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            [ImgImageBwdDisjunctive]

******************************************************************************/
bdd_ptr ImgImageFwdDisjunctive(bdd_ptr g)
{
  bdd_ptr cur_acc, vars;
  node_ptr t_r_d = dp_trans_bdd;
  bdd_ptr acc = bdd_zero(dd_manager);

#ifdef OPT_QUANTIFY_INPUT
  if (opt_quantify_input(options)) {
    vars = bdd_dup(state_variables_bdd);
  } else {
    vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
  }
#else
  vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
#endif
  while(cdr(t_r_d)){
    bdd_ptr tmp_1;
    
    if (opt_verbose_level_gt(options, 0))
      fprintf(stderr,"relational product: size of acc = %d\n",
              bdd_size(dd_manager, acc));
    tmp_1 = bdd_and_abstract(dd_manager, (bdd_ptr)car(t_r_d), g, vars);
    bdd_or_accumulate(dd_manager, &acc, tmp_1);
    bdd_free(dd_manager,tmp_1);
    t_r_d = cdr(t_r_d);
  }
  cur_acc = bdd_shift_backward(dd_manager, acc);
  bdd_free(dd_manager, acc);
  bdd_free(dd_manager, vars);
  return(cur_acc);
} 
