/**CFile***********************************************************************

  FileName    [dd.c]

  PackageName [dd]

  Synopsis [NuSMV interface to the Decision Diagram Package of the
  University of Colorado.]

  Description [This file implements the interface between the NuSMV
  system and the California University Decision Diagram (henceforth
  referred as CUDD). The CUDD package is a generic implementation of a
  decision diagram data structure. For the time being, only Boole
  expansion is implemented and the leaves in the in the nodes can be
  the constants zero, one or any arbitrary value. A coding standard
  has been defined. I.e all the functions acting on BDD and ADD have
  \"bdd\" and \"add\" respectively as prefix.
  <p><br>
  The BDD or ADD returned as a result of an operation are always
  referenced (see the CUDD User Manual for more details about this),
  and need to be dereferenced when the result is no more necessary to
  computation, in order to release the memory associated to it when
  garbage collection occurs.
  All the functions takes as first argument the decision diagram
  manager (henceforth referred as DdManager).]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "ddInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
node_ptr one_number = Nil;
node_ptr zero_number = Nil;
node_ptr boolean_type = Nil; 


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/
#define common_error(variable, message) \
 if ((variable) == NULL) { \
   rpterr((message));\
   nusmv_exit(1); \
 }

#define common_error2(dd, variable, variable2, message) \
 if ((variable) == NULL) { \
   rpterr((message));\
   Cudd_RecursiveDeref((dd),(variable2));\
   nusmv_exit(1); \
 }


/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void InvalidType(FILE *file, char *field, char *expected);


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Creates a new DD manager.]

  Description [Creates a new DD manager, initializes the table, the
  basic constants and the projection functions.<br>
  \"maxMemory\" (the last parameter of the function \"Cudd_Init\")
  is set to 0. In such a way \"Cudd_Init\" decides suitables values
  for the maximum size of the cache and for the limit for fast unique
  table growth based on the available memory. Returns a pointer to the
  manager if successful; else abort depending the mode (interactive or
  batch) the system is used.] 

  SideEffects        []

  SeeAlso            [quit_dd_package]

******************************************************************************/
DdManager * init_dd_package(){
  DdManager * dd;

  one_number = find_node(NUMBER,(node_ptr)1,Nil);
  zero_number = find_node(NUMBER,(node_ptr)0,Nil);
  boolean_type = cons(zero_number,cons(one_number,Nil));
  dd = Cudd_Init(0, 0, UNIQUE_SLOTS, CACHE_SLOTS, 0);
  common_error(dd, "init_dd_package: Unable to initialize the manager.");
  return(dd);
}

/**Function********************************************************************

  Synopsis           [Deletes resources associated with a DD manager.]

  Description        [Deletes resources associated with a DD manager and
  resets the global statistical counters. (Otherwise, another manager
  subsequently created would inherit the stats of this one.)]

  SideEffects        []

  SeeAlso            [init_dd_package]

******************************************************************************/
void quit_dd_package(DdManager * dd)
{
  Cudd_Quit(dd);
}

/**Function********************************************************************

  Synopsis [Checks the unique table for nodes with non-zero reference
  counts.]

  Description [Checks the unique table for nodes with non-zero
  reference counts. It is normally called before dd_quit to make sure
  that there are no memory leaks due to missing add/bdd_free's.
  Takes into account that reference counts may saturate and that the
  basic constants and the projection functions are referenced by the
  manager.  Returns the number of nodes with non-zero reference count.
  (Except for the cases mentioned above.)]

  SideEffects []

******************************************************************************/
int dd_checkzeroref(DdManager * dd)
{
  return Cudd_CheckZeroRef(dd);
} /* end of dd_checkzeroref */

/**Function********************************************************************

  Synopsis           [Returns the number of nodes in the unique table.]

  Description        [Returns the total number of nodes currently in the unique
  table, including the dead nodes.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int get_dd_nodes_allocated(DdManager * dd){
  return(Cudd_ReadKeys(dd));
}

/**Function********************************************************************

  Synopsis [Applies function <code>f</code> to the list of BDD/ADD <code>l</code>.]

  Description [This function acts like the Lisp <tt>mapcar</tt>. It returns
  the list of the result of the application of function \code>f</code> to each
  element of list <code>l</code>.]

  SideEffects        []

  SeeAlso            [map walk walk_dd]

******************************************************************************/
node_ptr map_dd(DdManager * dd, NPFDD f, node_ptr l)
{
  node_ptr t;

  if (l == Nil) return(Nil);
  t = (*f)(dd, (DdNode *)car(l));
  return(cons(t,map_dd(dd, f, cdr(l))));
}

/**Function********************************************************************

  Synopsis [Applies function <code>f</code> to the list of BDD/ADD <code>l</code>.]

  Description [This function acts like the <tt>map_dd</dd>. This functions
  applies the function <code>f</code> to each element of list
  <code>l</code>. Nothing is returned, performs side-effects on the elements.]

  SideEffects        []

  SeeAlso            [map walk map_dd]

******************************************************************************/
void walk_dd(DdManager * dd, VPFDD f, node_ptr l)
{
  if (l == Nil) return;
  (*f)(dd, (DdNode *)car(l));
  walk_dd(dd, f, cdr(l));
}

/**Function********************************************************************

  Synopsis    [Prints out statistic and setting of the DD manager.]

  Description [Prints out statistics and settings for a CUDD manager.]

  SideEffects []

******************************************************************************/
void dd_print_stats(DdManager *mgr, FILE *file)
{
    Cudd_PrintInfo(mgr, file);

    /* Print some guidance to the parameters */
    (void) fprintf(file, "\nMore detailed information about the semantics ");
    (void) fprintf(file, "and values of these parameters\n");
    (void) fprintf(file, "can be found in the documentation about the CU ");
    (void) fprintf(file, "Decision Diagram Package.\n");
  
    return;

} /* end of dd_print_stats */

/**Function********************************************************************

  Synopsis    [Builds a group of variables that should stay adjacent
  during reordering.]

  Description [Builds a group of variables that should stay adjacent
  during reordering. The group is made up of n variables. The first
  variable in the group is f. The other variables are the n-1
  variables following f in the order at the time of invocation of this
  function. Returns a handle to the variable group if successful else fail.]

  SideEffects [Modifies the variable tree.]

******************************************************************************/
dd_block * dd_new_var_block(DdManager * dd, int start_index, int offset)
{
  MtrNode *group;

  if (start_index == MAX_VAR_INDEX) return(NULL);
  /*
    We use MTR_FIXED because we want the internal order of variablesa
    in the groups to be preserved (MTR_FIXED does not preserve them.).
  */
  group = Cudd_MakeTreeNode(dd, start_index, offset, MTR_FIXED);
  common_error(group, "dd_new_var_block: group = NULL");
  return((dd_block *) group);
} /* end of dd_new_var_block */

/**Function********************************************************************

  Synopsis    [Returns the index of the variable currently in the i-th
  position of the order.]

  Description [Returns the index of the variable currently in the i-th
  position of the order. If the index is MAX_VAR_INDEX, returns
  MAX_VAR_INDEX; otherwise, if the index is out of bounds fails.]

  SideEffects []

******************************************************************************/
int dd_get_var_at_level(DdManager *dd, int level)
{
  int result;

  result = Cudd_ReadInvPerm(dd, level);
  if (result < 0) {
    rpterr("dd_get_var_at_level: level %d out of bound.", level);
    nusmv_exit(1);
  }
  return(result);
} /* end of dd_get_var_at_level */

/**Function********************************************************************

  Synopsis    [Returns the number of BDD variables in existance.]

  Description [Returns the number of BDD variables in existance.]

  SideEffects []

******************************************************************************/
int dd_get_size(DdManager *dd)
{
  return Cudd_ReadSize(dd);
} /* end of dd_get_size */

/**Function********************************************************************

  Synopsis    [Returns the number of BDD variables in existance.]

  Description [Returns the number of BDD variables in existance.]

  SideEffects [Yuan Lu]

******************************************************************************/
int get_dyn_node_allocated(DdManager *dd)
{
  return Cudd_ReadMaxSize(dd);
} /* end of dd_get_size */

/**Function********************************************************************

  Synopsis    [Returns the number of BDD variables in existance.]

  Description [Returns the number of BDD variables in existance.]

  SideEffects [Yuan Lu]

******************************************************************************/
void set_dyn_node_allocated(DdManager *dd)
{
  Cudd_SetMaxSize(dd);
}

/**Function********************************************************************

  Synopsis    [Enables automatic dynamic reordering of BDDs and ADDs.]

  Description [Enables automatic dynamic reordering of BDDs and
  ADDs. Parameter method is used to determine the method used for
  reordering. If REORDER_SAME is passed, the method is
  unchanged.]

  SideEffects []

  SeeAlso     [dd_autodyn_disable dd_reordering_status]

******************************************************************************/
void dd_autodyn_enable(DdManager * dd, int method)
{
  Cudd_AutodynEnable(dd, method);
} /* end of dd_autodyn_enable */

/**Function********************************************************************

  Synopsis    [Disables automatic dynamic reordering of BDD and ADD.]

  Description [Disables automatic dynamic reordering of BDD and ADD.]

  SideEffects []

  SeeAlso     [dd_autodyn_enable dd_reordering_status]

******************************************************************************/
void dd_autodyn_disable(DdManager *dd)
{
  Cudd_AutodynDisable(dd);
} /* end of dd_autodyn_disable */

/**Function********************************************************************

  Synopsis    [Reports the status of automatic dynamic reordering of BDDs
  and ADDs.]

  Description [Reports the status of automatic dynamic reordering of
  BDDs and ADDs. Parameter method is set to the reordering method
  currently selected. Returns 1 if automatic reordering is enabled; 0
  otherwise.]

  SideEffects [Parameter method is set to the reordering method currently
  selected.]

  SeeAlso     [dd_autodyn_disable dd_autodyn_enable]

******************************************************************************/
dd_reorderingtype dd_reordering_status(DdManager *dd, dd_reorderingtype * method)
{
  return(Cudd_ReorderingStatus(dd, method));
} /* end of dd_reordering_status */

/**Function********************************************************************

  Synopsis    [Main dynamic reordering routine.]

  Description [Main dynamic reordering routine.
  Calls one of the possible reordering procedures:
  <ul>
  <li>Swapping
  <li>Sifting
  <li>Symmetric Sifting
  <li>Group Sifting
  <li>Window Permutation
  <li>Simulated Annealing
  <li>Genetic Algorithm
  <li>Dynamic Programming (exact)
  </ul>

  For sifting, symmetric sifting, group sifting, and window
  permutation it is possible to request reordering to convergence.<p>

  Returns 1 in case of success; 0 otherwise. In the case of symmetric
  sifting (with and without convergence) returns 1 plus the number of
  symmetric variables, in case of success.<p>

  This functions takes as arguments:
  <ul>
  <li> <tt>dd</tt> the DD manager; 
  <li> <tt>heuristics</tt> method used for reordering; 
  <li> <tt>minsize</tt> bound below which no reordering occurs; 
  </ul>
  ]

  SeeAlso     [Cudd_ReduceHeap]

  SideEffects [Changes the variable order for all diagrams and clears
  the cache.]

******************************************************************************/
int dd_reorder(DdManager *dd, int method, int minsize)
{
  int result;
  
  result = Cudd_ReduceHeap(dd, method, minsize);
  if (result == 0) {
    rpterr("dd_reorder: reordering of ADD/BDD fails.");
    nusmv_exit(1);
  } 
  return(result);
} /* end of dd_reorder */

/**Function********************************************************************

  Synopsis [Gets the internal reordering method used.]

  Description [Returns the internal reordering method used.]

  SideEffects []

******************************************************************************/
dd_reorderingtype dd_get_ordering_method (DdManager * dd) 
{
  dd_reorderingtype method;

  (void) Cudd_ReorderingStatus(dd, &method);
  return(method);
}

/**Function********************************************************************

  Synopsis    [Converts a string to a dynamic ordering method type.]

  Description [Converts a string to a dynamic ordering method type. If string
  is not "sift" or "window", then returns REORDER_.]

  SideEffects []

******************************************************************************/
int StringConvertToDynOrderType(char *string)
{

  if (strcmp("random", string) == 0) { 
    return REORDER_RANDOM;
  }
  else if (strcmp("random_pivot", string) == 0) { 
    return  REORDER_RANDOM_PIVOT;
  }
  else if (strcmp("sift", string) == 0) { 
    return REORDER_SIFT;
  }
  else if (strcmp("sift_converge", string) == 0) { 
    return  REORDER_SIFT_CONV;
  }
  else if (strcmp("symmetry_sift", string) == 0) { 
    return  REORDER_SYMM_SIFT;
  }
  else if (strcmp("symmetry_sift_converge", string) == 0) { 
    return  REORDER_SYMM_SIFT_CONV;
  }
  else if (strcmp("window2", string) == 0) { 
    return REORDER_WINDOW2;
  }
  else if (strcmp("window3", string) == 0) { 
    return  REORDER_WINDOW3;
  }
  else if (strcmp("window4", string) == 0) { 
    return  REORDER_WINDOW4;
  }
  else if (strcmp("window2_converge", string) == 0) { 
    return  REORDER_WINDOW2_CONV;
  }
  else if (strcmp("window3_converge", string) == 0) { 
    return  REORDER_WINDOW3_CONV;
  }
  else if (strcmp("window4_converge", string) == 0) { 
    return  REORDER_WINDOW4_CONV;
  }
  else if (strcmp("group_sift", string) == 0) { 
    return  REORDER_GROUP_SIFT;
  }
  else if (strcmp("group_sift_converge", string) == 0) { 
    return  REORDER_GROUP_SIFT_CONV;
  }
  else if (strcmp("annealing", string) == 0) { 
    return  REORDER_ANNEALING;
  }
  else if (strcmp("genetic", string) == 0) { 
    return  REORDER_GENETIC;
  }
  else if (strcmp("exact", string) == 0) { 
    return  REORDER_EXACT;
  }
  else if (strcmp("linear", string) == 0) { 
    return  REORDER_LINEAR;
  }
  else if (strcmp("linear_converge", string) == 0) { 
    return  REORDER_LINEAR_CONV;
  }
  else {
    return REORDER_NONE;
  }
}

/**Function********************************************************************

  Synopsis    [Converts a dynamic ordering method type to a string.]

  Description [Converts a dynamic ordering method type to a string.  This
  string must NOT be freed by the caller.]

  SideEffects []

******************************************************************************/
char * DynOrderTypeConvertToString(int method)
{
  if (method == REORDER_NONE) {
    return ""; 
  }
  else if (method == REORDER_RANDOM) { 
    return "random";
  }
  else if (method == REORDER_RANDOM_PIVOT) { 
    return "random_pivot";
  }
  else if ((method == REORDER_SIFT) || (method == REORDER_GROUP_SIFT)) {
    return "sift"; 
  }
  else if (method == REORDER_SIFT_CONV) { 
    return "sift_converge";
  }
  else if (method == REORDER_SYMM_SIFT) { 
    return "symmetry_sift";
  }
  else if (method == REORDER_SYMM_SIFT_CONV) { 
    return "symmetry_sift_converge";
  }
  else if (method == REORDER_WINDOW2) {
    return "window2"; 
  }
  else if (method == REORDER_WINDOW3) { 
    return "window3";
  }
  else if (method == REORDER_WINDOW4) { 
    return "window4";
  }
  else if (method == REORDER_WINDOW2_CONV) { 
    return "window2_converge";
  }
  else if (method == REORDER_WINDOW3_CONV) { 
    return "window3_converge";
  }
  else if (method == REORDER_WINDOW4_CONV) { 
    return "window4_converge";
  }
  else if (method == REORDER_GROUP_SIFT) { 
    return "group_sift";
  }
  else if (method == REORDER_GROUP_SIFT_CONV) { 
    return "group_sift_converge";
  }
  else if (method == REORDER_ANNEALING) { 
    return "annealing";
  }
  else if (method == REORDER_GENETIC) { 
    return "genetic";
  }
  else if (method == REORDER_EXACT) { 
    return "exact";
  }
  else if (method == REORDER_LINEAR) { 
    return "linear";
  }
  else if (method == REORDER_LINEAR_CONV) { 
    return "linear_converge";
  }
  else if (method == REORDER_SAME) {
    return "same";
  }
  else {
    fail("unrecognized method");
  }
}


/**Function********************************************************************

  Synopsis [Sets the internal parameters of the package to the given values.]

  Description [The CUDD package has a set of parameters that can be assigned
  different values. This function receives a table which maps strings to
  values and sets the parameters represented by the strings to the pertinent
  values. Some basic type checking is done. It returns 1 if everything is
  correct and 0 otherwise.]

  SideEffects []

******************************************************************************/
int dd_set_parameters(DdManager *mgr, avl_tree *valueTable, FILE *file)
{
  int reorderMethod;
  dd_reorderingtype zddReorderMethod;
  st_table *newValueTable;
  st_generator *stgen;
  avl_generator *avlgen;
  char *paramName;
  char *paramValue;

  /* Initial value of the variables. */
  reorderMethod = REORDER_SAME;
  zddReorderMethod = REORDER_SAME;

  /* Build a new table with the parameter names but with
  ** the prefix removed. */
  newValueTable = st_init_table(st_ptrcmp, st_ptrhash);
  avl_foreach_item(valueTable, avlgen, AVL_FORWARD, (char **)&paramName, 
                   (char **)&paramValue) {
    if (strncmp(paramName, "BDD.", 4) == 0) {
      st_insert(newValueTable, (char *)&paramName[4],
                (char *)paramValue);
    }
  }

  st_foreach_item(newValueTable, stgen, (char **)&paramName, (char **)&paramValue) {
    unsigned int uvalue;
    char *invalidChar;

    invalidChar = (char *)NULL;
    if (strcmp(paramName, "Hard limit for cache size") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Hard limit for cache size", "unsigned integer");
      }
      else {
        Cudd_SetMaxCacheHard(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Cache hit threshold for resizing") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
		InvalidType(file, "Cache hit threshold for resizing",
			    "unsigned integer");
      }
      else {
        Cudd_SetMinHit(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Garbage collection enabled") == 0) {
      if (strcmp(paramValue, "yes") == 0) {
        Cudd_EnableGarbageCollection(mgr);
      }
      else if (strcmp(paramValue, "no") == 0) {
        Cudd_DisableGarbageCollection(mgr);
      }
      else {
        InvalidType(file, "Garbage collection enabled", "(yes,no)");
      }
    }
    else if (strcmp(paramName, "Limit for fast unique table growth") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Limit for fast unique table growth", "unsigned integer");
      }
      else {
        Cudd_SetLooseUpTo(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Maximum number of variables sifted per reordering") 
             == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Maximum number of variables sifted per reordering",
                    "unsigned integer");
      }
      else {
        Cudd_SetSiftMaxVar(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Maximum number of variable swaps per reordering")
             == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Maximum number of variable swaps per reordering", 
                    "unsigned integer");
      }
      else {
        Cudd_SetSiftMaxSwap(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, 
                    "Maximum growth while sifting a variable") == 0) {
      double value;

      value = strtod(paramValue, &invalidChar);
      if (*invalidChar) {
        InvalidType(file, "Maximum growth while sifting a variable", "real");
      }
      else {
        Cudd_SetMaxGrowth(mgr, value);
      }
    }
    else if (strcmp(paramName, "Dynamic reordering of BDDs enabled") == 0) {
      if (strcmp(paramValue, "yes") == 0) {
        Cudd_AutodynEnable(mgr, reorderMethod);
      }
      else if (strcmp(paramValue, "no") == 0) {
        Cudd_AutodynDisable(mgr);
      }
      else {
        InvalidType(file, "Dynamic reordering of BDDs enabled", "(yes,no)");
      }
    }
    else if (strcmp(paramName, "Default BDD reordering method") == 0) {
      dd_reorderingtype reorderInt;

      reorderMethod = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || reorderMethod < 0) {
        InvalidType(file, "Default BDD reordering method", "integer");
      }
      else {
        if (Cudd_ReorderingStatus(mgr, &reorderInt)) {
          Cudd_AutodynEnable(mgr, reorderMethod);
        }
      }
    }
    else if (strcmp(paramName, "Dynamic reordering of ZDDs enabled") == 0) {
      if (strcmp(paramValue, "yes") == 0) {
        Cudd_AutodynEnableZdd(mgr, zddReorderMethod);
      }
      else if (strcmp(paramValue, "no") == 0) {
        Cudd_AutodynDisableZdd(mgr);
      }
      else {
        InvalidType(file, "Dynamic reordering of ZDDs enabled", "(yes,no)");
      }
    }
    else if (strcmp(paramName, "Default ZDD reordering method") == 0) {
      dd_reorderingtype reorderInt;

      zddReorderMethod = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || zddReorderMethod < 0) {
        InvalidType(file, "Default ZDD reordering method", "integer");
      }
      else {
        if (Cudd_ReorderingStatusZdd(mgr, &reorderInt)) {
          Cudd_AutodynEnableZdd(mgr, zddReorderMethod);
        }
      }
    }
    else if (strcmp(paramName, "Realignment of ZDDs to BDDs enabled") == 0) {
      if (strcmp(paramValue, "yes") == 0) {
        Cudd_zddRealignEnable(mgr);
      }
      else if (strcmp(paramValue, "no") == 0) {
        Cudd_zddRealignDisable(mgr);
      }
      else {
        InvalidType(file, "Realignment of ZDDs to BDDs enabled", "(yes,no)");
      }
    }
    else if (strcmp(paramName, "Realignment of BDDs to ZDDs enabled") == 0) {
      if (strcmp(paramValue, "yes") == 0) {
        Cudd_bddRealignEnable(mgr);
      }
      else if (strcmp(paramValue, "no") == 0) {
        Cudd_bddRealignDisable(mgr);
      }
      else {
        InvalidType(file, "Realignment of BDDs to ZDDs enabled", "(yes,no)");
      }
    }
    else if (strcmp(paramName, "Dead nodes counted in triggering reordering") == 0) {
      if (strcmp(paramValue, "yes") == 0) {
        Cudd_TurnOnCountDead(mgr);
      }
      else if (strcmp(paramValue, "no") == 0) {
        Cudd_TurnOffCountDead(mgr);
      }
      else {
        InvalidType(file, "Dead nodes counted in triggering reordering", "(yes,no)");
      }
    }
    else if (strcmp(paramName, "Group checking criterion") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Group checking criterion", "integer");
      }
      else {
        Cudd_SetGroupcheck(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Recombination threshold") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Recombination threshold", "integer");
      }
      else {
        Cudd_SetRecomb(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Symmetry violation threshold") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Symmetry violation threshold", "integer");
      }
      else {
        Cudd_SetSymmviolation(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Arc violation threshold") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Arc violation threshold", "integer");
      }
      else {
        Cudd_SetArcviolation(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "GA population size") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar  || uvalue < 0) {
        InvalidType(file, "GA population size", "integer");
      }
      else {
        Cudd_SetPopulationSize(mgr, uvalue);
      }
    }
    else if (strcmp(paramName, "Number of crossovers for GA") == 0) {
      uvalue = (unsigned int) strtol(paramValue, &invalidChar, 10);
      if (*invalidChar || uvalue < 0) {
        InvalidType(file, "Number of crossovers for GA", "integer");
      }
      else {
        Cudd_SetNumberXovers(mgr, uvalue);
      }
    }
    else {
      (void) fprintf(file, "Warning: Parameter %s not recognized.", paramName);
      (void) fprintf(file, " Ignored.\n");
    }
  } /* end of st_foreach_item */

  /* Clean up. */
  st_free_table(newValueTable);

  return(1);

} /* end of dd_set_parameters */



/**Function********************************************************************

  Synopsis    [Prints a disjoint sum of products.]

  Description [Prints a disjoint sum of product cover for the function
  rooted at node. Each product corresponds to a path from node a leaf
  node different from the logical zero, and different from the
  background value. Uses the standard output.  Returns 1 if successful;
  0 otherwise.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
int dd_printminterm(
  DdManager * manager,
  DdNode * node)
{
  return(Cudd_PrintMinterm(manager, node));
}

/**Function********************************************************************

  Synopsis           [Writes a dot file representing the argument DDs.]

  Description        [Writes a dot file representing the argument
  DDs. For a better description see the \"Cudd_DumpDot\" documentation
  in the CUDD package.]

  SideEffects        []

  SeeAlso            [dd_dump_davinci]
******************************************************************************/
int dd_dump_dot(
  DdManager * dd /* manager */,
  int  n /* number of output nodes to be dumped */,
  bdd_ptr * f /* array of output nodes to be dumped */,
  char ** inames /* array of input names (or NULL) */,
  char ** onames /* array of output names (or NULL) */,
  FILE * fp /* pointer to the dump file */) 
{
  return(Cudd_DumpDot(dd, n, (DdNode **)f, inames, onames, fp));
}

/**Function********************************************************************

  Synopsis           [Writes a daVnci file representing the argument DDs.]

  Description        [Writes a daVnci file representing the argument
  DDs. For a better description see the \"Cudd_DumpDaVinci\" documentation
  in the CUDD package.]

  SideEffects        []

  SeeAlso            [dd_dump_davinci]
******************************************************************************/
int dd_dump_davinci(
  DdManager * dd /* manager */,
  int  n /* number of output nodes to be dumped */,
  bdd_ptr * f /* array of output nodes to be dumped */,
  char ** inames /* array of input names (or NULL) */,
  char ** onames /* array of output names (or NULL) */,
  FILE * fp /* pointer to the dump file */) 
{
  return(Cudd_DumpDaVinci(dd, n, (DdNode **)f, inames, onames, fp));
}

/**Function********************************************************************

  Synopsis           [Reads the constant 1 ADD of the manager.]

  Description        [Reads the constant 1 ADD of the manager.]

  SideEffects        []

  SeeAlso            [add_zero]
******************************************************************************/
add_ptr add_one(DdManager * dd)
{
  DdNode * result = Cudd_ReadOne(dd);

  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Reads the constant 0 ADD of the manager.]

  Description        [Reads the constant 0 ADD of the manager.]

  SideEffects        []

  SeeAlso            [add_one]
******************************************************************************/
add_ptr add_zero(DdManager * dd)
{
  DdNode * result = Cudd_ReadZero(dd);

  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Reference an ADD node.]

  Description        [Reference an ADD node.]

  SideEffects        [The reference count of the node is incremented by one.]

  SeeAlso            [add_deref add_free]
******************************************************************************/
void add_ref(add_ptr fn)
{
  Cudd_Ref(fn);
}

/**Function********************************************************************

  Synopsis           [Dereference an ADD node.]

  Description        [Dereference an ADD node.]

  SideEffects        [The reference count of the node is decremented by one.]

  SeeAlso            [add_ref add_free]
******************************************************************************/
void add_deref(add_ptr fn)
{
  Cudd_Deref(fn);
}

/**Function********************************************************************

  Synopsis           [Dereference an ADD node. If it dies, recursively decreases
  the reference count of its children.]

  Description        [Decreases the reference count of node. If the node dies,
  recursively decreases the reference counts of its children. It is used to
  dispose off an ADD that is no longer needed.]

  SideEffects        [The reference count of the node is decremented by one,
  and if the node dies a recursive dereferencing is applied to its children.]

  SeeAlso            []
******************************************************************************/
void add_free(DdManager * dd, add_ptr dd_node)
{
  common_error(dd_node, "add_free: dd_node = NULL");
  Cudd_RecursiveDeref(dd, (DdNode *)dd_node);
}

/**Function********************************************************************

  Synopsis           [Creates a copy of an ADD node.]

  Description        [Creates a copy of an ADD node.]

  SideEffects        [The reference count is increased by one unit.]

  SeeAlso            [add_ref add_free add_deref]

******************************************************************************/
add_ptr add_dup(add_ptr dd_node)
{
  Cudd_Ref(dd_node);
  return(dd_node);
}

/**Function********************************************************************

  Synopsis           [Return cofactors for an ADD node.]

  Description        [Return cofactors for an ADD node.]

  SideEffects        [Yuan Lu]

  SeeAlso            []

******************************************************************************/
add_ptr add_else(add_ptr f)
{
  add_ptr res = Cudd_E(f);
  Cudd_Ref(res);
  return res;
}

add_ptr add_then(add_ptr f)
{
  add_ptr res = Cudd_T(f);
  Cudd_Ref(res);
  return res;
}

/**Function********************************************************************

  Synopsis           [Creates an returns an ADD for constant leaf_node.]

  Description        [Retrieves the ADD for constant leaf_node if it already
  exists, or creates a new ADD.  Returns a pointer to the
  ADD if successful; fails otherwise.]

  SideEffects        [The reference count of the node is incremented by one unit.]

******************************************************************************/
add_ptr add_leaf(DdManager * dd, node_ptr leaf_node)
{
  DdNode * result;

  result = Cudd_addConst(dd,leaf_node);
  common_error(result, "add_leaf: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Returns 1 if the node is a constant node.]

  Description        [Returns 1 if the node is a constant node (rather than an
  internal node). All constant nodes have the same index (MAX_VAR_INDEX).]

  SideEffects        []

******************************************************************************/
int add_isleaf(add_ptr dd_node)
{
  return(Cudd_IsConstant(dd_node));
}

/**Function********************************************************************

  Synopsis           [Returns the value of a constant node.]

  Description        [Returns the value of a constant node. If <code>Leaf</code>
  is an internal node, a failure occurs.]

  SideEffects        []

******************************************************************************/
node_ptr add_get_leaf(DdManager * dd, add_ptr Leaf)
{
  if (!Cudd_IsConstant((DdNode *)Leaf)) {
    rpterr("add_get_leaf: not a leaf!");
    nusmv_exit(1);
  }
  return((node_ptr)Cudd_V(Leaf));
}
/**Function********************************************************************

  Synopsis           [Checks the unique table of the DdManager for the
  existence of an internal node.]

  Description        [Checks the unique table for the existence of an internal
  node. If it does not exist, it creates a new one. The reference
  count of whatever is returned is increased by one unit. For a newly
  created node, increments the reference counts of what T and E point
  to.  Returns a pointer to the new node if successful; a failure
  occurs if memory is exhausted or if reordering took place.]

  SideEffects        []

******************************************************************************/
add_ptr add_build(DdManager * dd, int level, add_ptr t, add_ptr e)
{
  DdNode * result;

  result = (t == e) ? (DdNode *)t : (DdNode *)cuddUniqueInter(dd, level, (DdNode *)t, (DdNode *)e);
  common_error(result, "add_build: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Returns the ADD variable with index <code>index</code>.]

  Description        [Retrieves the ADD variable with index
  <code>index</code> if it already exists, or creates a new ADD
  variable. Returns a pointer to the variable if successful; a failure
  is generated otherwise.  An ADD variable differs from a BDD variable
  because it points to the arithmetic zero, instead of having a
  complement pointer to 1. The returned value is referenced.]

  SideEffects        []

  SeeAlso            [add_new_var_at_level]

******************************************************************************/
add_ptr add_new_var_with_index(DdManager * dd, int index)
{
  add_ptr result;

  result = Cudd_addIthVar(dd, index);
  common_error(result, "add_new_var_with_index: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Returns a new ADD variable at a specified level.]

  Description        [Creates a new ADD variable. The new variable has an
  index equal to the largest previous index plus 1 and is positioned at
  the specified level in the order.  Returns a pointer to the new
  variable if successful; a failure is generated otherwise. The
  returned value is referenced.]

  SideEffects        []

  SeeAlso            [add_new_var_with_index]

******************************************************************************/
add_ptr add_new_var_at_level(DdManager * dd, int level)
{
  DdNode * result;

  result = Cudd_addNewVarAtLevel(dd,level);
  common_error(result, "add_new_var_at_level: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Converts an ADD to a BDD.]

  Description [Converts an ADD to a BDD by replacing all
  discriminants greater than or equal to value with 1, and all other
  discriminants with 0. Returns a pointer to the resulting BDD if
  successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_to_add]

******************************************************************************/
bdd_ptr add_to_bdd(DdManager * dd, add_ptr fn)
{
  DdNode * result;
  extern node_ptr zero_number;
  
  result = Cudd_addBddThreshold(dd,fn,zero_number);
  common_error(result, "add_to_bdd: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Converts a BDD to a 0-1 ADD.]

  Description [Converts a BDD to a 0-1 ADD. Returns a pointer to the
  resulting ADD if successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [add_to_bdd]

******************************************************************************/
add_ptr bdd_to_add(DdManager * dd, bdd_ptr fn)
{
  DdNode * result;

  result = Cudd_BddToAdd(dd, (DdNode *)fn);
  common_error(result, "bdd_to_add: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies AND to the corresponding discriminants of f and g.]

  Description [Applies logical AND to the corresponding discriminants
  of f and g. f and g must have only 0 or 1 as terminal nodes. Returns
  a pointer to the result if successful; a failure is generated
  otherwise.]

  SideEffects []

  SeeAlso     [add_or add_xor add_not]

******************************************************************************/
add_ptr add_and(DdManager * dd, add_ptr a, add_ptr b)
{
  DdNode * result;

  result = Cudd_addAnd(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "add_and: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies OR to the corresponding discriminants of f and g.]

  Description [Applies logical OR to the corresponding discriminants
  of f and g. f and g must have only 0 or 1 as terminal nodes. Returns
  a pointer to the result if successful; a failure is generated
  otherwise.]

  SideEffects []

  SeeAlso     [add_and add_xor add_not add_imply]

******************************************************************************/
add_ptr add_or(DdManager * dd, add_ptr a, add_ptr b)
{
  DdNode * result;

  result = Cudd_addOr(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "add_or: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies XOR to the corresponding discriminants of f and g.]

  Description [Applies logical XOR to the corresponding discriminants
  of f and g. f and g must have only 0 or 1 as terminal nodes. Returns
  a pointer to the result if successful; a failure is generated
  otherwise.]

  SideEffects []

  SeeAlso     [add_or add_and add_not add_imply]

******************************************************************************/
add_ptr add_xor(DdManager * dd, add_ptr a, add_ptr b)
{
  DdNode * result;

  result = Cudd_addXor(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "add_xor: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies NOT to the corresponding discriminant of f.]

  Description [Applies logical NOT to the corresponding discriminant of f.
  f must have only 0 or 1 as terminal nodes. Returns a pointer to the
  result if successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [add_and add_xor add_or add_imply]

******************************************************************************/
add_ptr add_not(DdManager * dd, add_ptr a)
{
  DdNode * result;

  result = Cudd_addNot(dd, (DdNode *)a);
  common_error(result, "add_not: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies IMPLY to the corresponding discriminants of f.]

  Description [Applies logical IMPLY to the corresponding discriminants of f.
  f must have only 0 or 1 as terminal nodes. Returns a pointer to the
  result if successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [add_and add_xor add_or add_not]

******************************************************************************/
add_ptr add_imply(DdManager * dd, add_ptr a, add_ptr b)
{
  DdNode * tmp_1;
  DdNode * result;

  tmp_1 = Cudd_addNot(dd,(DdNode *)a);
  common_error(tmp_1, "add_imply: not(a) = NULL");
  Cudd_Ref(tmp_1);
  result = Cudd_addOr(dd, tmp_1, (DdNode *)b);
  common_error2(dd, result, tmp_1, "add_imply: result = NULL");
  Cudd_Ref(result);
  Cudd_RecursiveDeref(dd, tmp_1);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies AND to the corresponding discriminants of f and g.]

  Description [Applies logical AND to the corresponding discriminants
  of f and g and stores the result in f. f and g must have only 0 or 1
  as terminal nodes.]

  SideEffects [The result is stored in the first operand.]

  SeeAlso     [add_and]

******************************************************************************/
void add_and_accumulate(DdManager * dd, add_ptr *a, add_ptr b)
{
  DdNode * result;
  
  result = Cudd_addAnd(dd,(DdNode *) *a, (DdNode *)b);
  common_error(result, "add_and_accumulate: result = NULL");
  Cudd_Ref(result);
  Cudd_RecursiveDeref(dd, (DdNode *) *a);
  *a = (add_ptr)result;
}

/**Function********************************************************************

  Synopsis    [Applies OR to the corresponding discriminants of f and g.]

  Description [Applies logical OR to the corresponding discriminants
  of f and g and stores the result in f. f and g must have only 0 or 1
  as terminal nodes.]

  SideEffects [The result is stored in the first operand.]

  SeeAlso     [add_and]

******************************************************************************/
void add_or_accumulate(DdManager * dd, add_ptr *a, add_ptr b)
{
  DdNode * result;
  
  result = Cudd_addOr(dd,(DdNode *) *a, (DdNode *)b);
  common_error(result, "add_or_accumulate: result = NULL");
  Cudd_Ref(result);
  Cudd_RecursiveDeref(dd, (DdNode *) *a);
  *a = (add_ptr)result;
}

/**Function********************************************************************

  Synopsis    [Applies op to the corresponding discriminants of f and g.]

  Description [Applies op to the corresponding discriminants of f and g.
  Returns a pointer to the result if successful; a failure is
  generated otherwise.]

  SideEffects []

******************************************************************************/
add_ptr add_apply(DdManager * dd, NPFNN op, add_ptr a, add_ptr b)
{
  DdNode * result;

  result = Cudd_addApply(dd, op, (DdNode *)a, (DdNode *)b);
  common_error(result, "add_apply: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies Set Inclusion to the corresponding
  discriminants of f and g.]

  Description [Applies logical Set Inclusion to the corresponding
  discriminants of f and g. Returns a pointer to the result if
  successful; a failure is generated otherwise.]

  SideEffects [.]

  SeeAlso     [add_apply]

******************************************************************************/
add_ptr add_setin(DdManager * dd, add_ptr a, add_ptr b)
{ return(add_apply(dd, node_setin, a, b)); }

/**Function********************************************************************

  Synopsis    [Applies equality to the corresponding discriminants of f and g.]

  Description [Applies equality to the corresponding discriminants of
  f and g. Returns a pointer to the result if successful; a failure is
  generated otherwise.] 

  SideEffects [.]

  SeeAlso     [add_apply]

******************************************************************************/
add_ptr add_equal(DdManager * dd, add_ptr a, add_ptr b)
{ return(add_apply(dd, node_equal, a, b)); }

/**Function********************************************************************

  Synopsis    [Implements ITE(f,g,h).]

  Description [Implements ITE(f,g,h). This procedure assumes that f is
  a 0-1 ADD.  Returns a pointer to the resulting ADD if successful; a
  failure is generated otherwise.]

  SideEffects []

******************************************************************************/
add_ptr add_ifthenelse(DdManager * dd, add_ptr If, add_ptr Then, add_ptr Else)
{
  DdNode * result;

  result = Cudd_addIte(dd, (DdNode *)If, (DdNode *)Then, (DdNode *)Else);
  common_error(result, "add_ifthenelse: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Computes the difference between two ADD cubes.]

  Description [Computes the difference between two ADD cubes, i.e. the
  cube of ADD variables belonging to cube a and not belonging to cube
  b. Returns a pointer to the resulting cube; a failure is generated
  otherwise.]

  SideEffects []

  SeeAlso     [bdd_cube_diff]

******************************************************************************/
add_ptr add_cube_diff(DdManager * dd, add_ptr a, add_ptr b)
{
  DdNode * result;

  result = Cudd_addCubeDiff(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "add_cube_diff: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Permutes the variables of an ADD.]

  Description [Given a permutation in array permut, creates a new ADD
  with permuted variables. There should be an entry in array permut
  for each variable in the manager. The i-th entry of permut holds the
  index of the variable that is to substitute the i-th variable.
  Returns a pointer to the resulting ADD if successful; a failure is
  generated otherwise. The reuslt is referenced.]

  SideEffects []

  SeeAlso     [bdd_permute]

******************************************************************************/
add_ptr add_permute(DdManager * dd, add_ptr fn, int * permut)
{
  DdNode *result;
  
  result = Cudd_addPermute(dd, (DdNode *)fn, permut);
  common_error(result, "add_permute: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Finds the variables on which an ADD depends on.]

  Description [Finds the variables on which an ADD depends on.
  Returns an ADD consisting of the product of the variables if
  successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_support]

*****************************************************************************/
add_ptr add_support(DdManager * dd, add_ptr fn)
{
  DdNode * tmp_1, * result;
  
  tmp_1 = Cudd_Support(dd, (DdNode *)fn);
  common_error(tmp_1, "add_support: tmp_1 = NULL");
  Cudd_Ref(tmp_1);
  result = Cudd_BddToAdd(dd, tmp_1);
  common_error2(dd, result, tmp_1, "add_support: result = NULL");
  Cudd_RecursiveDeref(dd, tmp_1);
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis [ADD restrict according to Coudert and Madre's algorithm (ICCAD90).]

  Description [ADD restrict according to Coudert and Madre's algorithm
  (ICCAD90). Returns the restricted ADD if successful; a failure is
  generated otherwise.
  If application of restrict results in an ADD larger than the input
  ADD, the input ADD is returned.]

  SideEffects []

******************************************************************************/
add_ptr add_simplify_assuming(DdManager * dd, add_ptr a, add_ptr b)
{
  DdNode * result;

  result = Cudd_addRestrict(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "add_simplify_assuming: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Counts the number of ADD nodes in an ADD.]

  Description [Counts the number of ADD nodes in an ADD. Returns the number
  of nodes in the graph rooted at node.]

  SideEffects []

  SeeAlso     [add_count_minterm]

******************************************************************************/
int add_size(DdManager * dd, add_ptr fn)
{
  return(Cudd_DagSize((DdNode *)fn));
}

/**Function********************************************************************

  Synopsis    [Counts the number of ADD minterms of an ADD.]

  Description [Counts the number of minterms of an ADD. The function is
  assumed to depend on nvars variables. The minterm count is
  represented as a double, to allow for a larger number of variables.
  Returns the number of minterms of the function rooted at node. The
  result is parameterized by the number of \"nvars\" passed as argument.]

  SideEffects []

  SeeAlso     [bdd_size bdd_count_minterm]

******************************************************************************/
double add_count_minterm(DdManager * dd, add_ptr fn, int nvars)
{
  return(Cudd_CountMinterm(dd, (DdNode *)fn, nvars));
}

/**Function********************************************************************

  Synopsis    [Given the result of add_if_then it returns the leaf corresponding.]

  Description [Given the result of add_if_then it returns the leaf
  corresponding. The ADD is traversed according to the rules given as
  a result of add_if_then. If it is costant, then the corresponding
  value is returned. The Else branch is recursively traversed, if the
  result of this travesring is an ELSE_CNST, then the result of the
  traversing of the Then branch is returned.]

  SideEffects []

  SeeAlso     [add_if_then]

******************************************************************************/
node_ptr add_value(DdManager * dd, add_ptr fn)
{
  node_ptr result;
 
  result = Cudd_add_value((DdNode *)fn);
  return(result);
}

/**Function********************************************************************

  Synopsis    [Given a minterm, it returns an ADD indicating the rules
  to traverse the ADD.]

  Description [Given a minterm, it returns an ADD indicating the rules
  to traverse the ADD.]

  SideEffects []

  SeeAlso     [add_value]

******************************************************************************/
add_ptr add_if_then(DdManager * dd, add_ptr I, add_ptr T)
{
  DdNode * result;

  result = Cudd_addIfThen(dd, (DdNode *)I, (DdNode *)T);
  common_error(result, "add_if_then: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies a generic function to constant nodes.]

  Description [Applies a generic function <tt>VPFCVT op</tt> to the
  constants nodes of <tt>f</tt>.]

  SideEffects []

******************************************************************************/
void add_walkleaves(VPFCVT op, add_ptr f)
{
 Cudd_addWalkLeaves(op, (DdNode *)f);
 return;
}

/**Function********************************************************************

  Synopsis           [Reads the constant 1 BDD of the manager.]

  Description        [Reads the constant 1 BDD of the manager.]

  SideEffects        []

  SeeAlso            [bdd_zero]
******************************************************************************/
bdd_ptr bdd_one(DdManager * dd)
{
  DdNode * result = Cudd_ReadOne(dd);

  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Reads the constant 0 BDD of the manager.]

  Description        [Reads the constant 0 BDD of the manager.]

  SideEffects        []

  SeeAlso            [bdd_one]
******************************************************************************/
bdd_ptr bdd_zero(DdManager * dd)
{
  DdNode * result = Cudd_ReadLogicZero(dd);

  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Reference an BDD node.]

  Description        [Reference an BDD node.]

  SideEffects        [The reference count of the node is incremented by one.]

  SeeAlso            [bdd_deref bdd_free]
******************************************************************************/
void bdd_ref(bdd_ptr dd_node)
{
  Cudd_Ref(dd_node);
}

/**Function********************************************************************

  Synopsis           [Dereference an BDD node.]

  Description        [Dereference an BDD node.]

  SideEffects        [The reference count of the node is decremented by one.]

  SeeAlso            [bdd_ref bdd_free]
******************************************************************************/
void bdd_deref(bdd_ptr dd_node)
{
  Cudd_Deref(dd_node);
}

/**Function********************************************************************

  Synopsis           [Dereference an BDD node. If it dies, recursively decreases
  the reference count of its children.]

  Description        [Decreases the reference count of node. If the node dies,
  recursively decreases the reference counts of its children. It is used to
  dispose off a BDD that is no longer needed.]

  SideEffects        [The reference count of the node is decremented by one,
  and if the node dies a recursive dereferencing is applied to its children.]

  SeeAlso            []
******************************************************************************/
void bdd_free(DdManager * dd, bdd_ptr dd_node) 
{
  common_error(dd_node, "bdd_free: dd_node = NULL");
  Cudd_RecursiveDeref(dd, (DdNode *)dd_node);
}

/**Function********************************************************************

  Synopsis           [Creates a copy of an BDD node.]

  Description        [Creates a copy of an BDD node.]

  SideEffects        [The reference count is increased by one unit.]

  SeeAlso            [bdd_ref bdd_free bdd_deref]

******************************************************************************/
bdd_ptr bdd_dup(bdd_ptr dd_node)
{
  Cudd_Ref(dd_node);
  return(dd_node);
}

/**Function********************************************************************

  Synopsis    [Applies NOT to the corresponding discriminant of f.]

  Description [Applies logical NOT to the corresponding discriminant of f.
  f must be a BDD. Returns a pointer to the result if successful; a
  failure is generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_and bdd_xor bdd_or bdd_imply]

******************************************************************************/
bdd_ptr bdd_not(DdManager * dd, bdd_ptr fn)
{
  DdNode * result;

  result = Cudd_Not(fn);
  common_error(result, "bdd_not: result == NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies AND to the corresponding discriminants of f and g.]

  Description [Applies logical AND to the corresponding discriminants
  of f and g. f and g must be BDDs. Returns a pointer to the result if
  successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_or bdd_xor bdd_not]

******************************************************************************/
bdd_ptr bdd_and(DdManager * dd, bdd_ptr a, bdd_ptr b)
{
  DdNode * result;

  result = Cudd_bddAnd(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "bdd_and: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies OR to the corresponding discriminants of f and g.]

  Description [Applies logical OR to the corresponding discriminants
  of f and g. f and g must be BDDs. Returns a pointer to the result if
  successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_and bdd_xor bdd_not]

******************************************************************************/
bdd_ptr bdd_or(DdManager * dd, bdd_ptr a, bdd_ptr b)
{
  DdNode * result;

  result = Cudd_bddOr(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "bdd_or: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies XOR to the corresponding discriminants of f and g.]

  Description [Applies logical XOR to the corresponding discriminants
  of f and g. f and g must be BDDs. Returns a pointer to the result if
  successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_or bdd_imply bdd_not]

******************************************************************************/
bdd_ptr bdd_xor(DdManager * dd, bdd_ptr a, bdd_ptr b)
{
  DdNode * result;

  result = Cudd_bddXor(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "bdd_xor: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies IMPLY to the corresponding discriminants of f and g.]

  Description [Applies logical IMPLY to the corresponding discriminants
  of f and g. f and g must be BDDs. Returns a pointer to the result if
  successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_or bdd_xor bdd_not]

******************************************************************************/
bdd_ptr bdd_imply(DdManager * dd, bdd_ptr a, bdd_ptr b)
{
  DdNode * tmp_1;
  DdNode * result;

  tmp_1 = Cudd_Not((DdNode *)a);
  common_error(tmp_1, "bdd_imply: not(a) = NULL");
  Cudd_Ref(tmp_1);
  result = Cudd_bddOr(dd, tmp_1, (DdNode *)b);
  common_error2(dd, result, tmp_1, "bdd_imply: result = NULL");
  Cudd_Ref(result);
  Cudd_RecursiveDeref(dd, tmp_1);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Applies AND to the corresponding discriminants of f and g.]

  Description [Applies logical AND to the corresponding discriminants
  of f and g and stores the result in f. f and g must be two BDDs. The
  result is referenced.]

  SideEffects [The result is stored in the first operand and referenced.]

  SeeAlso     [bdd_and]

******************************************************************************/
void bdd_and_accumulate(DdManager * dd, bdd_ptr * a, bdd_ptr b)
{
  DdNode * result;

  result = Cudd_bddAnd(dd, (DdNode *)*a, (DdNode *)b);
  common_error(result, "bdd_and_accumulate: result = NULL");
  Cudd_Ref(result);
  Cudd_RecursiveDeref(dd, (DdNode *)*a);
  *a = result;
  return;
}

/**Function********************************************************************

  Synopsis    [Applies OR to the corresponding discriminants of f and g.]

  Description [Applies logical OR to the corresponding discriminants
  of f and g and stores the result in f. f and g must be two BDDs. The
  result is referenced.]

  SideEffects [The result is stored in the first operand and referenced.]

  SeeAlso     [bdd_and]

******************************************************************************/
void bdd_or_accumulate(DdManager * dd, bdd_ptr * a, bdd_ptr b)
{
  DdNode * result;

  result = Cudd_bddOr(dd, (DdNode *)*a, (DdNode *)b);
  common_error(result, "bdd_or_accumulate: result = NULL");
  Cudd_Ref(result);
  Cudd_RecursiveDeref(dd, (DdNode *) *a);
  *a = result;
  return;
}

/**Function********************************************************************

  Synopsis [Existentially abstracts all the variables in cube from fn.]

  Description [Existentially abstracts all the variables in cube from fn.
  Returns the abstracted BDD if successful; a failure is generated
  otherwise.]

  SideEffects []

  SeeAlso     [bdd_forall]

******************************************************************************/
bdd_ptr bdd_forsome(DdManager * dd, bdd_ptr fn, bdd_ptr cube)
{
  DdNode * result;

  result = Cudd_bddExistAbstract(dd, (DdNode *)fn, (DdNode *)cube);
  common_error(result, "bdd_forsome: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Universally abstracts all the variables in cube from f.]

  Description [Universally abstracts all the variables in cube from f.
  Returns the abstracted BDD if successful; a failure is generated
  otherwise.]

  SideEffects []

  SeeAlso     [bdd_forsome]

******************************************************************************/
bdd_ptr bdd_forall(DdManager * dd, bdd_ptr fn, bdd_ptr cube)
{
  DdNode * result;

  result = Cudd_bddUnivAbstract(dd, (DdNode *)fn, (DdNode *)cube);
  common_error(result, "bdd_forall: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Permutes the variables of a BDD.]

  Description [Given a permutation in array permut, creates a new BDD
  with permuted variables. There should be an entry in array permut
  for each variable in the manager. The i-th entry of permut holds the
  index of the variable that is to substitute the i-th variable.
  Returns a pointer to the resulting BDD if successful; a failure is
  generated otherwise. The reuslt is referenced.]

  SideEffects []

  SeeAlso     [bdd_permute]

******************************************************************************/
bdd_ptr bdd_permute(DdManager * dd, bdd_ptr fn, int * permut)
{
  DdNode * result;

  result = Cudd_bddPermute(dd, (DdNode *)fn, permut);
  common_error(result, "bdd_permute: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis [Takes the AND of two BDDs and simultaneously abstracts the
  variables in cube.]

  Description [Takes the AND of two BDDs and simultaneously abstracts
  the variables in cube. The variables are existentially abstracted.
  Returns a pointer to the result if successful; a failure is
  generated otherwise.]

  SideEffects []

  SeeAlso     [bdd_and bdd_forsome]

******************************************************************************/
bdd_ptr bdd_and_abstract(DdManager *dd, bdd_ptr T, bdd_ptr S, bdd_ptr V)
{
  DdNode * result;

  result = Cudd_bddAndAbstract(dd, (DdNode *)T, (DdNode *)S, (DdNode *)V);
  common_error(result, "bdd_and_abstract: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis [BDD restrict according to Coudert and Madre's algorithm
  (ICCAD90).]

  Description [BDD restrict according to Coudert and Madre's algorithm
  (ICCAD90). Returns the restricted BDD if successful; a failure is
  generated otherwise.
  If application of restrict results in an BDD larger than the input
  BDD, the input BDD is returned.]

  SideEffects []

******************************************************************************/
bdd_ptr bdd_simplify_assuming(DdManager *dd, bdd_ptr fn, bdd_ptr c)
{
  DdNode * result;

  result = Cudd_bddRestrict(dd, (DdNode *)fn, (DdNode *)c);
  common_error(result, "bdd_simplify_assuming: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Restrict operator as described in Coudert et al. ICCAD90.]

  Description [Restrict operator as described in Coudert et
  al. ICCAD90.  Always returns a BDD not larger than the input
  <code>f</code> if successful; a failure is generated otherwise. The
  result is referenced.]

  SideEffects []

  SeeAlso     [bdd_simplify_assuming]

******************************************************************************/
bdd_ptr bdd_minimize(DdManager *dd, bdd_ptr fn, bdd_ptr c)
{
  DdNode * result;

  result = Cudd_bddRestrict(dd, (DdNode *)fn, (DdNode *)c);
  common_error(result, "bdd_minimize: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
} /* end of bdd_minimize */

/**Function********************************************************************

  Synopsis    [Computes f constrain c.]

  Description [Computes f constrain c (f @ c).
  Uses a canonical form: (f' @ c) = ( f @ c)'.  (Note: this is not true
  for c.)  List of special cases:
    <ul>
    <li> F @ 0 = 0
    <li> F @ 1 = F
    <li> 0 @ c = 0
    <li> 1 @ c = 1
    <li> F @ F = 1
    <li> F @ F'= 0
    </ul>
  Returns a pointer to the result if successful; a failure is
  generated otherwise.] 

  SideEffects []

  SeeAlso     [bdd_minimize bdd_simplify_assuming]

******************************************************************************/
bdd_ptr bdd_cofactor(DdManager * dd, bdd_ptr f, bdd_ptr g)
{
  DdNode *result;

  /* We use Cudd_bddConstrain instead of Cudd_Cofactor for generality. */
  result = Cudd_bddConstrain(dd, (DdNode *)f, (DdNode *)g);
  common_error(result, "bdd_cofactor: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
} /* end of bdd_cofactor */


/**Function********************************************************************

  Synopsis    [Return a minimum size BDD between bounds.]

  SideEffects []

  SeeAlso     [bdd_minimize bdd_simplify_assuming bdd_cofactor]

******************************************************************************/
bdd_ptr bdd_between(DdManager *dd, bdd_ptr f_min, bdd_ptr f_max)
{
  bdd_ptr care_set, ret;

  care_set = bdd_imply(dd, f_min, f_max);
  ret = bdd_minimize(dd, f_min, care_set);
  bdd_free(dd, care_set);
  /*
    The size of ret is never larger than the size of f_min. We need
    only to check ret against f_max.
  */ 
  if (bdd_size(dd, f_max) <= bdd_size(dd, ret)) {
    bdd_free(dd, ret);
    return(bdd_dup(f_max));
  } else {
    return(ret);
  }
} /* end of bdd_between */
  
/**Function********************************************************************

  Synopsis    [Counts the number of BDD nodes in an BDD.]

  Description [Counts the number of BDD nodes in an BDD. Returns the number
  of nodes in the graph rooted at node.]

  SideEffects []

  SeeAlso     [bdd_count_minterm]

******************************************************************************/
int bdd_size(DdManager * dd, bdd_ptr fn)
{
  return(Cudd_DagSize((DdNode *)fn));
}

/**Function********************************************************************

  Synopsis    [Counts the number of BDD minterms of an BDD.]

  Description [Counts the number of minterms of an BDD. The function is
  assumed to depend on nvars variables. The minterm count is
  represented as a double, to allow for a larger number of variables.
  Returns the number of minterms of the function rooted at node. The
  result is parameterized by the number of \"nvars\" passed as argument.]

  SideEffects []

  SeeAlso     [bdd_size bdd_count_minterm]

******************************************************************************/
double bdd_count_minterm(DdManager * dd, bdd_ptr fn, int nvars)
{
  return(floor(Cudd_CountMinterm(dd, (DdNode *)fn, nvars)));
}

/**Function********************************************************************

  Synopsis    [Finds the variables on which an BDD depends on.]

  Description [Finds the variables on which an BDD depends on.
  Returns an BDD consisting of the product of the variables if
  successful; a failure is generated otherwise.]

  SideEffects []

  SeeAlso     [add_support]

*****************************************************************************/
bdd_ptr bdd_support(DdManager *dd, bdd_ptr fn)
{
  DdNode * result;
  
  result = Cudd_Support(dd, (DdNode *)fn);
  common_error(result, "bdd_support: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis [Picks one on-set minterm deterministically from the given BDD.]

  Description [Picks one on-set minterm deterministically from the
  given DD. The minterm is in terms of vars. Builds a BDD for the
  minterm and returns a pointer to it if successful; a failure is
  generated otherwise. There are two reasons why the procedure may fail: It may
  run out of memory; or the function fn may be the constant 0. The
  result is referenced.]

  SideEffects []

******************************************************************************/
bdd_ptr bdd_pick_one_minterm(DdManager * dd, bdd_ptr fn, bdd_ptr * vars, int n)
{
  DdNode * result;
  bdd_ptr zero = bdd_zero(dd);
  
  if (fn == zero) {
    Cudd_Ref(fn);
    Cudd_Deref(zero);
    return(fn);
  }
  else {
    result = Cudd_bddPickOneMintermNR(dd, (DdNode *)fn, (DdNode **)vars, n);
    common_error(result, "bdd_pick_one_minterm: result = NULL");
    Cudd_Deref(zero);
    Cudd_Ref(result);
    return((bdd_ptr)result);
  }
}

/**Function********************************************************************

  Synopsis    [Picks one on-set minterm randomly from the given DD.]

  Description [Picks one on-set minterm randomly from the given DD. The
  minterm is in terms of vars. Builds a BDD for the minterm and returns a
  pointer to it if successful; a failure is generated otherwise. There
  are two reasons why the procedure may fail: It may run out of
  memory; or the function f may be the constant 0.]

  SideEffects []

******************************************************************************/
bdd_ptr bdd_pick_one_minterm_rand(DdManager * dd, bdd_ptr fn, bdd_ptr * vars, int n)
{
  DdNode * result;
  bdd_ptr zero = bdd_zero(dd);
  
  if (fn == zero) {
    Cudd_Ref(fn);
    Cudd_Deref(zero);
    return(fn);
  }
  else {
    result = Cudd_bddPickOneMinterm(dd, (DdNode *)fn, (DdNode **)vars, n);
    common_error(result, "bdd_pick_one_minterm_rand: result = NULL");
    Cudd_Deref(zero);
    Cudd_Ref(result);
    return((bdd_ptr)result);
  }
}

/**Function********************************************************************

  Synopsis           [Returns the BDD variable with index <code>index</code>.]

  Description        [Retrieves the BDD variable with index <code>index</code>
  if it already exists, or creates a new BDD variable. Returns a
  pointer to the variable if successful; a failure is generated
  otherwise. The returned value is referenced.]

  SideEffects        []

  SeeAlso            [bdd_new_var_at_level add_new_var_at_level]

******************************************************************************/
bdd_ptr bdd_new_var_with_index(DdManager * dd, int index)
{
  DdNode * result;
  
  result = Cudd_bddIthVar(dd, index);
  common_error(result, "bdd_new_var_with_index: result = NULL");
  /* bdd var does not require to be referenced when created */
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis           [TODO]

  Description        [TODO]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
add_ptr make_var_mask(DdManager * dd, add_ptr N, int * bit_ev, int offset)
{
  DdNode * result;

  result = Cudd_AddMakeVarMask(dd, N, bit_ev, offset);
  common_error(result, "make_var_mask: result = NULL");
  Cudd_Ref(result);
  return((add_ptr)result);
}

/**Function********************************************************************

  Synopsis    [Computes the difference between two BDD cubes.]

  Description [Computes the difference between two BDD cubes, i.e. the
  cube of BDD variables belonging to cube a and not belonging to cube
  b. Returns a pointer to the resulting cube; a failure is generated
  otherwise.]

  SideEffects []

  SeeAlso     [add_cube_diff]

******************************************************************************/
bdd_ptr bdd_cube_diff(DdManager * dd, bdd_ptr a, bdd_ptr b)
{
  DdNode * result;

  result = Cudd_bddCubeDiff(dd, (DdNode *)a, (DdNode *)b);
  common_error(result, "bdd_cube_diff: result = NULL");
  Cudd_Ref(result);
  return((bdd_ptr)result);
}

/**Function********************************************************************

  Synopsis           [Returns the index of a ADD.]

  Description        [Returns the index of the variable in an ADD.]

  SideEffects        [Yuan Lu]

******************************************************************************/
int add_read_index(add_ptr a)
{
  return Cudd_NodeReadIndex((DdNode *)a);
}

/**Function********************************************************************

  Synopsis           [Returns the index of the lowest variable in the ADD a.]

  Description        [Returns the index of the lowest variable in the
  ADD, i.e. the variable in ADD a with the highest position in the
  ordering. ]

  SideEffects        [Yuan Lu]

******************************************************************************/
int add_get_lowest_index(DdManager * dd, add_ptr a)
{
  int result;

  result = Cudd_AddGetLowestVar(dd, (DdNode *)a);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Returns the index of the lowest variable in the BDD a.]

  Description        [Returns the index of the lowest variable in the
  BDD, i.e. the variable in BDD a with the highest position in the
  ordering. ]

  SideEffects        []

******************************************************************************/
int bdd_get_lowest_index(DdManager * dd, bdd_ptr a)
{
  int result;
  
  result = Cudd_BddGetLowestVar(dd, (DdNode *)a);
  return(result);
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [Yuan Lu]

******************************************************************************/
int dd_check_manager(DdManager * dd)
{
  return Cudd_DebugCheck(dd);
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [Yuan Lu]

******************************************************************************/
int bdd_cofactor_equal(DdManager *dd_manager, bdd_ptr f, add_ptr rel1, add_ptr rel2)
{
  return Cudd_CofactorEqual(dd_manager, (DdNode *)f, (DdNode *)rel1, (DdNode *)rel2);
}

/**Function********************************************************************

  Synopsis           [Build a list of variables for a cube.]

  Description        [Build a list of variables for a cube.]

  SideEffects        [Yuan Lu]

******************************************************************************/
int bdd_cube_to_vars(DdManager *dd_manager, bdd_ptr cube, bdd_ptr *varl)
{
  return Cudd_CubeToVars(dd_manager, cube, varl);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis [Function to print a warning that an illegal value was read.]

  SideEffects        []

  SeeAlso            [bdd_set_parameters]

******************************************************************************/
static void InvalidType(FILE *file, char *field, char *expected)
{
    (void) fprintf(file, "Warning: In parameter \"%s\"\n", field);
    (void) fprintf(file, "Illegal type detected. %s expected\n", expected);

} /* end of InvalidType */

