/**CFile***********************************************************************

  FileName    [compileUtil.c]

  PackageName [compile]

  Synopsis    [Routines for model computation.]

  Description [This file contains the code for the compilation of the
  flattened hierarchy into BDD:
  <ul>
  <li> Creation of the boolean variables.</li>
  <li> Creation of the BDD representing the inertia of the system when
       there are processes. In fact when a process is running the
       other processes are stopped, and their state variables don't
       change.</li>
  <li> Creation of the BDD representing what does not change in the
       system, i.e. the set of invariance. These are introduced in the
       model by the keyword "<tt>INVAR</tt>" or by the <em>normal
       assignments</em> (i.e. "<tt>ASSIGN x : = y & z;</tt>"). These
       states are not stored in the transition relation, they are
       stored in an a doc variable.
  <li> Creation of the BDD representing the set of initial states.
  <li> Creation of the BDD representing the transition relation. 
       Various ways of representing the transition relation are offered
       the users.
       <ul>
       <li> <em>Monolithic</em>: the monolithic transition relation is
            computed.</li>
       <li> <em>Disjunctive Partioned</em>: the transition relation is
            stored as an implicitly disjoined list of transition relation.
            This kind of partitioning can be used only if the model
            considered is an asynchronous one.</li>
       <li> <em>Conjunctive Partioned</em>: the transition relation is
            stored as an implicitly conjoined list of transition relation.
            This kind of partitioning can be used only if the model
            considered is a synchronous one.</li>
       <li> <em>Conjunctive Partioned IWLS95</em>: as the above, but the
            heuristic proposed in \[1\] is used to order partition clusters. </li>
       </ul>
  <li> Computes the fairness constraints. I.e. each fairness constraint
       (which can be a CTL formula) is evaluated and the resulting BDD
       is stored in the list <tt>fairness_constraints_bdd</tt> to be
       then used in the model checking phase.
  </ul>
  \[1\] R. K. Ranjan and A. Aziz and B. Plessier and C. Pixley and R. K. Brayton,
      "Efficient BDD Algorithms for FSM Synthesis and Verification,
      IEEE/ACM Proceedings International Workshop on Logic Synthesis,
      Lake Tahoe (NV), May 1995.</li>
  ]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/
#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Macro***********************************************************************

  Synopsis     [The symbolic name of the input process selector variable.]

  Description  [This is the internal symbolic name of process selector
  variable. The range of this variable is the set of names of the
  instantiated processes.]

  SideEffects  []

******************************************************************************/
#define PROCESS_SELECTOR_VAR_NAME "_process_selector_"

/**Macro***********************************************************************

  Synopsis     [The "running" symbol.]

  Description  [The "running" symbol used to refer the internal
  variable "running" of  processes.]

  SeeAlso      []

******************************************************************************/
#define RUNNING_SYMBOL "running"

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

typedef enum {
  State_Instantiation_Mode,
  Input_Instantiation_Mode
} Instantiation_Vars_Mode_Type;

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The initial states.]

  Description [The {A|B}DD representing the initial states.]

  SeeAlso     []

******************************************************************************/
add_ptr init_add = (add_ptr)NULL;
bdd_ptr init_bdd = (bdd_ptr)NULL;

/* Yuan Lu */
add_ptr init_add_orig = (add_ptr)NULL;
bdd_ptr init_bdd_orig = (bdd_ptr)NULL;
add_ptr init_add_orig1 = (add_ptr)NULL;
bdd_ptr init_bdd_orig1 = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [The invariant states.]

  Description [The {A|B}DD representing the invariant states.]

  SeeAlso     []

******************************************************************************/
add_ptr invar_add = (add_ptr)NULL;
bdd_ptr invar_bdd = (bdd_ptr)NULL;

/* Yuan Lu */
add_ptr invar_add_orig = (add_ptr)NULL;
bdd_ptr invar_bdd_orig = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [ADD representing what does not change.]

  Description [ADD representing what does not change at each
  transition. It has a value different from <tt>TRUE</tt> iff there
  are processes. In this case it is conjoined with the trans.]

  SeeAlso     [optional]

******************************************************************************/
add_ptr frame_add = (add_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [Contains the list of process names.]

  Description [The list of process names. It represents the range of the 
  <tt>PROCESS_SELECTOR_VAR_NAME</tt> input variable.]

******************************************************************************/
static node_ptr process_selector_range = Nil;

/**Variable********************************************************************

  Synopsis    [The internal name of the <tt>PROCESS_SELECTOR_VAR_NAME</tt> input variable.]

  Description [Stores the internal name name of the
  <tt>PROCESS_SELECTOR_VAR_NAME</tt> input variable. It is a
  contextualized variable (i.e if
  <tt>PROCESS_SELECTOR_VAR_NAME="_process_selector_"</tt>, then the
  value of this variable is: "<tt>main._process_selector_</tt>.]

******************************************************************************/
node_ptr proc_selector_internal_vname = Nil;

/**Variable********************************************************************

  Synopsis    [The hash used to build frame axioms.]

  Description [This is an hash used to build the frame axioms. The
  index used is the variable name. In this hash for each variable name
  <tt>vname</tt> there is associated an ADD of this form:
  <tt>P1.running | P2.running | ... | Pn.running</tt> where in each
  body of the module instantiated as <tt>Pi</tt> occurs
  <tt>next(vname) := ...</tt>.]

  SeeAlso     [Compile_CompileFrame]

******************************************************************************/
static hash_ptr frame_hash;
void init_frame_hash() { frame_hash = new_assoc(); }
void insert_frame_hash(node_ptr x, node_ptr y) { insert_assoc(frame_hash, x, y);}
node_ptr lookup_frame_hash(node_ptr x) {return(find_assoc(frame_hash, x));}
void clear_frame_hash() {clear_assoc(frame_hash);}
static assoc_retval frame_hash_free(char *key, char *data, char * arg) {
  add_ptr element = (add_ptr)data;

  if (element != (add_ptr)NULL) {
    add_free(dd_manager, element);
  }
  return(ASSOC_DELETE);
}
void free_frame_hash() {clear_assoc_and_free_entries(frame_hash, frame_hash_free);}

/**Variable********************************************************************

  Synopsis    [The list of real state variables.]

  Description [It a list of ADD, one for each state variable,
  representing its encoding. It is used in <tt>print_reachable_states</tt>.]

  SeeAlso     [variables print_reachable_states]

******************************************************************************/
node_ptr real_state_variables = Nil;

/**Variable********************************************************************

  Synopsis    [The ADD of process selector.]

  Description [The ADD of process selector. Its an ADD whose leaves
  are the names of the processes. It is used to choose which process
  run and when. It is essentially the boolean encoding of the
  <tt>PROCESS_SELECTOR_VAR_NAME</tt>.]

  SeeAlso     [running_add running_atom PROCESS_SELECTOR_VAR_NAME]

******************************************************************************/
add_ptr process_selector_add; 

/**Variable********************************************************************

  Synopsis    [The NuSMV atom <tt>running</tt>.]

  Description [The internal representation of the NuSMV atom <tt>running</tt>.]

  SeeAlso     [process_selector_add running_add]

******************************************************************************/
node_ptr running_atom; 

/**Variable********************************************************************

  Synopsis    [The NuSMV ADD for atom <em>running</em>.]

  Description [The ADD corresponding to the boolean NuSMV variable <em>running</em>.]

  SeeAlso     [running_atom process_selector_add]

******************************************************************************/
add_ptr running_add;       

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void build_proc_selector_recur ARGS((node_ptr n, node_ptr context));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Builds an internal representation for a given string.]

  Description        [Builds an internal representation for a given
  string. If the conversion has been performed in the past, then the
  hashed value is returned back, else a new one is created, hashed and
  returned. We hash this in order to allow the following:
  <pre>
  VAR
     x : {a1, a2, a3};
     y : {a3, a4, a5};

  ASSIGN
     next(x) := case
                 x = y    : a2;
                 !(x = y) : a1;
                 1        : a3;
                esac;
  </pre>
  i.e. to allow the equality test between x and y. This can be
  performed because we internally have a unique representation of the
  atom <tt>a3</tt>.]

  SideEffects        []

  SeeAlso            [find_atom]

******************************************************************************/
node_ptr sym_intern(char *s)
{
  return(find_node(ATOM, (node_ptr)find_string(s), Nil));
}


/**Function********************************************************************

  Synopsis           [Creates the internal process selector variable.]

  Description        [Creates the internal process selector
  variable. The internal process selector variable is as default
  positioned at the top of the ordering and appended to
  <tt>input_variables</tt> and <tt>all_variables</tt>. ]

  SideEffects        [<tt>input_variables</tt> and
  <tt>all_variables</tt> are affected.]

  SeeAlso            []

******************************************************************************/
void create_process_symbolic_variables(node_ptr procs_expr)
{
  /* We create the internal representation of the process selector variable */
  proc_selector_internal_vname = find_node(DOT, Nil,
                                           sym_intern(PROCESS_SELECTOR_VAR_NAME));
  /* We extract the range of the process_selector_var_name */
  process_selector_range = map(car, procs_expr);

  /* We insert in the symbol hash the symbolic process variable */
  insert_symbol_hash(proc_selector_internal_vname,
                     find_node(VAR, Nil, process_selector_range));
  /*
    We add to the list of input variables the process selector
    variable, in such a way that as default it is at the top of the
    ordering.
  */
  input_variables = cons(proc_selector_internal_vname, input_variables);
  all_variables   = cons(proc_selector_internal_vname, all_variables);
}

/**Function********************************************************************

  Synopsis           [The ADD variables are built.]

  Description        [This function creates the ADD variables needed
  to build the automaton. If an input order file is given as option,
  then the ordering of variables is read from it, else the order in
  which the variables appear in a depth first traversal of the
  hierarchy is used. The symbolic name of the variable is stored in an
  array in a position corresponding to the first index of the boolean
  variable created to encode it. The boolean variables needed to
  encode a scalar symbolic variable are grouped together, so that the
  ordering methods do not modify the internal order of them. If the
  input order file contains a subset of the variables of the model,
  than the variables not specified in the ordering file are created at
  the end of the given ordering following the order in which they
  appear in a depth first traversal of the hierarchy. If the input
  ordering file is a superset of the variables declared in the model,
  than the variables not appearing in the model are discarded.]

  SideEffects        [<tt>state_variables_add</tt> <tt>next_state_variables_add</tt>
  <tt>input_vars_add</tt> <tt>num_of_state_variables</tt> are modified.]

  SeeAlso            [Compile_EncodeVar Compile_ReadOrder dd_new_var_block]

******************************************************************************/
void Compile_BuildVarsBdd(void)
{
  node_ptr read_variables, l_o_vars;

  if (get_input_order_file(options) != NULL) {
    /*
      If the input order file is defined, then the possibly partial
      variable order is read in, and stored in the variable
      "read_variables".
    */
    read_variables = Compile_ReadOrder(get_input_order_file(options));
    l_o_vars = read_variables;
  }
  else {
    l_o_vars = all_variables;
  }

  /* Initialization of the ADD variables */
  state_variables_add = add_one(dd_manager);
  next_state_variables_add = add_one(dd_manager);
  input_variables_add = add_one(dd_manager);
  next_input_variables_add = add_one(dd_manager);

  while (l_o_vars) { /* Loops over all the read or defined variables, depending the
                     input order file is provided or not, to create add/bdd variables. */
    node_ptr q;
    node_ptr vname = car(l_o_vars); /* The current variable */

    q = lookup_symbol_hash(vname);
    if (!q || node_get_type(q) != VAR) {
      /* The current variable does not appear in the read model !! */
      warning_variable_not_declared(vname);
      l_o_vars = cdr(l_o_vars);
      continue;
    }
    if (car(q)) {
      /* The current variable appear twice in the ordering file! */
      warning_var_appear_twice_in_order_file(vname);
      l_o_vars = cdr(l_o_vars);
      continue;
    }
    if (opt_verbose_level_gt(options, 1)) {
      print_node(nusmv_stderr, vname);
      fprintf(nusmv_stderr, ":\n");
    }

    q->left.bddtype = Compile_EncodeVar(vname, cdr(q), memberp(vname, input_variables));
    /* ^^^^^^^^^^^^^ Side effects on the hash */

    /* Yuan Lu : encode abstract variables */
    {
      node_ptr absf = lookup_abstract_hash(vname);
      if(absf) {
	node_ptr varl = car(car(absf));
	int count = 0;
	for(;varl;varl=cdr(varl)) {
	  node_ptr var = car(varl);
	  if(((unsigned)var) & 0x1) continue;
	  count++;
	  if(var == vname)
	    varl->left.inttype |= 0x1;
	}
	if(count == 0)
	  rpterr("SOMETHING WrONG");
	if(count == 1) {
	  varl = car(car(absf));
	  for(varl=car(car(absf));varl;varl=cdr(varl))
	    varl->left.inttype &= 0xfffffffe;
          cdr(cdr(absf))->left.bddtype = Compile_EncodeVar(vname, car(cdr(absf)), 0);
	}
      }
    }
    
    l_o_vars = cdr(l_o_vars); 
  }
  /* If the ordering is read from the ordering input file, then
     additional check have to be performed. */
  if (get_input_order_file(options) != NULL) {
    /*
      Check that all the variables defined in the source file (listed
      in global variable "all_variables") are listed in the order file. If
      not the case, then an a warning is printed out, and the variables
      are created at the current position in the ordering.
    */
    l_o_vars = all_variables;
    while (l_o_vars) {
      node_ptr vname = car(l_o_vars);
      node_ptr q = lookup_symbol_hash(vname);

      if (!q || node_get_type(q) != VAR) error_unknown_var_in_order_file(vname);
      if (!car(q)) { /* The current variable has not been specified in
                        the source file. We create it in the following
                        free position in the ordering, i.e. at the
                        end. */
        warning_missing_variable(vname);
        if (opt_verbose_level_gt(options, 1)) {
          print_node(nusmv_stderr, vname);
          fprintf(nusmv_stderr, ":\n");
        }

        q->left.bddtype = Compile_EncodeVar(vname, cdr(q), memberp(vname, input_variables));
        /* ^^^^^^^^^^^^^ Side effects on the hash */
      }
      l_o_vars = cdr(l_o_vars);
    }
  }
  if (!opt_reorder(options)) Compile_WriteOrder(get_output_order_file(options), 0);
  /*
    Creation of the array used to extract one minterm from a given
    set of states.
  */
  (void) make_state_vars();
}

/**Function********************************************************************

  Synopsis           [Creates the <tt>running</tt> ADD variable.]

  Description        [Creates the <tt>running</tt> ADD variable for
  each process declared.]

  SideEffects        []

******************************************************************************/
void build_proc_selector(node_ptr procs_expr)
{
  node_ptr l = procs_expr;
  node_ptr proc_selector_var;
  

  if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "computing process selector scheduling ......\n");

  running_atom = sym_intern(RUNNING_SYMBOL);

  if (opt_verbose_level_gt(options, 1)) {
    node_ptr p = process_selector_range;
    int j = 1;
    (void) fprintf(nusmv_stderr, "The list of processes is the following:\n");
    while(p) {
      (void) fprintf(nusmv_stderr, "process %d: ", j++);
      (void) print_node(nusmv_stderr, car(p));
      (void) fprintf(nusmv_stderr, "\n");
      p = cdr(p);
    }
  }
  /* proc_selector_var = find_node(VAR, BDD_OF_ENC, RANGE) */
  proc_selector_var = lookup_symbol_hash(proc_selector_internal_vname);

  if (!proc_selector_var || node_get_type(proc_selector_var) != VAR) {
    (void) fprintf(nusmv_stderr, "******* ERROR *******\n");
    (void) fprintf(nusmv_stderr, "The variable \"");
    (void) print_node(nusmv_stderr, proc_selector_internal_vname);
    (void) fprintf(nusmv_stderr, "\" has not yet been created.\n");
    nusmv_exit(1);
  }
  if (!car(proc_selector_var)) {
    (void) fprintf(nusmv_stderr, "******* ERROR *******\n");
    (void) fprintf(nusmv_stderr, "The variable \"");
    (void) print_node(nusmv_stderr, proc_selector_internal_vname);
    (void) fprintf(nusmv_stderr, "\" has not yet been encoded in BDD.\n");
    nusmv_exit(1);
  }
  
  /* Extract the Encoding of the process selector variable */ 
  process_selector_add = (add_ptr)car(proc_selector_var);

  while (l) { /* Loops over processes */
    add_ptr  cur_process;
    node_ptr tmp_node;
    node_ptr cur_context = car(car(l));
    node_ptr cur_assign_expr = cdr(car(l));

    /*
      Creates the running variable for the current process. If
      already defined then exit.
    */
    cur_process = add_leaf(dd_manager, cur_context);
    running_add = add_equal(dd_manager, process_selector_add, cur_process);

    /* We don't free running_add because its value is memoized in symbol_hash */
    add_free(dd_manager, cur_process);

    tmp_node = eval_struct(running_atom, cur_context);
    if (lookup_symbol_hash(tmp_node)) error_redefining(tmp_node);
    insert_symbol_hash(tmp_node, find_node(BDD, (node_ptr)running_add, Nil));

    build_proc_selector_recur(cur_assign_expr, Nil);
    l = cdr(l);
  }
  if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr, "done !\n");
}

/**Function********************************************************************

  Synopsis           [Computes the ADD representing what does not
  change between two consecutive states.]

  Description        [Computes the ADD representing what does not
  change between two consecutive states. i.e. it computes the frame
  axioms. It loops over the state variables. For each variable
  <tt>vname</tt> it consider the ADD <tt>Q</tt> stored in
  <tt>frame_hash</tt>, which is an disjunction of all the processes in
  whose body occurs <tt>next(vname) := ...</tt> and computes the ADD
  <tt> !Q -> (vname' = vname)</tt>, which is conjoined with
  <tt>frame_add</tt>. <br>
  The result is stored in the global variable <tt>frame_add</tt>.]

  SeeAlso            [frame_hash]

  SideEffects  []

******************************************************************************/
void Compile_CompileFrame()
{
  node_ptr var_list = state_variables;
  add_ptr  one      = add_one(dd_manager);
  
  frame_add = add_one(dd_manager);
  while (var_list) {
    node_ptr var = car(var_list);
    add_ptr q = (add_ptr)lookup_frame_hash(var);
    add_ptr dd_var = eval(var, Nil);

    if (q && (q != one)) {
      add_ptr next_dd_var = add_shift_forward(dd_manager, dd_var);
      add_ptr tmp_1 = add_equal(dd_manager, dd_var, next_dd_var);
      add_ptr tmp_2 = add_or(dd_manager, q, tmp_1);
      add_and_accumulate(dd_manager, &frame_add, tmp_2);

      add_free(dd_manager, next_dd_var);
      add_free(dd_manager, tmp_1);
      add_free(dd_manager, tmp_2);
    }
    add_free(dd_manager, dd_var);
    var_list = cdr(var_list);
  }
  add_free(dd_manager, one);
  /* It could free the contents of the frame_hash */
  if (opt_batch(options)) free_frame_hash();
}

/**Function********************************************************************

  Synopsis           [Computes the <tt>real_state_variables</tt>, a
  list of ADDs, one for each symbolic scalar variable.]

  Description        [Computes the <tt>real_state_variables</tt>, a
  list of ADDs, one for each symbolic scalar variable representing its
  boolean encoding.]

  SideEffects        [<tt>real_state_variables</tt> is computed.]

  SeeAlso            [eval]

******************************************************************************/
void build_real_state_variables()
{
  node_ptr var_list = state_variables;
  
  real_state_variables = Nil;
  while (var_list) {
    node_ptr var = car(var_list);
    add_ptr dd_var = eval(var, Nil);
 
    real_state_variables = cons((node_ptr)dd_var, real_state_variables);
    var_list = cdr(var_list);
  }
}

/**Function********************************************************************

  Synopsis           [Computes the <tt>init_add</tt>.]

  Description        [This function evaluates all the init statements
  listed in <tt>init_expr</tt>, then evaluates all the <tt>init()</tt>
  assignments in the list of processes <tt>procs<tt>. Moreover the
  number of states and the number of ADD nodes of the global initial
  set are printed out.]

  SideEffects        [The result is stored in the ADD <tt>init_add</tt>] 

******************************************************************************/
void Compile_CompileInit(node_ptr init_expr, node_ptr procs)
{
  node_ptr l=procs;

  if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating INIT statements", Nil);
  init_add_orig = eval(init_expr, Nil);

  if(abs_expr_pre) {
    node_ptr tmp = abs_expr_pre;
    abs_expr_pre = Nil;
    init_add_orig1 = eval(init_expr, Nil);
    abs_expr_pre = tmp;
  }
  
  while(l) { /* Loops over processes */
    add_ptr cur_init;
    node_ptr context = car(car(l));
    node_ptr assign_expr = cdr(car(l));

    l = cdr(l);
    if (opt_verbose_level_gt(options, 1))
        print_in_process("evaluating init() assignments", context);
    set_assignment_type_init();
    cur_init = eval(assign_expr, Nil);
    add_and_accumulate(dd_manager, &init_add_orig, cur_init);
    add_free(dd_manager, cur_init);
    if(abs_expr_pre) {
      node_ptr tmp = abs_expr_pre;
      abs_expr_pre = Nil;
      cur_init = eval(assign_expr, Nil);
      add_and_accumulate(dd_manager, &init_add_orig1, cur_init);
      add_free(dd_manager, cur_init);
      abs_expr_pre = tmp;
    }
  }
  if (opt_verbose_level_gt(options, 1))
    fprintf(nusmv_stderr, "size of global initial set = %g states, %d ADD nodes\n",
            add_count_states(dd_manager, init_add_orig), add_size(dd_manager, init_add_orig));

  if(!abs_expr_pre) init_add_orig1 = add_dup(init_add_orig);

  /* Conversion of ADD to BDD */
  init_bdd_orig = add_to_bdd(dd_manager, init_add_orig);
  init_bdd_orig1 = add_to_bdd(dd_manager, init_add_orig1);
  add_free(dd_manager, init_add_orig);

  /* Yuan Lu : clean the effect of next state variables */
  {
    bdd_ptr stvar = add_to_bdd(dd_manager, state_variables_add);
    bdd_ptr nstvar = bdd_shift_forward(dd_manager, stvar);
    bdd_ptr t = bdd_forsome(dd_manager, init_bdd_orig, nstvar);
    bdd_ptr t1 = bdd_forsome(dd_manager, init_bdd_orig1, nstvar);
    bdd_free(dd_manager, stvar);
    bdd_free(dd_manager, nstvar);
    bdd_free(dd_manager, init_bdd_orig);
    bdd_free(dd_manager, init_bdd_orig1);
    init_bdd_orig = t;
    init_bdd_orig1 = t1;
  }

  /* Yuan Lu : applying abstraction */
  {
    bdd_ptr t = Abs_AbsAbstractCur(init_bdd_orig, abs_expr_pre);
    init_bdd = Abs_AbsAbstractCur(t, abs_expr);
  }
}

/**Function********************************************************************

  Synopsis           [Builds the the transition relation of the model.]

  Description        [Builds the the transition relation of the model,
  accordingly with the partition method specified in the options.]

  SideEffects        []

  SeeAlso            [Compile_CompileModelConj Compile_CompileModelDisj Compile_CompileModelIwls95]

******************************************************************************/
void Compile_CompileModel(node_ptr trans_expr, node_ptr invar_expr, node_ptr procs,
                 add_ptr assumption)
{
  if (opt_disj_partitioning(options)) {
    Compile_CompileModelDisj(trans_expr, invar_expr, procs, assumption);
  }
  else 
    if (opt_conj_partitioning(options)) {
      Compile_CompileModelConj(trans_expr, invar_expr, procs, assumption);
    }
  else
    if (opt_monolithic(options)) {
      build_model_monolithic(trans_expr, invar_expr, procs, assumption);
    }
  else
    if (opt_iwls95cp_partitioning(options)) {
      Compile_CompileModelIwls95(trans_expr, invar_expr, procs, assumption);
    }
}

/**Function********************************************************************

  Synopsis           [Computes the set of fairness constraints BDD.]

  Description        [Computes the set of fairness constraints BDD, in
  order to use them during the model checking phase.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void compute_fairness_constraints(node_ptr fair_expressions) {
  fairness_constraints_bdd = eval_formula_list(fair_expressions, Nil);
}

/**Function********************************************************************

  Synopsis           [Print out model statistics.]

  Description        [Depending the partition method used this
  function prints out the statistics; i.e. the size of the transition
  relation or the size of the clusters used in conjunctive or
  disjunctive partitioning.]

  SideEffects        []

******************************************************************************/
void print_model_statistic(void) {
  if (opt_monolithic(options)) print_monolithic_info();
  else if (opt_conj_partitioning(options)) print_conj_part_info();
  else if (opt_disj_partitioning(options)) print_disj_part_info();
  else if (opt_iwls95cp_partitioning(options)) print_iwls95cp_part_info();
  else (void) fprintf(nusmv_stderr, "Unknown partitioning method\n");
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Perform the recursive step of <tt>build_proc_selector</tt>.]

  Description        [Perform the recursive step of
  <tt>build_proc_selector</tt>. It also inserts in <tt>frame_hash</tt>
  the information necessary to compute the frame axioms.]

  SideEffects        []

  SeeAlso            [frame_hash Compile_CompileFrame]

******************************************************************************/
static void build_proc_selector_recur(node_ptr n, node_ptr context)
{
  if (n == Nil) return;
  yylineno = n->lineno;
  switch(node_get_type(n)) {
  case AND:
    build_proc_selector_recur(car(n), context);
    build_proc_selector_recur(cdr(n), context);
    break;
  case CONTEXT:
    build_proc_selector_recur(cdr(n), car(n));
    break;
  case EQDEF: {
    node_ptr vname, tmp_1;
    if (node_get_type(car(n)) == SMALLINIT || node_get_type(car(n)) == NEXT) {
      vname = eval_struct(car(car(n)), context);
      tmp_1 = find_node(node_get_type(car(n)), vname, Nil);
    }
    else
      vname = tmp_1 = eval_struct(car(n), context);
    /*
      NOTE: In a recent release of SMV distributed at Carnegie Mellon
      Sergei Berezin (the new maintainer) says that here you have to do
      the following test:
      if (node_get_type(tmp_1) != SMALLINIT) {
      instead of the below test.
    */
    if (node_get_type(tmp_1) == NEXT) {
      add_ptr q = (add_ptr)lookup_frame_hash(vname);
      if (!q) q = running_add;
      else add_or_accumulate(dd_manager, &q, running_add);
      add_ref(q);
      insert_frame_hash(vname, (node_ptr)q);
    }
    break;
  }
  default:
    internal_error("build_proc_selector_recur: type = %d", node_get_type(n));
  }
}



