/**CFile***********************************************************************

  FileName    [compileOrd.c]

  PackageName [compile]

  Synopsis    [Routines for reading and writing variable orders.]

  Description [Routines for reading and writing variable orders.]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [reads from a file the variable order.]

  Description        [This function reads from a file the variable
  order to be used in the system. The grammar of the order file is as
  follows:
  <pre>
    orderfile    ::=  variable NL
    variable     ::=  complex_atom
    complex_atom ::=  simple_atom | simple_atom DOT complex_atom
    simple_atom  ::=  atom
  </pre>
  Where <tt>NL</tt> is a new line, and <tt>DOT</tt> is the period character.
  ]

  SideEffects        []

  SeeAlso            [write_order]

******************************************************************************/
node_ptr Compile_ReadOrder(const char *input_order_file)
{
  if (input_order_file != NULL) {
    extern int yylex();
    int token;
    node_ptr read_variables = Nil;

    Parser_OpenInput(input_order_file);
    token = yylex();
    while (token) {
      node_ptr var = Nil;

      while (1) {
	if (token != ATOM) rpterr("syntax error");
	var = find_node(DOT, var, (node_ptr)find_atom(yylval.node));
	token = yylex();
	while( token == LB ) {
	  token = yylex();
	  if ( token != NUMBER ) rpterr("syntax error");
	  var = find_node(ARRAY, var,
                                 find_node(NUMBER,
                                           (node_ptr)eval_num(yylval.node, Nil), Nil));
	  token = yylex();
	  if (token != RB ) rpterr("syntax error");
	  token = yylex();
	}
	if (token != DOT) break;
	token = yylex();
      }
      read_variables = cons(var, read_variables);
    }
    read_variables = reverse(read_variables);
    Parser_CloseInput();
    return(read_variables);
  }
  return(Nil);
}

/**Function********************************************************************

  Synopsis           [Writes on a file the variable order.]

  Description        [This function writes the variable order
  currently in use in the system in the specified output file. If the
  option "-reorder" has been used then, then this function exits (if
  the batch interaction mode is used). The file generated as output
  can be used as input order file for next computations.]

  SideEffects        []

  SeeAlso            [Compile_ReadOrder]

******************************************************************************/
void Compile_WriteOrder(const char *output_order_file_name, int force)
{
  if (!opt_reorder(options) && (is_default_order_file(options)) && (force == 0)) return;
  {
    FILE *oof;
    char * output_order_fname = get_output_order_file(options);

    if (output_order_file_name != NULL) {
      if ((oof = fopen(output_order_file_name, "w")) == NULL) {
        rpterr("output_order: unable to open file %s", output_order_file_name);
      }
    }
    else {
      if ((oof = fopen(output_order_fname, "w")) == NULL) {
        rpterr("output_order: unable to open file %s", output_order_fname);
      }
    } 
    {
      node_ptr ordering = Compile_GetOrdering();

      while(ordering != Nil) {
        node_ptr name = car(ordering);
        
        print_node(oof, name);
        fprintf(oof, "\n"); 
        ordering = cdr(ordering);
      }
    }
    if (fclose(oof) == EOF) {
      if (output_order_file_name != NULL)
        rpterr("cannot close %s", output_order_file_name);
      else
        rpterr("cannot close %s", get_output_order_file(options));
    }
    if (opt_verbose_level_gt(options, 0)) {
      if (output_order_fname != NULL)
        fprintf(nusmv_stderr, "NuSMV: variable order output to file %s\n", output_order_file_name);
    }
    if (opt_batch(options)) 
      if (!opt_reorder(options)) nusmv_exit(0);
  }
}
