/**CFile***********************************************************************

  FileName    [compileMono.c]

  PackageName [compile]

  Synopsis    [Routines to performs the computation of the monolithic
  transition relation.]

  Description [Routines to performs the computation of the monolithic
  transition relation.]

  SeeAlso     [compileIwls95.c, compileConj.c, compileDisj.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The monolithic transition relation]

  Description [The monolithic transition relation in ADD and BDD]

  SeeAlso     []

******************************************************************************/
add_ptr trans_add = (add_ptr)NULL;
bdd_ptr trans_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [The monolithic transition relation]

  Description [The monolithic transition relation in ADD and BDD]

  SeeAlso     [Yuan Lu]

******************************************************************************/
add_ptr trans_add_orig = (add_ptr)NULL;
bdd_ptr trans_bdd_orig = (bdd_ptr)NULL;

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Builds the monolithic transition relation.]

  Description        [Starting from the <tt>trans_expr</tt> and
  <tt>procs</tt> the monolithic transition relation is computed. If
  <tt>assumption</tt> is not NULL, then the resulting transition
  relation is restricted to the set <tt>assumption</tt> using the
  Coudert and Madre algorithm.]

  SideEffects        []

  SeeAlso            [Compile_CompileModel]

******************************************************************************/
void build_model_monolithic(node_ptr trans_expr, node_ptr invar_expr, node_ptr procs,
                            add_ptr assumption)
{
  add_ptr tmp_1, tmp_2, running_add;
  node_ptr l_of_p=procs;

  /* Initializes BDD of transition relation and invariant set. */
  trans_add_orig = add_one(dd_manager);
  invar_add_orig = add_one(dd_manager);

  while(l_of_p) { /* For all process in 'procs' */
    node_ptr cur_process = car(l_of_p);
    node_ptr context = car(cur_process);
    node_ptr assign_expr = cdr(cur_process);

    l_of_p = cdr(l_of_p);     
    /* evaluate 'running_atom' in 'context' */
    running_add = eval(running_atom, context);

    /* now evaluate the TRANS, ASSIGN statements */
    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating TRANS statements", context);
    tmp_1 = eval_simplify(trans_expr, Nil, assumption);
    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating next() assignments", context);
    set_assignment_type_trans();
    tmp_2 = eval_simplify(assign_expr, Nil, assumption);
    add_and_accumulate(dd_manager, &tmp_1, tmp_2);
    add_free(dd_manager, tmp_2);

    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "size of transition relation = %d  BDD nodes\n",
              add_size(dd_manager, tmp_1));
    tmp_2 = add_imply(dd_manager, running_add, tmp_1);
    add_and_accumulate(dd_manager, &trans_add_orig, tmp_2);
    add_free(dd_manager, tmp_1);
    add_free(dd_manager, tmp_2);

    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating normal assignments", context);
    set_assignment_type_assign();
    tmp_1 = eval_simplify(assign_expr, Nil, assumption);
    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating INVAR statements", context);
    tmp_2 = eval_simplify(invar_expr, Nil, assumption);
    add_and_accumulate(dd_manager, &tmp_1, tmp_2);
    add_free(dd_manager, tmp_2);

    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "size of invariant set = %g states, %d BDD nodes\n",
              add_count_states(dd_manager, tmp_1), add_size(dd_manager, tmp_1));

    tmp_2  = add_imply(dd_manager, running_add, tmp_1);
    add_and_accumulate(dd_manager, &invar_add_orig, tmp_2);
    add_free(dd_manager, tmp_1);
    add_free(dd_manager, tmp_2);
    add_free(dd_manager, running_add);
  }
  if (cdr(procs)) add_and_accumulate(dd_manager, &trans_add_orig, frame_add);
  if (opt_verbose_level_gt(options, 0) && cdr(procs)) {
    fprintf(nusmv_stderr, "size of global initial set = %d  BDD nodes\n",
            add_size(dd_manager, init_bdd));
    fprintf(nusmv_stderr, "size of global transition relation = %d  ADD nodes\n",
            add_size(dd_manager, trans_add_orig));
    fprintf(nusmv_stderr, "size of global invariant set = %d  ADD nodes\n",
            add_size(dd_manager, invar_add_orig));
  }
  /* Translation from ADD to BDD */
  trans_bdd_orig = add_to_bdd(dd_manager, trans_add_orig);
  invar_bdd_orig = add_to_bdd(dd_manager, invar_add_orig);

  /* Yuan Lu : applying abstraction */
  {
    trans_bdd = Abs_AbsAbstractCurNex(trans_bdd_orig, abs_expr);
    invar_bdd = Abs_AbsAbstractCur(invar_bdd_orig, abs_expr);
  }

  add_free(dd_manager, trans_add_orig);
  add_free(dd_manager, invar_add_orig);
  input_variables_bdd = add_to_bdd(dd_manager, input_variables_add);
  next_input_variables_bdd = add_to_bdd(dd_manager, next_input_variables_add);

  state_variables_bdd = add_to_bdd(dd_manager, state_variables_add);
  next_state_variables_bdd = add_to_bdd(dd_manager, next_state_variables_add);

#ifdef OPT_QUANTIFY_INPUT
  if (opt_quantify_input(options)) {
    bdd_ptr tmp;
    
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr,
              "Quantifying out input variables from the Transition Relation.\n");
      fprintf(nusmv_stderr, "Size of Transition Relation = %d BDD nodes\n",
                          bdd_size(dd_manager, trans_bdd));
    }
    tmp = bdd_forsome(dd_manager, trans_bdd, input_variables_bdd);
    bdd_free(dd_manager, trans_bdd);
    trans_bdd = tmp;
    if (opt_verbose_level_gt(options, 0))
      fprintf(nusmv_stderr, "Size of new Transition Relation = %d BDD nodes\n",
                          bdd_size(dd_manager, trans_bdd));
  }
#endif
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_monolithic_info()
{
  if (trans_bdd != NULL)
    (void)fprintf(nusmv_stdout, "No Partitioning:\n");
  if (invar_bdd != NULL)
    (void)fprintf(nusmv_stdout, "BDD nodes representing transition relation: %d + %d\n",
                  bdd_size(dd_manager, trans_bdd), bdd_size(dd_manager, invar_bdd));
}
