/**CFile***********************************************************************

  FileName    [compileIwls95.c]

  PackageName [compile]

  Synopsis    [Build a conjunctively decomposed transition relation
  using the clustering heuristic by Ranjan et al, Iwls95.]

  Description [Conjuncitively partitioned models are very sensitive to
  the way clusters are built and ordered. The algorithm implemented
  here takes into account the number of next state variables
  introduced, when choosing the next cluster in the order. The
  ordering of the clusters takes into account the \"index\" of the
  variables (because it was found that quantifying out a variable from
  a function becomes computationally less expensive as the depth (the
  index) of the variable in the BDD ordering increases.<br>

  <ol>
  <li> R. K. Ranjan and A. Aziz and B. Plessier and C. Pixley and R. K. Brayton,
      "Efficient BDD Algorithms for FSM Synthesis and Verification,
      IEEE/ACM Proceedings International Workshop on Logic Synthesis,
      Lake Tahoe (NV), May 1995.</li>
  </ol>]

  SeeAlso     [compileMono.c, compileDisj.c, compileConj.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";
#define DEBUG_IWLS95 0

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The conjunctively partitioned transition relation.]

  Description []

  SeeAlso     []

******************************************************************************/
node_ptr Iwls95_Fwd_trans = Nil;
node_ptr Iwls95_Bwd_trans = Nil;

/**Variable********************************************************************

  Synopsis    [The IWLS95 partitioning method options.]

  Description [The IWLS95 partitioning method options.]

******************************************************************************/
Iwls95OptionStruct_t * Iwls95Options = NIL(Iwls95OptionStruct_t);

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static node_ptr compileCompileModelIwls95Recur ARGS((node_ptr, node_ptr, node_ptr, add_ptr));
static int CheckQuantificationSchedule ARGS ((DdManager * dd, node_ptr Cl_list));
static float Iwls95ComputeBenefit ARGS((Iwls95Cluster_Info_t * Ci, Iwls95OptionStruct_t * Options));
static void Iwls95ClusterInfoFree ARGS((DdManager *dd, Iwls95Cluster_Info_t * CI));
static int Iwls95CLuster_equal ARGS((Iwls95Cluster_Info_t * A, Iwls95Cluster_Info_t * B));
static int IwlsCheckMonolithic ARGS((DdManager *dd, node_ptr Clist, bdd_ptr Mon_T));
static node_ptr Iwls95DeleteCluster ARGS((node_ptr source, Iwls95Cluster_Info_t * el));
static void Iwls95PrintCluster ARGS((DdManager *dd, Iwls95Cluster_Info_t * Ci));
static void Iwls95ComputeClusterInfoAux ARGS((DdManager * dd, node_ptr Q, double * x_c, double * z_c, double * M_c, bdd_ptr PSPI, bdd_ptr NS));
static bdd_ptr Iwls95Compute_Supp_Q_Ci ARGS((DdManager *dd, node_ptr Q, Iwls95Cluster_Info_t * Ci));


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [The implicitly conjoined transition relation is
  computed accordingly to the heuristic described in IWLS95.]

  Description        [Computes an implicitly conjoined transition
  relation, in which clusters are ordered accordingly the heuristic
  defined in IWLS95.<br>

  First computes the BDD representing the transition relation of the
  next function of each scalar variable (<code>next(x) := expr
  </code>). This initializes the set of clusters.<br> 

  Then, the list of clusters is ordered, and if necesessary recombine
  clusters to obtain a small number of clusters whose size in BDD
  nodes is smaller than a given threshold. Then clusters are ordered
  again.]

  SideEffects        []

  SeeAlso            [Compile_CompileModel Compile_CompileModelConj Compile_CompileModelDisj]

******************************************************************************/
void Compile_CompileModelIwls95(node_ptr trans_expr, node_ptr invar_expr, node_ptr procs,
                        add_ptr assumption)
{
  add_ptr tmp_1, tmp_2;
  add_ptr Test  = eval(trans_expr, Nil);
  add_ptr one   = add_one(dd_manager);
  node_ptr List = Nil;
  node_ptr context = car(car(procs));
  node_ptr assign_expr = cdr(car(procs));
  node_ptr Clist = Nil;
  
  /* one and Test only needed for equality test */
  add_free(dd_manager, one); 
  add_free(dd_manager, Test);
  if (cdr(procs) != Nil)
    rpterr("Processes not allowed with IWLS95 partitioning.");
  if (Test != one) {
    rpterr("TRANS statements not allowed with IWLS95 partitioning.");
  }

  /* initializes invar_add */
  invar_add = add_one(dd_manager);

  /* evaluates the running atom in current context */
  running_add = eval(running_atom, context);
  
  /*
    List is a simple list of ADD, each element representing the transition
    relation next(v) = f(V). If "v" is a symbolic variable assuming k values,
    it is encoded with log_2(k) boolean variables, which are kept together.
    According to the definition given in IWLS'95, next(v) = f(V) is
    a cluster.
  */
  if (opt_verbose_level_gt(options, 1))
    print_in_process("evaluating next() assignments", context);
  set_assignment_type_trans();
  List = compileCompileModelIwls95Recur(List, assign_expr, Nil, assumption);
  /* Yuan Lu : store max_part_list */
  max_part_list = List;

  if (opt_verbose_level_gt(options, 0))
    print_in_process("evaluating normal assignments", context);
  set_assignment_type_assign();
  tmp_1 = eval_simplify(assign_expr, Nil, assumption);
  if (opt_verbose_level_gt(options, 1))
    print_in_process("evaluating INVAR statements", context);
  tmp_2 = eval_simplify(invar_expr, Nil, assumption);
  add_and_accumulate(dd_manager, &tmp_1, tmp_2);
  add_free(dd_manager, tmp_2);

  if (opt_verbose_level_gt(options, 0))
    fprintf(nusmv_stderr, "size of invariant set = %g states, %d BDD nodes\n",
            add_count_states(dd_manager, tmp_1), add_size(dd_manager, tmp_1));

  tmp_2 = add_imply(dd_manager, running_add, tmp_1);
  add_and_accumulate(dd_manager, &invar_add, tmp_2);
  
  add_free(dd_manager, tmp_1);
  add_free(dd_manager, tmp_2);
  add_free(dd_manager, running_add);

  Iwls95Options = Iwls95GetOptions();
  
  { /* Creation of the cluster list to be handled by the IWLS95 routines */
    node_ptr l = List;

    while(l != Nil) {
      add_ptr Ti = (add_ptr)(car(l));
      Iwls95Cluster_Info_t * Ci = Iwls95ClusterInfoAlloc();

      Ci->Ti = add_to_bdd(dd_manager, Ti);
      Clist = cons((node_ptr)Ci, Clist);
      l = cdr(l);
    }
  }
  /* Free List */
  walk_dd(dd_manager, (VPFDD)add_free, List);
  /* conversion from Add to BDD */
  input_variables_bdd      = add_to_bdd(dd_manager, input_variables_add);
  next_input_variables_bdd = add_to_bdd(dd_manager, next_input_variables_add);
  state_variables_bdd      = add_to_bdd(dd_manager, state_variables_add);
  next_state_variables_bdd = add_to_bdd(dd_manager, next_state_variables_add);
  invar_bdd                = add_to_bdd(dd_manager, invar_add);
  add_free(dd_manager, invar_add);
  {
    bdd_ptr PI = bdd_one(dd_manager);
    bdd_ptr PS = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
    bdd_ptr NS = bdd_and(dd_manager, next_input_variables_bdd, next_state_variables_bdd);

    /* The main algorithm for generating and ordering clusters */
    Iwls95MakePartition(dd_manager, Clist, &Iwls95_Fwd_trans, &Iwls95_Bwd_trans,
                        Iwls95Options, PS, PI, NS);
    bdd_free(dd_manager, PS);
    bdd_free(dd_manager, PI);
    bdd_free(dd_manager, NS);
    Iwls95FreeClustersList(dd_manager, Clist);
  }

  if (opt_verbose_level_gt(options, 3))
    print_iwls95cp_detailed_info();
}

/**Function********************************************************************

  Synopsis    [Gets the necessary options for computing the image and returns
               in the option structure.]

  Description [Gets the necessary options for computing the image and returns
               in the option structure.]

  SideEffects []

******************************************************************************/
Iwls95OptionStruct_t * Iwls95GetOptions(void)
{
  char *flagValue;
  Iwls95OptionStruct_t *option;
  
  option = ALLOC(Iwls95OptionStruct_t, 1);

  option->clusterSize = get_image_cluster_size(options);

  flagValue = Cmd_FlagReadByName("image_verbosity");
  if (flagValue == NIL(char)){
    option->verbosity = 0; /* the default value */
  }
  else {
    option->verbosity = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W1");
  if (flagValue == NIL(char)){
    option->W1 = 6; /* the default value */
  }
  else {
    option->W1 = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W2");
  if (flagValue == NIL(char)){
    option->W2 = 1; /* the default value */
  }
  else {
    option->W2 = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W3");
  if (flagValue == NIL(char)){
    option->W3 = 1; /* the default value */
  }
  else {
    option->W3 = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W4");
  if (flagValue == NIL(char)){
    option->W4 = 2; /* the default value */
  }
  else {
    option->W4 = atoi(flagValue);
  }
  return option;
}

/**Function********************************************************************

  Synopsis           [Given a list of cluster, this function
  initializes the data structures to perform image computation.]

  Description        [This function performs the initialization of the
  data structures to perform image computation. <br>
  This process consists of the following steps:<br>
  <ol>
  <li> Ordering of the clusters given as input accordingly with the
       heuristic described in IWLS95.</li>
  <li> Clustering of the result of previous step accordingly the
       threshold value stored in the option \"image_cluster_size\".</li>
  <li> Ordering of the result of previous step accordingly with the
       heuristic described in IWLS95.</li>
  </ol>]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Iwls95MakePartition(DdManager * dd, node_ptr Ti_list, node_ptr *Fwd_Ti_list,
                         node_ptr *Bwd_Ti_list, Iwls95OptionStruct_t * Options,
                         bdd_ptr PS, bdd_ptr PI, bdd_ptr NS)
{
  node_ptr FwdClustered = Nil;
  node_ptr BwdClustered = Nil;
  node_ptr NewFwdClustered = Nil;
  node_ptr NewBwdClustered = Nil;
  
  *Fwd_Ti_list = Nil;
  *Bwd_Ti_list = Nil;

  /* Here it may be interesting to use heuristic when the Cluster Threshold is 1 */
  Iwls95OrderClusters(dd, Ti_list, &FwdClustered, PS, PI, NS, Options);
  Iwls95OrderClusters(dd, Ti_list, &BwdClustered, NS, PI, PS, Options);

  Iwls95MakeClusters(dd, FwdClustered, &NewFwdClustered, Options);
  Iwls95MakeClusters(dd, BwdClustered, &NewBwdClustered, Options);

  /* We free intermadaite results */
  Iwls95FreeClustersList(dd, FwdClustered);
  Iwls95FreeClustersList(dd, BwdClustered);

  Iwls95OrderClusters(dd, NewFwdClustered, Fwd_Ti_list, PS, PI, NS, Options);
  Iwls95OrderClusters(dd, NewBwdClustered, Bwd_Ti_list, NS, PI, PS, Options);

  Iwls95FreeClustersList(dd, NewFwdClustered);
  Iwls95FreeClustersList(dd, NewBwdClustered);

#if DEBUG_IWLS95
  if (CheckQuantificationSchedule(dd, *Fwd_Ti_list) == 0) {
    (void) fprintf(nusmv_stderr, "\n\n###########################################.\n");
    (void) fprintf(nusmv_stderr, "The order of clusters appear to be wrong.\n");
    (void) fprintf(nusmv_stderr, "############################################\n\n");
  }
  if (CheckQuantificationSchedule(dd, *Bwd_Ti_list) == 0) {
    (void) fprintf(nusmv_stderr, "\n\n###########################################.\n");
    (void) fprintf(nusmv_stderr, "The order of clusters appear to be wrong.\n");
    (void) fprintf(nusmv_stderr, "############################################\n\n");
  }
#endif
}


/**Function********************************************************************

  Synopsis           [This function orders the clusters given as input
  accordingly with the order heuristic described in IWLS95.]

  Description        [This function orders the clusters given as input
  accordingly with the order heuristic described in IWLS95. The
  ordering is performed according to a cost function attached to each
  cluster.]

  SideEffects        [The result is stored in \"Ordered\"]
  SeeAlso            [Iwls95ClusterInfo_struct,  Iwls95OptionStruct]

******************************************************************************/
void Iwls95OrderClusters(DdManager * dd, node_ptr Unordered, node_ptr *Ordered,
                         bdd_ptr PS, bdd_ptr PI, bdd_ptr NS,
                         Iwls95OptionStruct_t *Options)
{
  node_ptr clist = Unordered;
  node_ptr Olist = Nil;

  while (clist != Nil) {
    double best_benefit = -9999;
    Iwls95Cluster_Info_t * Best_Cluster = NIL(Iwls95Cluster_Info_t);
    node_ptr C_info_list  = Iwls95ComputeClusterInfo(dd, clist, PS, PI, NS);

    while (C_info_list != Nil) {
      Iwls95Cluster_Info_t * C_info = (Iwls95Cluster_Info_t *)car(C_info_list);
      int benefit = Iwls95ComputeBenefit(C_info, Options);

      if (benefit > best_benefit) {
        best_benefit = benefit;
        Iwls95ClusterInfoFree(dd, Best_Cluster);
        Best_Cluster = C_info;
      } else {
        Iwls95ClusterInfoFree(dd, C_info);
      }
      C_info_list = cdr(C_info_list);
    }
    clist = Iwls95DeleteCluster(clist, Best_Cluster);
    Olist = cons((node_ptr)Best_Cluster, Olist);
  }
  *Ordered = reverse(Olist);
}

/**Function********************************************************************

  Synopsis           [Computes the parameters necessary to use iwls95]

  Description        [Fill all the fields of the list of clusters taken as input.]

  SideEffects        [Side effects on <code>Q</code>]

  SeeAlso            []

******************************************************************************/
node_ptr Iwls95ComputeClusterInfo(
 DdManager *dd /* The DD manager */,
 node_ptr Q /* The list of clusters whose fields have to be filled */,
 bdd_ptr PS /* Cube of present state variables */,
 bdd_ptr PI /* Cube of input variables */,
 bdd_ptr NS /* Cube of next state variables */)
{
  node_ptr C_list = Q;
  node_ptr Iwls95Cluster_Info_list = Nil;
  bdd_ptr  PSPI = bdd_and(dd, PS, PI);
  double x_c, z_c, M_c;

  Iwls95ComputeClusterInfoAux(dd, Q, &x_c, &z_c, &M_c, PSPI, NS);
  while(C_list != Nil) {
    Iwls95Cluster_Info_t * Cur_Cluster = (Iwls95Cluster_Info_t *)car(C_list);
    bdd_ptr Ti = Cur_Cluster->Ti;
    Iwls95Cluster_Info_t * CI_Ti = Iwls95ClusterInfoAlloc();

    CI_Ti->Ti          = bdd_dup(Ti);
    CI_Ti->Supp_Ti     = bdd_support(dd, Ti);
    CI_Ti->PSPI_Ti     = Iwls95CubeAnd(dd, CI_Ti->Supp_Ti, PSPI);
    CI_Ti->NS_Ti       = Iwls95CubeAnd(dd, CI_Ti->Supp_Ti, NS);
    CI_Ti->Supp_Q_Ci   = Iwls95Compute_Supp_Q_Ci(dd, Q, CI_Ti);
    CI_Ti->Ei          = bdd_cube_diff(dd, CI_Ti->PSPI_Ti, CI_Ti->Supp_Q_Ci);
    {
      double v_c = bdd_size(dd, CI_Ti->Ei) - 1;
      double w_c = bdd_size(dd, CI_Ti->PSPI_Ti) - 1;
      double y_c = bdd_size(dd, CI_Ti->NS_Ti) - 1;
      double m_c = bdd_get_lowest_index(dd, CI_Ti->PSPI_Ti);

      CI_Ti->v_c = (v_c < 0) ? 0 : v_c;
      CI_Ti->w_c = (w_c < 0) ? 0 : w_c;
      CI_Ti->x_c = (x_c < 0) ? 0 : x_c;
      CI_Ti->y_c = (y_c < 0) ? 0 : y_c;
      CI_Ti->z_c = (z_c < 0) ? 0 : z_c;
      CI_Ti->m_c = m_c;
      CI_Ti->M_c = M_c;
    }
    Iwls95Cluster_Info_list = cons((node_ptr)CI_Ti, Iwls95Cluster_Info_list);
    C_list = cdr(C_list);
  }
  bdd_free(dd, PSPI);
  Iwls95Cluster_Info_list = reverse(Iwls95Cluster_Info_list);
  return(Iwls95Cluster_Info_list);
}

/**Function********************************************************************

  Synopsis           [Computes the set Supp_Q_Ci.]

  Description        [Computes the set of present an primary input variables
  that belong to the set of support of cluster Ci, and do not belong to the
  set of support of each cluster Cj, for j != i and Cj belonging to the set
  of the not yet ordered clusters. The set Supp_Q_Ci is formally defined as:
  Supp_Q_Ci = {v \in (PS U PI) /\ v \not\in S(T_Cj), Cj != Ci, Cj \in Q}]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr Iwls95Compute_Supp_Q_Ci(DdManager *dd, node_ptr Q,
                                Iwls95Cluster_Info_t * Ci)
{
  bdd_ptr Acc = bdd_one(dd);

  while (Q != Nil) {
    Iwls95Cluster_Info_t * Cj = (Iwls95Cluster_Info_t *)car(Q);

    if (Iwls95CLuster_equal(Cj, Ci) == 0) {
      bdd_ptr Supp = bdd_support(dd, Cj->Ti);

      bdd_and_accumulate(dd, &Acc, Supp);
      bdd_free(dd, Supp);
    }
    Q = cdr(Q);
  }
  return(Acc);
}

/**Function********************************************************************

  Synopsis           [Computes some global parameters necessary in the
  ordering of clusters.]

  Description        [Computes some global parameters necessary in the
  ordering of clusters.]

  SideEffects        [The results are returned in x_c, z_c, M_c]

  SeeAlso            [Iwls95ComputeClusterInfo, Iwls95OptionStruct, Iwls95ClusterInfo_struct]

******************************************************************************/
static void Iwls95ComputeClusterInfoAux(DdManager * dd, node_ptr Q, double * x_c,
                                        double * z_c, double * M_c,
                                        bdd_ptr PSPI, bdd_ptr NS)
{
  bdd_ptr Acc = Iwls95ComputeCLustersCube(dd, Q);
  bdd_ptr Acc_PSPI = Iwls95CubeAnd(dd, Acc, PSPI);
  bdd_ptr Acc_NS = Iwls95CubeAnd(dd, Acc, NS);

  *x_c = bdd_size(dd, Acc_PSPI);
  *z_c = bdd_size(dd, Acc_NS);
  *M_c = bdd_get_lowest_index(dd, Acc_PSPI);
  bdd_free(dd, Acc);
  bdd_free(dd, Acc_PSPI);
  bdd_free(dd, Acc_NS);
}

/**Function********************************************************************

  Synopsis           [Computes the cube of the set of support of all
  the clusters.]

  Description        [Given a list of clusters, it computes their set
  of support.] 

  SideEffects        []

******************************************************************************/
bdd_ptr Iwls95ComputeCLustersCube(DdManager * dd, node_ptr set_of_clusters)
{
  node_ptr soc = set_of_clusters;
  
  bdd_ptr result = bdd_one(dd);
  while (soc != Nil) {
    Iwls95Cluster_Info_t * Cur_Cluster = (Iwls95Cluster_Info_t *)car(soc);
    bdd_ptr Supp_Ti = bdd_support(dd, Cur_Cluster->Ti);

    bdd_and_accumulate(dd, &result, Supp_Ti);
    bdd_free(dd, Supp_Ti);
    soc = cdr(soc);
  }
  return(result);
}

/**Function********************************************************************

  Synopsis           [Return the intersection of two cubes.]

  Description        [Given the cubes A and B, compute the cube of the
                      variables which are both in A and in B.]

  SideEffects        []

******************************************************************************/
bdd_ptr Iwls95CubeAnd(DdManager *dd, bdd_ptr A, bdd_ptr B)
{
  bdd_ptr tmp = bdd_cube_diff(dd, A, B);
  bdd_ptr result = bdd_cube_diff(dd, A, tmp);

  bdd_free(dd, tmp);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Forms the clusters of relations based on BDD
  size heuristic.]

  Description        [The clusters are formed by taking the product in
  order. Once the BDD size of the current cluster reaches a threshold, a new
  cluster is created.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Iwls95MakeClusters(DdManager * dd, node_ptr RelationList,
                        node_ptr * NewRelationList,
                        Iwls95OptionStruct_t * Options)
{
  int flag;
  node_ptr List = RelationList;
  bdd_ptr cluster, tmp_cluster, relation;

  *NewRelationList = Nil;
  while (List != Nil) {
    flag = 0;
    cluster = bdd_one(dd);
    do {
      Iwls95Cluster_Info_t * Ci = (Iwls95Cluster_Info_t *)car(List);

      relation = Ci->Ti;
      tmp_cluster = bdd_and(dd, cluster, relation);
      if ((bdd_size(dd, tmp_cluster) <= Options->clusterSize) || (flag == 0)){
        bdd_free(dd, cluster);
        cluster = tmp_cluster;
        flag = 1;
        List = cdr(List);
      }
      else {
        bdd_free(dd, tmp_cluster);
        break;
      }
    } while (List != Nil); /* do */
    {
      Iwls95Cluster_Info_t * New_Ci = Iwls95ClusterInfoAlloc();

      New_Ci->Ti = cluster;
      *NewRelationList = cons((node_ptr)New_Ci, *NewRelationList);
    }
  } /* while (List != Nil) */
}


/**Function********************************************************************

  Synopsis    [Prints the option values used in IWLS95 technique for
  image computation.]

  Description [Prints the option values used in IWLS95 technique for
  image computation.]

  SideEffects []

******************************************************************************/
int Iwls95PrintOption(FILE * fp)
{
  Iwls95OptionStruct_t * option = Iwls95GetOptions();

  if (option == NIL(Iwls95OptionStruct_t)) {
    (void) fprintf(fp, "Iwls95 Option appear to be NULL\n");
    return(1);
  }
  (void) fprintf(fp, "Printing Information about Image method: IWLS95\n"); 
  (void) fprintf(fp, "   Threshold Value of Bdd Size For Creating Clusters = %d\n", option->clusterSize);
  (void) fprintf(fp, "   Verbosity = %d\n", option->verbosity);
  (void) fprintf(fp, "   W1 =%3d\n", option->W1);
  (void) fprintf(fp, "   W2 =%2d\n", option->W2);
  (void) fprintf(fp, "   W3 =%2d\n", option->W3);
  (void) fprintf(fp, "   W4 =%2d\n", option->W4);
  (void) fprintf(fp, "Use \"set image_cluster_size value\" to set this to desired value.\n"); 
  (void) fprintf(fp, "Use \"set image_verbosity value\" to set this to desired value.\n");

  (void) fprintf(fp, "Use \"set image_W? value\" to set these to the desired values.\n");
  return(0);
}

/**Function********************************************************************

  Synopsis           [Prints out the info for each cluster.]

  Description        [Prints out the information of each cluster, i.e.
  the size of the cluster in BDD nodes, the set of support of the
  cluster (distinguishing between current and next state variables),
  the variable existentially quantified when the current cluster is
  multiplied in the product. If debugging is activated then also the
  parameters necessary to compute the cost function are printed
  out.] 

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Iwls95PrintClustersInfo(FILE * fp, DdManager * dd,
                             node_ptr ClusterList, node_ptr var_list)
{
  node_ptr cl = ClusterList;
  int c_id = 0;
  
  while (cl != Nil) {
    Iwls95Cluster_Info_t * Ci = (Iwls95Cluster_Info_t *)car(cl);

    c_id++;
    (void) fprintf(fp, "############################################################\n");
    (void) fprintf(fp, "Cluster #%d:\n", c_id);
    (void) fprintf(fp, "Size of C_%d = %d\n", c_id, bdd_size(dd, Ci->Ti));
    (void) fprintf(fp, "The variables in C_%d are:\n", c_id);
    (void) print_state_vars(dd, Ci->Supp_Ti, var_list);
    (void) fprintf(fp, "\nSize of E_%d = %d Boolean Variables\n", c_id, bdd_size(dd, Ci->Ei) - 1);
    (void) fprintf(fp, "The variables in E_%d are:\n", c_id);
    (void) print_state_vars(dd, Ci->Ei, var_list);
    (void) fprintf(fp, "\n############################################################\n");
    cl = cdr(cl);
#if DEBUG_IWLS95
    (void) fprintf(fp, "\n############################################################\n");
    (void) Iwls95PrintCluster(dd, Ci);
    (void) fprintf(fp, "\n############################################################\n");
#endif
  }
}


/**Function********************************************************************

  Synopsis           [Allocates and initializes memory for Iwls95Cluster_Info struct.]

  Description        [Allocates and initializes memory for Iwls95Cluster_Info struct.]

  SideEffects        []

******************************************************************************/
Iwls95Cluster_Info_t * Iwls95ClusterInfoAlloc(void)
{
  Iwls95Cluster_Info_t * CI;

  CI = ALLOC(Iwls95Cluster_Info_t, 1);
  CI->Ti          = (bdd_ptr)NULL;
  CI->Ei          = (bdd_ptr)NULL;
  CI->Supp_Ti     = (bdd_ptr)NULL; 
  CI->Supp_Q_Ci   = (bdd_ptr)NULL;
  CI->PSPI_Ti     = (bdd_ptr)NULL;
  CI->NS_Ti       = (bdd_ptr)NULL;
  CI->v_c         = -1;
  CI->w_c         = -1;
  CI->x_c         = -1;
  CI->y_c         = -1;
  CI->z_c         = -1;
  CI->m_c         = -1;
  CI->M_c         = -1;
  return(CI);
}

/**Function********************************************************************

  Synopsis           [Frees the  memory allocated for Iwls95Cluster_Info struct.]

  Description        [Frees the  memory allocated for Iwls95Cluster_Info struct.]

  SideEffects        []

******************************************************************************/
void Iwls95ClusterInfoFree(DdManager *dd, Iwls95Cluster_Info_t * CI)
{
  if (CI != (Iwls95Cluster_Info_t *)NULL) {
    if (CI->Ti        != (bdd_ptr)NULL) bdd_free(dd, CI->Ti);
    if (CI->Ei        != (bdd_ptr)NULL) bdd_free(dd, CI->Ei);
    if (CI->Supp_Ti   != (bdd_ptr)NULL) bdd_free(dd, CI->Supp_Ti);
    if (CI->Supp_Q_Ci != (bdd_ptr)NULL) bdd_free(dd, CI->Supp_Q_Ci);
    if (CI->PSPI_Ti   != (bdd_ptr)NULL) bdd_free(dd, CI->PSPI_Ti);
    if (CI->NS_Ti     != (bdd_ptr)NULL) bdd_free(dd, CI->NS_Ti);
    FREE(CI);
  }
}

/**Function********************************************************************

  Synopsis           [Frees a list of Iwls95Cluster_Info struct.]

  Description        [Frees a list of Iwls95Cluster_Info struct.]

  SideEffects        []

******************************************************************************/
void Iwls95FreeClustersList(DdManager *dd, node_ptr CL)
{
  node_ptr l = CL;

  while (l != Nil) {
    Iwls95Cluster_Info_t * c = (Iwls95Cluster_Info_t *)car(l);

    Iwls95ClusterInfoFree(dd, c);
    l = cdr(l);
  }
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_iwls95cp_part_info(void)
{
  int i = 0;
  node_ptr l = Iwls95_Fwd_trans;
  
  (void) fprintf(nusmv_stdout, "Iwls95 Conjunctive Partitioning:\n");
  (void) fprintf(nusmv_stdout, "Forward:\n");
  while(l != Nil) {
    bdd_ptr Ti = (bdd_ptr)((Iwls95Cluster_Info_t *)car(l))->Ti;

    (void) fprintf(nusmv_stdout, "partition %d: %d BDD nodes.\n", i++,
                   (bdd_ptr)bdd_size(dd_manager, Ti));
    l = cdr(l);
  }

  i = 0;
  l = Iwls95_Bwd_trans;
  (void) fprintf(nusmv_stdout, "Backward:\n");
  while(l != Nil) {
    bdd_ptr Ti = (bdd_ptr)((Iwls95Cluster_Info_t *)car(l))->Ti;
    
    (void) fprintf(nusmv_stdout, "partition %d: %d BDD nodes.\n", i++,
                   (bdd_ptr)bdd_size(dd_manager, Ti));
    l = cdr(l);
  }

  (void)fprintf(nusmv_stdout, "Size of invariant: %d BDD nodes.\n",
                bdd_size(dd_manager, invar_bdd));
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_iwls95cp_detailed_info(void) {
  fprintf(nusmv_stdout, "\n########################################\n");
  fprintf(nusmv_stdout, "# Forward clusters: \n");
  fprintf(nusmv_stdout, "########################################\n");
  Iwls95PrintClustersInfo(nusmv_stdout, dd_manager, Iwls95_Fwd_trans, state_variables);
  fprintf(nusmv_stdout, "\n########################################\n");
  fprintf(nusmv_stdout, "# Backward clusters: \n");
  fprintf(nusmv_stdout, "########################################\n");
  Iwls95PrintClustersInfo(nusmv_stdout, dd_manager, Iwls95_Bwd_trans, state_variables);
  fprintf(nusmv_stdout, "\n");  
}
/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Given a list of expression, then the
  corresponding list of ADD is returned.]

  Description        [Given a list of expression, then the
  corresponding list of ADD is returned.]

  SideEffects        [None]

  SeeAlso            [Compile_CompileModelDisj Compile_CompileModelIwls95]

******************************************************************************/
static node_ptr compileCompileModelIwls95Recur(node_ptr Acc, node_ptr sexp,
                                    node_ptr cxt, add_ptr assumption)
{
  if (sexp == Nil) return(Acc);
  yylineno = sexp->lineno;
  switch(node_get_type(sexp)){
  case AND:
    {
      Acc = compileCompileModelIwls95Recur(Acc, car(sexp), cxt, assumption);
      Acc = compileCompileModelIwls95Recur(Acc, cdr(sexp), cxt, assumption);
      return(Acc);
    }
  case CONTEXT:
    {
      return(compileCompileModelIwls95Recur(Acc, cdr(sexp), car(sexp), assumption));
    }
  default:
    {
      add_ptr one = add_one(dd_manager);
      add_ptr sexp_add = eval(sexp, cxt);

      add_free(dd_manager, one);
      if (sexp_add == one) {
        add_free(dd_manager, sexp_add);
        return(Acc);
      }
      {
        add_ptr cur_ti = add_simplify_assuming(dd_manager, sexp_add, assumption);
        
        add_free(dd_manager, sexp_add);
        return(cons((node_ptr)cur_ti, Acc));
      }
    }
  }
}

/**Function********************************************************************

  Synopsis     [Computes the benefit of the given cluster.]

  Description  [Computes the benefit function associated to each cluster.
       The objective function attached with each Ti is
       Ci =  W1*C1 + W2*C2 - W3*C3 + W4*C4
       where:
       W1 = weight attached with variables getting smoothed
       W2 = weight attached with the support count of the Ti
       W3 = weight attached with variables getting introduced
       W4 = weight attached with the max bdd id of the Ti
       C1 = v_c/w_c
       C2 = w_c/x_c
       C3 = y_c/z_c
       C4 = m_c/M_c
       v_c = number of ps and pi variables which can be existentially
             quantified when Ti is multiplied in the product;
       w_c = number of ps and pi variables in the support of Ti;
       x_c = total number of ps and pi variables remaining which have not yet
             been smoothed out;
       y_c = number of ns variables that would be introduced in the product
             by multiplying Ti;
       z_c = total number of ns variables which have not been introduced
             so far.
       m_c = value of Max bdd id of Ti
       M_c = value of Max bdd id across all the Ti's remaining to be multiplied

       Get the weights from the global option.]

  SideEffects  []

******************************************************************************/
static float Iwls95ComputeBenefit(Iwls95Cluster_Info_t * Ci,
                                  Iwls95OptionStruct_t * Option)
{
  int W1, W2, W3, W4;
  float benefit;
  
  W1 = Option->W1;
  W2 = Option->W2;
  W3 = Option->W3;
  W4 = Option->W4;
 
  benefit = 0;
  benefit += (Ci->w_c ? (W1 * (float)(Ci->v_c / Ci->w_c)) : 0);
  benefit += (Ci->x_c ? (W2 * (float)(Ci->w_c / Ci->x_c)) : 0);
  benefit -= (Ci->z_c ? (W2 * (float)(Ci->y_c / Ci->z_c)) : 0);
  benefit += (Ci->M_c ? (W2 * (float)(Ci->m_c / Ci->M_c)) : 0);

  return(benefit);
}

/**Function********************************************************************

  Synopsis    [Given a list of clusters representing the relations this
               function checks if the schedule is correct.]

  Description [Let Ci and Ti be the ith cube and relation in the list.
               The schedule is correct iff<br>
  <ol>
  <li> For all Tj: j > i, S(Tj) and S(Ci) do not intersect, i.e., the
       variables which are quantified in Ci should not appear in the
       Tj for j>i.<li>
  <li> For any j, S(Ci) and S(Cj) do not intersect.</li>
  </ol><br>

  where S(T) is the set of support of the BDD T.
  Returns 1 if the schedule is correct, 0 otherwise.
  This function is implemented for checking the correctness of the
  clustering algorithm only.]

  SideEffects []

******************************************************************************/
static int CheckQuantificationSchedule(DdManager *dd, node_ptr Cl_list)
{
  node_ptr Clist = Cl_list;
  
  while (Clist != Nil) {
    Iwls95Cluster_Info_t * Cj = (Iwls95Cluster_Info_t *)car(Clist);
    node_ptr Nlist = cdr(Clist);

    while (Nlist!= Nil) {
      Iwls95Cluster_Info_t * Ci = (Iwls95Cluster_Info_t *)car(Nlist);
      bdd_ptr intersection = bdd_cube_diff(dd, Cj->Ei, Ci->Supp_Ti);

      bdd_free(dd, intersection);
      if (intersection !=  Cj->Ei)
        /* Violates the condition "a" */
        return(0);
      intersection = bdd_cube_diff(dd, Cj->Ei,  Ci->Ei);
      bdd_free(dd, intersection);
      if (intersection !=  Cj->Ei)
        /* Violates the condition "b" */
        return(0);
      Nlist = cdr(Nlist);
    } /* while (Nlist!= Nil) */
    Clist = cdr(Clist);
  } /* while (Clist != Nil) */
  return(1);
}

/**Function********************************************************************

  Synopsis           [Checks that the And_i Ti == Mon_T]

  Description        [Checks whether the conjunction of the transition
                      relations of all the clusters equals the the
                      monolithic Transition relation.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int IwlsCheckMonolithic(DdManager *dd, node_ptr Clist, bdd_ptr Mon_T)
{
  bdd_ptr Acc = bdd_one(dd);

  while(Clist != Nil) {
    Iwls95Cluster_Info_t * Ci = (Iwls95Cluster_Info_t *)car(Clist);

    bdd_and_accumulate(dd, &Acc, Ci->Ti);
  }
  bdd_free(dd, Acc);
  return((Acc == Mon_T));
}

/**Function********************************************************************

  Synopsis           [Checks if two clusters are equal.]

  Description        [Checks if two clusters are equal. If it is the
  case then return 1 else return 0. <br>
  Notice that the check is performed only using the \"Ti\" field of the
  Iwls95Cluster_Info structure.] 

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int Iwls95CLuster_equal(Iwls95Cluster_Info_t * A, Iwls95Cluster_Info_t * B)
{
  assert(A != NULL);
  assert(B != NULL);

  return((A->Ti == B->Ti));
}

/**Function********************************************************************

  Synopsis           [Deletes a given cluster from a list of clusters.]

  Description        [Deletes a given cluster from a list of clusters.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr Iwls95DeleteCluster(node_ptr source, Iwls95Cluster_Info_t * el)
{
  node_ptr result = Nil;
  
  /* source = reverse(source); */
  while(source != Nil) {
    if (!Iwls95CLuster_equal((Iwls95Cluster_Info_t *)(car(source)), el))
      result = cons(car(source), result);
    source = cdr(source);
  }
  return(result);
}

/**Function********************************************************************

  Synopsis           [Prints the cluster information.]

  Description        [Prints the cluster information.]

  SideEffects        []

  SeeAlso            [Iwls95Cluster_Info_t]

******************************************************************************/
static void Iwls95PrintCluster(DdManager *dd, Iwls95Cluster_Info_t * Ci)
{
  /* Be careful, this can be only used to print to standard output */
  (void) fprintf(nusmv_stdout, "Ei = \n");
  (void) dd_printminterm(dd, Ci->Ei);
  (void) fprintf(nusmv_stdout, "Supp_Ti = \n");
  (void) dd_printminterm(dd, Ci->Supp_Ti);
  (void) fprintf(nusmv_stdout, "Supp_Q_Ci = \n");
  (void) dd_printminterm(dd, Ci->Supp_Q_Ci);
  (void) fprintf(nusmv_stdout, "PSPI_Ti = \n");
  (void) dd_printminterm(dd, Ci->PSPI_Ti);
  (void) fprintf(nusmv_stdout, "NS_Ti = \n");
  (void) dd_printminterm(dd, Ci->NS_Ti);
  (void) fprintf(nusmv_stdout, "v_c = %g\nw_c = %g\nx_c = %g\ny_c = %g\nz_c = %g\nm_c= %g\nM_c = %g\n\n",Ci->v_c, Ci->w_c, Ci->x_c, Ci->y_c, Ci->z_c, Ci->m_c, Ci->M_c);
}

