/**CFile***********************************************************************

  FileName    [compileFlatten.c]

  PackageName [compile]

  Synopsis    [Flattening of the model.]

  Description [Performs the flattening of the model. We start from the
  module <code>main</code> and we recursively instantiate all the modules
  or processes declared in it.<br>
  Consider the following example:
  <blockquote>
  MODULE main<br>
   ...<br>
   VAR<br>
     a : boolean;<br>
     b : foo;<br>
   ...<br><br>

  MODULE foo<br>
   VAR <br>
     z : boolean;<br>
   ASSIGN<br>
     z := 1;<br>
  </blockquote>
  The flattening instantiate the module foo in the <code>main</code>
  module. You can refer to the variables "<code>z</code>" declared in the
  module <code>foo</code> after the flattening by using the dot notation
  <code>b.z</code>.]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/
typedef enum {
  State_Variables_Instantiation_Mode,
  Input_Variables_Instantiation_Mode
} Instantiation_Variables_Mode_Type;

typedef enum {
  Normal_Instantiation_Mode,
  Implement_Instantiation_Mode,
  Spec_Instantiation_Mode
} Instantiation_Mode_Type;

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [Names of state variables.]

  Description [This list contains the symbolic name of state variables.]

  SeeAlso     []

******************************************************************************/
node_ptr state_variables = Nil;

/**Variable********************************************************************

  Synopsis    [Names of input variables.]

  Description [This list contains the symbolic name of input variables.]

  SeeAlso     []

******************************************************************************/
node_ptr input_variables = Nil;

/**Variable********************************************************************

  Synopsis    [Names of all variables.]

  Description [This list contains the symbolic name of all
  variables. It is the union of <code>input_variables</code> and
  <code>state_variables</code>.]

  SeeAlso     []

******************************************************************************/
node_ptr all_variables = Nil;


/**Variable********************************************************************

  Synopsis    [All the model symbols.]

  Description [This list contains the name of all the variables of the
  model, both input and state variables, plus the symbolic name of all
  the defined symbols (i.e. those declared with <code>DEFINE</code>).]

  SeeAlso     []

******************************************************************************/
node_ptr all_symbols = Nil;

/**Variable********************************************************************

  Synopsis    [The mode to perform instantiation.]

  Description [The value of this variable modifies the behavior of
  the instantiation routines, i.e. <code>Compile_FlattenHierarchy</code>.]

  SeeAlso     []

******************************************************************************/
static Instantiation_Mode_Type instantiate_mode = Normal_Instantiation_Mode;
static void set_instantiatiation_mode_normal(void) {
  instantiate_mode = Normal_Instantiation_Mode;
}
static void set_instantiatiation_mode_implement(void) {
  instantiate_mode = Implement_Instantiation_Mode;
}
static void set_instantiatiation_mode_spec(void) {
  instantiate_mode = Spec_Instantiation_Mode;
}
static int instantiation_mode_is_normal (void) {
  return(instantiate_mode == Normal_Instantiation_Mode);
}
static int instantiation_mode_is_implement() {
  return(instantiate_mode == Implement_Instantiation_Mode);
}
static int instantiation_mode_is_spec() {
  return(instantiate_mode == Spec_Instantiation_Mode);
}

/**Variable********************************************************************

  Synopsis    [The mode to perform variable instantiation.]

  Description [Depending the value of this variable we perform
  instantiation of state variables or input variables.]

******************************************************************************/
static Instantiation_Variables_Mode_Type variable_instantiate_mode = State_Variables_Instantiation_Mode;

void set_variable_instantiation_to_input () {
  variable_instantiate_mode = Input_Variables_Instantiation_Mode;
}
static void set_variable_instantiation_to_state (void) {
  variable_instantiate_mode = State_Variables_Instantiation_Mode;
}
static int variable_instantiation_is_input (void) {
  return(variable_instantiate_mode == Input_Variables_Instantiation_Mode);
}

/**Variable********************************************************************

  Synopsis    [The symbol hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>. 
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     [instantiate_var compileEncodeVar Compile_BuildVarsBdd]

******************************************************************************/
static hash_ptr symbol_hash;
void init_symbol_hash() {symbol_hash = new_assoc();}
void insert_symbol_hash(node_ptr x, node_ptr y) { insert_assoc(symbol_hash, x, y);}
node_ptr lookup_symbol_hash(node_ptr x) {return(find_assoc(symbol_hash, x));}
static assoc_retval free_symbol_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_symbol_hash() {clear_assoc_and_free_entries(symbol_hash, free_symbol_hash);}

/**Variable********************************************************************

  Synopsis    [The has containing the definition of each module read in.]

  Description [This hash uses the name of the module as index, and for
  each module it stores the following data structure:<br>
  <center><code>&lt;LAMBDA , arguments, module_body&gt;</code></center><br>
  I.e. it is a node, whose type is <code>LAMBDA</code> and whose "car" are
  the module arguments and the "cdr" is the module body (i.e. normal
  assignments, init assignments and next assignments.
  ]

******************************************************************************/
static hash_ptr module_hash;
void init_module_hash()
{
  /* Auxiliary variable used to traverse the parse tree. */
  node_ptr m; 
  /* The parse tree representing the input files. */
  extern node_ptr parse_tree;

  module_hash = new_assoc();
  m = parse_tree;
  while(m){
    node_ptr cur_module = car(m);
    node_ptr name = find_atom(car(car(cur_module)));
    node_ptr params = cdr(car(cur_module));
    node_ptr def = cdr(cur_module);

    m = cdr(m);
    if (find_assoc(module_hash, name)) error_redefining(name);
    insert_module_hash(name, new_node(LAMBDA, params, reverse(def)));
  }
}
void insert_module_hash(node_ptr x, node_ptr y) { insert_assoc(module_hash, x, y);}
node_ptr lookup_module_hash(node_ptr x) {return(find_assoc(module_hash, x));}
void clear_module_hash() {if (module_hash != NULL) clear_assoc(module_hash);}

/**Variable********************************************************************

  Synopsis    [The <code>param_hash</code> associates actual to original
  parameter of a module.]

  Description [This hash is used by <code>make_params</code> to detect
  multiple substitution for parameters.]

  SeeAlso     [make_params]

******************************************************************************/
static hash_ptr param_hash;
void init_param_hash() { param_hash = new_assoc(); }
void insert_param_hash(node_ptr x, node_ptr y) { insert_assoc(param_hash, x, y);}
node_ptr lookup_param_hash(node_ptr x) {return(find_assoc(param_hash, x));}
void clear_param_hash() {clear_assoc(param_hash);}

/**Variable********************************************************************

  Synopsis    [Variable containing the current context in the
  instantiation phase.]

  Description [Variable containing the current context in the
  instantiation phase. It is used in the instantiation of the
  arguments of modules or processes.]

******************************************************************************/
static node_ptr param_context;

/**Variable********************************************************************

  Synopsis    [Flag to indicate if the current processed variable is
  an input variable or not.]

  Description [Flag to indicate if the current processed variable is
  an input variable or not. It is used only by the check_implement
  package, which has not yet been ported.]

******************************************************************************/
int is_input_decl = 0;

/**Variable********************************************************************

  Synopsis    [The stack containing the nesting for modules.]

  Description [This variable contains the nesting of modules. It is
  used in the instantiation phase to check for recursively defined modules.]

******************************************************************************/
static node_ptr module_stack = Nil;

/* Unused variable */ /* WARNING TO BE REMOVED ASAP */
node_ptr the_impl = Nil;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void instantiate ARGS((node_ptr, node_ptr, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr));
static void instantiate_by_name ARGS((node_ptr, node_ptr, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr));
static node_ptr put_in_context ARGS((node_ptr));
static void instantiate_var ARGS((node_ptr, node_ptr, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr));

static void instantiate_vars ARGS((node_ptr, node_ptr, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *, node_ptr *));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Traverse the module hierarchy and extracts the
  information needed to compile the automaton.]

  Description        [Traverses the module hierarchy and extracts the
  information needed to compile the automaton. The hierarchy of modules
  is flattened, the variables are contextualized, the various parts of
  the model read in are extracted (i.e. the formulae to be verified,
  the initial expressions, ...)...<br>
  Notice that this function only manage s_expr and not ADD or BDD.]

  SideEffects        [None]

******************************************************************************/
void Compile_FlattenHierarchy(
 node_ptr root_name /* the <code>ATOM</code> representing the module at the top of the hierarchy. */,
 node_ptr name /* the name of the module at the top of the hierarchy. */,
 node_ptr *trans /* the list of TRANS actually recognized */,
 node_ptr *init /* the list of INIT actually recognized */,
 node_ptr *invar /* the list of INVAR actually recognized */,
 node_ptr *spec /* the list of SPEC actually recognized */,
 node_ptr *ltl_spec /* the list of LTLSPEC actually recognized */,
 node_ptr *invar_spec /* the list of INVARSPEC actually recognized */,
 node_ptr *invar_fb /* the list of CHECKINVARIANTFB actually recognized */,
 node_ptr *invar_strong /* the list of CHECKINVARIANTSTRONG actually recognized */,
 node_ptr *after /* the list of AFTER actually recognized */,
 node_ptr *fair /* the list of FAIR actually recognized */,
 node_ptr *abst /* Yuan Lu : abstraction statements */,
 node_ptr *assign /* the list of ASSIGN actually recognized */,
 node_ptr *procs /* the list of processes actually recognized */,
 node_ptr actual /* the actual module arguments */)
{
  node_ptr tmp_assign = Nil;

  instantiate_by_name(root_name, name, trans, init, invar, spec, ltl_spec, invar_spec,
                      invar_fb, invar_strong, after, fair, abst, &tmp_assign, procs, actual);

  *procs = cons(cons(name, tmp_assign), *procs);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Put a variable in the current "context"]

  Description        [Put a variable in the current "context", which
  is stored in <code>param_context</code>.]

  SideEffects        [None]

  SeeAlso            [param_context]

******************************************************************************/
static node_ptr put_in_context(node_ptr v)
{
  return(find_node(CONTEXT, param_context, v));
}

/**Function********************************************************************

  Synopsis           [Instantiate the given variable.]

  Description        [It takes as input a variable and a context, and
  depending on the type of the variable some operation are performed in order to
  instantiate it in the given context:
  <br><br>
  <ul>
    <li><b>BOOLEAN</b><br>
        if the variable is of type boolean, then we add an entry in
        <code>symbol_hash</code> saying that the variable range is <code>{0,1}</code>.</li>
    <li><b>RANGE</b><br>
        if the variable is a range of the form <code>M..N</code>, then
        we add an entry in the <code>symbol_hash</code> saying that the
        variable range is <code>{N, N+1, ..., M-1, M}</code>. If
        <code>M</code> is less or equal to <code>N</code>, than an error occurs.</li>
    <li><b>SCALAR</b><br>
        if the variable is a scalar variable whose possible values are
        <code>{a1, a2, ..., aN}</code>, then we add an entry in the
        <code>symbol_hash</code> saying that the variable range is
        <code>{a1, ..., aN}</code>. </li>
    <li><b>ARRAY</b><br>
        for each element of the array it is created the corresponding
        symbol. Suppose you have the following definition "<code>VAR
        x : array 1..4 of boolean;</code>". We call this function
        for 4 times, asking at each call <code>i</code> (<code>i</code> from 1
        to 4) to declare the boolean variable <code>x\[i\]</code>.</li>
    <li><b>MODULE</b><br>
        If the variable is an instantiation of a module, than their
        arguments (if any) are contextualized, and passed as actual
        parameter to <code>instantiate_by_name<code> with the name of the
        instantiated module as root name (to extract its definition)
        and as variable name as the name of the module (to perform
        flattening).</li>
    <li><b>PROCESS</b><br>
        If the variable is of type process, than we extract the
        module name and args, we perform the contextualization of the
        process arguments and we perform a call to
        <tt>Compile_FlattenHierarchy</tt> using the variable name as process
        name (to perform flattening), the module name as root name (to
        extract its definition) and the computed actual parameters.</li>
  </ul><br>
  The variables of type boolean, scalar, and array depending the kind
  of variable instantiation mode are appended to
  <tt>input_variables</tt> or to <tt>state_variables</tt>.
]

  SideEffects        []

  SeeAlso            [instantiate_vars]

******************************************************************************/
static void instantiate_var(node_ptr name, node_ptr type, node_ptr *trans,
                            node_ptr *init, node_ptr *invar, node_ptr *spec,
                            node_ptr *ltl_spec, node_ptr *invar_spec, node_ptr *invar_fb,
                            node_ptr *invar_strong, node_ptr *after, node_ptr *fair,
                            node_ptr *abst, node_ptr *assign, node_ptr *procs, node_ptr context)
{
  yylineno = type->lineno;
  switch(node_get_type(type)) {
  case BOOLEAN: {
    insert_symbol_hash(name, new_node(VAR, Nil, boolean_type));
    if (variable_instantiation_is_input()) {
      input_variables = cons(name, input_variables);
    } else {
      state_variables = cons(name, state_variables);
    }
    all_symbols = cons(name, all_symbols);
    break;
  }
  case TWODOTS: {
    node_ptr expanded_range = Nil;
    int dim1, dim2, i;

    dim1 = eval_num(car(type), context);
    dim2 = eval_num(cdr(type), context);
    /* Checks if the range is a "range", a range is from "a" to "b"
       with the constraint that "b >= a" */
    if (dim2 >= dim1)
      for(i=dim2 ; i>=dim1 ; i--)
        expanded_range = cons(find_node(NUMBER, (node_ptr)i, Nil), expanded_range);
    else error_empty_range(name, dim1, dim2);
    insert_symbol_hash(name, new_node(VAR, Nil, expanded_range));
    if (variable_instantiation_is_input()) {
      input_variables = cons(name, input_variables);
    } else {
      state_variables = cons(name, state_variables);
    }
    all_symbols = cons(name, all_symbols);
    break;
  }   
  case SCALAR: {
    insert_symbol_hash(name, new_node(VAR, Nil, car(type)));
    if (variable_instantiation_is_input()) {
      input_variables = cons(name, input_variables);
    } else {
      state_variables = cons(name, state_variables);
    }
    all_symbols = cons(name, all_symbols);
    break;
  }
  case MODTYPE: {
      node_ptr actual;

      param_context = context;
      actual = map(put_in_context, cdr(type));
      instantiate_by_name(car(type), name, trans, init, invar, spec, ltl_spec,
                          invar_spec, invar_fb, invar_strong, after, fair, abst,
			  assign, procs, actual);
      break;
  }
  case PROCESS: {
    node_ptr actual;
    node_ptr pmod_name = car(car(type));
    node_ptr pmod_args = cdr(car(type));

    param_context = context;
    actual = map(put_in_context, pmod_args);

    Compile_FlattenHierarchy(pmod_name, name, trans, init, invar, spec, ltl_spec, invar_spec,
                    invar_fb, invar_strong, after, fair, abst, assign, procs, actual);
    break;
  }
  case ARRAY: {
    int dim1, dim2, i;

    dim1 = eval_num(car(car(type)), context);
    dim2 = eval_num(cdr(car(type)), context);
    for(i=dim1; i<=dim2; i++)
      /* Creates the name[i] variable */
      instantiate_var(find_node(ARRAY, name, find_node(NUMBER, (node_ptr)i, Nil)),
                      cdr(type), trans, init, invar, spec, ltl_spec, invar_spec,
                      invar_fb, invar_strong, after, fair, abst, assign, procs, context);
    break;
  }
  default:
    internal_error("instantiate_vars: type = %d", node_get_type(type));
  }
}

/**Function********************************************************************

  Synopsis           [Recursively applies <tt>instantiate_var</tt>.]

  Description        [Recursively applies <tt>instantiate_var</tt> to
  a given list of variables declaration, and performs some check for
  multiple variable definitions.] 

  SideEffects        []

  SeeAlso            [instantiate_var]

******************************************************************************/
static void instantiate_vars(node_ptr var_list, node_ptr mod_name, node_ptr *trans,
                             node_ptr *init, node_ptr *invar, node_ptr *spec,
                             node_ptr *ltl_spec, node_ptr *invar_spec,
                             node_ptr *invar_fb, node_ptr *invar_strong,
                             node_ptr *after, node_ptr *fair, node_ptr *abst,
			     node_ptr *assign, node_ptr *procs)
{
  if (var_list == Nil) return;
  instantiate_vars(cdr(var_list), mod_name, trans, init, invar, spec, ltl_spec,
                   invar_spec, invar_fb, invar_strong, after, fair, abst, assign, procs);
  {
    node_ptr cur_var = car(var_list);
    node_ptr name = eval_struct(car(cur_var), mod_name);
    node_ptr type = cdr(cur_var);

    if (lookup_symbol_hash(name)) error_redefining(name);
    instantiate_var(name, type, trans, init, invar, spec, ltl_spec, invar_spec,
                    invar_fb, invar_strong, after, fair, abst, assign, procs, mod_name);
  }
}


/**Function********************************************************************

  Synopsis           [Builds the parameters of a module from the list of formal parameters of the module itself.]

  Description        [Builds the parameters of a module from the list
  of formal parameters of the module itself and a <tt>basename</tt>.<br>
  There must be a one to one correspondence between the elements of
  <tt>actual</tt> and <tt>formal</tt> parameters. If the number of elements of
  the lists are different then, an error occurs.]

  SideEffects        [In the <tt>param_hash</tt>, the new parameter is
  associated to the old one.]

******************************************************************************/
static void make_params(node_ptr basename, node_ptr actual, node_ptr formal)
{
  while(formal) {
    node_ptr old, new; 

    if (!actual) rpterr("too few actual parameters");
    new = find_node(DOT, basename, find_atom(car(formal)));
    old = car(actual);
    formal = cdr(formal);
    actual = cdr(actual);
    if (lookup_param_hash(new)) error_multiple_substitution(new);
    insert_param_hash(new, old);
  }
  if (actual) rpterr("too many actual parameters");
}

/**Function********************************************************************

  Synopsis           [Instantiate all in the body of a module.]

  Description        [This function is responsible of the
  instantiation of the body of a module. The module definition is
  <tt>mod_def</tt> and the module name <tt>mod_name</tt> are passed as
  arguments. First we instantiate the arguments of the given
  module. Then it loops over the module definition searching for
  defined symbols (i.e. those introduced by the keyword
  <tt>DEFINE</tt>) and inserts their definition in the
  <tt>symbol_hash</tt>. After this preliminary phase it loops again
  over module body in order to performs the other instantiation, and
  to extract all the information needed to compile the automaton,
  i.e. the list of processes, the TRANS statements, the INIT
  statements, ... and so on.]

  SideEffects        []

  SeeAlso            [instantiate_var instantiate_vars]

******************************************************************************/
static void instantiate(node_ptr mod_def, node_ptr mod_name, node_ptr *trans,
                        node_ptr *init, node_ptr *invar, node_ptr *spec,
                        node_ptr *ltl_spec, node_ptr *invar_spec, node_ptr *invar_fb,
                        node_ptr *invar_strong, node_ptr *after, node_ptr *fair,
                        node_ptr *abst, node_ptr *assign, node_ptr *procs,
			node_ptr actual)
{
  node_ptr mod_body_decls;
  node_ptr tmp_trans        = Nil; 
  node_ptr tmp_init         = Nil;
  node_ptr tmp_invar        = Nil;
  node_ptr tmp_spec         = Nil;
  node_ptr tmp_ltlspec      = Nil;
  node_ptr tmp_invar_spec   = Nil;
  node_ptr tmp_invar_fb     = Nil;
  node_ptr tmp_invar_strong = Nil;
  node_ptr tmp_after        = Nil;
  node_ptr tmp_fair         = Nil;
  node_ptr tmp_abst         = Nil;
  node_ptr tmp_assign       = Nil;
  node_ptr tmp_procs        = Nil;
  node_ptr mod_formal_args  = car(mod_def); /* Module formal parameters */
  node_ptr mod_body         = cdr(mod_def); /* Module body */

  make_params(mod_name, actual, mod_formal_args); /* creates local parameters */
  /*
    We first instantiate all the definitions, in case they are
    constants used in the array declarations
  */
  mod_body_decls = mod_body;
  while(mod_body_decls != Nil) { /* loop over module declaration */
    node_ptr cur_decl = car(mod_body_decls);

    mod_body_decls = cdr(mod_body_decls);
    switch(node_get_type(cur_decl)) {
    case DEFINE: {
      node_ptr define_list = car(cur_decl);

      while(define_list != Nil) { /* loop over DEFINE declaration */
        node_ptr cur_define = car(define_list);
        node_ptr name = eval_struct(car(cur_define), mod_name);
        node_ptr definition = cdr(cur_define);

        yylineno = define_list->lineno;
        if (lookup_symbol_hash(name)) error_redefining(name);
        insert_symbol_hash(name, find_node(CONTEXT, mod_name, definition));
        all_symbols = cons(name, all_symbols);
        define_list = cdr(define_list);
      }
    }
    break;
    default:
      break;
    }
  }

  /* Now, we instantiate all the other elements of a module. */
  mod_body_decls = mod_body;
  while(mod_body_decls != Nil) { /* loop again over module declaration */
    node_ptr cur_decl = car(mod_body_decls);

    mod_body_decls = cdr(mod_body_decls);
    switch(node_get_type(cur_decl)) {
    case ISA:
      instantiate_by_name(car(cur_decl), mod_name, &tmp_trans, &tmp_init, &tmp_invar,
                          &tmp_spec, &tmp_ltlspec, &tmp_invar_spec, &tmp_invar_fb,
                          &tmp_invar_strong, &tmp_after, &tmp_fair, &tmp_abst,
			  &tmp_assign, &tmp_procs, Nil);
      break;
    case INPUT:
      if (instantiation_mode_is_implement()) break;
      is_input_decl = 1;
      instantiate_vars(car(cur_decl), mod_name, &tmp_trans, &tmp_init,
		       &tmp_invar, &tmp_spec, &tmp_ltlspec, &tmp_invar_spec,
		       &tmp_invar_fb, &tmp_invar_strong, &tmp_after,
		       &tmp_fair, &tmp_abst, &tmp_assign, &tmp_procs);
      is_input_decl = 0;
      break;
    case VAR:
      instantiate_vars(car(cur_decl), mod_name, &tmp_trans, &tmp_init,
		       &tmp_invar, &tmp_spec, &tmp_ltlspec, &tmp_invar_spec,
		       &tmp_invar_fb, &tmp_invar_strong, &tmp_after,
		       &tmp_fair, &tmp_abst, &tmp_assign, &tmp_procs);
      break;
    case IVAR:
      set_variable_instantiation_to_input();
      instantiate_vars(car(cur_decl), mod_name, &tmp_trans, &tmp_init,
		       &tmp_invar, &tmp_spec, &tmp_ltlspec, &tmp_invar_spec,
		       &tmp_invar_fb, &tmp_invar_strong, &tmp_after,
		       &tmp_fair, &tmp_abst, &tmp_assign, &tmp_procs);
      set_variable_instantiation_to_state();
      break;
    case TRANS:
      tmp_trans = find_node(AND, tmp_trans, find_node(CONTEXT, mod_name, car(cur_decl)));
      break;
    case INIT:
      tmp_init = find_node(AND, tmp_init, find_node(CONTEXT, mod_name, car(cur_decl)));
      break;
    case INVAR:
      tmp_invar = find_node(AND, tmp_invar, find_node(CONTEXT, mod_name, car(cur_decl)));
      break;
    case SPEC:
      if (instantiation_mode_is_implement()) break;
      if (instantiation_mode_is_spec())
        rpterr("sorry -- can't check a SPEC in an implementation (yet)");
      tmp_spec = cons(find_node(CONTEXT, mod_name, car(cur_decl)), tmp_spec);
      break;
    case LTLSPEC:
      if (instantiation_mode_is_implement()) break;
      if (instantiation_mode_is_spec())
        rpterr("sorry -- can't check a LTLSPEC in an implementation (yet)");
      tmp_ltlspec = cons(find_node(CONTEXT, mod_name, car(cur_decl)), tmp_ltlspec);
      break;
    case INVARSPEC:
      if (instantiation_mode_is_implement()) break;
      if (instantiation_mode_is_spec())
        rpterr("sorry -- can't check an INVARIANT in an implementation (yet)");
      tmp_invar_spec = cons(find_node(CONTEXT, mod_name, car(cur_decl)), tmp_invar_spec);
      break;
    case CHECKINVARIANTFB:
      if (instantiation_mode_is_implement()) break;
      if (instantiation_mode_is_spec())
        rpterr("sorry -- can't check a FB_INVARIANT in an implementation (yet)");
      tmp_invar_fb = cons(find_node(CONTEXT, mod_name, car(cur_decl)), tmp_invar_fb);
      break;
    case CHECKINVARIANTSTRONG:
      if (instantiation_mode_is_implement()) break;
      if (instantiation_mode_is_spec())
        rpterr("sorry -- can't check an STRONG_INVARIANT in an implementation (yet)");
      tmp_invar_strong = cons(find_node(CONTEXT, mod_name, car(cur_decl)), tmp_invar_strong);
      break;
    case AFTER:
      if (instantiation_mode_is_implement()) break;
      if (instantiation_mode_is_spec())
        rpterr("sorry -- can't check an AFTER (Temp. Proj.) in an implementation (yet)");
      tmp_after = cons(find_node(CONTEXT, mod_name, cur_decl), tmp_after);
      break;
    case COMPUTE:
      if (instantiation_mode_is_implement()) break;
      if (instantiation_mode_is_spec())
        rpterr("sorry -- can't compute a COMPUTE in an implementation (yet)");
      tmp_spec = cons(find_node(CONTEXT, mod_name, car(cur_decl)), tmp_spec);
      break;
    case FAIRNESS:
      tmp_fair = cons(find_node(CONTEXT, mod_name, car(cur_decl)), tmp_fair);
      break;
    /* Yuan Lu */
    case ABSTRACT: {
      node_ptr avar = eval_struct(car(car(cur_decl)), mod_name);
      node_ptr abs = find_node(IMPLIES, avar, cdr(car(cur_decl)));
      tmp_abst = cons(abs, tmp_abst);
      break;
    }
    case ASSIGN:
      tmp_assign = find_node(AND, find_node(CONTEXT, mod_name, car(cur_decl)), tmp_assign);
      break;
    case OUTPUT:
      if (!instantiation_mode_is_implement()) break;
      {
	node_ptr out_list = car(cur_decl);

	while(out_list != Nil) {
	  tmp_spec = find_node(AND, tmp_spec,
                             find_node(EQUAL,
                                       find_node(CONTEXT, mod_name, car(out_list)),
                                       find_node(CONTEXT, the_impl, car(out_list))));
	  out_list = cdr(out_list);
	}
      }
      break;
    default:
      break;
    }
  }
  *trans = find_node(AND, *trans, tmp_trans);
  *init  = find_node(AND, *init, tmp_init);
  *invar = find_node(AND, *invar, tmp_invar);
  *spec  = append(tmp_spec, *spec);
  *ltl_spec = append(tmp_ltlspec, *ltl_spec);
  *invar_spec = append(tmp_invar_spec, *invar_spec);
  *invar_fb = append(tmp_invar_fb, *invar_fb);
  *invar_strong = append(tmp_invar_strong, *invar_strong);
  *after = append(tmp_after, *after);
  *fair  = append(tmp_fair, *fair);
  *abst  = append(tmp_abst, *abst);
  *assign = find_node(AND, *assign, tmp_assign);
  *procs = append(*procs, tmp_procs);
}

/**Function********************************************************************

  Synopsis           [Starts the flattening from a given point in the
  module hierarchy.]

  Description        [Uses <tt>root_name</tt> as root module in the
  flattening of the hierarchy. The name of the module in the hierarchy
  is <tt>name</tt>. First checks if the module exists. Then it checks
  if the module is recursively defined, and if the case an error is
  printed out. If these checks are passed, then it proceeds in the
  instantiation of the body of the module.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void instantiate_by_name(node_ptr root_name, node_ptr name, node_ptr *trans,
                                node_ptr *init, node_ptr *invar, node_ptr *spec,
                                node_ptr *ltl_spec, node_ptr *invar_spec,
                                node_ptr *invar_fb, node_ptr *invar_strong,
                                node_ptr *after, node_ptr *fair, node_ptr *abst,
				node_ptr *assign, node_ptr *procs, node_ptr actual)
{
  node_ptr s;
  node_ptr mod_name = find_atom(root_name);         /* find module name */
  node_ptr mod_def  = lookup_module_hash(mod_name); /* find module definition  */

  yylineno = root_name->lineno;
  if (!mod_def) error_undefined(root_name); /* The module is undefined */
  s = module_stack;
  /* scans module_stack in order to find if there are recursively defined modules. */
  while(s != Nil) { 
    if (car(s) == mod_name)
      rpterr("module \"%s\" is recursively defined",
             get_text((string_ptr)car(root_name)));
    s = cdr(s);
  }
  /* append current module to module_stack */
  module_stack = cons(mod_name, module_stack); 

  instantiate(mod_def, name, trans, init, invar, spec, ltl_spec, invar_spec, invar_fb,
              invar_strong, after, fair, abst, assign, procs, actual);
  /* elimination of current module form module_stack */
  s = cdr(module_stack);
  free_node(module_stack);
  module_stack = s;
}
