/**CFile***********************************************************************

  FileName    [compileEncode.c]

  PackageName [compile]

  Synopsis    [Routines that performs the encoding of the symbolic variables
  into BDD.]

  Description [Routines that performs the encoding of the symbolic variables
  into BDD. At now the only available encoding is the binary one.]

  SeeAlso     [compileEval.c, compileOrd.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The cube of input variables.]

  Description [The cube of input variables to be used in image forward.]

  SeeAlso     []

******************************************************************************/
add_ptr input_variables_add = (add_ptr)NULL;
bdd_ptr input_variables_bdd = (bdd_ptr)NULL;


/**Variable********************************************************************

  Synopsis    [The cube of input variables.]

  Description [The cube of input variables to be used in image backward.]

  SeeAlso     []

******************************************************************************/
add_ptr next_input_variables_add = (add_ptr)NULL;
bdd_ptr next_input_variables_bdd = (bdd_ptr)NULL;


/**Variable********************************************************************

  Synopsis    [The cube of state variables.]

  Description [The cube of state variables to be used in image forward.]

  SeeAlso     []

******************************************************************************/
add_ptr state_variables_add = (add_ptr)NULL;
bdd_ptr state_variables_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [The cube of state variables.]

  Description [The cube of state variables to be used in image backward.]

  SeeAlso     []

******************************************************************************/
add_ptr next_state_variables_add = (add_ptr)NULL;
bdd_ptr next_state_variables_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [The ADD representing input variables.]

  Description [The ADD representing input variables. It is used in
  checking the simulation of two automata. It is used only by the check_implement
  package, which has not yet been ported.]

  SeeAlso     [check_implement]

******************************************************************************/
add_ptr input_vars_add;

/**Variable********************************************************************

  Synopsis    [The array of symbolic variable names.]

  Description [The array of symbolic variable names. Each element <tt>i</tt>
  contains the symbolic name associated to the boolean variable with
  index <tt>i</tt>.]

******************************************************************************/
static node_ptr variable_names[MAX_VAR_INDEX];

/**Variable********************************************************************

  Synopsis    [Correspondence between current and next variables.]

  Description [These arrays are used to maintain correspondence
  between current and next variables. Position <em>i</em> contains the
  index of the corresponding next state variable. They are used to
  perform forward and backward shifting respectively.]

  SeeAlso     []

******************************************************************************/
static int current2next[MAX_VAR_INDEX];
static int next2current[MAX_VAR_INDEX];


/**Variable********************************************************************

  Synopsis    [Array used to pick up a minterm from a given BDD.]

  Description [This array should contain at least all variables in the
  support of the BDD of which we want extract a minterm.]

  SeeAlso     []

******************************************************************************/
static bdd_ptr state_vars[MAX_VAR_INDEX];
static int state_vars_dim = 0;

/**Variable********************************************************************

  Synopsis    [The hash used to take care of already printed variable value.]

  Description [The hash used to take care of already printed variable
  value. It is used by <tt>print_state</tt>.]

  SeeAlso     [print_state]

******************************************************************************/
static hash_ptr print_hash;
void init_print_hash() { print_hash = new_assoc(); }
void insert_print_hash(node_ptr x, node_ptr y) { insert_assoc(print_hash, x, y);}
node_ptr lookup_print_hash(node_ptr x) {return(find_assoc(print_hash, x));}
void clear_print_hash() {clear_assoc(print_hash);}

/**Variable********************************************************************

  Synopsis    [The number of boolean state variables created.]

  Description [It is the number of boolean state variables created to
  encode symbolic state variables, current and next.]

  SeeAlso     []

******************************************************************************/
static int num_of_state_variables = 0; 
static int num_of_state_variables_get(void) { return(num_of_state_variables); }
static void num_of_state_variables_set(int n) { num_of_state_variables = n; }
static void num_of_state_variables_inc(void) { num_of_state_variables++; }

/**Variable********************************************************************

  Synopsis    [The number of boolean input variables created.]

  Description [It is the number of boolean input state variables
  created to encode symbolic input variables, both current and next.]

  SeeAlso     []

******************************************************************************/
static int num_of_input_variables = 0;
static int num_of_input_variables_get(void) { return(num_of_input_variables); }
static void num_of_input_variables_set(int n) { num_of_input_variables = n; }
static void num_of_input_variables_inc(void) { num_of_input_variables++; }


/**Variable********************************************************************

  Synopsis    [The number of boolean variables created.]

  Description [It is the number of boolean variables created to encode
  both input and state variables. It is the sum of
  <tt>num_of_input_variables</tt> plus <tt>num_of_state_variables</tt>.]

  SeeAlso     []

******************************************************************************/
static int num_of_variables = 0;
static int num_of_variables_get(void) { return(num_of_variables); }
static void num_of_variables_set(int n) { num_of_variables = n; }
static void num_of_variables_inc(void) { num_of_variables++; }

/**Variable********************************************************************

  Synopsis    [The number of current state boolean variables created.]

  Description [It is the number of current state boolean variables created.]

  SeeAlso     []

******************************************************************************/
static int real_num_of_state_vars = 0;
static int real_num_of_state_vars_get(void) { return(real_num_of_state_vars); }
static void real_num_of_state_vars_set(int n) { real_num_of_state_vars = n; }
static void real_num_of_state_vars_inc(void) { real_num_of_state_vars++; }

/**Variable********************************************************************

  Synopsis    [The constant hash.]

  Description [This hash associates to an atom the corresponding ADD
  leaf if defined. Suppose to have a declaration of this kind:<br> 
  <pre>
  VAR 
    state : {idle, stopped}
  <pre>
  then in the constant hash for the atom <tt>idle</tt> there is the
  corresponding leaf ADD, i.e. the ADD whose value is the symbol
  <tt>idle</tt>. This hash is used by the evaluator.]

  SeeAlso     [eval]

******************************************************************************/
static hash_ptr constant_hash;
void init_constant_hash() { constant_hash = new_assoc(); }
void insert_constant_hash(node_ptr x, node_ptr y) { insert_assoc(constant_hash, x, y);}
node_ptr lookup_constant_hash(node_ptr x) {return(find_assoc(constant_hash, x));}
static assoc_retval constant_hash_free(char *key, char *data, char * arg) {
  add_ptr element = (add_ptr)data;

  if (element != (add_ptr)NULL) {
    add_free(dd_manager, element);
  }
  return(ASSOC_DELETE);
}

void clear_constant_hash() {clear_assoc_and_free_entries(constant_hash, constant_hash_free);}

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static add_ptr CompileEncodeVarRecur ARGS((node_ptr, int, int));
static void get_bdd_state_var ARGS((void));
static void get_bdd_input_var ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the data structure relative to BDD variables.]

  Description        [Initializes the data structure relative to BDD
  variables, i.e. the array used to store correspondence between
  variable names and bdd variables, and the array used to shift
  forward and backward during image computation.]

  SideEffects        []

******************************************************************************/
void Compile_EncodeVarsInit(void)
{
  int j;

  for (j = 0; j < MAX_VAR_INDEX; j++) {
    variable_names[j] = Nil;
    current2next[j] = j;
    next2current[j] = j;
  }
}

/**Function********************************************************************

  Synopsis           [Encode a variable into BDD.]

  Description        [This function encodes the variable <tt>name</tt>,
  whose range is <tt>range</tt>, into BDD variables. The boolean
  variables created are grouped together, in such a way that the BDD
  dynamic reordering consider them as a single block.]

  SideEffects        []

  SeeAlso            [CompileEncodeVarRecur]

******************************************************************************/
add_ptr Compile_EncodeVar(node_ptr name, node_ptr range, int is_input)
{
  int start_index, offset;
  add_ptr encoding;
  
  start_index = num_of_variables_get() + 1;
  encoding = CompileEncodeVarRecur(range, num_of_variables_get(), is_input);
  offset = num_of_variables_get() - start_index; /* offset <= 0 no variables created */
  if (offset > 0) {
    variable_names[start_index] = name;
    dd_new_var_block(dd_manager, start_index, offset + 1);
    if (opt_verbose_level_gt(options, 2)) {
      (void) fprintf(nusmv_stderr, "\n*****************************************************\n");
      (void) fprintf(nusmv_stderr, "Built block from var %d to var %d enclosed \n", start_index, start_index + offset);
      (void) fprintf(nusmv_stderr, "The number of vars in this group is %d\n", offset + 1);
      (void) fprintf(nusmv_stderr, "*******************************************************\n");
    }
  } else {
    if (opt_verbose_level_gt(options, 2)) {
      (void) fprintf(nusmv_stderr, "The current variable has no associated values,\n");
      (void) fprintf(nusmv_stderr, "so no BDD variables were created for it.\n\n");
    }
  }
  return(encoding);
}

/**Function********************************************************************

  Synopsis           [Encodes a scalar variables into boolean variables.]

  Description        [This function computes the ADD for representing the
  boolean encoding of a scalar variable with values listed in "range". If
  X has range {a, b, c}, the propositions (X == a), (X == b), (X == c)
  are encoded by means of two binary variables. E.g. <br>
  <pre>
              x1
            1/  \\0 
            x2   \\ 
           /  \\   \\
          a    c   b
  </pre>
  Where (x1 and x2) represents (X == a), while (x1 and not x2) represents (X
  == b), and (x1 = 0) represents (X == c).]

  SideEffects        [<tt>num_of_state_variables</tt>,
  <tt>num_of_input_variables</tt> are  modified depending the kind of
  variable we are encoding.]

  SeeAlso            []

******************************************************************************/
static add_ptr CompileEncodeVarRecur(node_ptr range, int curlevel, int is_input)
{
  if (range == Nil) internal_error("CompileEncodeVarRecur: range = Nil");

  /* Base case: We reached a leaf */
  if (cdr(range) == Nil) {
    node_ptr v = find_atom(car(range));
    add_ptr temp = (add_ptr)lookup_constant_hash(v);
    if (temp) {
      add_ref(temp);
      return(temp);
    }
    temp = add_leaf(dd_manager, v);
    if (v && node_get_type(v) == ATOM) {
      insert_constant_hash(v, (node_ptr)temp);
      insert_check_constant_hash(v, v);
    }
    add_ref(temp);
    return(temp);
  }

  /* Step case: */
  if (is_input) { /* next variables are not created */
    curlevel += 2;
    if (curlevel > num_of_variables) get_bdd_input_var();
    {
      /*
        The element list are decomposed and left and right are
        allocated in the bdd
      */

      /*
	Yuan Lu : I change the following line
      add_ptr p0 = CompileEncodeVarRecur(odd_elements(range), curlevel, is_input);
      add_ptr p1 = CompileEncodeVarRecur(even_elements(range), curlevel, is_input);
      add_ptr result = add_build(dd_manager, curlevel - 1, p0, p1);
	to
      */
      node_ptr lhalf, rhalf;
      add_ptr p0, p1, result;
      split_power_half(range, &lhalf, &rhalf);
      p0 = CompileEncodeVarRecur(lhalf, curlevel, is_input);
      p1 = CompileEncodeVarRecur(rhalf, curlevel, is_input);
      result = add_build(dd_manager, curlevel - 1, p1, p0);

      add_free(dd_manager, p0);
      add_free(dd_manager, p1);
      return(result);
    }
  } else { /* next variables created */
    curlevel += 2;
    if (curlevel > num_of_variables) get_bdd_state_var();
    {
      /*
        The element list are decomposed and left and right are
        allocated in the bdd
      */
      /*
        Yuan Lu : I change the following line
      add_ptr p0 = CompileEncodeVarRecur(odd_elements(range), curlevel, is_input);
      add_ptr p1 = CompileEncodeVarRecur(even_elements(range), curlevel, is_input);
      add_ptr result = add_build(dd_manager, curlevel - 1, p0, p1);
        to
      */
      node_ptr lhalf, rhalf;
      add_ptr p0, p1, result;
      split_power_half(range, &lhalf, &rhalf);
      p0 = CompileEncodeVarRecur(lhalf, curlevel, is_input);
      p1 = CompileEncodeVarRecur(rhalf, curlevel, is_input);
      result = add_build(dd_manager, curlevel - 1, p1, p0);

      add_free(dd_manager, p0);
      add_free(dd_manager, p1);
      return(result);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Return the list of variables corresponding
  to the current order of variables in the system.]

  Description        [It returns the list of variables corresponding
  to the current order of variables in the system.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Compile_GetOrdering(void)
{
  int cvl, max_level;
  int flag = 0;
  node_ptr current_ordering = Nil;

  max_level = dd_get_size(dd_manager);
  if (opt_verbose_level_gt(options, 4))
    fprintf(nusmv_stderr, "Number of variables: %d\n", max_level);
  for (cvl = 0; (cvl < max_level) ; cvl++) {
    int index = dd_get_var_at_level(dd_manager, cvl);
    node_ptr name = variable_names[index];

   if (opt_verbose_level_gt(options, 4))
     fprintf(nusmv_stderr,"level %d with variable %d\n", cvl, index);
   if (name == proc_selector_internal_vname) flag = 1;
   if (name != Nil) current_ordering = cons(name, current_ordering);
  }
  if (flag == 0) current_ordering = cons(proc_selector_internal_vname, current_ordering);
  current_ordering = reverse(current_ordering);
  return(current_ordering);
}


/**Function********************************************************************

  Synopsis           [Prints out the state represented by the given minterm.]

  Description        [Prints out the state represented by the given
  minterm. If <tt>changes_only</tt> is 1, than only state variables
  which assume a different value from the previous printed one were
  printed out.]

  SideEffects        [<tt>print_hash</tt> is modified.]

******************************************************************************/
void print_state(bdd_ptr s, int changes_only)
{
  node_ptr proc_name;
  add_ptr add_state, tmp;
  node_ptr los = all_symbols;

  add_state = bdd_to_add(dd_manager, s);
  tmp = add_if_then(dd_manager, add_state, process_selector_add);
  proc_name = add_value(dd_manager, tmp);
  add_free(dd_manager, tmp);

  if (proc_name != Nil) indent_node(nusmv_stdout, "[executing process ", proc_name, "]\n");
  while (los) {
    node_ptr cur_sym_value;
    node_ptr cur_sym = car(los);
    add_ptr cur_sym_vals = eval(cur_sym, Nil);

    los = cdr(los);
    tmp = add_if_then(dd_manager, add_state, cur_sym_vals);
    cur_sym_value = add_value(dd_manager, tmp);
    add_free(dd_manager, tmp);
    if (changes_only) {
      if (cur_sym_value == lookup_print_hash(cur_sym)) continue;
      insert_print_hash(cur_sym, cur_sym_value);
    }
    indent_node(nusmv_stdout, "", cur_sym, " = ");
    print_node(nusmv_stdout, cur_sym_value);
    fprintf(nusmv_stdout, "\n");
    add_free(dd_manager, cur_sym_vals);
  }
  add_free(dd_manager, add_state);
}

/**Function********************************************************************

  Synopsis           [Prints out the symbolic names of boolean
  variables stored in a cube.]

  Description        [Given a cube of boolean BDD variables, this
  function prints out the symbolic names of the corresponding
  variables. The symbolic name of the variables to be printed out are
  listed in <tt>list_of_sym</tt>.]

  SideEffects        [None]

******************************************************************************/
void print_state_vars(DdManager * dd, bdd_ptr cube, node_ptr list_of_sym)
{
  node_ptr los = list_of_sym;
  add_ptr a_cube = bdd_to_add(dd, cube);
  add_ptr one = add_one(dd);
  extern int current2next[];

  fprintf(nusmv_stdout, "Current VARS:\n");
  while (los) {
    node_ptr cur_sym = car(los);
    add_ptr cur_sym_vals = eval(cur_sym, Nil);
    add_ptr cur_sym_cube = add_support(dd, cur_sym_vals);
    add_ptr is_in = add_cube_diff(dd, a_cube, cur_sym_cube);

    los = cdr(los);
    if (is_in != a_cube) {
      indent_node(nusmv_stdout,"", cur_sym, " ");
    }
    add_free(dd, cur_sym_vals);
    add_free(dd, cur_sym_cube);
    add_free(dd, is_in);
  }
  fprintf(nusmv_stdout, "\nNext VARS:\n");  
  los = list_of_sym;
  while (los) {
    node_ptr cur_sym = car(los);
    add_ptr cur_sym_vals = eval(cur_sym, Nil);
    add_ptr next_cur_sym_vals = add_permute(dd, cur_sym_vals, current2next);
    add_ptr next_cur_sym_cube = add_support(dd, next_cur_sym_vals);
    add_ptr next_is_in = add_cube_diff(dd, a_cube, next_cur_sym_cube);

    los = cdr(los);
    if (next_is_in != a_cube) {
      indent_node(nusmv_stdout,"", find_node(NEXT, cur_sym, Nil), " ");
    }
    add_free(dd, cur_sym_vals);
    add_free(dd, next_cur_sym_vals);
    add_free(dd, next_cur_sym_cube);
    add_free(dd, next_is_in);
  }
  add_free(dd, a_cube);
}

/**Function********************************************************************

  Synopsis           [Return the number of states of a given BDD.]

  Description        [Return the number of minterms (i.e. states)
  represented by a BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
double bdd_count_states(DdManager *dd, add_ptr fn) 
{
  return(bdd_count_minterm(dd, fn, real_num_of_state_vars));
}

/**Function********************************************************************

  Synopsis           [Return the number of states of a given ADD.]

  Description        [Return the number of minterms (i.e. states)
  represented by a ADD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
double add_count_states(DdManager *dd, bdd_ptr fn) 
{
  return(add_count_minterm(dd, fn, real_num_of_state_vars));
}

/**Function********************************************************************

  Synopsis           [Shift the given set from current variables to next ones.]

  Description        [Shift the given set from current variables to next ones.]

  SideEffects        []

  SeeAlso            [add_shift_backward]

******************************************************************************/
add_ptr add_shift_forward(DdManager * dd, add_ptr fn)
{ return(add_permute(dd, fn, current2next)); }

/**Function********************************************************************

  Synopsis           [Shift the given set from next variables to current ones.]

  Description        [Shift the given set from next variables to current ones.]

  SideEffects        []

  SeeAlso            [add_shift_forward]

******************************************************************************/
add_ptr add_shift_backward(DdManager * dd, add_ptr fn)
{ return(add_permute(dd, fn, next2current)); }



/**Function********************************************************************

  Synopsis           [Shift the given set from current variables to next ones.]

  Description        [Shift the given set from current variables to next ones.]

  SideEffects        []

  SeeAlso            [bdd_shift_backward, bdd_permute]

******************************************************************************/
bdd_ptr bdd_shift_forward(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_permute(dd, fn, current2next);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Shift the given set from next variables to current ones.]

  Description        [Shift the given set from next variables to current ones.]

  SideEffects        []

  SeeAlso            [bdd_shift_forward, bdd_permute]

******************************************************************************/
bdd_ptr bdd_shift_backward(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_permute(dd, fn, next2current);
  return(result);
}
    

/**Function********************************************************************

  Synopsis           [Extracts a minterm from a given BDD.]

  Description        [Extracts a minterm from a given BDD.]

  SideEffects        []

  SeeAlso            [bdd_pick_one_minterm]

******************************************************************************/
bdd_ptr bdd_pick_one_state(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_pick_one_minterm(dd, fn, state_vars, state_vars_dim);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Extracts a random minterm from a given BDD.]

  Description        [Extracts a random minterm from a given BDD.]

  SideEffects        []

  SeeAlso            [bdd_pick_one_minterm_rand]

******************************************************************************/
bdd_ptr bdd_pick_one_state_rand(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_pick_one_minterm_rand(dd, fn, state_vars, state_vars_dim);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Initializes the array necesary to extract minterms from BDDs]

  Description        [Initializes the array necesary to extract minterms from BDDs]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void init_state_vars(void)
{
  int i;
  for (i = 0; i < MAX_VAR_INDEX; i++) state_vars[i] = (bdd_ptr)NULL;
  state_vars_dim = 0;
}

/**Function********************************************************************

  Synopsis           [Creates the array necessary to extract minterms
  from a BDD.]

  Description        [Creates the array necessary to extract minterms
  from a BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void make_state_vars(void)
{ 
  int i, k = 0;

  /* We need to extract only current state variables */
  for (i = 1; i < num_of_variables; i+=2) {
    state_vars[k++] = bdd_new_var_with_index(dd_manager, i);
  }
  state_vars_dim = k;
}

/**Function********************************************************************

  Synopsis           [Free the array necessary to extract minterms.]
  from a BDD.]

  Description        [Free the array necessary to extract minterms
  from a BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void free_state_vars(void)
{
  int i;

  for (i = 0; i < state_vars_dim; i++) bdd_free(dd_manager, state_vars[i]);
}


/**Function********************************************************************

  Synopsis           [Reset the counters necessary to the encoder.]

  Description        [Reset the counters necessary to the encoder.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void reset_encode()
{
  num_of_variables_set(0);
  num_of_input_variables_set(0);
  num_of_state_variables_set(0);
  real_num_of_state_vars_set(0);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Adds a new boolean variable to the DD package.]

  Description        [Adds a new boolean variable to the DD manager.
  If it is a state variable, then also the "next" variable is created,
  and the data structure to perform substitution of "current" and "next"
  variables is filled in.]

  SideEffects        []

  SeeAlso            [get_bdd_input_var]

******************************************************************************/
static void get_bdd_state_var(void)
{
  add_ptr tmp_1;
  
  if (num_of_variables == MAX_VAR_INDEX) error_too_many_vars();  
  if (!is_input_decl) {
    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "  BDD variable %d, ", num_of_variables+1);
    tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
    add_and_accumulate(dd_manager, &state_variables_add, tmp_1);
    add_free(dd_manager, tmp_1);
    if (num_of_variables == MAX_VAR_INDEX) error_too_many_vars();  
    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, " next BDD variable %d\n", num_of_variables+1);
    tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
    add_and_accumulate(dd_manager, &next_state_variables_add, tmp_1);
    add_free(dd_manager, tmp_1); 
    current2next[num_of_variables - 1] = num_of_variables;
    next2current[num_of_variables]     = num_of_variables - 1;
    num_of_state_variables += 2;
    real_num_of_state_vars_inc();
  }
  else { /* This is necessary in the checking of the implement */
    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "  BDD impl input variable %d\n", num_of_variables+1);
    tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
    add_and_accumulate(dd_manager, &input_vars_add, tmp_1);
    num_of_state_variables += 1;
  }   
  return;
}

/**Function********************************************************************

  Synopsis           [Adds a new boolean variable to the DD package.]

  Description        [Adds a new boolean variable to the DD
  manager. This function is used to create the boolean variables
  needed to encode input variables.]

  SideEffects        []

  SeeAlso            [get_bdd_state_var]

******************************************************************************/
static void get_bdd_input_var(void)
{
  add_ptr tmp_1;
  
  if (num_of_variables == MAX_VAR_INDEX) error_too_many_vars();
  if (opt_verbose_level_gt(options, 1))
    fprintf(nusmv_stderr, "  BDD input variable %d ", num_of_variables+1);
  tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
  add_and_accumulate(dd_manager, &input_variables_add, tmp_1);
  add_free(dd_manager, tmp_1);
  if (num_of_variables == MAX_VAR_INDEX) error_too_many_vars();
  if (opt_verbose_level_gt(options, 1))
    fprintf(nusmv_stderr, " next BDD variable %d\n", num_of_variables+1);
  tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
  add_and_accumulate(dd_manager, &next_input_variables_add, tmp_1);
  add_free(dd_manager, tmp_1); 

  current2next[num_of_variables - 1] = num_of_variables;
  next2current[num_of_variables]     = num_of_variables - 1;
  num_of_input_variables += 2;

  return;
}

