/**CFile***********************************************************************

  FileName    [compileDisj.c]

  PackageName [compile]

  Synopsis    [Routines to performs the computation of the
  disjunctively decomposed transition relation.]

  Description [Routines to performs the computation of the
  conjunctively decomposed transition relation.]

  SeeAlso     [compileMono.c, compileIwls95.c, compileConj.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The disjunctively partitioned transition relation.]

  Description [The disjunctive partitioned transition relation in ADD and BDD.]

  SeeAlso     []

******************************************************************************/
node_ptr dp_trans_add = Nil;
node_ptr dp_trans_bdd = Nil;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
#ifdef OPT_QUANTIFY_INPUT
static bdd_ptr add_to_bdd_and_abstract_input ARGS((DdManager *, add_ptr));
#endif
/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Build the disjunctive partitioned transition relation.]

  Description        [This function takes as input a list of
  processes. For each process a partition is performed and the
  implicitly disjoined transition relation is computed.]

  SideEffects        [<tt>dp_trans_add</tt> is computed.]

  SeeAlso            [ Compile_CompileModel Compile_CompileModelConj Compile_CompileModeliwls95]

******************************************************************************/
void Compile_CompileModelDisj(node_ptr trans_expr, node_ptr invar_expr,
                              node_ptr procs_expr, add_ptr assumption)
{
  add_ptr running_add;
  int partition_num=0;
  node_ptr procs_list = procs_expr;
  
  if (cdr(procs_list) == Nil)
    rpterr("You cannot have only one process if you want to use disjunctive partitioning.\n");

  invar_add = add_one(dd_manager);
  dp_trans_add = cons((node_ptr)add_zero(dd_manager), Nil);

  while(procs_list != Nil) {
    add_ptr current_trans;
    add_ptr current_frame;
    add_ptr tmp_1, tmp_2;
    node_ptr current_context = car(car(procs_list));
    node_ptr current_assign_exp = cdr(car(procs_list));

    partition_num++; /* Increment partition number counter */

    /* evaluate 'running_atom' in 'context' */
    running_add=eval(running_atom, current_context);

    if (opt_verbose_level_gt(options, 1))
        print_in_process("evaluating TRANS statements", current_context);
    current_trans = eval_simplify(trans_expr, Nil, assumption);

    if (opt_verbose_level_gt(options, 1))
        print_in_process("evaluating next(*) assignments", current_context);
    set_assignment_type_trans();
    tmp_1 = eval_simplify(current_assign_exp, Nil, assumption);
    add_and_accumulate(dd_manager, &current_trans, tmp_1);
    add_free(dd_manager, tmp_1);
    
    if (opt_verbose_level_gt(options, 1))
        fprintf(nusmv_stderr, "size of transition relation = %d BDD nodes\n",
                add_size(dd_manager, current_trans));

    /* inserts current process frame axioms */
    current_frame = add_and(dd_manager, running_add, frame_add);
    add_and_accumulate(dd_manager, &current_trans, current_frame);

    /* Running conditions */
    add_and_accumulate(dd_manager, &current_trans, running_add);

    procs_list = cdr(procs_list);
    
    dp_trans_add=cons((node_ptr)current_trans, dp_trans_add);

    if (opt_verbose_level_gt(options, 0))
        print_in_process("evaluating normal assignments", current_context);
    set_assignment_type_assign();
    tmp_1 = eval_simplify(current_assign_exp, Nil, assumption);
    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating INVAR statements", current_context);
    tmp_2 = eval_simplify(invar_expr, Nil, assumption);
    add_and_accumulate(dd_manager, &tmp_1, tmp_2);
    add_free(dd_manager, tmp_2);

    if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr, "size of invariant set = %g states, %d ADD nodes\n",
                add_count_states(dd_manager, tmp_1), add_size(dd_manager, tmp_1));
    
    tmp_2 = add_imply(dd_manager, running_add, tmp_1);
    add_and_accumulate(dd_manager, &invar_add, tmp_2);

    add_free(dd_manager, tmp_1);
    add_free(dd_manager, tmp_2);
    add_free(dd_manager, running_add);
    add_free(dd_manager, current_frame);
  }
  /* Conversion from ADD to BDD */
  invar_bdd = add_to_bdd(dd_manager, invar_add);
  input_variables_bdd = add_to_bdd(dd_manager, input_variables_add);
  next_input_variables_bdd = add_to_bdd(dd_manager, next_input_variables_add);
  state_variables_bdd = add_to_bdd(dd_manager, state_variables_add);
  next_state_variables_bdd = add_to_bdd(dd_manager, next_state_variables_add);
#ifdef OPT_QUANTIFY_INPUT
  if (opt_quantify_input(options)) {
    dp_trans_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd, dp_trans_add);
  } else {
    dp_trans_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd_and_abstract_input, dp_trans_add);
  }
#else
  dp_trans_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd, dp_trans_add);
#endif
  add_free(dd_manager, invar_add);
  walk_dd(dd_manager, add_free, dp_trans_add);
  if (opt_verbose_level_gt(options, 0)) {
    (void) print_disj_part_info();
    (void) fprintf(nusmv_stderr, "size of invariant set = %g states, %d BDD nodes\n",
                   bdd_count_states(dd_manager, invar_bdd), bdd_size(dd_manager, invar_bdd));
  }
}


/**Function********************************************************************

  Synopsis           [Prints the BDD size of the partitions.]

  Description        [Prints the BDD size of the partitions.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_disj_part_info()
{
  int i = 0;
  node_ptr l = dp_trans_bdd;
  
  if (dp_trans_bdd != Nil)
    (void) fprintf(nusmv_stdout, "Disjunctive Partitioning:\n");
  while(l != Nil) {
    (void) fprintf(nusmv_stdout, "partition %d: %d BDD nodes.\n", i++,
                   (bdd_ptr)bdd_size(dd_manager, (bdd_ptr)car(l)));
    l = cdr(l);
  }
  if (invar_bdd != NULL)
    (void)fprintf(nusmv_stdout, "Size of invariant: %d BDD nodes.\n",
                  bdd_size(dd_manager, invar_bdd));
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
#ifdef OPT_QUANTIFY_INPUT
/**Function********************************************************************

  Synopsis           [Abstract the input variables from the given BDD.]

  Description        [Abstract the input variables from the given BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static bdd_ptr add_to_bdd_and_abstract_input(DdManager * dd, add_ptr fn)
{
  bdd_ptr result;
  bdd_ptr tmp;
  
  tmp = add_to_bdd(dd, fn);
  result = bdd_forsome(dd, tmp, input_variables_bdd);
  bdd_free(dd, tmp);
  return(result);
}
#endif
