/**CFile***********************************************************************

  FileName    [compileConj.c]

  PackageName [compile]

  Synopsis    [Routines to performs the computation of the
  conjunctively decomposed transition relation.]

  Description [Routines to performs the computation of the
  conjunctively decomposed transition relation.]

  SeeAlso     [compileMono.c, compileIwls95.c, compileDisj.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
/**Variable********************************************************************

  Synopsis    [The conjunctively partitioned transition relation]

  Description [The conjunctively partitioned transition relation in ADD and BDD.]

  SeeAlso     []

******************************************************************************/
node_ptr cp_trans_add = Nil;
node_ptr cp_trans_bdd = Nil;

/* Yuan Lu : backup the orginal machine's transition relation */
node_ptr cp_trans_add_orig = Nil;
node_ptr cp_trans_bdd_orig = Nil;

/**Variable********************************************************************

  Synopsis    [The list of the cube to be abstracted for each cluster.]

  Description [The list of the ADD cube to be abstracted for each
  cluster in the image forward computation.]

  SeeAlso     []

******************************************************************************/
node_ptr forward_quantifiers_add = Nil;
node_ptr forward_quantifiers_bdd = Nil;

/* Yuan Lu : backup the original machine's forward quantifier list */
node_ptr forward_quantifiers_add_orig = Nil;
node_ptr forward_quantifiers_bdd_orig = Nil;

/**Variable********************************************************************

  Synopsis    [The list of the cube to be abstracted for each cluster.]

  Description [The list of the ADD cube to be abstracted for each
  cluster in the image backward computation.]

  SeeAlso     []

******************************************************************************/
node_ptr reverse_quantifiers_add = Nil;
node_ptr reverse_quantifiers_bdd = Nil;

/* Yuan Lu : backup the original machine's backward quantifier list */
node_ptr reverse_quantifiers_add_orig = Nil;
node_ptr reverse_quantifiers_bdd_orig = Nil;

/**Variable********************************************************************

  Synopsis    [List of partitioned transitions for each variable.]

  Description [List of partitioned transitions for each variable.]

  SeeAlso     [Yuan Lu]

******************************************************************************/
node_ptr max_part_list;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static node_ptr make_quantifiers ARGS((node_ptr l, add_ptr vars));
static node_ptr compileCompileModelConjRecur ARGS((node_ptr l, node_ptr n, node_ptr c, add_ptr a));
static node_ptr make_support_list ARGS((node_ptr l));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Build an implicitly conjoined transition relation.]

  Description        [This function builds the BDD representing the
  implicitly conjoined transition relation. The third argument (that's
  "assumption") is the set of states to which the transition relation
  has to be restricted to.]

  SideEffects        []

  SeeAlso            [Compile_CompileModelDisj Compile_CompileModelMono Compile_CompileModelIwls95]

******************************************************************************/
void Compile_CompileModelConj(node_ptr trans_expr, node_ptr invar_expr, node_ptr procs,
                    add_ptr assumption)
{
  add_ptr tmp_1, tmp_2, running_add;

  if (cdr(procs))
    rpterr("Processes not allowed with conjunctive partitioning.");

  /* Initializes BDD of transition relation and invariant set. */
  invar_add_orig = add_one(dd_manager);

  { 
    node_ptr context = car(car(procs));
    node_ptr assign_expr = cdr(car(procs));

    /* evaluate 'running_atom' in 'context' */
    running_add = eval(running_atom, context);

    /* now evaluate the TRANS, ASSIGN statements */
    cp_trans_add_orig = cons((node_ptr)add_one(dd_manager), Nil);
  
    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating TRANS statements", context);
    cp_trans_add_orig = compileCompileModelConjRecur(cp_trans_add_orig, trans_expr, Nil, assumption);
  
    if (opt_verbose_level_gt(options, 1)) print_in_process("evaluating next() assignments", context);
    set_assignment_type_trans();
    cp_trans_add_orig = compileCompileModelConjRecur(cp_trans_add_orig, assign_expr, Nil, assumption);

#ifdef ABS_DEBUG
    if(ABS_DEBUG > 1) dd_check_manager(dd_manager);
#endif
    /* Yuan Lu : applying abstraction */
    cp_trans_add = Abs_AbsAbstractTransPartADD(cp_trans_add_orig);

#ifdef ABS_DEBUG
    if(ABS_DEBUG > 1) dd_check_manager(dd_manager);
#endif

    /* Yuan Lu : build max_part_list */
    max_part_list = Abs_AbsBuildMaxPartList(max_part_list, assign_expr, Nil, assumption);

#ifdef ABS_DEBUG
    if(ABS_DEBUG > 1) dd_check_manager(dd_manager);
#endif

    if (opt_verbose_level_gt(options, 1)) {
      int i = 1;
      node_ptr q = cp_trans_add;

      fprintf(nusmv_stderr, "transition relation:\n");
      while(q) {
        fprintf(nusmv_stderr, "partition %d: size = %d\n", i++,
                add_size(dd_manager, (add_ptr)car(q)));
        q = cdr(q);
      }
    }

    /* Yuan Lu : generate quantifier list for *_orig */
    {
      add_ptr cp_vars = add_dup(state_variables_add);
      add_ptr next_cp_vars = add_dup(next_state_variables_add);
      node_ptr sl = make_support_list(cp_trans_add_orig);

      /* Quantification should also consider the input variables. */
      add_and_accumulate(dd_manager, &cp_vars, input_variables_add);
      add_and_accumulate(dd_manager, &next_cp_vars, next_input_variables_add);

      forward_quantifiers_add_orig = make_quantifiers(sl, cp_vars);
      reverse_quantifiers_add_orig = make_quantifiers(sl, next_cp_vars);
      add_free(dd_manager, cp_vars);
      add_free(dd_manager, next_cp_vars);
      walk_dd(dd_manager, (VPFDD)add_free, sl);
    }

    { 
      add_ptr cp_vars = add_dup(state_variables_add);
      add_ptr next_cp_vars = add_dup(next_state_variables_add);
      node_ptr sl = make_support_list(cp_trans_add);

      /* Quantification should also consider the input variables. */
      add_and_accumulate(dd_manager, &cp_vars, input_variables_add);
      add_and_accumulate(dd_manager, &next_cp_vars, next_input_variables_add);

      forward_quantifiers_add = make_quantifiers(sl, cp_vars);
      reverse_quantifiers_add = make_quantifiers(sl, next_cp_vars);
      add_free(dd_manager, cp_vars);
      add_free(dd_manager, next_cp_vars);
      walk_dd(dd_manager, (VPFDD)add_free, sl);
    }

    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating normal assignments", context);
    set_assignment_type_assign();
    tmp_1 = eval_simplify(assign_expr, Nil, assumption);
    if (opt_verbose_level_gt(options, 1))
      print_in_process("evaluating INVAR statements", context);
    tmp_2 = eval_simplify(invar_expr, Nil, assumption);
    add_and_accumulate(dd_manager, &tmp_1, tmp_2);
    add_free(dd_manager, tmp_2);

    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "size of invariant set = %g states, %d BDD nodes\n",
              add_count_states(dd_manager, tmp_1), add_size(dd_manager, tmp_1));

    tmp_2  = add_imply(dd_manager, running_add, tmp_1);
    add_and_accumulate(dd_manager, &invar_add_orig, tmp_2);
    add_free(dd_manager, tmp_1);
    add_free(dd_manager, tmp_2);
    add_free(dd_manager, running_add);
  }
  if (opt_verbose_level_gt(options, 0) && cdr(procs)) {
    fprintf(nusmv_stderr, "size of global initial set = %d  BDD nodes\n",
            add_size(dd_manager, init_add));
    print_conj_part_info();
  }
  invar_bdd_orig = add_to_bdd(dd_manager, invar_add_orig);
  /* Yuan Lu */
  invar_bdd = Abs_AbsAbstractCur(invar_bdd_orig, abs_expr);

  input_variables_bdd = add_to_bdd(dd_manager, input_variables_add);
  next_input_variables_bdd = add_to_bdd(dd_manager, next_input_variables_add);
  state_variables_bdd = add_to_bdd(dd_manager, state_variables_add);
  next_state_variables_bdd = add_to_bdd(dd_manager, next_state_variables_add);
  forward_quantifiers_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd, forward_quantifiers_add);
  reverse_quantifiers_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd, reverse_quantifiers_add);
  cp_trans_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd, cp_trans_add);

  /* Yuan Lu */
  forward_quantifiers_bdd_orig = map_dd(dd_manager, (NPFDD)add_to_bdd, forward_quantifiers_add_orig);
  reverse_quantifiers_bdd_orig = map_dd(dd_manager, (NPFDD)add_to_bdd, reverse_quantifiers_add_orig);
  cp_trans_bdd_orig = map_dd(dd_manager, (NPFDD)add_to_bdd, cp_trans_add_orig);

  add_free(dd_manager, invar_add_orig);
  walk_dd(dd_manager, (VPFDD)add_free, cp_trans_add);
  walk_dd(dd_manager, (VPFDD)add_free, forward_quantifiers_add);
  walk_dd(dd_manager, (VPFDD)add_free, reverse_quantifiers_add);
  if (opt_verbose_level_gt(options, 3)) print_conj_part_detailed_info();
}

/**Function********************************************************************

  Synopsis           [Prints the BDD size of the transitions.]

  Description        [Prints the BDD size of the transitions.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_conj_part_info()
{
  int i = 0;
  node_ptr l = cp_trans_bdd;
  
  if (cp_trans_bdd != Nil)
    (void) fprintf(nusmv_stdout, "Conjunctive Partitioning:\n");
  while(l != Nil) {
    (void) fprintf(nusmv_stdout, "partition %d: %d BDD nodes.\n", i++,
                   (bdd_ptr)bdd_size(dd_manager, (bdd_ptr)car(l)));
    l = cdr(l);
  }
  if (invar_bdd != NULL)
    (void)fprintf(nusmv_stdout, "Size of invariant: %d BDD nodes.\n",
                  bdd_size(dd_manager, invar_bdd));
}

/**Function********************************************************************

  Synopsis           [Prints the symbolic variables of each partition and each cluster.]

  Description        [Prints the symbolic variables of each partition and each cluster.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_conj_part_detailed_info(void) 
{
  fprintf(nusmv_stdout, "\n**********************************\n");
  fprintf(nusmv_stdout, "The set of FWD variables are:\n");
  {
    node_ptr l = forward_quantifiers_bdd;
    int i = 0;

    while (l != Nil) {
      i++;
      fprintf(nusmv_stdout, "\n*************** EXISTS FWD Partition %d ********************\n",i);
      print_state_vars(dd_manager, (bdd_ptr)car(l), state_variables);
      fprintf(nusmv_stdout, "\n");
      l = cdr(l);
    }
  }
  fprintf(nusmv_stdout, "\n**********************************\n");
  fprintf(nusmv_stdout, "The set of support of CP_TRANS are :\n");
  {
    node_ptr l = cp_trans_bdd;
    int i = 0;

    while (l != Nil) {
      bdd_ptr a = bdd_support(dd_manager, (bdd_ptr)car(l));

      i++;
      fprintf(nusmv_stdout, "\n*************** CP TRANS Partition %d ********************\n",i);
      print_state_vars(dd_manager, a, state_variables);
      fprintf(nusmv_stdout, "\n");
      l = cdr(l);
      add_free(dd_manager, a);
    }
  }
  fprintf(nusmv_stdout, "\n**********************************\n");
  fprintf(nusmv_stdout, "The set of current BWD variables are:\n");
  {
    node_ptr l = reverse_quantifiers_bdd;
    int i = 0;

    while (l != Nil) {
      i++;
      fprintf(nusmv_stdout, "\n*************** EXIST BWD Partition %d ********************\n",i);
      print_state_vars(dd_manager, (bdd_ptr)car(l), state_variables);
      fprintf(nusmv_stdout, "\n");
      l = cdr(l);
    }
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Recursively build the portioned transition relation.]

  Description        [Recursively build the portioned transition
  relation. When the size of the current cluster is greater than the
  threshold found in the options, than a new cluster is created. The
  partitions are simplified using the given <tt>assumption<tt>.]

  SideEffects        []

  SeeAlso            [Compile_CompileModelConj]

******************************************************************************/
static node_ptr compileCompileModelConjRecur(node_ptr trans_list, node_ptr n, node_ptr context,
                                     add_ptr assumption)
{
  
  if (n == Nil) return(trans_list);
  yylineno = n->lineno;
  switch(node_get_type(n)) {
  case AND: {
      trans_list = compileCompileModelConjRecur(trans_list, car(n), context, assumption);
      trans_list = compileCompileModelConjRecur(trans_list, cdr(n), context, assumption);
      return(trans_list);
  }
  case CONTEXT:
    return(compileCompileModelConjRecur(trans_list, cdr(n), car(n), assumption));
  default:
    {
      add_ptr one = add_one(dd_manager);
      add_ptr z   = eval(n, context);

      add_free(dd_manager, one);
      if (z == one) {
        add_free(dd_manager, z);
        return(trans_list);
      }
      {
        add_ptr tmp  = add_simplify_assuming(dd_manager, z, assumption);
        int threshold = get_conj_part_threshold(options);

        add_free(dd_manager, z);
        z = tmp;
        if ((add_size(dd_manager, z) > threshold) ||
            (add_size(dd_manager, (add_ptr)car(trans_list)) > threshold))
          return(cons((node_ptr)z, trans_list));
        else{
          add_ptr tmp = add_and(dd_manager, (add_ptr)car(trans_list), z);

          add_free(dd_manager, z);
          add_free(dd_manager, trans_list->left.bddtype);
          trans_list->left.bddtype = tmp;
          return(trans_list);
        }
      }
    }
  }
}

/**Function********************************************************************

  Synopsis           [Computes the cube to be quantified for each cluster.]

  Description        [Computes the cube to be quantified for each
  cluster. Given a list of cubes <tt>l</tt>, and a cube
  <tt>vars</tt>, then for each element in the list, the cube
  difference from it and <tt>vars</tt> is computed and returned back.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr make_quantifiers(node_ptr l, add_ptr vars)
{
  add_ptr tmp;

  if (l == Nil) {
    add_ref(vars);
    return(cons((node_ptr)vars, Nil));
  }
  tmp = add_cube_diff(dd_manager, vars, (add_ptr)car(l));
  return(cons((node_ptr)tmp, make_quantifiers(cdr(l), vars)));
}

/**Function********************************************************************

  Synopsis           [Given a list of {A|B}DDs then return the list of
  the corresponding set of support.]

  Description        [Given a list of {A|B}DDs then return the list of
  the corresponding set of support, i.e. the cube of the {A|B}DD
  boolean variables the {A|B}DD depends on.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr make_support_list(node_ptr l)
{
  if (l == Nil) return(l);
  {
    node_ptr r = make_support_list(cdr(l));
    add_ptr s = add_support(dd_manager, (add_ptr)car(l));

    if (r != Nil) {
      add_ptr tmp = add_and(dd_manager, (add_ptr)car(r), s);

      add_free(dd_manager, s);
      return(cons((node_ptr)tmp, r));
    }
    return(cons((node_ptr)s, Nil));
  }
}
