/**CFile***********************************************************************

  FileName    [compileCmd.c]

  PackageName [compile]

  Synopsis    [Shell interface for the compile package.]

  Description [This file contains the interface of the compile package
  with the interactive shell.]

  SeeAlso     [cmdCmd.c]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@@irst.itc.it ]

******************************************************************************/

#include "compileInt.h"

static char rcsid[] UTIL_UNUSED = "$Id:$";

/* prototypes of the command functions */
int CommandProcessModel ARGS((int argc, char **argv));
int CommandFlattenHierarchy ARGS((int argc, char **argv));
int CommandBuildVariables ARGS((int argc, char **argv));
int CommandBuildModel ARGS((int argc, char **argv));
int CommandComputeFairnessConstraints ARGS((int argc, char **argv));
int CommandDumpModel ARGS((int argc, char **argv));
int CommandAddTrans ARGS((int argc, char **argv));
int CommandAddInit ARGS((int argc, char **argv));
int CommandAddFairness ARGS((int argc, char **argv));
int CommandRestoreModel ARGS((int argc, char **argv));
int CommandWriteOrder ARGS((int argc, char **argv));
int CommandIwls95PrintOption ARGS((int argc, char **argv));
int CommandIwls95PrintClusterInfo ARGS((int argc, char **argv));
int CommandGo ARGS((int argc, char **argv));
int CommandGetInternalStatus ARGS((int argc, char **argv));

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [It's used to store the original value of the <tt>trans_bdd</tt>.]

  Description [It's used to store the original value of the <tt>trans_bdd</tt>.]

  SeeAlso     [trans_bdd]

******************************************************************************/
static bdd_ptr orig_trans_bdd;

/**Variable********************************************************************

  Synopsis    [It's used to store the original value of the <tt>init_bdd</tt>.]

  Description [It's used to store the original value of the <tt>init_bdd</tt>.]

  SeeAlso     [init_bdd]

******************************************************************************/
static bdd_ptr orig_init_bdd;

/**Variable********************************************************************

  Synopsis    [It's used to store the original value of the
  <tt>fairness_constraints_bdd</tt>.]

  Description [It's used to store the original value of the
  <tt>fairness_constraints_bdd</tt>.]

  SeeAlso     [init_bdd]

******************************************************************************/
static node_ptr orig_fairness_constraints_bdd;

/**Variable********************************************************************

  Synopsis    [Flag to indicate if the interactive model inspection
  feature is enabled.]

  Description [Flag to indicate if the interactive model inspection
  feature is enabled.]

******************************************************************************/
static int interactive_mode = 0;
static void enable_interactive_mode() {interactive_mode = 1;}
static void disable_interactive_mode() {interactive_mode = 0;}
static int interactive_mode_enabled() {return((interactive_mode == 1));}
void reset_interactive_mode() {interactive_mode = 0;}

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int UsageProcessModel ARGS((void));
static int UsageFlattenHierarchy ARGS((void));
static int UsageBuildModel ARGS((void));
static int UsageBuildVariables ARGS((void));
static int UsageComputeFairnessConstraints ARGS((void));
static int UsageDumpModel ARGS((void));
static int UsageAddTrans ARGS((void));
static int UsageAddInit ARGS((void));
static int UsageAddFairness ARGS((void));
static int UsageRestoreModel ARGS((void));
static int UsageWriteOrder ARGS((void));
static int UsageIwls95PrintOption ARGS((void));
static int UsageIwls95PrintClusterInfo ARGS((void));
static int UsageGo ARGS((void));
static int UsageGetInternalStatus ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the compile package.]

  Description        [Initializes the compile package.]

  SideEffects        []

******************************************************************************/
void Compile_Init(void)
{
  Compile_EncodeVarsInit();
  cmps = cmp_struct_init();
  Cmd_CommandAdd("process_model", CommandProcessModel, 0);
  Cmd_CommandAdd("flatten_hierarchy", CommandFlattenHierarchy, 0);
  Cmd_CommandAdd("build_variables", CommandBuildVariables, 0);
  Cmd_CommandAdd("build_model", CommandBuildModel, 0);
  Cmd_CommandAdd("compute_fairness", CommandComputeFairnessConstraints, 0);
  Cmd_CommandAdd("dump_model", CommandDumpModel, 0);
  Cmd_CommandAdd("add_trans", CommandAddTrans, 0);
  Cmd_CommandAdd("add_init", CommandAddInit, 0);
  Cmd_CommandAdd("add_fairness", CommandAddFairness, 0);
  Cmd_CommandAdd("restore_model", CommandRestoreModel, 0);
  Cmd_CommandAdd("write_order", CommandWriteOrder, 0);
  Cmd_CommandAdd("print_iwls95options", CommandIwls95PrintOption, 0);
  Cmd_CommandAdd("print_iwls95info", CommandIwls95PrintClusterInfo, 0);
  Cmd_CommandAdd("go", CommandGo, 0);
  Cmd_CommandAdd("get_internal_status", CommandGetInternalStatus, 0);
}

/**Function********************************************************************

  Synopsis           [Shut down the compile package]

  Description        [Shut down the compile package]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Compile_End(void)
{
  if (frame_add != (add_ptr)NULL) add_free(dd_manager, frame_add);
  if (invar_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, invar_bdd);
  if (init_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, init_bdd);
  if (input_variables_add != (add_ptr)NULL) add_free(dd_manager, input_variables_add);
  if (next_input_variables_add != (add_ptr)NULL) add_free(dd_manager, next_input_variables_add);
  if (state_variables_add != (add_ptr)NULL) add_free(dd_manager, state_variables_add);
  if (next_state_variables_add != (add_ptr)NULL) add_free(dd_manager, next_state_variables_add);
  if (input_variables_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, input_variables_bdd);
  if (next_input_variables_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, next_input_variables_bdd);
  if (state_variables_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, state_variables_bdd);
  if (next_state_variables_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, next_state_variables_bdd);
  walk_dd(dd_manager, add_free, real_state_variables);
  if (trans_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, trans_bdd);
  walk_dd(dd_manager, bdd_free, dp_trans_bdd);
  walk_dd(dd_manager, bdd_free, cp_trans_bdd);
  walk_dd(dd_manager, bdd_free, forward_quantifiers_bdd);
  walk_dd(dd_manager, bdd_free, reverse_quantifiers_bdd);
  Iwls95FreeClustersList(dd_manager, Iwls95_Fwd_trans);
  Iwls95FreeClustersList(dd_manager, Iwls95_Bwd_trans);
  free_state_vars();
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Performs the batch steps and then returns
  control to the interactive shell.]

  CommandName        [process_model] 	   

  CommandSynopsis    [Performs the batch steps and then return control
  to the interactive shell.]  

  CommandArguments   [\[-h\] \[-i model-file\] \[-m Method\]]  

  CommandDescription [ Reads the model, compiles it into BDD and
  performs the model checking of all the specification contained in
  it. If the environment variable <tt>forward_search</tt> has been set
  before, then the set of reachable states is computed. If the
  environment variables <tt>enable_reorder</tt> and
  <tt>reorder_method</tt> are set, then the reordering of variables is
  performed accordingly. This command simulates the batch behavior of
  NuSMV and then returns the control to the interactive shell.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-i model-file</tt>
       <dd> Sets the environment variable <tt>input_file</tt> to file
           <tt>model-file</tt>, and reads the model from file
           <tt>model-file</tt>.
    <dt> <tt>-m Method</tt>
       <dd> Sets the environment variable <tt>partition_method</tt> to
       <tt>Method</tt> and uses it as partitioning method.
  </dl>
  ]  

  SideEffects        []

******************************************************************************/
int CommandProcessModel(int argc, char **argv)
{
  int c;
  char * partition_method = NIL(char);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"i:m:h")) != EOF){
    switch(c){
    case 'i': {
      set_input_file(options, util_optarg);
      break;
    }
    case 'm': {
      partition_method = ALLOC(char, strlen(util_optarg)+1);
      strcpy(partition_method, util_optarg);
      break;
    }
    case 'h': return(UsageProcessModel());
    default:  return(UsageProcessModel());
    }
  }

  if (argc != util_optind) return(UsageProcessModel());

  if (get_input_file(options) == (char *)NULL) {
    fprintf(nusmv_stderr, "Input file is (null). You must set the input file before.\n");
    return 1;
  }

  if (partition_method != NIL(char)) {
    if (str2method(partition_method) != -1) {
      set_partition_method(options, str2method(partition_method));
    } else {
      fprintf(nusmv_stderr, "The only possible values for \"-m\" option are:\n\t");
      print_partition_method(nusmv_stderr);
      fprintf(nusmv_stderr, "\n");
      return 1;
    }
  }
  
  if (Cmd_CommandExecute("read_model")) return 1;

  if (Cmd_CommandExecute("flatten_hierarchy")) return 1;

  if (Cmd_CommandExecute("build_variables")) return 1;

  if (Cmd_CommandExecute("build_model")) return 1;

  /* We avoid calling the corresponding shell command for simplicity */
  if (opt_forward_search(options)) 
    if (Cmd_CommandExecute("compute_reachable")) return 1;

  if (Cmd_CommandExecute("compute_fairness")) return 1;

  if (opt_check_trans(options)) 
    if (Cmd_CommandExecute("check_trans")) return 1;

  if (Cmd_CommandExecute("check_spec")) return 1;

  if (Cmd_CommandExecute("compute")) return 1;

  if (Cmd_CommandExecute("check_ltlspec")) return 1;

  if (Cmd_CommandExecute("check_invar")) return 1;

  if (Cmd_CommandExecute("_check_invar_fb")) return 1;

  if (Cmd_CommandExecute("_check_invar_strong")) return 1;

  if (Cmd_CommandExecute("_check_after")) return 1;

  if (Cmd_CommandExecute("print_usage")) return 1;

  if (opt_reorder(options)) { /* If the case activate reordering */
    fprintf(nusmv_stdout, "\n========= starting reordering ============\n");
    dd_reorder(dd_manager, get_reorder_method(options), DEFAULT_MINSIZE);

    if (Cmd_CommandExecute("write_order")) return 1;

    fprintf(nusmv_stdout, "\n========= after reordering ============\n");

    if (Cmd_CommandExecute("print_usage")) return 1;

  }
  return 0;
}

static int UsageProcessModel()
{
  (void) fprintf(nusmv_stderr, "usage: process_model [-h] [-i ifile] [-m method]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -i ifile \tReads the model from file \"ifile\".\n");
  (void) fprintf(nusmv_stderr, "   -m method\t Uses \"method\" as partition method in model construction.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Flattens the hierarchy of modules]

  CommandName        [flatten_hierarchy] 	   

  CommandSynopsis    [Flattens the hierarchy of modules]  

  CommandArguments   [\[-h\]]  

  CommandDescription [
  This command is responsible of the instantiation of modules and
  processes. The instantiation is performed by substituting the actual
  parameters for the formal parameters, and then prefixing the result by
  the instance name.]  

  SideEffects        []

******************************************************************************/
int CommandFlattenHierarchy(int argc, char ** argv)
{
  int c;

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageFlattenHierarchy());
    default:  return(UsageFlattenHierarchy());
    }
  }
  if (argc != util_optind) return(UsageFlattenHierarchy());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr, "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }

  if (cmp_struct_get_flatten_hrc(cmps)) {
    fprintf(nusmv_stderr, "The hierarchy has been already flattened.\n");
    return 1;
  }
  if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, "Flattening hierarchy.....");

  init_module_hash();

  /*
    Processing of the parse tree and constructions of all the
    expressions for the state machine(s). Here the expansions are
    performed so that modules and processes are created. The expansion
    of modules is such that the formal parameters (if any) are
    replaced by the actual ones and the machine is replicated.
  */
  {
    node_ptr init_expr         = Nil;
    node_ptr invar_expr        = Nil;
    node_ptr trans_expr        = Nil;
    node_ptr assign_expr       = Nil;
    node_ptr procs_expr        = Nil;
    node_ptr fair_expr         = Nil;
    node_ptr abs_expr          = Nil;
    node_ptr spec_expr         = Nil;
    node_ptr ltlspec_expr      = Nil;
    node_ptr invar_spec_expr   = Nil;
    node_ptr invar_fb_expr     = Nil;
    node_ptr invar_strong_expr = Nil;
    node_ptr after_expr        = Nil;

    Compile_FlattenHierarchy(sym_intern("main"), Nil, &trans_expr, &init_expr, &invar_expr,
                    &spec_expr, &ltlspec_expr, &invar_spec_expr, &invar_fb_expr,
                    &invar_strong_expr, &after_expr, &fair_expr, &abs_expr,
		    &assign_expr, &procs_expr, Nil);

    all_symbols = reverse(all_symbols);
    state_variables   = reverse(state_variables);
    input_variables = reverse(input_variables);
    all_variables = append_ns(input_variables, state_variables);
    cmp_struct_set_init(cmps, init_expr);
    cmp_struct_set_invar(cmps, invar_expr);
    cmp_struct_set_trans(cmps, trans_expr);
    cmp_struct_set_assign(cmps, assign_expr);
    cmp_struct_set_procs(cmps, procs_expr);
    cmp_struct_set_fair(cmps, reverse(fair_expr));
    cmp_struct_set_spec(cmps, reverse(spec_expr));
    cmp_struct_set_ltlspec(cmps, reverse(ltlspec_expr));
    cmp_struct_set_invar_spec(cmps, reverse(invar_spec_expr));
    cmp_struct_set_invar_fb(cmps, reverse(invar_fb_expr));
    cmp_struct_set_invar_strong(cmps, reverse(invar_strong_expr));
    cmp_struct_set_after(cmps, reverse(after_expr));
  }
  cmp_struct_set_flatten_hrc(cmps);
  if (opt_verbose_level_gt(options, 0))
      (void) fprintf(nusmv_stderr, ".....done\n");

  return 0;
}


static int UsageFlattenHierarchy()
{
  (void) fprintf(nusmv_stderr, "usage: flatten_hierarchy [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  return 1;
}
  
/**Function********************************************************************

  Synopsis           [Builds the BDD variables necessary to compile the
  model into BDD.]

  CommandName        [build_variables]

  CommandSynopsis    [Builds the BDD variables necessary to compile the
  model into BDD.]  

  CommandArguments   [\[-h\] \[-i order-file\]]  

  CommandDescription [
  Generates the boolean BDD variables and ADD needed to encode
  propositionally the (symbolic) variables declared in the model.<br>

  The variables are created as default in the order in which they
  appear in a depth first traversal of the hierarchy.<p>

  The input order file can be partial and can contain variables not
  declared in the model. Variables not declared in the model are
  simply discarded. The variables declared in the model not
  listed in the input file are created at the end of the given
  ordering following the default ordering.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-i order-file</tt>
       <dd> Sets the environment variable <tt>input_order_file</tt> to
       <tt>order-file</tt>, and reads the variable ordering to be used from
       file <tt>order-file</tt>. This can be combined with the
       <tt>write_order</tt> command. The variable ordering is written to a file,
       which can be inspected and reordered by the user, and then read back in.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandBuildVariables(int argc, char ** argv)
{
  int c;
  char * input_order_file_name = NIL(char);
  char * tmp_input_order_file_name = NIL(char);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"i:h")) != EOF){
    switch(c){
    case 'i': {
      input_order_file_name = ALLOC(char, strlen(util_optarg)+1);
      strcpy(input_order_file_name, util_optarg);
      break;
    }    
    case 'h': return(UsageBuildVariables());
    default:  return(UsageBuildVariables());
    }
  }
  if (argc != util_optind) return(UsageBuildVariables());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps)) {
    fprintf(nusmv_stderr, "The variables appear to be already built.\n");
    return 1;
  }

  if (opt_verbose_level_gt(options, 0)) {
    (void) fprintf(nusmv_stderr, "Building variables.....\n");
  }

  if (input_order_file_name != NIL(char))
    set_input_order_file(options, input_order_file_name);
  /*
    Builds the ADD containing the list of state variables
    (i.e. vars).
    Side effect on symbol_hash.
  */
  create_process_symbolic_variables(cmp_struct_get_procs(cmps));
  Compile_BuildVarsBdd();

  /* Builds the ADD list representing real_state_variables */
  build_real_state_variables();
  
  cmp_struct_set_build_variables(cmps);
  if (opt_verbose_level_gt(options, 0)) {
    (void) fprintf(nusmv_stderr, "....done\n");
  }
  return 0;
}

static int UsageBuildVariables()
{
  (void) fprintf(nusmv_stderr, "usage: build_variables [-h] [-i <file>]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -i <file> \tReads variable ordering from file <file>.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Compiles the flattened hierarchy into BDD]

  CommandName        [build_model] 	   

  CommandSynopsis    [Compiles the flattened hierarchy into BDD]  

  CommandArguments   [\[-h\] \[-f\] \[-m Method\]]  

  CommandDescription [
  Compiles the flattened hierarchy into BDD (initial states, invariants,
  and transition relation) using the method specified in the environment
  variable <tt>partition_method</tt> for building the transition relation.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m Method</tt>
       <dd> Sets the environment variable <tt>partition_method</tt> to
           the value <tt>Method</tt>, and then builds the transition
           relation. Available methods are <code>Monolithic</code>,
           <code>Disjunctive</code>, <code>Conjunctive</code> and
           <code>Iwls95CP</code>.
    <dt> <tt>-f</tt>
       <dd> Forces model construction. By default, only one partition
            method is allowed. This option allows to overcome this
            default, and to build the transition relation with different
            partitionings.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandBuildModel(int argc, char ** argv)
{
  int c;
  int Build_Model_Force = 0;
  char * partition_method = NIL(char);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"m:fh")) != EOF){
    switch(c){
    case 'm': {
      partition_method = ALLOC(char, strlen(util_optarg)+1);
      strcpy(partition_method, util_optarg);
      break;
    }
    case 'f': {
      if (cmp_struct_get_build_model(cmps) == 0) {
        fprintf(nusmv_stderr, "You cannot use the \"-f\" without having built a model before.\n");
        return(1);
      }
      Build_Model_Force = 1;
      break;
    }
    case 'h': return(UsageBuildModel());
    default:  return(UsageBuildModel());
    }
  }
  if (argc != util_optind) return(UsageBuildModel());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }

  if (Build_Model_Force != 1) {
    if (cmp_struct_get_build_model(cmps)) {
      fprintf(nusmv_stderr, "A model appear to be already built from file: %s.\n",
              get_input_file(options));
      return 1;
    }
  }

  if (partition_method != NIL(char)) {
    if (str2method(partition_method) != -1) {
      set_partition_method(options, str2method(partition_method));
    } else {
      fprintf(nusmv_stderr, "The only possible values for \"-m\" option are:\n\t");
      print_partition_method(nusmv_stderr);
      fprintf(nusmv_stderr, "\n");
      return 1;
    }
  }
  
  /*
    Builds the ADD containing the list of state variables.
    Side effects on:
    - process_selector_add
    - symbol_hash (adds all <process>.running ADD leaves)
    If we force a model construction we does not need to rebuild
    process selector (the forcing can be only performed if a previous
    call to build_model has been performed).
  */
  if (cmp_struct_get_process_selector(cmps) == 0)
    build_proc_selector(cmp_struct_get_procs(cmps));
  cmp_struct_set_process_selector(cmps);

  /* Builds the ADD representing frame-axioms */
  if (cmp_struct_get_build_frames(cmps) == 0)
    Compile_CompileFrame();
  cmp_struct_set_build_frames(cmps);
  /*
    Checking for the correctness of the state machines which have
    been declared. It should not mention DD manipulation.
  */
  if (cmp_struct_get_compile_check(cmps) == 0)
    Compile_CheckProgram(cmp_struct_get_procs(cmps), cmp_struct_get_spec(cmps),
                         cmp_struct_get_ltlspec(cmps), cmp_struct_get_invar_spec(cmps),
                         cmp_struct_get_fair(cmps));
  cmp_struct_set_compile_check(cmps);
  /*
    Builds the ADD representing the Initial States. This is obtained
    by conjoining the init in different modules and processes.
  */
  if (cmp_struct_get_build_init(cmps) == 0)
    Compile_CompileInit(cmp_struct_get_init(cmps), cmp_struct_get_procs(cmps));
  cmp_struct_set_build_init(cmps);
  /* Builds the ADD representing the Transition Relation */
  {
    add_ptr one = add_one(dd_manager);

    Compile_CompileModel(cmp_struct_get_trans(cmps), cmp_struct_get_invar(cmps),
                cmp_struct_get_procs(cmps), one);
    add_free(dd_manager, one);
  }

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr,
            "\nThe model has been built from file %s.\n", get_input_file(options));
  }
  cmp_struct_set_build_model(cmps);
  return 0;
}

static int UsageBuildModel()
{
  (void) fprintf(nusmv_stderr, "usage: build_model [-h] [-f] [-m Method]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m Method \tUses this as partitioning method, and set it as default method\n");
  (void) fprintf(nusmv_stderr, "\t\tto be used in the following image computations.\n");
  (void) fprintf(nusmv_stderr, "\t\tActually the available methods are:\n\t\t");
  (void) print_partition_method(nusmv_stderr);
  (void) fprintf(nusmv_stderr, "\n   -f \t\tForces the model construction (can be only used if a model has been previously built).\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Evaluates the fairness constraints]

  CommandName        [compute_fairness] 	   

  CommandSynopsis    [Evaluates the fairness constraints]  

  CommandArguments   [\[-h\]]  

  CommandDescription [
  For each fairness constraint, computes the corresponding set of
  states.]  

  SideEffects        []

******************************************************************************/
int CommandComputeFairnessConstraints(int argc, char ** argv)
{
  int c;
  

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageComputeFairnessConstraints());
    default:  return(UsageComputeFairnessConstraints());
    }
  }
  if (argc != util_optind) return(UsageComputeFairnessConstraints());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (cmp_struct_get_fairness(cmps) == 0) {
    if (opt_verbose_level_gt(options, 0)){
      fprintf(nusmv_stderr, "Starting computation of Fairness Constraints....\n");
    }
    compute_fairness_constraints(cmp_struct_get_fair(cmps));
    if (opt_verbose_level_gt(options, 0)){
      fprintf(nusmv_stderr, "done.\n");
    }
    cmp_struct_set_fairness(cmps);
  }
  else {
    fprintf(nusmv_stderr, "Fairness Constraints already computed.\n");
  }
  return 0;
}


static int UsageComputeFairnessConstraints()
{
  (void) fprintf(nusmv_stderr, "usage: compute_fairness_constraints [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Saves the current internal representation of the model]

  CommandName        [dump_model]

  CommandSynopsis    [Saves the current internal representation of the model]  

  CommandArguments   [\[-h\]]  

  CommandDescription [Initializes the system to allow interactive
  model modification. This command is only available if it
  has been chosen the <code>Monolithic</code> partition method.]

  SideEffects        []

******************************************************************************/
int CommandDumpModel(int argc, char **argv)
{
  int c;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageDumpModel());
    default:  return(UsageDumpModel());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
    
  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }
  if (!opt_monolithic(options)) {
    (void) fprintf(nusmv_stderr, "The dump of the model can be used only with \"Monolithic\" partition method.\n");
    (void) fprintf(nusmv_stderr, "Use \"build_model -f -m Monolithic\" to build the transition relation.\n");
    return(1);
  }
  if (interactive_mode_enabled() == 0) {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Activated Interactive Modification of the model.\n");
    }
    enable_interactive_mode();
    orig_trans_bdd = bdd_dup(trans_bdd);
    orig_init_bdd  = bdd_dup(init_bdd);
    orig_fairness_constraints_bdd = fairness_constraints_bdd;
  }
  else {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Interactive Modification of the model already activated.\n");
    }
  }
  return(0);
}


static int UsageDumpModel()
{
  (void) fprintf(nusmv_stderr, "usage: dump_model [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Modifies the transition relation]

  CommandName        [add_trans] 	   

  CommandSynopsis    [Modifies the transition relation]  

  CommandArguments   [\[-h\] | trans-expression]  

  CommandDescription [Modifies the transition relation.
  <tt>trans-expression</tt> is evaluated and then conjoined with the
  current transition relation. This command can be used only if the
  partitioning method is <code>Monolithic</code>. This command has the
  same effect of a <tt>TRANS</tt> statement in the original input
  file. Before executing this command you have to execute the
  <code>dump_model</code> command.]  

  SideEffects        []

******************************************************************************/
int CommandAddTrans(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int status = 0;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c){
    case 'h': return(UsageAddTrans());
    default:  return(UsageAddTrans());
    }
  }
  if (argc == 1) return(UsageAddTrans());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
  if (!interactive_mode_enabled()) {
    fprintf(nusmv_stderr, "You must execute the \"dump_model\" command before.\n");
    return 1;
  }
  {
    int i;
    node_ptr parsed_command = Nil;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "TRANS ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == TRANS)) {
        node_ptr trans_expr = car(parsed_command);
        add_ptr t = eval(trans_expr, Nil);
        bdd_ptr t_bdd = add_to_bdd(dd_manager, t);

        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "The size of the original TRANS is: %d BDD nodes\n", bdd_size(dd_manager, trans_bdd));
          fprintf(nusmv_stderr, "The size of the evaluated formula is: %d BDD nodes\n", bdd_size(dd_manager, t_bdd));
        }
        bdd_and_accumulate(dd_manager, &trans_bdd, t_bdd);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "The size of the new TRANS is: %d BDD nodes\n", bdd_size(dd_manager, trans_bdd));
        }
        add_free(dd_manager, t);
        bdd_free(dd_manager, t_bdd);
        if (fair_states_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, fair_states_bdd);
        fair_states_bdd = (bdd_ptr)NULL;
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"add_trans trans-expr\".\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"add_trans trans-expr\".\n");
      status = 1;
    }
  }
  return(status);
}

static int UsageAddTrans()
{
  (void) fprintf(nusmv_stderr, "usage: add_trans [-h] | trans-expr\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  (void) fprintf(nusmv_stderr, "   trans-expr\tEvaluates the \"trans-expr\" and updates the transition relation accordingly.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Modifies the set of initial states.]

  CommandName        [add_init] 	   

  CommandSynopsis    [Modifies the set of initial states.]  

  CommandArguments   [\[-h\] | simple-expression]  

  CommandDescription [Modifies the initial states. 
  <tt>simple-expression</tt> is evaluated and conjoined with
  the set of initial states to form the new set of initial
  states. This command has the same effect of an <tt>INIT</tt>
  declaration in the original input file. Before executing this
  command you have to execute the <code>dump_model</code> command.]

  SideEffects        []

******************************************************************************/
int CommandAddInit(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int status = 0;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c) {
    case 'h': return(UsageAddInit());
    default:  return(UsageAddInit());
    }
  }
  if (argc == 1) return(UsageAddInit());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
  if (!interactive_mode_enabled()) {
    fprintf(nusmv_stderr, "You must execute the \"dump_model\" command before.\n");
    return 1;
  }
  {
    int i;
    node_ptr parsed_command = Nil;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "INIT ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == INIT)) {
        node_ptr init_expr = car(parsed_command);
        add_ptr i = eval(init_expr, Nil);
        bdd_ptr i_bdd = add_to_bdd(dd_manager, i);

        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "The size of the original INIT is: %d BDD nodes\n", bdd_size(dd_manager, init_bdd));
          fprintf(nusmv_stderr, "The size of the evaluated formula is: %d BDD nodes\n", bdd_size(dd_manager, i_bdd));
        }
        bdd_and_accumulate(dd_manager, &init_bdd, i_bdd);
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "The size of the new INIT is: %d BDD nodes\n", bdd_size(dd_manager, init_bdd));
        }
        add_free(dd_manager, i);
        bdd_free(dd_manager, i_bdd);
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"add_init simple-expr\".\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"add_init simple-expr\".\n");
      status = 1;
    }
  }
  return(status);
}
  
static int UsageAddInit()
{
  (void) fprintf(nusmv_stderr, "usage: add_init [-h] | simple-expr\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  (void) fprintf(nusmv_stderr, "   simple-expr\tEvaluates \"simple-expr\" and updates the initial states accordingly.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Adds a fairness constraint to the model]

  CommandName        [add_fairness]

  CommandSynopsis    [Adds a fairness constraint to the model]

  CommandArguments   [\[-h\] | fair-expression]

  CommandDescription [Adds a new fairness constraint to the existing
  list of fairness constraints. <tt>fair-expression</tt> is evaluated and
  added to the current list of fairness constraints. This command has
  the same effect of a <tt>FAIRNESS</tt> statement in the original
  input file. Before executing this command you have to execute the
  <code>dump_model</code> command.]  

  SideEffects        []

******************************************************************************/
int CommandAddFairness(int argc, char **argv)
{
  int c;
  int offSet = 0;
  int status = 0;

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c) {
    case 'h': return(UsageAddFairness());
    default:  return(UsageAddFairness());
    }
  }
  if (argc == 1) return(UsageAddFairness());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
  if (!interactive_mode_enabled()) {
    fprintf(nusmv_stderr, "You must execute the \"dump_model\" command before.\n");
    return 1;
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }
  {
    int i;
    node_ptr parsed_command = Nil;

    for (i = offSet; i>0; i--) argv++;
    if (Parser_ReadFromString(argc - offSet, argv, "FAIRNESS ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == FAIRNESS)) {
        node_ptr fair_expr = car(parsed_command);
        bdd_ptr f_bdd = eval_spec(fair_expr, Nil);

        if (fair_states_bdd) bdd_free(dd_manager, fair_states_bdd);
        fair_states_bdd = (bdd_ptr)NULL;
        if (opt_verbose_level_gt(options, 0)) {
          fprintf(nusmv_stderr, "The size of the new FAIRNESS is: %d BDD nodes\n", bdd_size(dd_manager, f_bdd));
        }
        fairness_constraints_bdd = cons(find_node(BDD, (node_ptr)f_bdd, Nil),
                                        fairness_constraints_bdd);
        bdd_free(dd_manager, f_bdd);
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"add_fairness fair-expr\".\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"add_fairness fair-expr\".\n");
      status = 1;
    }
  }
  return(status);
}
  
static int UsageAddFairness()
{
  (void) fprintf(nusmv_stderr, "usage: add_fairness [-h] | fair-expr\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  (void) fprintf(nusmv_stderr, "   fair-expr\tEvaluates the \"fair-expr\" and updates the bdd of fairness constraints accordingly.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Restores the model after interactive model modification.]

  CommandName        [restore_model] 	   

  CommandSynopsis    [Restores the model after interactive model modification.]  

  CommandArguments   [\[-h\]]  

  CommandDescription [This command cancels the effects of all
  <code>add_fairness</code>, <code>add_init</code> and
  <code>add_trans</code> commands issued inside the model inspection
  session, i.e. after the <code>dump_model</code> command.]  

  SideEffects        []

******************************************************************************/
int CommandRestoreModel(int argc, char **argv)
 {
  int c;
  
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"h")) != EOF){
    switch(c) {
    case 'h': return(UsageRestoreModel());
    default:  return(UsageRestoreModel());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"build_variables\" command.\n");
    return 1;
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return 1;
  }
  if (cmp_struct_get_fairness(cmps) == 0) {
    fprintf(nusmv_stderr,
            "Fairness constraints have to be previously computed. Use the \"compute_fairness\" command.\n");
    return 1;
  }
  if (!interactive_mode_enabled()) {
    fprintf(nusmv_stderr, "You must execute the \"dump_model\" command before.\n");
    return 1;
  }

  if (fair_states_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, fair_states_bdd);
  fair_states_bdd = (bdd_ptr)NULL;
  bdd_free(dd_manager, trans_bdd);
  trans_bdd = orig_trans_bdd;
  bdd_free(dd_manager, init_bdd);
  init_bdd = orig_init_bdd;
  fairness_constraints_bdd = orig_fairness_constraints_bdd;
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Performed the restoring of the model.\n");
  }
  disable_interactive_mode();
  return(0);
}
  
static int UsageRestoreModel()
{
  (void) fprintf(nusmv_stderr, "usage: restore_model [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Writes variable order to file.]

  CommandName        [write_order] 	   

  CommandSynopsis    [Writes variable order to file.]  

  CommandArguments   [\[-h\] \[-f order-file\]]  

  CommandDescription [Writes the current order of BDD variables in the
  file specified in the environment variable <tt>output_order_file</tt>.

  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-f order-file</tt>
       <dd> Sets the environment variable <tt>output_order_file</tt>
       to <tt>order-file</tt> and then writes the ordering into it.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandWriteOrder(int argc, char **argv)
{
  int c;
  char * order_output_fname = NIL(char);
  /*
   * Parse the command line.
  */
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "f:h")) != EOF) {
    switch (c) {
    case 'h': return(UsageWriteOrder());
    case 'f':
      order_output_fname = ALLOC(char, strlen(util_optarg)+1);
      strcpy(order_output_fname, util_optarg);
      break;
    default:
      return(UsageWriteOrder());
    }
  }

  if (dd_manager == NIL(DdManager)) {
    (void) fprintf(nusmv_stderr, "The DD Manager has not been created yet.\n");
    return 1;
  }

  if (order_output_fname != NIL(char)) {
    set_output_order_file(options, order_output_fname);
    FREE(order_output_fname);
  }
  Compile_WriteOrder(get_output_order_file(options), 1);

  return 0;
}

static int UsageWriteOrder()
{
  (void) fprintf(nusmv_stderr, "usage: write_order [-h] | [-f <file>]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  (void) fprintf(nusmv_stderr, "   -f <file>\tWrites ordering to file <file>.\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Prints out the information of the Iwls95 clustering.]

  CommandName        [print_iwls95info]

  CommandSynopsis    [Print out the information of the Iwls95 clustering.]  

  CommandArguments   [\[-h\] \| \[-m\] \| \[-o output-file\]]

  CommandDescription [This command prints out the information
  regarding each cluster. In particular for each cluster it prints
  out, the cluster number, the size of the cluster (in BDD nodes), the
  variables occurring in it, the size of the cube that has to be
  quantified out relative to the cluster and the variables to be
  quantified out.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command through the
            program specified by the <tt>PAGER</tt> shell variable if
            defined, or through the UNIX utility "more".
    <dt> <tt>-o output-file</tt>
       <dd> Redirects the generated output to the file
            <tt>output-file</tt>.
  </dl>          
  ]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int CommandIwls95PrintClusterInfo(int argc, char ** argv)
{
  int c;
  int useMore = 0;
 char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);

  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "hmo")) != EOF) {
    switch (c) {
    case 'h':
      return(UsageIwls95PrintClusterInfo());
    case 'o':
      if (useMore == 1) return(UsageIwls95PrintClusterInfo());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageIwls95PrintClusterInfo());
      useMore = 1;
      break;
    default:
      return(UsageIwls95PrintClusterInfo());
    }
  } /* while */
  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
      nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        (void) fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      (void) fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  (void) fprintf(nusmv_stdout, "The forward Clusters are:\n");
  (void) Iwls95PrintClustersInfo(nusmv_stdout, dd_manager, Iwls95_Fwd_trans, state_variables);
  (void) fprintf(nusmv_stdout, "The backward Clusters are:\n");
  (void) Iwls95PrintClustersInfo(nusmv_stdout, dd_manager, Iwls95_Bwd_trans, state_variables);
  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  return(0);
}

static int UsageIwls95PrintClusterInfo()
{
  (void) fprintf(nusmv_stderr, "usage: print_iwls95info [-h] [-m] [-o file]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  (void) fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  (void) fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" shell variable if defined,\n");
  (void) fprintf(nusmv_stderr, "      \t\t else through UNIX \"more\"\n");
  (void) fprintf(nusmv_stderr, "   -o file\tWrites the cluster info to \"file\".\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Prints the Iwls95 Options.]

  CommandName        [print_iwls95options] 	   

  CommandSynopsis    [Prints the Iwls95 Options.]  

  CommandArguments   [\[-h\]]  

  CommandDescription [This command prints out the configuration
  parameters of the IWLS95 clustering algorithm, i.e.
  <tt>image_verbosity</tt>, <tt>image_cluster_size</tt> and
  <tt>image_W{1,2,3,4}</tt>.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int CommandIwls95PrintOption(int argc, char ** argv)
{
  int c;

  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch (c) {
    case 'h':
      return(UsageIwls95PrintOption());
    default:
      return(UsageIwls95PrintOption());
    }
  } /* while */
  return(Iwls95PrintOption(nusmv_stdout));
}

static int UsageIwls95PrintOption()
{
  (void) fprintf(nusmv_stderr, "usage: print_iwls95options [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrint the command usage\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Implements the go command]

  CommandName        [go]

  CommandSynopsis    [Initializes the system for the verification.]

  CommandArguments   [\[-h\]]

  CommandDescription [This command initializes the system for
  verification. It is equivalent to the command sequence
  <tt>read_model</tt>, <tt>flatten_hierarchy</tt>,
  <tt>build_variables</tt>, <tt>build_model</tt>,
  <tt>compute_fairness</tt>. If some commands have already been
  executed, then only the remaining ones will be invoked.<p>
  Command options:<p>  
  <dl><dt> -h
  <dd> Prints the command usage.<p> 
  </dl>
  ]

  SideEffects        []
******************************************************************************/
int CommandGo(int argc, char ** argv)
{
  int c;

  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch (c) {
    case 'h':
      return(UsageGo());
    default:
      return(UsageGo());
    }
  } /* while */
  
  if (cmp_struct_get_read_model(cmps) == 0)
    if (Cmd_CommandExecute("read_model")) return 1;
  if (cmp_struct_get_flatten_hrc(cmps) == 0)
    if (Cmd_CommandExecute("flatten_hierarchy")) return 1;
  if (cmp_struct_get_build_variables(cmps) == 0)
    if (Cmd_CommandExecute("build_variables")) return 1;
  if (cmp_struct_get_build_model(cmps) == 0)
    if(Cmd_CommandExecute("build_model")) return 1;
  if (cmp_struct_get_fairness(cmps) == 0)
    if(Cmd_CommandExecute("compute_fairness")) return 1;
  return 0;
}

static int UsageGo()
{
  (void) fprintf(nusmv_stderr, "usage: go [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  return 1;
}

/**Function********************************************************************

  Synopsis           [Implements the get_internal_status command]

  CommandName        [get_internal_status]

  CommandSynopsis    [Returns the internal status of the system.]

  CommandArguments   [\[-h\]]

  CommandDescription [Prints out the internal status of the system. i.e.
  <ul>
  <li> -1 : read_model has not yet been executed or an error occured
            during its execution. </li>
  <li>  0 : flatten_hierarchy has not yet been executed or an error
            occured during its execution. </li>
  <li>  1 : build_variables has not yet been executed or an error
            occured during its execution. </li>
  <li>  2 : build_model has not yet been executed or an error occured
            during its execution. </li>
  <li>  3 : compute_fairness has not yet been executedor an error
            occured during its execution.  </li>
  </ul>
  Command options:<p>  
  <dl><dt> -h
  <dd> Prints the command usage.<p> 
  </dl>
  ]

  SideEffects        []
******************************************************************************/
int CommandGetInternalStatus(int argc, char ** argv)
{
  int c;

  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch (c) {
    case 'h':
      return(UsageGetInternalStatus());
    default:
      return(UsageGetInternalStatus());
    }
  } /* while */
  
  if (cmp_struct_get_read_model(cmps) == 0){
    fprintf(nusmv_stderr, "The internal status is: -1\n");
    return(0);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0){
    fprintf(nusmv_stderr, "The internal status is: 0\n");
    return(0);
  }
  if (cmp_struct_get_build_variables(cmps) == 0){
    fprintf(nusmv_stderr, "The internal status is: 1\n");
    return(0);
  }
  if (cmp_struct_get_build_model(cmps) == 0){
    fprintf(nusmv_stderr, "The internal status is: 2\n");
    return(0);
  }
  if (cmp_struct_get_fairness(cmps) == 0){
    fprintf(nusmv_stderr, "The internal status is: 3\n");
    return(0);
  }
  return 0;
}

static int UsageGetInternalStatus()
{
  (void) fprintf(nusmv_stderr, "usage: get_internal_status [-h]\n");
  (void) fprintf(nusmv_stderr, "   -h \t\tPrints the command usage\n");
  return 1;
}
