/**CFile***********************************************************************

  FileName    [compileCheck.c]

  PackageName [compile]

  Synopsis    [Perform semantic checks on the model.]

  Description [The routines to perform some the semantic checks.<br>
  The performed checks are:
  <ul>
  <li>undefined symbols</li>
  <li>multiple definition of symbols</li>
  <li>circular definition of symbols</li>
  </ul>]

  Author      [Marco Roveri]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The constant hash used in the checking phase.]

  Description [The role is similar to that of <tt>constant_hash</tt>,
  but the associated value is not an ADD, it's the atom itself. It is
  used only for checking purpose.]

  SeeAlso     [constant_hash]

******************************************************************************/
static hash_ptr check_constant_hash;
void init_check_constant_hash() {
  check_constant_hash = new_assoc();
 }
void insert_check_constant_hash(node_ptr x, node_ptr y) {
  insert_assoc(check_constant_hash, x, y);
}
node_ptr lookup_check_constant_hash(node_ptr x) {
  return(find_assoc(check_constant_hash, x));
}
void clear_check_constant_hash() {
  clear_assoc(check_constant_hash);
}

/**Variable********************************************************************

  Synopsis    [Hash used to check assignments.]

  Description [This hash is used to detect multiple definitions. It
  associates to each symbol of the form <em>next(x)</em>,
  <em>init(x)</em>, <em>x</em>, the line number of the input file
  where a value is assigned to it (eg. <em>next(x) := expr</em>).]

  SeeAlso     []

******************************************************************************/
static hash_ptr global_assign_hash;
static void init_global_assign_hash() { global_assign_hash = new_assoc(); }
static void insert_global_assign_hash(node_ptr x, node_ptr y) {
  insert_assoc(global_assign_hash, x, y);
}
static node_ptr lookup_global_assign_hash(node_ptr x) {
  return(find_assoc(global_assign_hash, x));
}
static void clear_global_assign_hash() {clear_assoc(global_assign_hash);}


/**Variable********************************************************************

  Synopsis    [Hash used to check multiple and circular assignments.]

  Description [This hash is used in two different phases of the
  checking. 
  <ol>
  <li>The first deal with multiple definition. During this phase
      the data associated to symbols of the form <em>next(x)</em>,
      <em>init(x)</em> is the body of the assignment (eg, if you have the
      following assignment <em>next(x) := x & y;</em>, then the data
      associated to the symbol <em>next(x)</em> is <em>x & y</em>).</li>
  <li>The second deal with circular definition. In this phase the data
       associated to each symbol is extracted to check the body, and it is
       replaced with <tt>FAILURE_NODE</tt> or <tt>CLOSED_NODE</tt>.</li>
  </ol>]

  SeeAlso     []

******************************************************************************/
static hash_ptr assign_hash;
static void init_assign_hash() { assign_hash = new_assoc(); }
static void insert_assign_hash(node_ptr x, node_ptr y) { insert_assoc(assign_hash, x, y);}
static node_ptr lookup_assign_hash(node_ptr x) {return(find_assoc(assign_hash, x));}
static void clear_assign_hash() {clear_assoc(assign_hash);}

static node_ptr heuristic_order = Nil;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void check_circular_assign ARGS((node_ptr, node_ptr, int));
static void init_check_program_recur ARGS((node_ptr l));
static void init_check_program ARGS((node_ptr procs_exp));
static void check_circ ARGS((node_ptr n, node_ptr context, int));
static void check_assign ARGS((node_ptr n, node_ptr context, int mode));
static void check_assign_both ARGS((node_ptr v, int node_type, int lineno));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Semantic checks on the model.]

  Description        [The main routine performing all the
  check on the read model.<br>
  First it loops on the body of processes to verify that there are no
  multiple assignments and circular definitions.<br>
  Then performs circularity checks on the list of SPECS,
  LTL SPECS, INVARIANT SPECS, FAIRNESS constraints.<br>
  The last operation performed consists in detecting multiple
  assignments between different modules.]

  SideEffects  []

******************************************************************************/
void Compile_CheckProgram(node_ptr procs, node_ptr spec_expr, node_ptr ltlspec_expr,
                   node_ptr invar_expr, node_ptr fair_expr)
{
  node_ptr l=procs;

  /* Initialization of the hashes */
  init_global_assign_hash();
  init_assign_hash();

  /* Initializes check_constant_hash with process_selector elements. */
  init_check_program(procs);
  
  while(l) { /* Loops over processes */
    node_ptr context = car(car(l));
    node_ptr assign_expr = cdr(car(l));

    /* It checks for multiple or circular assignments */
    if (opt_verbose_level_gt(options, 0))
      print_in_process("checking for multiple assignments", context);
    check_assign(assign_expr, Nil, 0);
    if (opt_verbose_level_gt(options, 0))
      print_in_process("checking for circular assignments", context);
    check_assign(assign_expr, Nil, 1);

    clear_assign_hash();

    l = cdr(l);
  }

  /* It checks SPEC */
  l = spec_expr;
  while(l) { /*  Loops over spec expressions */
    check_circ(car(l), Nil, 0);
    l = cdr(l);
  }

  /* It checks LTL SPEC */
  l = ltlspec_expr;
  while(l) { /*  Loops over ltlspec expressions */
    check_circ(car(l), Nil, 0);
    l = cdr(l);
  }
  /* It checks FAIRNESS */
  l = fair_expr;
  while(l) { /* Loops over fairness expressions */
    check_circ(car(l), Nil, 0);
    l = cdr(l);
  }
  l = invar_expr;
  while(l) { /* Loops over check_invariant expressions */
    check_circ(car(l), Nil, 0);
    l = cdr(l);
  }
  
  l = state_variables;
  while(l) { /* Loops over variable list */
    node_ptr v = car(l);
    int lineno = (int)lookup_global_assign_hash(v);
  
    if (lineno) {
      check_assign_both(v, NEXT, lineno);
      check_assign_both(v, SMALLINIT, lineno);
    }
    l = cdr(l);
  }
  clear_global_assign_hash();
  clear_assign_hash();
  if (opt_batch(options)) clear_check_constant_hash();
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the data structure to perform semantic checks.]

  Description        [Initializes the data structure to perform
  semantic checks. The check constant hash is filled in with the names
  of the declared processes.]

  SideEffects        []

******************************************************************************/
static void init_check_program(node_ptr procs_exp)
{
  node_ptr procs_name_list = map(car, procs_exp);
  init_check_program_recur(procs_name_list);
}

/**Function********************************************************************

  Synopsis           [Recursive step of <tt>init_check_program</tt>]

  Description        [Performs recursive step of
  <tt>init_check_program</tt>. Loops over the list of process names
  and inserts the process symbolic name in the <tt>check_constant_hash</tt>.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void init_check_program_recur(node_ptr l)
{
  if (l == Nil) internal_error("init_check_program_recur: l = Nil");
  if (cdr(l) == Nil) {
    node_ptr v = find_atom(car(l));
    if (lookup_check_constant_hash(v)) return;
    if (v && (node_get_type(v) == ATOM)) insert_check_constant_hash(v, v);
    return;
  }
  else {
    init_check_program_recur(odd_elements(l));
    init_check_program_recur(even_elements(l));
  }
  return;
}

/**Function********************************************************************

  Synopsis           [Checks for circular definitions.]

  Description        [This function checks for circular definition of
  any kind. This function is able to detect circularity of the
  following kinds:
  <ul>
     <li><code>next(x) := alpha(next(x))<code></li>
     <li><code>next(x) := next(alpha(x))<code></li<
     <li>any combination of the two above.</li>
     <li><code>x := alpha(x)</code>
  </ul>
  where <code>alpha(x)</code> (<code>alpha(next(x))</code>) is a
  formula in which the variable <code>x</code> (<code>next(x)</code>)
  occurs. Notice that <code>next(alpha(x))</code> can be rewritten in
  term of <code>next(x)</code>, since the next operator distributes
  along all kind of operators.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void check_circ(node_ptr n, node_ptr context, int is_next)
{
  if (n == Nil) return;
  yylineno = node_get_lineno(n);
  switch(node_get_type(n)) {
  case NUMBER:
  case BDD:
  case VAR:
    return;
  case ATOM:
    if (is_next == 0) {
      node_ptr name = find_node(DOT, context, find_atom(n));
      node_ptr temp1 = lookup_param_hash(name);
      node_ptr temp2 = lookup_symbol_hash(name);
      node_ptr temp3 = lookup_check_constant_hash(find_atom(n));
      if ((temp1 && temp2) || (temp2 && temp3) || (temp3 && temp1))
        rpterr("atom \"%s\" is ambiguous", str_get_text(node_get_lstring(n)));
      if (temp1) { check_circ(temp1, context, is_next); return;}
      if (temp3) return;
    } else {
      node_ptr temp1 = lookup_check_constant_hash(find_atom(n));

      /* if it a constant does not matter */
      if (temp1) return;
      else {
        node_ptr name = eval_struct(n, context);
        node_ptr t = find_node(NEXT, name, Nil);

        check_circular_assign(t, context, 0); 
        return;
      }
    }
  case DOT:
  case ARRAY: {
    node_ptr t = eval_struct(n, context);
    check_circular_assign(t, context, is_next);
    return;
  }
  case CONTEXT:
    check_circ(cdr(n), car(n), is_next);
    return;
  case NEXT: {
    /* to handle with nesting of formulas */
    check_circ(car(n), context, 1);
    return;
  }
  default:
    check_circ(car(n), context, is_next);
    check_circ(cdr(n), context, is_next);
    return;
  }
}

/**Function********************************************************************

  Synopsis           [Performs circular assignment checking]

  Description        [Checks for circular assignments in the model. If
  there are any, then an error is generated.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void check_circular_assign(node_ptr n, node_ptr context, int is_next)
{
  node_ptr t = lookup_assign_hash(n);

  if (t == CLOSED_NODE) return;
  if (t == FAILURE_NODE) error_circular(n);
  if (t == Nil) t = lookup_symbol_hash(n);
  if ((t == Nil) && (is_next == 0)) t = lookup_symbol_hash(car(n));
  if (t == Nil) {
    if (lookup_check_constant_hash(n)) return;
    error_undefined(n);
  }
  insert_assign_hash(n, FAILURE_NODE);
  io_atom_push(n);
  check_circ(t, context, is_next);
  io_atom_pop();
  insert_assign_hash(n, CLOSED_NODE);
  switch(node_get_type(n)) {
  case NEXT:
    heuristic_order = cons(car(n), heuristic_order); break;
  case SMALLINIT:
    break;
  default:
    heuristic_order = cons(n, heuristic_order);
  }
}

/**Function********************************************************************

  Synopsis           [Checks for multiple or circular assignments.]

  Description        [This function detects either multiple or
  circular assignments in "context" regarding to "mode".
  If mode is equal to 0 (zero) then it checks for multiple assignments
  or symbols redefinition. Otherwise it performs checks for circular
  assignments.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void check_assign(node_ptr n, node_ptr context, int mode)
{
  if (n == Nil) return;
  yylineno = node_get_lineno(n);
  switch(node_get_type(n)) {
  case AND:
    check_assign(car(n), context, mode);
    check_assign(cdr(n), context, mode);
    break;
  case CONTEXT:
    check_assign(cdr(n), car(n), mode);
    break;
  case EQDEF:
    {
      node_ptr t1, t2, t3;
      if ((node_get_type(car(n)) == SMALLINIT) || (node_get_type(car(n)) == NEXT)) {
	t1 = eval_struct(car(car(n)), context);
	t3 = find_node(node_get_type(car(n)), t1, Nil);
      }
      else
        t1 = t3 = eval_struct(car(n), context);
      if (mode == 0) { /* Checking for multiple assignments */
	t2 = lookup_symbol_hash(t1);
	if (t2 == Nil) error_undefined(t1);
	if (node_get_type(t2) != VAR) error_redefining(t1);
	if (lookup_assign_hash(t3)) error_multiple_assignment(t3);
	insert_assign_hash(t3, find_node(CONTEXT, context, cdr(n)));
	insert_global_assign_hash(t3, (node_ptr)yylineno);
      }
      else { /* Checking for circular assignments */
        check_circular_assign(t3, context, 0);
      }
      break;
    }
  default:
    internal_error("check_assign: type = %d", node_get_type(n));
  }
}

/**Function********************************************************************

  Synopsis           [Given a variable, it checks if there are
  multiple assignments to it.]

  Description        [Checks if there exists in the model an
  assignments of type <tt>node_type</tt> for variable <tt>v</tt>. If
  such an assignment exists, then an error is generated.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void check_assign_both(node_ptr v, int node_type, int lineno)
{
  node_ptr v1 = find_node(node_type, v, Nil);
  int lineno2 = (int) lookup_global_assign_hash(v1);

  if (lineno2) error_assign_both(v, v1, lineno, lineno2);
}
