/**CHeaderFile*****************************************************************

  FileName    [compile.h]

  PackageName [compile]

  Synopsis    [Compilation of NuSMV input language into BDD.]

  Description [This package contains the compiler of NuSMV code into
  BDD. It works on a flattened/instantiated structure. Performs the
  checks on the parse tree and generates the encoding of scalar
  variables into BDDs. Then, the transition relation is created with
  different methods.]

  Author      [Marco Roveri]

  Copyright [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

  Revision    [$Id: $]

******************************************************************************/
#ifndef _COMPILE
#define _COMPILE

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Data structure used to store the results of compilation.]

  Description [Data structure used to store the results of compilation.]

******************************************************************************/
struct cmp_struct {
  int      read_model;
  int      flatten_hierarchy;
  int      build_variables;
  int      process_selector;
  int      build_frames;
  int      compile_check;
  int      build_init;
  int      build_model;
  int      fairness_constraints;
  node_ptr init_expr;
  node_ptr invar_expr;
  node_ptr trans_expr;
  node_ptr assign_expr;
  node_ptr procs_expr;
  node_ptr fair_expr;
  node_ptr spec_expr;
  node_ptr ltlspec_expr;
  node_ptr invar_spec_expr;
  node_ptr invar_fb_expr;
  node_ptr invar_strong_expr;
  node_ptr after_expr;
};

/**Struct**********************************************************************

  Synopsis    [This structure contains the information about a particular
               cluster.]

  Description [There is one Iwls95ClusterInfo_struct per cluster. The
  various fields of this structure contain the information for
  ordering the clusters for image computation purposes.]

  SeeAlso     []

******************************************************************************/
struct Iwls95ClusterInfo_struct {
  int i;
  bdd_ptr Ti;           /* The current Cluster */
  bdd_ptr Ei;           /* Variables that can be existentially
			   quantified when Ti is multiplied in the
			   product */ 
  bdd_ptr Supp_Ti;      /* Set of Support of Cluster T_i = S(T_i)     */
  bdd_ptr Supp_Q_Ci;    /* {v : v \in S(T_j), j != i, j \in Q}        */
  bdd_ptr PSPI_Ti;      /* {v : v \in (PS U PI) & v \in S(T_i)}       */
  bdd_ptr NS_Ti;        /* {v : v \in NS & v \in S(T_i)}              */
  double v_c;           /* Number of variables which can be existentially
                           quantified when T_i is multiplied in the product */
  double w_c;           /* Number of current and input variables in
                           the support of T_i */
  double x_c;           /* Number of current and input variables which
                           have not yet been quantified */
  double y_c;           /* Number of next variables that would be
                           introduced in the product by multiplying T_i */
  double z_c;           /* Number of next variables not yet introduced
                           in the product */
  double m_c;           /* The maximum "index" of a variable to be
                           quantified in S(T_i) */
  double M_c;           /* The maximum BDD index of a variable to be
                           quantified out in the remaining clusters */
};

/**Struct**********************************************************************

  Synopsis    [The parameters controlling the ordering of clusters.]

  Description [This structure contains the options to perform ordering
  of clusters in the IWLS95 partitioning method. <br>
  <code>clusterSize</code> is the threshold value used to create
  clusters. <code>W1</code>, <code>W2</code> etc are the weights used
  in the heuristic algorithms to order the clusters for early
  quantification. <code>verbosity</code> controls the amount of
  information printed during IWLS95 image computation.]

******************************************************************************/
struct Iwls95OptionStruct {
  int clusterSize;
  int verbosity; /* To choose if we want local verbosity in the image iwls computation */
  int W1;
  int W2; /* Weights attached to various parameters. */
  int W3; /* For details please refer to the paper */
  int W4; /* (insert reference of the paper IWLS95) */
};

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

typedef struct Iwls95ClusterInfo_struct Iwls95Cluster_Info_t;
typedef struct Iwls95OptionStruct Iwls95OptionStruct_t;
typedef struct cmp_struct cmp_struct_rec;
typedef struct cmp_struct * cmp_struct_ptr;

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

void Compile_Init ARGS((void));
void Compile_End  ARGS((void));
void Compile_CheckProgram ARGS((node_ptr, node_ptr, node_ptr, node_ptr, node_ptr));
void set_assignment_type_init ARGS((void));
void set_assignment_type_trans ARGS((void));
void set_assignment_type_assign ARGS((void));
int eval_num ARGS((node_ptr, node_ptr));
node_ptr eval_struct ARGS((node_ptr, node_ptr));
node_ptr eval_tree ARGS((node_ptr, node_ptr));
node_ptr node_plus ARGS((node_ptr, node_ptr));
node_ptr node_plus1 ARGS((node_ptr));
add_ptr eval ARGS((node_ptr, node_ptr));
add_ptr eval_simplify ARGS((node_ptr, node_ptr, add_ptr));
node_ptr Compile_ReadOrder ARGS((const char *));
void Compile_WriteOrder ARGS((const char *, int));
node_ptr sym_intern ARGS((char *));

/* Yuan Lu : I add ABSTRACT statements */
void Compile_FlattenHierarchy ARGS((node_ptr, node_ptr, node_ptr *, node_ptr *,
                                    node_ptr *, node_ptr *, node_ptr *, node_ptr *,\
                                    node_ptr *, node_ptr *, node_ptr *, node_ptr *,\
                                    node_ptr *, node_ptr *, node_ptr *, node_ptr));
void create_process_symbolic_variables ARGS((node_ptr));
void Compile_EncodeVarsInit ARGS((void));
void Compile_BuildVarsBdd ARGS((void));
void build_real_state_variables ARGS((void));
double bdd_count_states ARGS((DdManager *, bdd_ptr fn));
double add_count_states ARGS((DdManager *, bdd_ptr fn)); 
void Compile_CompileFrame ARGS((void));
void build_proc_selector ARGS((node_ptr));
void Compile_CompileInit ARGS((node_ptr, node_ptr));
void Compile_CompileModel ARGS((node_ptr, node_ptr, node_ptr, add_ptr));
void compute_fairness_constraints ARGS((node_ptr));

void reset_encode ARGS((void));

void print_state ARGS((bdd_ptr, int));
void init_print_hash ARGS((void));
void insert_print_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_print_hash ARGS((node_ptr));
void clear_print_hash ARGS((void));

void init_symbol_hash ARGS((void));
void insert_symbol_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_symbol_hash ARGS((node_ptr));
void clear_symbol_hash ARGS((void));

void init_param_hash ARGS((void));
void insert_param_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_param_hash ARGS((node_ptr));
void clear_param_hash ARGS((void));

void init_constant_hash ARGS((void));
void insert_constant_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_constant_hash ARGS((node_ptr));
void clear_constant_hash ARGS((void));

void init_frame_hash ARGS((void));
void insert_frame_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_frame_hash ARGS((node_ptr));
void clear_frame_hash ARGS((void));

void init_value_hash ARGS((void));
void insert_value_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_value_hash ARGS((node_ptr));
void clear_value_hash ARGS((void));

void init_check_constant_hash ARGS((void));
void insert_check_constant_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_check_constant_hash ARGS((node_ptr));
void clear_check_constant_hash ARGS((void));

void print_state_vars ARGS((DdManager *, bdd_ptr, node_ptr));

void init_state_vars ARGS((void));
void make_state_vars ARGS((void));
void free_state_vars ARGS((void));

void print_model_statistic ARGS((void));
void print_monolithic_info ARGS((void));
void print_conj_part_info ARGS((void));
void print_conj_part_detailed_info ARGS((void));
void print_disj_part_info ARGS((void));
void print_iwls95cp_part_info ARGS((void));

add_ptr add_shift_forward ARGS((DdManager * dd, add_ptr fn));
add_ptr add_shift_backward ARGS((DdManager * dd, add_ptr fn));
bdd_ptr bdd_shift_forward ARGS((DdManager * dd, bdd_ptr fn));
bdd_ptr bdd_shift_backward ARGS((DdManager * dd, bdd_ptr fn));
bdd_ptr bdd_pick_one_state ARGS((DdManager * dd, bdd_ptr fn));
bdd_ptr bdd_pick_one_state_rand ARGS((DdManager * dd, bdd_ptr fn));

cmp_struct_ptr cmp_struct_init ARGS((void));

int cmp_struct_get_read_model ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_read_model ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_flatten_hrc ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_flatten_hrc ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_build_variables ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_build_variables ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_process_selector ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_process_selector ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_build_frames ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_build_frames ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_compile_check ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_compile_check ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_build_init ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_build_init ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_build_model ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_build_model ARGS((cmp_struct_ptr cmp));

int cmp_struct_get_fairness ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_fairness ARGS((cmp_struct_ptr cmp));

node_ptr cmp_struct_get_init ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_init ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_invar ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_invar ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_trans ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_trans ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_assign ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_assign ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_procs ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_procs ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_fair ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_fair ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_spec ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_spec ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_ltlspec ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_ltlspec ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_invar_spec ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_invar_spec ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_invar_fb ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_invar_fb ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_invar_strong ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_invar_strong ARGS((cmp_struct_ptr cmp, node_ptr n));

node_ptr cmp_struct_get_after ARGS((cmp_struct_ptr cmp));
void cmp_struct_set_after ARGS((cmp_struct_ptr cmp, node_ptr n));

Iwls95OptionStruct_t * Iwls95GetOptions ARGS((void));
void Iwls95FreeClustersList ARGS((DdManager *, node_ptr));

void init_module_hash ARGS((void));
void insert_module_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_module_hash ARGS((node_ptr));
void clear_module_hash ARGS((void));

void reset_interactive_mode ARGS((void));
#endif /* _COMPILE */
