/**CFile***********************************************************************

  FileName    [absVariables.c]

  PackageName [abs]

  Synopsis    [

  Description [
  ]

  SeeAlso     [absAutoGenAbs.c]

  Author      [Yuan Lu]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "absInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";
#define DEBUG_IWLS95 0

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The dependency hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>.
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     []

******************************************************************************/
static hash_ptr depend_hash;
void init_depend_hash() {depend_hash = new_assoc();}
void insert_depend_hash(node_ptr x, node_ptr y) { insert_assoc(depend_hash, x, y);}
node_ptr lookup_depend_hash(node_ptr x) {return(find_assoc(depend_hash, x));}
static assoc_retval free_depend_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_depend_hash() {clear_assoc_and_free_entries(depend_hash, free_depend_hash);}

/**Variable********************************************************************

  Synopsis    [The interest hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>.
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     []

******************************************************************************/
static hash_ptr interest_hash;
void init_interest_hash() {interest_hash = new_assoc();}
void insert_interest_hash(node_ptr x, node_ptr y) { insert_assoc(interest_hash, x, y);}
node_ptr lookup_interest_hash(node_ptr x) {return(find_assoc(interest_hash, x));}
static assoc_retval free_interest_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_interest_hash() {clear_assoc_and_free_entries(interest_hash, free_interest_hash);}

/**Variable********************************************************************

  Synopsis    [The cone-of-influence hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>.
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     []

******************************************************************************/
static hash_ptr coi_hash;
void init_coi_hash() {coi_hash = new_assoc();}
void insert_coi_hash(node_ptr x, node_ptr y) { insert_assoc(coi_hash, x, y);}
node_ptr lookup_coi_hash(node_ptr x) {return(find_assoc(coi_hash, x));}
static assoc_retval free_coi_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_coi_hash() {clear_assoc_and_free_entries(coi_hash, free_coi_hash);}

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
/**Variable********************************************************************

  Synopsis    [The x hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>.
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     []

******************************************************************************/
static hash_ptr x_hash;
void init_x_hash() {x_hash = new_assoc();}
void insert_x_hash(node_ptr x, node_ptr y) { insert_assoc(x_hash, x, y);}
node_ptr lookup_x_hash(node_ptr x) {return(find_assoc(x_hash, x));}
static assoc_retval free_x_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_x_hash() {clear_assoc_and_free_entries(x_hash, free_x_hash);}

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void absBuildInterestListRecur ARGS((node_ptr, int));
static void absBuildCoiListRecur ARGS((node_ptr));
static void absExtractVarFromProc ARGS((node_ptr));
static void absExtractVarOneProc ARGS((node_ptr, node_ptr));
static node_ptr absGenVarListRecur ARGS((node_ptr, node_ptr, node_ptr));
static int absInList ARGS((node_ptr, node_ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Check two ATOM nodes are same or not.]

  Description        [Check two ATOM nodes are same or not.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Abs_AbsPrintVarOnLevel(node_ptr spec_expr, node_ptr fair_expr)
{
  node_ptr list1 = absGenVarListRecur(spec_expr, Nil, Nil);
  node_ptr tmp, dep, list2;
  int level=1;
  FILE *fp = fopen("depend", "w");

  init_x_hash();

  list1 = absGenVarListRecur(fair_expr, Nil, list1);

  while(list1) {
    list2 = Nil;
    fprintf(fp, "Level %d: ", level++);
    for(tmp=list1;tmp;tmp=cdr(tmp)) {
      node_ptr var = car(tmp);
      if(lookup_x_hash(var)) continue;
      fprintf_node_exp(fp, var);
      fprintf(fp, " ");
      dep = lookup_depend_hash(var);
      insert_x_hash(var, var);
      for(;dep;dep=cdr(dep)) {
	node_ptr dvar = car(dep);
	if(lookup_x_hash(dvar)) continue;
	list2 = find_node(CONS, dvar, list2);
      }
    }
    fprintf(fp, "\n");
    list1 = list2;
  }
  fclose(fp);
  clear_x_hash();
}

/**Function********************************************************************

  Synopsis           [Build the influence variables upto a certain level.]

  Description        [Build the influence variables upto a certain level.]

  SideEffects        [The interest_hash and coi_hash are built.]

  SeeAlso            [Abs_AbsExtractVariableGraph]

******************************************************************************/
void Abs_AbsBuildInterestList(node_ptr spec_expr, node_ptr fair_expr, int level)
{
  node_ptr list = absGenVarListRecur(spec_expr, Nil, Nil);
  node_ptr tmp, dep;

  list = absGenVarListRecur(fair_expr, Nil, list);
  init_coi_hash();
  init_interest_hash();

  for(tmp=list;tmp;tmp=cdr(tmp)) {
    node_ptr var = car(tmp);
    if(!lookup_coi_hash(var)) {
      insert_coi_hash(var,var);
      absBuildCoiListRecur(var);
    }
  }

  for(;level;level--) {
    tmp = Nil;
    for(;list;list=cdr(list)) {
      node_ptr var = car(list);
      if(lookup_interest_hash(var)) continue;
      dep = lookup_depend_hash(var);
      insert_interest_hash(var, var);
      for(;dep;dep=cdr(dep)) {
	node_ptr dvar = car(dep);
	if(lookup_interest_hash(dvar)) continue;
	tmp = find_node(CONS, dvar, tmp);
      }
    }
    list = tmp;
  }
}

/**Function********************************************************************

  Synopsis           [Extract variable dependenc graph.]

  Description        [Extract variable dependenc graph.]

  SideEffects        [depend_hash is built.]

  SeeAlso            [Abs_AbsBuildInterestList]

******************************************************************************/
void Abs_AbsExtractVariableGraph(node_ptr procs)
{
  init_depend_hash();

  absExtractVarFromProc(procs);
#ifdef ABS_DEBUG
  if(ABS_DEBUG > 1) {
    FILE *fp = fopen("depend", "w");
    node_ptr vars = all_variables;
    for(;vars;vars=cdr(vars)) {
      node_ptr dep = lookup_depend_hash(car(vars));
      if(dep) {
	/* fprintf_node_exp(fp, car(vars)); */
	debug_fprint_node(fp, car(vars));
	fprintf(fp, " **********DEPENDS ON******** :\n\t");
	for(;dep;dep=cdr(dep)) {
	  fprintf(fp, "\t");
	  fprintf_node_exp(fp, car(dep));
	}
	fprintf(fp, "\n\n");
      }
    }
    fclose(fp);
  }
#endif
}

/**Function********************************************************************

  Synopsis           [Given an assignment, return the appeared variables.]

  Description        [Given an assignment, return the appeared variables.]

  SideEffects        [None.]

  SeeAlso            [Abs_AbsExtractVariableGraph]

******************************************************************************/
node_ptr Abs_AbsGenVarList(node_ptr assignment, node_ptr context)
{
  return absGenVarListRecur(assignment, context, Nil);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Build list of Coi variables recursively.]

  Description        [Build list of Coi variables recursively.]

  SideEffects        [coi_hash is built.]

  SeeAlso            [absBuildInterestListRecur]

******************************************************************************/
static void absBuildCoiListRecur(node_ptr var)
{
  node_ptr list = lookup_depend_hash(var);
  for(;list;list=cdr(list)) {
    node_ptr varn = car(list);
    if(!lookup_coi_hash(varn)) {
      insert_coi_hash(varn, varn);
      absBuildCoiListRecur(varn);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Build list of the influential variables recursively.]

  Description        [Build list of the influential variables recursively.]

  SideEffects        [interest_hash]

  SeeAlso            [absBuildCoiListRecur]

******************************************************************************/
static void absBuildInterestListRecur(node_ptr var, int level)
{
  node_ptr list;

  if(!level) return;
  level--;
  list = lookup_depend_hash(var);
  for(;list;list=cdr(list)) {
    node_ptr varn = car(list);
    if(!lookup_interest_hash(varn)) {
      insert_interest_hash(varn, varn);
      absBuildInterestListRecur(varn, level);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Build variable dependence relation.]

  Description        [Build variable dependence relation. For each proc,
  variable dependence relation is built by calling absExtractVarOneProc.]

  SideEffects        [depend_hash]

  SeeAlso            []

******************************************************************************/
static void absExtractVarFromProc(node_ptr procs)
{
  if(!procs) return;
  while(procs) {
    node_ptr cur_process = car(procs);
    node_ptr context = car(cur_process);
    node_ptr assign_expr = cdr(cur_process);

    absExtractVarOneProc(assign_expr, context);
    procs = cdr(procs);
  }
}

/**Function********************************************************************

  Synopsis           [Build variable dependence relation for one proc.]

  Description        [Build variable dependence relation for one proc.
                      When a node is assignment (EQDEF), the dependence
                      is extracted.]

  SideEffects        [None]

  SeeAlso            [absAutoGenAbsPreRecur]

******************************************************************************/
static void absExtractVarOneProc(node_ptr procs, node_ptr context)
{
  if(!procs) return;
  switch(node_get_type(procs)) {
    case EQDEF : {
      node_ptr res, name;
      node_ptr var = car(procs);
      switch(node_get_type(var)) {
	case SMALLINIT : return;
	case NEXT :
          name = eval_struct(car(var), context);
	  res = lookup_depend_hash(name);
          res = absGenVarListRecur(cdr(procs), context, res);
	  if(res) insert_depend_hash(name, res);
	  return;
	default : 
          name = eval_struct(var, context);
	  res = lookup_depend_hash(name);
          res = absGenVarListRecur(cdr(procs), context, Nil);
	  if(res) insert_depend_hash(name, res);
	  return;
      }
    }
    case CONTEXT :
      absExtractVarOneProc(cdr(procs), car(procs));
      return;
    case CONS :
        rpterr("Something Wrong");
    default :
      absExtractVarOneProc(car(procs), context);
      absExtractVarOneProc(cdr(procs), context);
  }
}

/**Function********************************************************************

  Synopsis           [Add dependence variables into a list.]

  Description        [Add dependence variables into a list. Note that the
  the variables which appear in data context are also considered.]

  SideEffects        [None]

  SeeAlso            [absAutoGenAbsPreRecur]

******************************************************************************/
static node_ptr absGenVarListRecur(node_ptr ass, node_ptr context, node_ptr list)
{
  if(!ass) return list;
  switch(node_get_type(ass)) {
    case ATOM: {
      node_ptr name  = find_node(DOT, context, find_atom(ass));
      node_ptr temp1 = lookup_symbol_hash(name);
      node_ptr temp2 = lookup_param_hash(name);
      node_ptr temp3 = (temp1) ? temp1 : temp2;
      if(temp3) {
	if(node_get_type(temp3)==VAR)
	  if(!absInList(name, list)) return cons(name, list);
	  else return list;
	else
	  return absGenVarListRecur(temp3, context, list);
      }
      return list;
    }
    case ARRAY: {
      node_ptr name = eval_struct(ass, context);
      if(!absInList(name, list)) return cons(name, list);
    }
    case DOT: {
      node_ptr name = eval_struct(ass, context);
      node_ptr temp1 = lookup_symbol_hash(name);
      node_ptr temp2 = lookup_param_hash(name);
      node_ptr temp3 = (temp1) ? temp1 : temp2;
      if(temp3) {
	if(node_get_type(temp3) == VAR)
	  if(!absInList(name, list)) return cons(name, list);
	  else return list;
	else
	  return absGenVarListRecur(temp3, context, list);
      }
      return list;
    }
    case CONTEXT:
      return absGenVarListRecur(cdr(ass), car(ass), list);
    case NUMBER: return list;
    case EQDEF:
    case SMALLINIT:
      rpterr("Yuan Lu : SOMething wrong");
    default : {
      node_ptr ln = absGenVarListRecur(car(ass), context, list);
      return absGenVarListRecur(cdr(ass), context, ln);
    }
  }
  return list;
}

/**Function********************************************************************

  Synopsis           [Check if name appear in the list.]

  Description        [Check if name appear in the list.]

  SideEffects        [None]

  SeeAlso            [absAutoGenAbsPreRecur]

******************************************************************************/
static int absInList(node_ptr name, node_ptr list)
{
  for(;list;list=cdr(list)) {
    if(isamevar(name, car(list))) return 1;
  }
  return 0;
}

