/**CFile***********************************************************************

  FileName    [absUtil.c]

  PackageName [abs]

  Synopsis    [High-level routines to perform automatic abstraction.]

  Description [This file provides the interface to the abstraction
  routines.]

  Author      [Yuan Lu]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "absInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/
static node_ptr make_support_list ARGS((node_ptr));
static node_ptr make_quantifiers ARGS((node_ptr, add_ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Check two ATOM nodes are same or not.]

  Description        [Check two ATOM nodes are same or not.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
Abs_AbsEncodePost()
{
  Abs_BuildAbstractRelation(abs_expr_pre);
  Abs_BuildAbstractRelation(abs_expr);
}

/**Function********************************************************************

  Synopsis           [Abstract all the variables corresponding to the abstraction
		      set.]

  Description        [Abstract all the variables corresponding to the abstraction
		      set.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Abs_AbsExAbsAll(bdd_ptr *g)
{
  node_ptr abs = abs_expr_pre;
  bdd_ptr f = *g;
  for(;abs;abs=cdr(abs)) {
    node_ptr varl = car(car(car(abs)));
    add_ptr allsupp;
    bdd_ptr suppb, res;

    allsupp = add_one(dd_manager);
    for(;varl;varl=cdr(varl)) {
      add_ptr var = (add_ptr) car(lookup_symbol_hash(car(varl)));
      add_ptr supp = add_support(dd_manager, var);
      add_and_accumulate(dd_manager, &allsupp, supp);
      add_free(dd_manager, supp);
    }
    suppb = add_to_bdd(dd_manager, allsupp);
    res = bdd_forsome(dd_manager, f, suppb);
    bdd_free(dd_manager, suppb); bdd_free(dd_manager, f);
    f = res;
  }
  for(abs=abs_expr;abs;abs=cdr(abs)) {
    node_ptr varl = car(car(car(abs)));
    add_ptr allsupp;
    bdd_ptr suppb, res;

    allsupp = add_one(dd_manager);
    for(;varl;varl=cdr(varl)) {
      add_ptr var = (add_ptr) car(lookup_symbol_hash(car(varl)));
      add_ptr supp = add_support(dd_manager, var);
      add_and_accumulate(dd_manager, &allsupp, supp);
      add_free(dd_manager, supp);
    }
    suppb = add_to_bdd(dd_manager, allsupp);
    res = bdd_forsome(dd_manager, f, suppb);
    bdd_free(dd_manager, suppb); bdd_free(dd_manager, f);
    f = res;
  }
  *g = f;
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Abs_AbsFreeAbs(node_ptr abs)
{
  node_ptr list = car(car(abs));

  for(;list;list=cdr(list))
    free_node(list);

  for(list=cdr(car(abs));list;list=cdr(list))
    free_node(list);

  free_node(car(abs));

  for(list = car(cdr(abs));list;list=cdr(list))
    free_node(list);

  free_node(cdr(cdr(abs)));
  free_node(cdr(abs));
  free_node(abs);
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Abs_AbsRebuild()
{
  bdd_free(dd_manager, init_bdd);
  init_bdd= Abs_AbsAbstractCur(init_bdd_orig, abs_expr);

  if(invar_bdd) {
    bdd_free(dd_manager, invar_bdd);
    invar_bdd= Abs_AbsAbstractCur(invar_bdd_orig, abs_expr);
  }

  if (opt_disj_partitioning(options)) {
  } else if (opt_conj_partitioning(options)) {
    node_ptr t1 = cp_trans_bdd_orig;
    node_ptr t2 = cp_trans_bdd;
    node_ptr t3 = cp_trans_add;
    for(;t1;t1=cdr(t1), t2=cdr(t2), t3=cdr(t3)) {
      bdd_ptr f = Abs_AbsAbstractCurNex((bdd_ptr)car(t1), abs_expr);
      bdd_free(dd_manager, (bdd_ptr)car(t2));
      t2->left.bddtype = f;
      t3->left.bddtype = bdd_to_add(dd_manager, f);
    }

    {
      add_ptr cp_vars = add_dup(state_variables_add);
      add_ptr next_cp_vars = add_dup(next_state_variables_add);
      node_ptr sl = make_support_list(cp_trans_add);

      /* Quantification should also consider the input variables. */
      add_and_accumulate(dd_manager, &cp_vars, input_variables_add);
      add_and_accumulate(dd_manager, &next_cp_vars, next_input_variables_add);


      walk_dd(dd_manager, (VPFDD)bdd_free, forward_quantifiers_bdd);
      walk_dd(dd_manager, (VPFDD)bdd_free, reverse_quantifiers_bdd);

      forward_quantifiers_add = make_quantifiers(sl, cp_vars);
      reverse_quantifiers_add = make_quantifiers(sl, next_cp_vars);
      add_free(dd_manager, cp_vars);
      add_free(dd_manager, next_cp_vars);


      walk_dd(dd_manager, (VPFDD)add_free, sl);


      forward_quantifiers_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd, forward_quantifiers_add);
      reverse_quantifiers_bdd = map_dd(dd_manager, (NPFDD)add_to_bdd, reverse_quantifiers_add);

      walk_dd(dd_manager, (VPFDD)add_free, cp_trans_add);
      walk_dd(dd_manager, (VPFDD)add_free, forward_quantifiers_add);
      walk_dd(dd_manager, (VPFDD)add_free, reverse_quantifiers_add);
    }

    for(;abs_rel_prod;abs_rel_prod=cdr(abs_rel_prod)) {
      bdd_free(dd_manager, (bdd_ptr)car(car(abs_rel_prod)));
      bdd_free(dd_manager, (bdd_ptr)cdr(car(abs_rel_prod)));
    }
  } else if (opt_monolithic(options)) {
    bdd_free(dd_manager, trans_bdd);
    trans_bdd= Abs_AbsAbstractCurNex(trans_bdd_orig, abs_expr);
  } else if (opt_iwls95cp_partitioning(options)) {
  }

  /* free fair_states_bdd */
  if(fair_states_bdd) {
    bdd_free(dd_manager, fair_states_bdd);
    fair_states_bdd = (bdd_ptr) Nil;
  }
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void domain_exchange()
{
  bdd_ptr t = invar_bdd;
  invar_bdd = invar_bdd_orig;
  invar_bdd_orig = t;

  t = init_bdd; init_bdd = init_bdd_orig;
  init_bdd_orig = t;

  if(opt_monolithic(options)) {
    bdd_ptr tmp = trans_bdd;
    trans_bdd = trans_bdd_orig;
    trans_bdd_orig = tmp;

  } else if(opt_conj_partitioning(options)) {
    node_ptr tmp = cp_trans_bdd; 
    cp_trans_bdd = cp_trans_bdd_orig; 
    cp_trans_bdd_orig = tmp;

    tmp = cp_trans_add; 
    cp_trans_add = cp_trans_add_orig; 
    cp_trans_add_orig = tmp;

    tmp = forward_quantifiers_bdd; 
    forward_quantifiers_bdd = forward_quantifiers_bdd_orig; 
    forward_quantifiers_bdd_orig = tmp;

    tmp = forward_quantifiers_add; 
    forward_quantifiers_add = forward_quantifiers_add_orig; 
    forward_quantifiers_add_orig = tmp;

    tmp = reverse_quantifiers_bdd; 
    reverse_quantifiers_bdd = reverse_quantifiers_bdd_orig; 
    reverse_quantifiers_bdd_orig = tmp;

    tmp = reverse_quantifiers_add; 
    reverse_quantifiers_add = reverse_quantifiers_add_orig; 
    reverse_quantifiers_add_orig = tmp;
  } else if(opt_disj_partitioning(options)) {
  } else if(opt_iwls95cp_partitioning(options)) {
  }
}

/**Function********************************************************************

  Synopsis           [Check two ATOM nodes are same or not.]

  Description        [Check two ATOM nodes are same or not.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int isameatom(node_ptr var1, node_ptr var2)
{
  return (!strcmp(get_text((string_ptr) car(var1)),
          get_text((string_ptr) car(var2))));
}

/**Function********************************************************************

  Synopsis           [Check two value lists are same or not.]

  Description        [Check two value lists are same or not.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int isameval(node_ptr val1, node_ptr val2)
{
  if(node_get_type(val1) != node_get_type(val2)) return 0;
  switch(node_get_type(val1)) {
    case ATOM : return (!strcmp(get_text((string_ptr) car(val1)),
                        get_text((string_ptr) car(val2))));
    case NUMBER : return (car(val1)==car(val2));
    default : return 0;
  }
}

/**Function********************************************************************

  Synopsis           [Check two variables are the same or not.]

  Description        [Check two variables are the same or not.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int isamevar(node_ptr var1, node_ptr var2)
{
  if(!var1)
    if(!var2) return 1;
    else return 0;
  if(!var2) return 0;
  if(var1 == var2) return 1;
  if(node_get_type(var1) != node_get_type(var2)) return 0;

  switch(node_get_type(var1)) {
    case ATOM :
      if(isameatom(var1, var2))
        return 1;
      return 0;
    case DOT :
      if(!isamevar(car(var1), car(var2))) return 0;
      if(!isamevar(cdr(var1), cdr(var2))) return 0;
      return 1;
    case ARRAY :
      if(!isamevar(car(var1), car(var2))) return 0;
      if(!isamevar(cdr(var1), cdr(var2))) return 0;
    case NUMBER :
      return isameval(var1, var2);
    default :
      rpterr("Type I dont understand %d", node_get_type(var1));
  }
  return 0;
}

/**Function********************************************************************

  Synopsis           [Check .]

  Description        [Check .]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int isnextvar(node_ptr var, node_ptr assign, node_ptr context)
{
  if(node_get_type(assign) == NEXT) {
    node_ptr asvar = eval_struct(car(assign), context);
    if(isamevar(asvar, var)) return 1;
  }
  return 0;
}

/**Function********************************************************************

  Synopsis           [Check .]

  Description        [Check .]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int isFunNoAbs(node_ptr abs_val)
{
  node_ptr p = abs_val;
  node_ptr q;

  for(;cdr(p);p=cdr(p)) {
    int k = (int) car(p);
    for(q=cdr(p);q;q=cdr(q)) {
      if(k == ((int) car(q))) return 0;
    }
  }
  return 1;
}

/**Function********************************************************************

  Synopsis           [.]

  Description        [.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Abs_AbsBuildMaxPartList(node_ptr Acc, node_ptr sexp,
				 node_ptr cxt, add_ptr assumption)
{
  if (sexp == Nil) return(Acc);
  switch(node_get_type(sexp)){
  case AND:
    {
      Acc = Abs_AbsBuildMaxPartList(Acc, car(sexp), cxt, assumption);
      Acc = Abs_AbsBuildMaxPartList(Acc, cdr(sexp), cxt, assumption);
      return(Acc);
    }
  case CONTEXT:
    {
      return(Abs_AbsBuildMaxPartList(Acc, cdr(sexp), car(sexp), assumption));
    }
  default:
    {
      add_ptr one = add_one(dd_manager);
      add_ptr sexp_add = eval(sexp, cxt);

      add_free(dd_manager, one);
      if (sexp_add == one) {
        add_free(dd_manager, sexp_add);
        return(Acc);
      }
      {
        add_ptr cur_ti = add_simplify_assuming(dd_manager, sexp_add, assumption);
	bdd_ptr cur_bdd = add_to_bdd(dd_manager, cur_ti);

        add_free(dd_manager, sexp_add);
        add_free(dd_manager, cur_ti);
        return(cons((node_ptr)cur_bdd, Acc));
      }
    }
  }
}

/**Function********************************************************************

  Synopsis           [Given a list of {A|B}DDs then return the list of
  the corresponding set of support.]

  Description        [Given a list of {A|B}DDs then return the list of
  the corresponding set of support, i.e. the cube of the {A|B}DD
  boolean variables the {A|B}DD depends on.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr make_support_list(node_ptr l)
{
  if (l == Nil) return(l);
  {
    node_ptr r = make_support_list(cdr(l));
    add_ptr s = add_support(dd_manager, (add_ptr)car(l));

    if (r != Nil) {
      add_ptr tmp = add_and(dd_manager, (add_ptr)car(r), s);

      add_free(dd_manager, s);
      return(cons((node_ptr)tmp, r));
    }
    return(cons((node_ptr)s, Nil));
  }
}

/**Function********************************************************************

  Synopsis           [Computes the cube to be quantified for each cluster.]

  Description        [Computes the cube to be quantified for each
  cluster. Given a list of cubes <tt>l</tt>, and a cube
  <tt>vars</tt>, then for each element in the list, the cube
  difference from it and <tt>vars</tt> is computed and returned back.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr make_quantifiers(node_ptr l, add_ptr vars)
{
  add_ptr tmp;

  if (l == Nil) {
    add_ref(vars);
    return(cons((node_ptr)vars, Nil));
  }
  tmp = add_cube_diff(dd_manager, vars, (add_ptr)car(l));
  return(cons((node_ptr)tmp, make_quantifiers(cdr(l), vars)));
}

