/**CFile***********************************************************************

  FileName    [absSimulate.c]

  PackageName [abs]

  Synopsis    [High-level routines to automatically refine abstractions.]

  Description []

  Author      [Yuan Lu]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "absInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
static int q_pointer;
static int queue[1024];

/*---------------------------------------------------------------------------*/
/* Static function declarations                                              */
/*---------------------------------------------------------------------------*/
static void init_queue ARGS((void));
static int push ARGS((node_ptr));
static bdd_ptr absTranslateVar ARGS((node_ptr, node_ptr));
static node_ptr absSimulateTraceMono ARGS((node_ptr));
static node_ptr absSimulateTraceConj ARGS((node_ptr)); 
static node_ptr absSimulateTraceConj2 ARGS((node_ptr)); 
static node_ptr absSimulateTraceConj3 ARGS((node_ptr)); 

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Given a trace, simulate the concrete machine.]

  Description        [Given a trace, simulate the concrete machine. It returns
		      the state from which the transition is fake; if return
		      value is Nil, the trace is a legal trace in the concrete
		      machine.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
node_ptr Abs_AbsSimulateTrace(node_ptr trace)
{
  domain_exchange();
  if(opt_monolithic(options)) {
    return absSimulateTraceMono(trace);
  } else if (opt_conj_partitioning(options)) {
    return absSimulateTraceConj3(trace);
    /* return absSimulateTraceConj2(trace); */
  } else if (opt_disj_partitioning(options)) {
    rpterr("Not implemented yet");
  } else if (opt_iwls95cp_partitioning(options)) {
    rpterr("Not implmeneted yet");
  }
  domain_exchange();
  return Nil;
}

/**Function********************************************************************

  Synopsis           [Read a trace from a file, and generate BDDs for
		      this trace.]

  Description        [Read a trace from a file, and generate BDDs for
		      this trace.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
node_ptr Abs_AbsReadTrace(char *file_name)
{
  if (file_name!= NULL) {
    extern int yylex();
    node_ptr t=Nil;
    bdd_ptr f = bdd_one(dd_manager);
    bdd_ptr g;
    int token;

    Parser_OpenInput(file_name);
    token = yylex();
    while (token) {
      node_ptr var = Nil;
      while (token != NUMBER) {
        if (token != ATOM) rpterr("syntax error");
        var = find_node(DOT, var, find_atom(yylval.node));
        token = yylex();
        while( token == LB ) {
          token = yylex();
          if ( token != NUMBER ) rpterr("syntax error");
          var = find_node(ARRAY, var, 
		  find_node(NUMBER, (node_ptr)eval_num(yylval.node, Nil), Nil));
          token = yylex();
          if (token != RB ) rpterr("syntax error");
          token = yylex();
        }
        if (token != DOT) break;
        token = yylex();
      }
      if(var) {
	node_ptr val = (token==NUMBER) ?
		find_node(NUMBER, (node_ptr)eval_num(yylval.node, Nil), Nil):
		find_atom(yylval.node);
	g = absTranslateVar(var, val);
        bdd_and_accumulate(dd_manager, &f, g);
        bdd_free(dd_manager, g);
      } else {
        t = cons((node_ptr)f, t);
        f = bdd_one(dd_manager);
      }
      token = yylex();
    }
    t = cons((node_ptr)f, t);
    t = reverse(t);
    Parser_CloseInput();
    return(t);
  }
  return Nil;
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
static void init_queue()
{
  q_pointer = 0;
}

static int push(node_ptr val)
{
  int k = q_pointer;
  int *q = queue;
  int i = (int) car(val);
  for(;k;k--,q++)
    if(*q==i) return 0;
  *q = i;
  q_pointer++;
}

static bdd_ptr absTranslateVar(node_ptr vname, node_ptr vval)
{
  add_ptr var = eval(vname, Nil);
  add_ptr val = add_leaf(dd_manager, find_atom(vval));
  add_ptr rel = add_equal(dd_manager, var, val);
  add_ptr res = add_to_bdd(dd_manager, rel);

  add_free(dd_manager, val); add_free(dd_manager, rel);
  return res;
}

static node_ptr absSimulateTraceMono(node_ptr trace)
{
  node_ptr p = cdr(trace);
  bdd_ptr zero = bdd_zero(dd_manager);

  for(;cdr(p);p=cdr(p)) {
    bdd_ptr cur_state = (bdd_ptr) car(p);
    bdd_ptr nex_state = (bdd_ptr) car(cdr(p));
    bdd_ptr f;
    f = Img_ImageFwd(cur_state);
    bdd_and_accumulate(dd_manager, &f, nex_state);
    if(f == zero) {
      bdd_free(dd_manager, zero); bdd_free(dd_manager, f);
      return p;
    }
  }
  return Nil;
}

static node_ptr absSimulateTraceConj(node_ptr trace)
{
  node_ptr p = cdr(trace);
  bdd_ptr  zero = bdd_zero(dd_manager);

  for(;cdr(p);p=cdr(p)) {
    bdd_ptr cur_state = (bdd_ptr) car(p);
    bdd_ptr nex_state = (bdd_ptr) car(cdr(p));
    bdd_ptr f, g;
    f = Img_ImageFwd(cur_state);
    g = bdd_and(dd_manager, f, nex_state);
    if(g == zero) {
      bdd_free(dd_manager, zero); bdd_free(dd_manager, f);
      return p;
    }
  }
  return Nil;
}

static node_ptr absSimulateTraceConj2(node_ptr trace)
{
  node_ptr p = cdr(trace);
  node_ptr q = forward_quantifiers_bdd;
  bdd_ptr  zero = bdd_zero(dd_manager);
  bdd_ptr  allv = bdd_one(dd_manager);

  for(;q;q=cdr(q))
    bdd_and_accumulate(dd_manager, &allv, (bdd_ptr)car(q));
  for(;cdr(p);p=cdr(p)) {
    bdd_ptr  cur_state = (bdd_ptr) car(p);
    bdd_ptr  nex_state = (bdd_ptr) car(cdr(p));
    node_ptr t_r_c = cp_trans_bdd;
    while(t_r_c){
      bdd_ptr tmp_1, tmp_2, tmp_3, tmp_4;
      tmp_1 = bdd_and(dd_manager, cur_state, (bdd_ptr)car(t_r_c));
      tmp_2 = bdd_forsome(dd_manager, tmp_1, allv);
      tmp_3 = bdd_shift_backward(dd_manager, tmp_2);
      tmp_4 = bdd_and(dd_manager, tmp_3, nex_state);
      bdd_free(dd_manager, tmp_1); bdd_free(dd_manager, tmp_2);
      bdd_free(dd_manager, tmp_3); bdd_free(dd_manager, tmp_4);
      if(tmp_4 == zero) {
        printf("Not Real Trace\n");
	exit(0);
      }
      t_r_c = cdr(t_r_c);
    }
  }
  return Nil;
}

static node_ptr absSimulateTraceConj3(node_ptr trace)
{
  node_ptr p = cdr(trace);
  node_ptr q = forward_quantifiers_bdd;
  bdd_ptr  zero = bdd_zero(dd_manager);
  bdd_ptr  allv = bdd_one(dd_manager);

/*
  p = cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(cdr(p)))))))))))))));
*/
  for(;q;q=cdr(q))
    bdd_and_accumulate(dd_manager, &allv, (bdd_ptr)car(q));
  for(;cdr(p);p=cdr(p)) {
    bdd_ptr cur_state = (bdd_ptr) car(p);
    bdd_ptr nex_state = (bdd_ptr) car(cdr(p));
    bdd_ptr f, g;
    node_ptr vars;
    f = Img_ImageFwd(cur_state);
    g = bdd_and(dd_manager, f, nex_state);
    if(g!=zero) continue;
    {
      int c;
      g = bdd_dup(f);
      for(vars=all_variables;vars;vars=cdr(vars)) {
        add_ptr var = (add_ptr)car(lookup_symbol_hash(car(vars)));
        add_ptr varSa = add_support(dd_manager, var);
        bdd_ptr varSb = add_to_bdd(dd_manager, varSa);
        bdd_ptr var_ = bdd_forsome(dd_manager, allv, varSb);
        bdd_ptr onevar = bdd_forsome(dd_manager, nex_state, var_);
        bdd_ptr res = bdd_and(dd_manager, g, onevar);
        if(res == zero) {
          c++;
          debug_print_node(car(vars));
        }
        add_free(dd_manager, varSa);
        bdd_free(dd_manager, varSb);
        bdd_free(dd_manager, var_);
        bdd_free(dd_manager, onevar);
        bdd_free(dd_manager, g);
        g = res;
      }
    }
/*
    while(1) {
      bdd_ptr one_state;
      bdd_ptr one_state_;
      int c = 0;

      one_state = bdd_pick_one_state(dd_manager, f);
      one_state_ = bdd_not(dd_manager, one_state);
      bdd_and_accumulate(dd_manager, &f, one_state_);
      g = bdd_and(dd_manager, one_state, nex_state);
      if(f==zero) {
        printf("finished\n");
        break;
      }
      for(vars=all_variables;vars;vars=cdr(vars)) {
        add_ptr var = (add_ptr)car(lookup_symbol_hash(car(vars)));
        add_ptr varSa = add_support(dd_manager, var);
        bdd_ptr varSb = add_to_bdd(dd_manager, varSa);
        bdd_ptr var_ = bdd_forsome(dd_manager, allv, varSb);
        bdd_ptr onevar = bdd_forsome(dd_manager, nex_state, var_);
        g = bdd_and(dd_manager, one_state, onevar);
        if(g == zero) {
	  c++;
	  debug_print_node(car(vars));
	}
        add_free(dd_manager, varSa);
        bdd_free(dd_manager, varSb);
        bdd_free(dd_manager, var_);
        bdd_free(dd_manager, onevar);
        bdd_free(dd_manager, g);
      }
      if(c==0) {
        printf("Great"); break;
      }
    }
*/
  }
  return Nil;
}

