/**CFile***********************************************************************

  FileName    [absRefine.c]

  PackageName [abs]

  Synopsis    [High-level routines to automatically refine abstractions.]

  Description []

  Author      [Yuan Lu]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "absInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
static int q_pointer;
static int queue[1024];
static node_ptr abs_ref_rel=Nil;

/*---------------------------------------------------------------------------*/
/* Static function declarations                                              */
/*---------------------------------------------------------------------------*/
static node_ptr absMapStates ARGS((node_ptr));
static node_ptr absTraceImage ARGS((node_ptr, bdd_ptr*));
static node_ptr absExistLoop ARGS((node_ptr));
static void absPickSingleTrace ARGS((node_ptr));
static void absRefineAbs ARGS((bdd_ptr, bdd_ptr, bdd_ptr));
static void absRefineAbsAll ARGS((bdd_ptr));
static int  absCheckLoop ARGS((node_ptr, node_ptr, node_ptr));
static void absRefineAbsOptimal ARGS((bdd_ptr, bdd_ptr, bdd_ptr));
static node_ptr absTraceLoop ARGS((node_ptr, bdd_ptr*));
static add_ptr  absReEncode ARGS((node_ptr, int));
static void absRefineRelation ARGS(());
static node_ptr absRefineRelationRecur ARGS((node_ptr, node_ptr));
static bdd_ptr absBuildCube ARGS((int, int));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Refine the abstraction.]

  Description        [Refine the abstraction. When the counterexample is
		      a real counterexample, return 1; otherwise, return 0.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
int Abs_AbsRefineAbstraction(node_ptr atr)
{
  bdd_ptr f, p;
  node_ptr ast, t;
  node_ptr ctr, cst;
  int rtn=0;

  /* Clean state sets */
  for(ast=atr;ast;ast=cdr(ast)) {
    bdd_ptr g = (bdd_ptr) car(ast);
    Abs_AbsExAbsAll(&g);
    ast->left.bddtype = g;
  }

  /* change domain to concrete */
  domain_exchange();

  /* build refinement relation */
  absRefineRelation();
  abs_ref_rel = reverse(abs_ref_rel);

  /* Map to concrete states */
  ctr = absMapStates(atr);

  /* Symbolic simulate <ctr> */
  f = bdd_and(dd_manager, init_bdd, (bdd_ptr)car(ctr));
  cst = absTraceImage(ctr, &f);

  if(cst) { /* not reachable to last abstract state */
    for(ast=atr,t=ctr;t!=cst;ast=cdr(ast),t=cdr(t));
    absRefineAbs((bdd_ptr)car(ast), (bdd_ptr)car(cst), f);
  } else if(!(ast=absExistLoop(atr))) {
    printf("There is a counterexample\n");
    return 1; 
  } else  /* may include a real loop */
    rtn = absCheckLoop(atr, ast, ctr);

  /* free the BDDs in the array */
  walk_dd(dd_manager, (VPFDD)bdd_free, ctr);

  /* ???? change domain back to abstract */
  domain_exchange();
  return rtn;
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Refine abstraction given the original values.]

  Description	     [Refine abstraction given the original values.]

  SeeAlso            []

******************************************************************************/
static node_ptr absMapStates(node_ptr trace)
{
  if(!trace) return Nil;
  {
    node_ptr abs;
    node_ptr list = absMapStates(cdr(trace));
    bdd_ptr  f = bdd_dup((bdd_ptr)car(trace));
    bdd_ptr  one = bdd_one(dd_manager);

    for(abs=abs_ref_rel;abs;abs=cdr(abs)) {
      bdd_ptr absRl = (bdd_ptr) car(car(abs));
      bdd_ptr absVr = (bdd_ptr) cdr(car(abs));
      bdd_ptr  res = bdd_and_abstract(dd_manager, f, absRl, absVr);
      bdd_free(dd_manager, f);
      f = res;
    }
    return cons((node_ptr) f, list);
  }
}

/**Function********************************************************************

  Synopsis           [Given a sequence of states, compute a trace constrainted 
  by the sequence.]

  Description	     [Given a sequence of states, compute a trace constrainted 
  by the sequence.]

  SeeAlso            []

******************************************************************************/
static node_ptr absTraceImage(node_ptr trace, bdd_ptr *subset)
{
  node_ptr tr = trace;
  bdd_ptr  s2, s1 = *subset;
  bdd_ptr  zero = bdd_zero(dd_manager);

  for(;cdr(tr);tr=cdr(tr)) {
    node_ptr ntr = cdr(tr);
    bdd_and_accumulate(dd_manager, &s1, invar_bdd);
    s2 = Img_ImageFwd(s1);
    bdd_and_accumulate(dd_manager, &s2, (bdd_ptr)car(ntr));
    if(s2==zero) {*subset=s1; return tr;}
    ntr->left.bddtype = s2;
    s1 = bdd_dup(s2);
  }
  return Nil;
}

/**Function********************************************************************

  Synopsis           [Check if there is a loop in the trace.]

  Description        [Check if there is a loop in the trace. If true, return
  the first state which the loop starts from; otherwise, return NULL.]

  SeeAlso            []

******************************************************************************/
static node_ptr absExistLoop(node_ptr trace)
{
  node_ptr last_state = last(trace);
  node_ptr p = trace;
  for(;cdr(p);p=cdr(p))
    if(car(p) == last_state) return p;
  return Nil;
}

/**Function********************************************************************

  Synopsis           [Pick a single trace tr.]

  Description        [Pick a single trace tr.]

  SeeAlso            []

******************************************************************************/
static void absPickSingleTrace(node_ptr tr)
{
  bdd_ptr t = init_bdd_orig1;
  for(;cdr(tr);tr=cdr(tr)) {
    bdd_ptr s;
    bdd_free(dd_manager, (bdd_ptr)car(tr));
    tr->left.bddtype = t;
    s = Img_ImageFwd(t);
    bdd_and_accumulate(dd_manager, &s, (bdd_ptr)car(cdr(tr)));
    bdd_free(dd_manager, t);
    t = bdd_pick_one_state(dd_manager, s);
    bdd_free(dd_manager, s);
  }
}

/**Function********************************************************************

  Synopsis           [Refining the abstraction.]

  Description        [Refining the abstraction. abs_state is the failure
  state while crt_state is the set of states corresponding to the failure
  state. par is the deadend states.]

  SeeAlso            []

******************************************************************************/
static void absRefineAbs(bdd_ptr abs_state, bdd_ptr crt_state, bdd_ptr par)
{
  /* releasing all the abstractions dumply */
  /* absRefineAbsAll(abs_state); */

  /* Using two-level logic minimization technique */
  absRefineAbsOptimal(abs_state, crt_state, par);
}

/**Function********************************************************************

  Synopsis           [All the variables involved in the abstract state are released.]

  Description        [All the variables involved in the abstract state are released.]

  SeeAlso            []

******************************************************************************/
static void absRefineAbsAll(bdd_ptr abstate)
{
  node_ptr abs = abs_expr_pre;
  add_ptr  abs_state = bdd_to_add(dd_manager, abstate);
  for(;abs;abs=cdr(abs)) {
    node_ptr aval = car(cdr(car(abs)));
    node_ptr nval = cdr(lookup_symbol_hash(car(car(abs))));
    add_ptr var = (add_ptr) car(lookup_abstract_hash(car(car(abs))));
    add_ptr tmp = add_if_then(dd_manager, abs_state, var);
    int x, abs_val = (int) car(add_value(dd_manager, tmp));
    node_ptr t=aval;

    add_free(dd_manager, tmp);
    for(x=0;t;t=cdr(t))
      if(x < ((unsigned)car(car(t)))) x = ((unsigned)car(car(t)));
    for(;aval;aval=cdr(aval),nval=cdr(nval)) {
      if(abs_val == ((unsigned)car(car(aval))))
        aval->left.nodetype = find_node(NUMBER, (node_ptr)++x, 0);
    }
  }
  for(abs=abs_expr;abs;abs=cdr(abs)) {
    node_ptr aval = car(cdr(car(abs)));
    node_ptr nval = cdr(lookup_symbol_hash(car(car(abs))));
    add_ptr var = (add_ptr) car(lookup_abstract_hash(car(car(abs))));
    add_ptr tmp = add_if_then(dd_manager, abs_state, var);
    int x, abs_val = (int) car(add_value(dd_manager, tmp));
    node_ptr t=aval;

    add_free(dd_manager, tmp);
    for(x=0;t;t=cdr(t))
      if(x < ((unsigned)car(car(t)))) x = ((unsigned)car(car(t)));
    for(;aval;aval=cdr(aval),nval=cdr(nval)) {
      if(abs_val == ((unsigned)car(car(aval))))
	aval->left.nodetype = find_node(NUMBER, (node_ptr)++x, 0);
    }
  }
  add_free(dd_manager, abs_state);
}

/**Function********************************************************************

  Synopsis           [Use CAV00 refining algorithm to refine abstraction.]

  Description        [Use CAV00 refining algorithm to refine abstraction.]

  SeeAlso            []

******************************************************************************/
static void absRefineAbsOptimal(bdd_ptr abs_state, bdd_ptr crt_state, bdd_ptr par)
{
  node_ptr allvars = all_variables;
  bdd_ptr  supp = bdd_one(dd_manager);
  add_ptr  abstate = bdd_to_add(dd_manager, abs_state);
  bdd_ptr  f;

  /* quantify out un-abstracted variables to improve the performance */
  for(;allvars;allvars=cdr(allvars)) {
    if(!lookup_abstract_hash(car(allvars))) {
      add_ptr var = (add_ptr) car(lookup_symbol_hash(car(allvars)));
      add_ptr sa = add_support(dd_manager, var);
      bdd_ptr sb = add_to_bdd(dd_manager, sa);
      bdd_and_accumulate(dd_manager, &supp, sb);
      add_free(dd_manager, sa); bdd_free(dd_manager, sb);
    }
  }
  f = bdd_forsome(dd_manager, par, supp);

  /* check if f(x_i=d_1) == f(x_i=d_2) */
  {
    node_ptr abs;
    int vrc;
    static bdd_ptr vrl[63];

    for(abs=abs_expr;abs;abs=cdr(abs)) {
      node_ptr name_list = car(car(car(abs)));
      for(;name_list;name_list=cdr(name_list))
        if(lookup_coi_hash(car(name_list))) break;
      if(name_list) {
	node_ptr lp, lq, min_list=Nil;
	add_ptr  af, ag, ah;
	bdd_ptr  g, h, p, r, bddvarc, bddvara, fixval;
	add_ptr  addvara = (add_ptr) car(cdr(cdr(car(abs))));
	bdd_ptr  rel = (bdd_ptr) cdr(cdr(cdr(car(abs))));
	bdd_ptr  zero = bdd_zero(dd_manager);

	name_list = car(car(car(abs)));

	/* build the the cube for concrete variables */
	bddvarc = bdd_one(dd_manager);
	for(;name_list;name_list=cdr(name_list)) {
	  add_ptr addv = (add_ptr) car(lookup_symbol_hash(car(name_list)));
	  ag = add_support(dd_manager, addv);
	  g = add_to_bdd(dd_manager, ag);
	  r = bdd_and(dd_manager, bddvarc, g);
	  add_free(dd_manager, ag); bdd_free(dd_manager, g);
	  bdd_free(dd_manager, bddvarc);
	  bddvarc = r;
	}

	/* build the cube for the abstract variables */
	ag = add_support(dd_manager, addvara);
	bddvara = add_to_bdd(dd_manager, ag);
	add_free(dd_manager, ag);

	/* extract the set of concrete values for this abstract state 
	   this set is stored in a list <min_list>.
	*/
	af = add_if_then(dd_manager, abstate, addvara);
	ag = add_leaf(dd_manager, add_value(dd_manager, af));
	ah = add_equal(dd_manager, addvara, ag);
	h = add_to_bdd(dd_manager, ah);
	g = bdd_and(dd_manager, h, rel);
	fixval = bdd_not(dd_manager, g);

	add_free(dd_manager, af); add_free(dd_manager, ag); 
	add_free(dd_manager, ah); bdd_free(dd_manager, h);

	h = bdd_forsome(dd_manager, g, bddvara);
	bdd_free(dd_manager, g); g = h;

	/* translate the cube into a list of variables */
	vrc = bdd_cube_to_vars(dd_manager, bddvarc, vrl);

	/* enumerate each minterm and store them in min_list */
	while(g != zero) {
	  r = bdd_pick_one_minterm(dd_manager, g, vrl, vrc);
	  h = bdd_not(dd_manager, r);
	  p = bdd_and(dd_manager, g, h);

	  bdd_free(dd_manager, h); bdd_free(dd_manager, g);
	  g = p;

	  min_list = cons((node_ptr) r, min_list);
	}

	/* cluster the set of concrete values into equivalence classes */
        lp = min_list;
	for(lp=min_list;lp;lp=cdr(lp)) {
	  add_ptr rel1 = bdd_to_add(dd_manager, (bdd_ptr)car(lp));
	  lq = lp;
	  while(cdr(lq)) {
	    node_ptr lq1 = cdr(lq);
	    add_ptr rel2 = bdd_to_add(dd_manager, (bdd_ptr)car(lq1));
	    if(bdd_cofactor_equal(dd_manager, f, rel1, rel2)) {
	      bdd_ptr lpa = (bdd_ptr) car(lp);
	      bdd_ptr lpb = (bdd_ptr) car(lq1);
	      bdd_or_accumulate(dd_manager, &lpa, lpb);
	      lp->left.bddtype = lpa;
	      lq->right.nodetype = cdr(lq1);
	      bdd_free(dd_manager, lpb);
	      free_node(lq1);
	    } else 
	      lq = cdr(lq);
	    add_free(dd_manager, rel2);
	  }
	  add_free(dd_manager, rel1);
	}

	/* generate values for each equivalence class and build
	   abstraction relation h(x)=x".
	 */
	g = bdd_forsome(dd_manager, rel, bddvarc);
	h = bdd_not(dd_manager, g);
	bdd_free(dd_manager, g);
	lp = min_list;
	r = bdd_zero(dd_manager);

	vrc = bdd_cube_to_vars(dd_manager, bddvara, vrl);

	while(h != zero && lp != 0) {
	  bdd_ptr h1 = bdd_pick_one_minterm(dd_manager, h, vrl, vrc);
	  bdd_ptr h2 = bdd_not(dd_manager, h1);
	  g = bdd_and(dd_manager, (bdd_ptr) car(lp), h1);
	  p = bdd_or(dd_manager, g, r);

	  bdd_free(dd_manager, h1); bdd_free(dd_manager, g);
	  bdd_free(dd_manager, r); 

	  h1 = bdd_and(dd_manager, h, h2);
	  bdd_free(dd_manager, h); bdd_free(dd_manager, h2);

	  h = h1; r = p;
	  lp = cdr(lp);
	}

	/* generate the abstraction relation */
	bdd_free(dd_manager, h);
	if(lp) { /* too much refinement implies that not abstract */
	  bdd_free(dd_manager, rel);
	  cdr(cdr(car(abs)))->right.bddtype = Nil;
	} else { /* refine the abstraction relation */
	  g = bdd_and(dd_manager, fixval, rel);
	  cdr(cdr(car(abs)))->right.bddtype = bdd_or(dd_manager, g, r);

	  bdd_free(dd_manager, g); bdd_free(dd_manager, rel);
	}
      }
    }
    for(abs=abs_expr;abs;abs=cdr(abs)) {
      if(cdr(cdr(cdr(car(abs)))) != Nil) break;
      add_free(dd_manager, (add_ptr) car(cdr(cdr(car(abs)))));
      /* Abs_AbsFreeAbs(car(abs)); */
      free_node(abs);
    }
    abs_expr = abs;
    if(abs)
      while(cdr(abs)) {
        if(cdr(cdr(cdr(car(cdr(abs))))) == Nil) {
	  node_ptr t = cdr(abs);
	  abs->right.nodetype = cdr(t);
	  add_free(dd_manager, (add_ptr) car(cdr(cdr(car(t)))));
	  /* Abs_AbsFreeAbs(car(t)); */
	  free_node(t);
        } else
          abs = cdr(abs);
      }
  }
  bdd_free(dd_manager, supp);
  add_free(dd_manager, abstate);
}

/**Function********************************************************************

  Synopsis           [Check if there is a loop.]

  Description        [Check if there is a loop.]

  SeeAlso            []

******************************************************************************/
static int absCheckLoop(node_ptr atr, node_ptr ast, node_ptr ctr)
{
  bdd_ptr sum = bdd_zero(dd_manager);
  bdd_ptr loop;
  node_ptr t, t1;
  for(t=ctr, t1=atr;t1!=ast;t=cdr(t), t1=cdr(t1));
  t1 = t;
  for(t=cdr(t);t;t=cdr(t))
    bdd_or_accumulate(dd_manager, &sum, (bdd_ptr)car(t));
  loop = eg(sum);
  bdd_and_accumulate(dd_manager, &loop, sum);
  if(loop == bdd_zero(dd_manager)) {
    bdd_ptr  f;
    node_ptr ast1, t;
    node_ptr bad_state = absTraceLoop(t1, &f);
    for(ast1=ast,t=t1;t!=bad_state;ast1=cdr(ast1),t=cdr(t));
    absRefineAbs((bdd_ptr)car(ast1), (bdd_ptr)car(bad_state), f);
  } else {	/* concrete counterexample */
    printf("There is a loop counterexample\n");
    return 1;
  }
  bdd_free(dd_manager, sum);
  bdd_free(dd_manager, loop);
  return 0;
}

/**Function********************************************************************

  Synopsis           [Check if there is a loop.]

  Description        [Check if there is a loop.]

  SeeAlso            []

******************************************************************************/
static node_ptr absTraceLoop(node_ptr start, bdd_ptr *part)
{
  bdd_ptr zero = bdd_zero(dd_manager);
  node_ptr loop = start;
  while(1) {
    node_ptr nloop = (cdr(cdr(loop))) ? cdr(loop) : start;
    bdd_ptr curst = (bdd_ptr) car(loop);
    bdd_ptr nexst = (bdd_ptr) car(nloop);
    bdd_ptr sp = Img_ImageFwd(curst);
    bdd_and_accumulate(dd_manager, &sp, nexst);
    if(sp==zero) {
      *part = bdd_dup(curst);
      return loop;
    } else {
      bdd_free(dd_manager, nexst);
      nloop->left.bddtype = sp;
    }
    loop = nloop;
  }
}

/**Function********************************************************************

  Synopsis           [Generate a new add variable.]

  Description        [Generate a new add variable.]

  SeeAlso            []

******************************************************************************/
static add_ptr absReEncode(node_ptr range, int curlevel)
{
  if (range == Nil) internal_error("absReEncode: range = Nil");

  /* Base case: We reached a leaf */
  if (cdr(range) == Nil) {
    node_ptr v = find_atom(car(range));
    add_ptr temp = (add_ptr)lookup_constant_hash(v);
    if (temp) {
      add_ref(temp);
      return(temp);
    }
    temp = add_leaf(dd_manager, v);
    if (v && node_get_type(v) == ATOM) {
      insert_constant_hash(v, (node_ptr)temp);
      insert_check_constant_hash(v, v);
    }
    add_ref(temp);
    return(temp);
  }

  /* Step case: */
  curlevel += 2;
  {
    /*
        The element list are decomposed and left and right are
        allocated in the bdd
      */
    node_ptr lhalf, rhalf;
    add_ptr p0, p1, result;
    split_power_half(range, &lhalf, &rhalf);
    p0 = absReEncode(lhalf, curlevel);
    p1 = absReEncode(rhalf, curlevel);
    result = add_build(dd_manager, curlevel - 2, p1, p0);

    for(;lhalf;lhalf=cdr(lhalf)) free_node(lhalf);
    add_free(dd_manager, p0);
    add_free(dd_manager, p1);
    return(result);
  }
}

/**Function********************************************************************

  Synopsis           [Return log |list|.]

  Description        [Return log |list|.]

  SeeAlso            []

******************************************************************************/
int absCountPower(node_ptr list)
{
  int i=0;
  for(;list;list=cdr(list),i++);
  if(i<3) return 1;
  else if(i<5) return 2;
  else if(i<9) return 3;
  else if(i<17) return 4;
  else if(i<33) return 5;
  else if(i<65) return 6;
  else if(i<129) return 7;
  else if(i<257) return 8;
  else if(i<513) return 9;
  else if(i<1025) return 10;
  else if(i<2049) return 11;
  else if(i<4097) return 12;
  else if(i<8193) return 13;
  else if(i<16385) return 14;
  rpterr("Run our of range");
}

/**Function********************************************************************

  Synopsis           [Build partitioned product of abstraction-refinement
  relation : h(x) = y.]

  Description        [Build partitioned product of abstraction-refinement
  relation : h(x) = y.]

  SeeAlso            [Abs_AbsClusterAbs]

******************************************************************************/
static void absRefineRelation()
{
  bdd_ptr one = bdd_one(dd_manager);
  node_ptr last = cons(cons((node_ptr)one, (node_ptr)bdd_dup(one)), Nil);
  while(abs_ref_rel) {
    bdd_free(dd_manager, (bdd_ptr)cdr(car(abs_ref_rel)));
    bdd_free(dd_manager, (bdd_ptr)car(car(abs_ref_rel)));
    abs_ref_rel = cdr(abs_ref_rel);
  }
  abs_ref_rel = absRefineRelationRecur(abs_expr, last);
}

/**Function********************************************************************

  Synopsis           [Recursively build partitioned produce of abstraction
  relation : h(x) = y.]

  Description        [Recursively build partitioned produce of abstraction
  relation : h(x) = y.]

  SeeAlso            [Abs_AbsClusterAbs]

******************************************************************************/
static node_ptr absRefineRelationRecur(node_ptr abs, node_ptr last)
{
  node_ptr t;
  bdd_ptr one = bdd_one(dd_manager);
  if(!abs) {
    return cons(cons((node_ptr)one, (node_ptr)bdd_dup(one)), last);
  }
  t = absRefineRelationRecur(cdr(abs), last);
  {
    node_ptr abs1 = car(abs);
    bdd_ptr  absR = (bdd_ptr) cdr(cdr(cdr(abs1)));

    if(absR == one) {
      node_ptr var, varl = car(car(abs1));
      node_ptr form = cdr(car(abs1));
      bdd_ptr vsuppb, f;
      int aid, max=-1, pl;

      for(;varl;varl=cdr(varl)) {
	node_ptr v = lookup_symbol_hash(car(varl));
	int cid = add_read_index((add_ptr)car(v));
	if(cid > max) {
	  max = cid; var = v;
	}
      }
      aid = max + absCountPower(cdr(var))*2;
      for(pl=0;form;pl++,form=cdr(form));
      vsuppb = absBuildCube(aid, pl);
      f = bdd_and(dd_manager, vsuppb, (bdd_ptr)cdr(car(last)));
      last->left.nodetype->right.bddtype = f;
      bdd_free(dd_manager, vsuppb);
      return t;
    } else {
      add_ptr  var = (add_ptr) car(cdr(cdr(abs1)));
      add_ptr  vsuppa = add_support(dd_manager, var);
      bdd_ptr  vsuppb = add_to_bdd(dd_manager, vsuppa);
      bdd_ptr g = (bdd_ptr) car(car(t));
      bdd_ptr f = bdd_and(dd_manager, absR, g);
      if(bdd_size(dd_manager, f) < 5000) {
	bdd_free(dd_manager, g);
        t->left.nodetype->left.bddtype = f;
	g = (bdd_ptr) cdr(car(t));
        t->left.nodetype->right.bddtype = bdd_and(dd_manager, vsuppb, g);
	bdd_free(dd_manager, g);
	return t;
      } else
	return cons(cons((node_ptr)f, (node_ptr)vsuppb), t);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Build a cube]

  Description        [Build a cube. Given the first index (startidx) and 
  the length (lenidx), build a cube whose indices range from startidx to
  startidx+lenidx.]

  SeeAlso            []

******************************************************************************/
static bdd_ptr absBuildCube(int startidx, int lenidx)
{
  int p;
  bdd_ptr f = bdd_one(dd_manager);

  startidx-=2;
  for(p=startidx+lenidx;p>startidx;p-=2) {
    bdd_ptr g = bdd_new_var_with_index(dd_manager, p);
    bdd_ptr t = bdd_and(dd_manager, f, g);
    bdd_free(dd_manager, f);
    bdd_free(dd_manager, g);
    f = t;
  }
  return f;
}

