/**CFile***********************************************************************

  FileName    [absAutoGenAbs.c]

  PackageName [abs]

  Synopsis    [automatically generate abstraction functions;
  Given a variable x \in {d1, d2, ...} and M = M_1(x) || N,
  if N(x=d1) \equiv N(x=d2), then h(d1) = h(d2)

  Description [
  ]

  SeeAlso     [compileMono.c, compileDisj.c, compileConj.c]

  Author      [Yuan Lu]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "absInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: $";
#define DEBUG_IWLS95 0

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
int num_val;
node_ptr abs_expr_pre=Nil;
node_ptr abs_expr=Nil;
node_ptr list_atom=Nil;

static node_ptr st[1024];

/**Variable********************************************************************

  Synopsis    [The abstract hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>.
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     [instantiate_var compileEncodeVar Compile_BuildVarsBdd]

******************************************************************************/
static hash_ptr abstract_hash;
void init_abstract_hash() {abstract_hash = new_assoc();}
void insert_abstract_hash(node_ptr x, node_ptr y) { insert_assoc(abstract_hash, x, y);}
node_ptr lookup_abstract_hash(node_ptr x) {return(find_assoc(abstract_hash, x));}
static assoc_retval free_abstract_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_abstract_hash() {clear_assoc_and_free_entries(abstract_hash, free_abstract_hash);}

/**Variable********************************************************************

  Synopsis    [The constant hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>.
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     [instantiate_var compileEncodeVar Compile_BuildVarsBdd]

******************************************************************************/
static hash_ptr constant2_hash;
void init_constant2_hash() {constant2_hash = new_assoc();}
void insert_constant2_hash(node_ptr x, node_ptr y) { insert_assoc(constant2_hash,
x, y);}
node_ptr lookup_constant2_hash(node_ptr x) {return(find_assoc(constant2_hash, x));
}
static assoc_retval free_constant2_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_constant2_hash() {clear_assoc_and_free_entries(constant2_hash, free_constant2_hash);}

/**Variable********************************************************************

  Synopsis    [The range hash.]

  Description [This hash associates the following data structure to
  each variable <code>vname</code>:<br>
       <center><code>&lt;VAR, encoding, range&gt;</code></center><br>
  where <code>range</code> is the range of the variable <code>vname</code> and
  <code>encoding</code> is the ADD relative to the boolean encoding of the
  range <code>range</code>.
  And to each <code>Pi.running</code> the following:
       <center><code>&lt;BDD, ADD, Nil&gt;</code></center><br>
  where ADD is the ADD representing <code>process_selector == Pi</code>.
  And to each <code>DEFINE vname = ...</code> the following:
       <center><code>&lt;CONTEXT, context, definition&gt;</code></center>
  where context is the context in which the definition appear, and
  definition is the body of the definition itself.]

  SeeAlso     [instantiate_var compileEncodeVar Compile_BuildVarsBdd]

******************************************************************************/
static hash_ptr range_hash;
void init_range_hash() {range_hash = new_assoc();}
void insert_range_hash(node_ptr x, node_ptr y) { insert_assoc(range_hash, x, y);}
node_ptr lookup_range_hash(node_ptr x) {return(find_assoc(range_hash, x));}
static assoc_retval free_range_hash(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  if (element != (node_ptr)NULL) {
    switch(node_get_type(element)){
    case VAR:
    case BDD: {
      add_ptr enc = (add_ptr)car(element);

      if (enc != (add_ptr)NULL) add_free(dd_manager, enc);
      break;
    }
    default:
      break;
    }
  }
  return(ASSOC_DELETE);
}
void clear_range_hash() {clear_assoc_and_free_entries(range_hash, free_range_hash);}

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static node_ptr absGenListFormulaRecur ARGS((node_ptr, node_ptr, int, node_ptr *));
static void absElabFormula ARGS((node_ptr));
static node_ptr absChooseAbs ARGS((node_ptr));
static node_ptr absClusterVarFormula ARGS((node_ptr));
static node_ptr absGenListVarRecur ARGS((node_ptr));
static void absSplitAbs ARGS((node_ptr));
static int absCheckEquivForTwoValue ARGS((add_ptr, add_ptr, add_ptr));
static node_ptr absAutoGenAbsPreRecur ARGS((node_ptr, node_ptr, int*, node_ptr, node_ptr, int));
static int absFindPos ARGS((node_ptr, node_ptr));
static void absNoAbs ARGS((node_ptr, int*));
static void absEqual ARGS((node_ptr, int, int*));
static void absBisect ARGS((node_ptr, int, int*));
static void absDivide ARGS((node_ptr, int, int*));
static void absEnum ARGS((node_ptr, node_ptr, node_ptr, int*));
static void absModulus ARGS((node_ptr, int, int*));
static int absExistVar ARGS((node_ptr, node_ptr, node_ptr));

static node_ptr absPowerList ARGS((node_ptr, node_ptr, int, int));
static void absPowerListRecur ARGS((add_ptr, int, node_ptr*));
static node_ptr absTurn2Fun ARGS((node_ptr, node_ptr, int));
static int isFunIdentity ARGS((node_ptr));
static node_ptr absRangeFun ARGS((node_ptr, int, int));
static node_ptr isFunRange ARGS((node_ptr));
static node_ptr isFunResidue ARGS((node_ptr));
static node_ptr isFunSymmetry ARGS((node_ptr));

static void absRearrange ARGS((node_ptr));
static int absRearrange1 ARGS((node_ptr));
/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Check the ABSTRACT list to see if the
  provided abstraction functions by users are reasonable or not.]

  Description        [Check the ABSTRACT list to see if the
  provided abstraction functions by users are reasonable or not.
  For a variable x \in {d1, d2, ...} and M = M1(x) || N, if
  N(x=d1) \= N(x=d2) and h(d1) = h(d2), then the abstraction
  function is not reasonable, and return 0; otherwise, return 1.
  ]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Abs_AbsCheckAbsFunc(node_ptr abs_expr, node_ptr invar_expr, node_ptr procs,
                        add_ptr assumption)
{
/* to be implemented */
}

/**Function********************************************************************

  Synopsis           [Given a list of values, translate into possible
		      functions.]

  Description        [Given a list of values, translate into possible
		      functions. This is only useful for using aBDDs.]

  SideEffects        []

  SeeAlso            [absPowerList, absTurn2Fun]

******************************************************************************/
void Abs_AbsList2Fun()
{
#ifdef ABS_ABDD
  node_ptr abs = abs_expr;
  for(;abs;abs=cdr(abs)) {
    node_ptr abs_var = car(car(abs));
    node_ptr abs_list = cdr(car(abs));
    if(node_get_type(abs_list) == CONS) {
      int num_list = num_in_list(abs_list);
      int exponent = log_2(num_list);
      abs_list = absPowerList(abs_var, abs_list, num_list, exponent);
      abs->left.nodetype = absTurn2Fun(abs_var, abs_list, power2(exponent));
      debug_print_node(car(abs)); printf("\n");
    }
  }
#endif
}

/**Function********************************************************************

  Synopsis           [Automatically generate abstraction functions.]

  Description        [Automatically generate abstraction functions.
  First, we check all the variables (including boolean variables???).
  If for two values v1, v2 in a variable var N(var=v1) = N(var=v2),
  then we defines h_var(v1)=h_var(v2). The difference of this function
  and Abs_AbsAutoGenAbsPost() is that the check is before the transition
  relation is built. This helps to decide how to encode the variables.]

  SideEffects        []

  SeeAlso            [Abs_AbsAutoGenAbsPost]

******************************************************************************/
void Abs_AbsAutoGenAbs(node_ptr procs, node_ptr specs, node_ptr fairs)
{
  node_ptr vars, abs;
  node_ptr list_atomf=Nil;

  init_abstract_hash();
  init_constant2_hash();
  init_range_hash();

  if(!abs_expr && !get_abs_level_threshold(options)) return;

  /* record all the constants with enumeration type in constant2_hash */
  for(vars=all_variables;vars;vars=cdr(vars)) {
    node_ptr var = car(vars);
    node_ptr vals = cdr(lookup_symbol_hash(find_atom(var)));
    int count = 0;
    for(;vals;vals=cdr(vals)) {
      node_ptr val = car(vals);
      if(node_get_type(val)==ATOM && !lookup_constant2_hash(val))
	insert_constant2_hash(val, val);
      count++;
    }
    insert_range_hash(var, (node_ptr) count);
  }

  /* generate list of atomic formulas */
  absGenListFormulaRecur(specs, Nil, 0, &list_atomf);
  absGenListFormulaRecur(fairs, Nil, 0, &list_atomf);
  absGenListFormulaRecur(procs, Nil, 0, &list_atomf);

  /* Some Boolean variables are briefed, therefore, we need to extend their
     meaning */
  absElabFormula(list_atomf);

  /* clustering the variables into equivalence classes */
  abs = absClusterVarFormula(list_atomf);

  /* decide which clusters we apply abstractions */
  abs = absChooseAbs(abs);

  /* decide when to apply abstractions */
  absSplitAbs(abs);

  /* add those variables which are not in the interest cone */
  for(vars=all_variables;vars;vars=cdr(vars)) {
    node_ptr var = car(vars);
    if(!lookup_interest_hash(var)) {
      node_ptr varl = cons(var, Nil);
      node_ptr forml = cons(Nil, Nil);
      node_ptr vall = cons(Nil, Nil);
      add_ptr addvar = add_one(dd_manager);
      node_ptr newabs= find_node(IMPLIES, cons(varl, forml),
                         cons(vall, cons((node_ptr)addvar, Nil)));
      abs_expr = cons(newabs, abs_expr);
    }
  }

  /* insert abstraction into hash */
  for(abs=abs_expr_pre;abs;abs=cdr(abs)) {
    node_ptr absf = car(abs);
    node_ptr var_list = car(car(absf));
    for(;var_list; var_list=cdr(var_list)) {
      node_ptr var = car(var_list);
      insert_abstract_hash(var, absf);
    }
  }

  for(abs=abs_expr;abs;abs=cdr(abs)) {
    node_ptr absf = car(abs);
    node_ptr var_list = car(car(absf));
    for(;var_list; var_list=cdr(var_list)) {
      node_ptr var = car(var_list);
      insert_abstract_hash(var, absf);
    }
  }

#ifdef ABS_DEBUG
  {
    int cabs=0;
    FILE *fp = fopen("depend", "a");
    for(abs=abs_expr;abs;abs=cdr(abs),cabs++) {
      debug_fprint_node(fp, car(abs));
      fprintf(fp, "\n");
    }
    for(abs=abs_expr_pre;abs;abs=cdr(abs),cabs++) {
      fprintf(fp, "large variables \n");
      debug_fprint_node(fp, car(car(abs)));
    }
    fprintf(fp, "The number of abstracted variables is %d\n", cabs);
    fclose(fp);
  }
#endif
}

/**Function********************************************************************

  Synopsis           [Automatically generate abstraction functions.]

  Description        [Automatically generate abstraction functions.
  First, we check all the variables (including boolean variables???).
  If for two values v1, v2 in a variable var N(var=v1) = N(var=v2),
  then we defines h_var(v1)=h_var(v2). The difference of this function
  and Abs_AbsAutoGenAbsPost() is that the check is before the transition
  relation is built. This helps to decide how to encode the variables.]

  SideEffects        []

  SeeAlso            [Abs_AbsAutoGenAbsPost]

******************************************************************************/
void Abs_AbsAutoGenAbsPre(node_ptr procs, node_ptr specs, node_ptr fairs)
{
  node_ptr vars, absf;
  node_ptr tmp;
  int i, inc;
  int d;

#ifdef ABS_DEBUG
  int n_coi=0;
  int n_abs=0;

  FILE *fp = fopen("depend", "a");
#endif

  init_abstract_hash();
  init_constant2_hash();

  if(!abs_expr && !get_abs_level_threshold(options)) return;
#ifdef ABS_DEBUG
  if(ABS_DEBUG > 1) {
    int count=0;
    for(vars=all_variables; vars; vars=cdr(vars)) {
      node_ptr var = car(vars);
      if(lookup_interest_hash(var)) {
        fprintf_node_exp(fp, var);
        if(++count!=3) fprintf(fp, "\t");
	else { count=0; fprintf(fp, "\n"); }
      }
    }
    printf("\n\n");
  }
#endif

  for(vars=all_variables;vars;vars=cdr(vars)) {
    node_ptr var = car(vars);
    node_ptr vals = cdr(lookup_symbol_hash(find_atom(var)));
    for(;vals;vals=cdr(vals)) {
      node_ptr val = car(vals);
      if(node_get_type(val)==ATOM && !lookup_constant2_hash(val))
	insert_constant2_hash(val, val);
    }
  }

  {
    node_ptr abs = abs_expr;
    for(;abs;abs=cdr(abs))
      insert_abstract_hash(car(car(abs)), new_node(VAR, Nil, cdr(car(abs))));
  }

  for(vars=all_variables; vars; vars=cdr(vars)) {
    node_ptr var = car(vars);
    node_ptr val = lookup_symbol_hash(var);

    if(get_coi(options)==1 && lookup_interest_hash(var)!=0) continue;

    for(tmp=cdr(val), num_val=0; tmp; tmp=cdr(tmp)) num_val++;
    if(lookup_interest_hash(var) && num_val < 3) continue;
    for(i=0, tmp=Nil;i<num_val;i++) tmp = cons(0, tmp);

    inc = 0;
    if(lookup_interest_hash(var)) {
      absAutoGenAbsPreRecur(var, tmp, &inc, specs, Nil, 0);
      absAutoGenAbsPreRecur(var, tmp, &inc, fairs, Nil, 0);
      absAutoGenAbsPreRecur(var, tmp, &inc, procs, Nil, 0);
    }

    if(isFunNoAbs(tmp)) continue;

    absRearrange(tmp);
    absf = new_node(IMPLIES, var, tmp);

    {
      int max=0;
      node_ptr t = tmp;
      for(;t;t=cdr(t)) {
	int val = (int)car(car(t));
	if(max < val) max = val;
      }
      if(num_val> 2 && (max+1)*3 > num_val) continue;
    }

#ifdef ABS_DEBUG
    n_abs++;
    if(!lookup_interest_hash(var)) {
      fprintf(fp, "This variable can be removed using COI\n");
      n_coi++;
    }
    if(num_val < 512) {
      debug_fprint_node(fp, absf); fprintf(fp, "\n");
    } else {
      debug_fprint_node(fp, car(absf));
      fprintf(fp, "Too many elements : %d\n", num_val);
    }
#endif
/*
    scanf("%d", &d);
    if (d==0) continue;
*/
    if(num_val > 256) abs_expr_pre = cons(absf, abs_expr_pre);
    else abs_expr = cons(absf, abs_expr);

    insert_abstract_hash(var, new_node(VAR, Nil, tmp));
  }

#ifdef ABS_DEBUG
  fprintf(fp, "Number of Abs is %d\n", n_abs);
  fprintf(fp, "Number of Coi is %d\n", n_coi);
  fclose(fp);
#endif
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Given a list of expression, then the
  corresponding list of ADD is returned.]

  Description        [Check if the proc is equivalent for different
  values (var=value1 and var=value2). When proc(var=value1) \equiv
  proc(var=value2), return 1; otherwise, return 0. Currently, I do
  not consider TRANS statements.]

  SideEffects        [None]

  SeeAlso            [compileCompileModelIwls95Recur]

******************************************************************************/
/*
static int absCheckEquivForTwoValue(add_ptr var, add_ptr val1, add_ptr val2)
{
  node_ptr list = max_part_list;
  add_ptr var_supp = add_support(dd_manager, var);
  add_ptr nvar_supp = add_shift(dd_manager, var_supp);
  add_ptr f1, f2;
  
  while(list) {
    add_ptr trans = (add_ptr) car(list);
    if(!add_is_support(dd_manager, trans, nvar_supp) &&
       add_is_support(dd_manager, trans, var_supp)) {
      f1 = add_cofactor(trans, var, val1);
      f2 = add_cofactor(trans, var, val2);
      if(f1!=f2) return 0;
    }
    list = cdr(list);
  }
  f1 = add_cofactor(invar, var, val1);
  f2 = add_cofactor(invar, var, val2);
  return 1;
}
*/


/**Function********************************************************************

  Synopsis           [Recursively go through the parse tree to generate
		      abstraction functions.]

  Description        [Recursively go through the parse tree to generate
		      abstraction functions.In the paramer list, <var>
                      reflects the variable we want to generate abstraction
                      for; <val> is the partition of the value list; <diff>
                      is a dynamic counter used for partitioning; <procs>
                      is the program we are visiting; <context> is the
                      the context of <procs>; <data>=1 implies that
                      procs is the datapath of a certain variable.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static node_ptr absAutoGenAbsPreRecur(node_ptr var, node_ptr val, int *diff,
				      node_ptr procs, node_ptr context, int data)
{
  int k;

  if(!procs) return;
  switch(node_get_type(procs)) {
    case ATOM: {
      node_ptr name  = find_node(DOT, context, find_atom(procs));
      node_ptr temp1 = lookup_symbol_hash(name);
      node_ptr temp2 = lookup_param_hash(name);
      node_ptr temp3 = lookup_constant2_hash(find_atom(procs));
      if(temp1) {
	if(node_get_type(temp1) == VAR)
	  return name;
	else
	  return absAutoGenAbsPreRecur(var, val, diff, temp1, context, data);
      }
      if(temp2)
	return absAutoGenAbsPreRecur(var, val, diff, temp2, context, data);
      if(temp3) return procs;
      else return name;
    }
    case ARRAY: return eval_struct(procs, context);
    case NUMBER: return procs;
    case DOT: {
      node_ptr name = eval_struct(procs,  context);
      node_ptr temp1 = lookup_symbol_hash(name);
      node_ptr temp2 = lookup_param_hash(name);
      if(temp1) {
        if(node_get_type(temp1) == VAR) return name;
	else return absAutoGenAbsPreRecur(var, val, diff, temp1, context, data);
      }
      if(temp2)
	return absAutoGenAbsPreRecur(var, val, diff, temp2, context, data);
      return name;
    }
    case NEXT:
    case SMALLINIT: return Nil;
    case EQUAL: {
      node_ptr ln = absAutoGenAbsPreRecur(var, val, diff, car(procs), context, 0);
      node_ptr rn = absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, 0);
      if(!ln || !rn) return Nil;
      if(isamevar(ln, var)) {
	switch(node_get_type(rn)) {
	  case NUMBER :
	  case ATOM :
	    k = absFindPos(var, rn);
	    if(k != num_val) {
	      ++(*diff);
	      absEqual(val, k, diff); return Nil;
	    }
	  default :
	    absNoAbs(val, diff); return Nil;
	}
      }
      if(isamevar(rn ,var)) {
        switch(node_get_type(ln)) {
          case NUMBER :
          case ATOM :
	    k = absFindPos(var, ln);
	    if(k != num_val) {
	      ++(*diff);
	      absEqual(val, k, diff); return Nil;
	    }
 	  default :
	    absNoAbs(val, diff); return Nil;
        }
      }
      break;
    }
    case PLUS :
    case MINUS :
    case TIMES :
    case DIVIDE : {
      node_ptr ln, rn;
      if(data) return Nil;
      ln = absAutoGenAbsPreRecur(var, val, diff, car(procs), context, 0);
      rn = absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, 0);
      if(!ln || !rn) return Nil;
      if(isamevar(ln, var)) {
        absNoAbs(val, diff); return Nil;
      }
      if(isamevar(rn, var)) {
        absNoAbs(val, diff); return Nil;
      }
      return Nil;
    }
    case MOD : {
      node_ptr ln, rn;
      if(data) return Nil;
      ln = absAutoGenAbsPreRecur(var, val, diff, car(procs), context, 0);
      rn = absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, 0);
      if(!ln || !rn) return Nil;
      if(isamevar(ln, var)) {
	switch(node_get_type(rn)) {
	  case NUMBER :
	    k = (int) car(rn);
	    absModulus(val, k, diff); return Nil;
	  default :
	    absNoAbs(val, diff); return Nil;
	}
      }
      return Nil;
    }
    case LT :
    case GE : {
      node_ptr ln = absAutoGenAbsPreRecur(var, val, diff, car(procs), context, 0);
      node_ptr rn = absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, 0);
      if(!ln || !rn) return Nil;
      if(isamevar(ln, var)) {
	switch(node_get_type(rn)) {
          case NUMBER :
	    k = (int) car(rn);
	    if(!k) {++(*diff); absEqual(val, 0, diff);}
	    else absBisect(val, k-1, diff);
	    return Nil;
	  default :
	    absNoAbs(val, diff); return Nil;
        }
      }
      if(isamevar(rn, var)) {
	switch(node_get_type(ln)) {
	  case NUMBER :
	    k = (int) car(ln);
	    if(!k) {++(*diff); absEqual(val, 0, diff);}
	    else absBisect(val, k-1, diff);
	    return Nil;
	  default :
	    absNoAbs(val, diff); return Nil;
	}
      }
      return Nil;
    }
    case GT :
    case LE : {
      node_ptr ln = absAutoGenAbsPreRecur(var, val, diff, car(procs), context, 0);
      node_ptr rn = absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, 0);
      if(!ln || !rn) return Nil;
      if(isamevar(ln, var)) {
        switch(node_get_type(rn)) {
          case NUMBER :
            k = (int) car(rn);
	    if(!k) {++(*diff); absEqual(val, 0, diff);}
            else absBisect(val, k, diff);
	    return Nil;
          default :
            absNoAbs(val, diff); return Nil;
        }
      }
      if(isamevar(rn, var)) {
        switch(node_get_type(ln)) {
          case NUMBER :
            k = (int) car(ln);
	    if(!k) {++(*diff); absEqual(val, 0, diff);}
            else absBisect(val, k, diff);
	    return Nil;
          default :
            absNoAbs(val, diff); return Nil;
        }
      }
      return Nil;
    }
    /* I assume the UNION is only used for values at this point */
    case UNION: return procs;
    case SETIN: {
      node_ptr ln = absAutoGenAbsPreRecur(var, val, diff, car(procs), context, 0);
      node_ptr rn = absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, 0);
      if(!ln || !rn) return Nil;
      if(isamevar(ln, var)) {
        switch(node_get_type(rn)) {
	  case UNION :
            ++(*diff);
	    absEnum(val, ln, rn, diff); return Nil;
	  default :
	    rpterr("Yuan Lu: Something wrong");
	}
      }
      return Nil;
    }
    case COLON:
      absAutoGenAbsPreRecur(var, val, diff, car(procs), context, 0);
      absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, 1);
      return Nil;
    case CONTEXT:
      return absAutoGenAbsPreRecur(var, val, diff, cdr(procs), car(procs), 0);
    case EQDEF: {
      node_ptr varn, procn = car(procs);
      switch(node_get_type(procn)) {
	case DOT:
	case ATOM:
	  varn = eval_struct(procn, context);
	  /* if(isamevar(var, varn)) return Nil; */
	  if(!lookup_interest_hash(varn)) return Nil;
	  break;
	case NEXT:
#ifdef ABS_DEBUG
	  if(ABS_DEBUG>1 && node_get_type(car(procn)) == ATOM &&
	     strcmp(car(procn)->left.strtype->text, "st_lc")==0)
	    printf("");
#endif
	  varn = eval_struct(car(procn), context);
	  /* if(isamevar(var, varn)) return Nil; */
	  if(!lookup_interest_hash(varn)) return Nil;
      }
    }
    default:
      absAutoGenAbsPreRecur(var, val, diff, car(procs), context, data);
      absAutoGenAbsPreRecur(var, val, diff, cdr(procs), context, data);
      break;
  }
  return Nil;
}

/**Function********************************************************************

  Synopsis           [Recursively go through the parse tree to generate
  list of atomic formulas.]

  Description        [Recursively go through the parse tree to generate
  list of atomic formulas. In the paramer list, <procs> is the program 
  we are visiting; <context> is the context of <procs>; <data>=1 implies
  that procs is the datapath of a certain variable; <laf> is the generated
  list of atomic formulas.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static node_ptr absGenListFormulaRecur(node_ptr procs, node_ptr context, 
				       int data, node_ptr *laf)
{
  int k;

  if(!procs) return;
  switch(node_get_type(procs)) {
    case ATOM: {
      node_ptr name  = find_node(DOT, context, find_atom(procs));
      node_ptr temp1 = lookup_symbol_hash(name);
      node_ptr temp2 = lookup_param_hash(name);
      node_ptr temp3 = lookup_constant2_hash(find_atom(procs));
      if(temp1) {
	if(node_get_type(temp1) == VAR)
	  return name;
	else
	  return absGenListFormulaRecur(temp1, context, data, laf);
      }
      if(temp2)
	return absGenListFormulaRecur(temp2, context, data, laf);
      if(temp3) 
	return temp3;
      else return name;
    }
    case ARRAY: return eval_struct(procs, context);
    case NUMBER: return find_node(NUMBER, car(procs), Nil);
    case DOT: {
      node_ptr name = eval_struct(procs,  context);
      node_ptr temp1 = lookup_symbol_hash(name);
      node_ptr temp2 = lookup_param_hash(name);
      if(temp1) {
        if(node_get_type(temp1) == VAR) return name;
	else return absGenListFormulaRecur(temp1, context, data, laf);
      }
      if(temp2)
	return absGenListFormulaRecur(temp2, context, data, laf);
      return name;
    }
    case NEXT: return absGenListFormulaRecur(car(procs), context, data, laf);
    case SMALLINIT: return Nil;
    case EQUAL: {
      node_ptr ln = absGenListFormulaRecur(car(procs), context, data, laf);
      node_ptr rn = absGenListFormulaRecur(cdr(procs), context, data, laf);
      node_ptr cur;

      if(!ln || !rn) return Nil;
      cur = find_node(EQUAL, ln, rn);
      if(!in_list(cur, *laf)) *laf = cons(cur, *laf);
      return Nil;
    }
    case PLUS :
    case MINUS :
    case TIMES :
    case DIVIDE :
    case MOD : {
      node_ptr ln = absGenListFormulaRecur(car(procs), context, data, laf);
      node_ptr rn = absGenListFormulaRecur(cdr(procs), context, data, laf);

      if(!ln || !rn) return Nil;
      return find_node(node_get_type(procs), ln, rn);
    }
    case LT :
    case LE :
    case GT :
    case GE : {
      node_ptr ln = absGenListFormulaRecur(car(procs), context, data, laf);
      node_ptr rn = absGenListFormulaRecur(cdr(procs), context, data, laf);
      node_ptr cur;

      if(!ln || !rn) return Nil;
      cur = find_node(node_get_type(procs), ln, rn);
      if(!in_list(cur, *laf)) *laf = cons(cur, *laf);
      return Nil;
    }
    /* I assume the UNION is only used for values at this point */
    case UNION: return procs;
    case SETIN: {
      node_ptr ln = absGenListFormulaRecur(car(procs), context, data, laf);
      node_ptr rn = absGenListFormulaRecur(cdr(procs), context, data, laf);
      node_ptr cur;

      if(!ln || !rn) return Nil;
      cur = find_node(SETIN, ln, rn);
      if(!in_list(cur, *laf)) *laf = cons(cur, *laf);
      return Nil;
    }
    case CONTEXT:
      return absGenListFormulaRecur(cdr(procs), car(procs), 0, laf);
    case EQDEF: {
      node_ptr varn, procn = car(procs);
      switch(node_get_type(procn)) {
	case DOT:
	case ATOM:
	  varn = eval_struct(procn, context);
	  if(!lookup_interest_hash(varn)) return Nil;
	  break;
	case NEXT:
	  varn = eval_struct(car(procn), context);
	  if(!lookup_interest_hash(varn)) return Nil;
      }
      absGenListFormulaRecur(cdr(procs), context, data, laf);
      return Nil;
    }
    case COLON: {
      node_ptr ln = absGenListFormulaRecur(car(procs), context, 0, laf);
      if(ln && node_get_type(ln) != NUMBER && !in_list(ln, *laf)) 
	*laf = cons(ln, *laf);
      absGenListFormulaRecur(cdr(procs), context, 1, laf);
      return Nil;
    }
    case CONS: {
      absGenListFormulaRecur(car(procs), context, data, laf);
      absGenListFormulaRecur(cdr(procs), context, data, laf);
      return Nil;
    }
    default: {
      node_ptr ln = absGenListFormulaRecur(car(procs), context, data, laf);
      node_ptr rn = absGenListFormulaRecur(cdr(procs), context, data, laf);
      if(ln && node_get_type(ln) != NUMBER && !in_list(ln, *laf)) 
	*laf = cons(ln, *laf);
      if(rn && node_get_type(rn) != NUMBER && !in_list(rn, *laf)) 
	*laf = cons(rn, *laf);
      return Nil;
    }
  }
  return Nil;
}

/**Function********************************************************************

  Synopsis           [Elaborate a formula by extending its Boolean expressions.]

  Description        [Elaborate a formula by extending its Boolean expressions.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
void absElabFormula(node_ptr forms)
{
  node_ptr one = find_node(NUMBER, (node_ptr) 1, 0);
  for(;forms;forms=cdr(forms)) {
    node_ptr formula = car(forms);
    switch(node_get_type(formula)) {
      case ATOM :
      case DOT :
	forms->left.nodetype = find_node(EQUAL, formula, one);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Build formula clusters and variable clusters.]

  Description        [Build formula clusters and variable clusters.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr absClusterVarFormula(node_ptr laf)
{
  node_ptr lafp, lafq;
  node_ptr var_list, varp, varq;
  node_ptr vars, form;
  node_ptr tmp;
  node_ptr abs=Nil;

  if(!laf) return Nil;

  /* eliminate the formulas whose variables are in the specification
     list or are involved uninteresting variables */
  for(lafq=laf;lafq;lafq=cdr(lafq)) {
    vars = Abs_AbsGenVarList(car(lafq), Nil);
    for(;vars;vars=cdr(vars))
      if(!lookup_interest_hash(car(vars)))
	break;
    if(!vars) break;
  }
  laf = lafq;
  for(lafp=cdr(laf);lafp;lafp=cdr(lafp)) {
    vars = Abs_AbsGenVarList(car(lafp), Nil);
    for(;vars;vars=cdr(vars))
      if(!lookup_interest_hash(car(vars)))
        break;
    if(vars)
      lafq->right.nodetype = cdr(lafp);
    else
      lafq = lafp;
  }

  /* build formula and variable clusters */
  var_list = absGenListVarRecur(laf);
  while(var_list) {
    vars = car(var_list);
    form = cons(car(laf), Nil);
    while(1) {
      int sign = 1;
      varq = var_list; varp = cdr(varq);
      lafq = laf; lafp = cdr(lafq);
      while(varp) {
	if(is_list_intersect(vars, car(varp))) {
	  sign = 0;
	  varq->right.nodetype = cdr(varp);
	  lafq->right.nodetype = cdr(lafp);
	  vars = list_union(vars, car(varp));
	  form = cons(car(lafp), form);
	} else {
	  varq = varp; lafq = lafp;
	}
	varp = cdr(varq); lafp = cdr(lafq);
      }
      if(sign) break;
    }
    tmp = cons(Nil, cons(Nil, Nil));
    abs = cons(new_node(IMPLIES, new_node(CONS, vars, form), tmp), abs);
    var_list = cdr(var_list);
    laf = cdr(laf);
  }
  return abs;
}

/**Function********************************************************************

  Synopsis           [Decide whether we apply abstraction on a cluster.]

  Description        [Decide whether we apply abstraction on a cluster.]

  SideEffects        [abs is modified.]

  SeeAlso            []

******************************************************************************/
static node_ptr absChooseAbs(node_ptr abs)
{
  if(!abs) return Nil;
  {
    int count=0, range=1, pc;
    node_ptr absp, form, vars;
    node_ptr tmp = absChooseAbs(cdr(abs));

    for(vars = car(car(car(abs)));vars;vars=cdr(vars))
      range *= (int) lookup_range_hash(car(vars));

    form = cdr(car(car(abs)));
    for(;form;form=cdr(form)) count++;

    pc = power2(count+1);
/*    if((count>0 & range<5)|| count > 6 || range <= power2(count)) return tmp; */
    if((count>0 & range<5)|| count > 6 || range <= pc) return tmp;
    else {
      /* count = power2(count+1)-1; */
      count = (range < 2*pc) ? range : 2*pc - 1;
      form = gen_list_integer(count, count);
      cdr(car(abs))->left.nodetype = form;
      abs->right.nodetype = tmp;
      return abs;
    }
  }
}

/**Function********************************************************************

  Synopsis           [Decide when to apply abstraction.]

  Description        [Decide when to apply abstraction.]

  SideEffects        [abs is modified.]

  SeeAlso            []

******************************************************************************/
static void absSplitAbs(node_ptr abs)
{
  if(!abs) return;
  absSplitAbs(cdr(abs));

  {
    int sign=0;
    node_ptr varl = car(car(car(abs)));
    for(;varl;varl=cdr(varl)) {
      int range = (int) lookup_range_hash(car(varl));
      if(range > 255) {
	sign = 1;
	break;
      }
    }
    if(sign) {
      abs->right.nodetype = abs_expr_pre; 
      abs_expr_pre = abs;
    } else {
      abs->right.nodetype = abs_expr;
      abs_expr = abs;
    }
  }
}


/**Function********************************************************************

  Synopsis           [Generate variable lists for each sub-structure in the list.]

  Description        [Generate variable lists for each sub-structure in the list.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr absGenListVarRecur(node_ptr list)
{
  node_ptr varl;

  if(!list) return Nil;
  varl = absGenListVarRecur(cdr(list));
  varl = cons(Abs_AbsGenVarList(car(list), Nil), varl);
  return varl;
}

/**Function********************************************************************

  Synopsis           [Look for the position of val in value list of var.]

  Description        [Look for the position of val in value list of var.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int absFindPos(node_ptr var, node_ptr val)
{
  int k = 0;
  node_ptr vals = lookup_symbol_hash(var);

  for(vals=cdr(vals); vals; vals=cdr(vals), k++)
    if(isameval(car(vals), val)) break;
  return k;
}

/**Function********************************************************************

  Synopsis           [Keep the values of val different.]

  Description        [Keep the values of val different.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void absNoAbs(node_ptr val, int *inc)
{
  node_ptr var;
  for(*inc=0, var=cdr(val);val;val=cdr(val), (*inc)++)
    val->left.inttype = *inc;
}

/**Function********************************************************************

  Synopsis           [In the linklist <val>, at position <pos>, mark IT!]

  Description        [In the linklist <val>, at position <pos>, mark IT!]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void absEqual(node_ptr val, int pos, int *inc)
{
  if(*inc > 1048575) {
    *inc = absRearrange1(val);
    ++(*inc);
  }
  for(;pos;val=cdr(val),pos--);
  val->left.inttype = *inc;
}

/**Function********************************************************************

  Synopsis           [In the linklist <val>, from position <pos>, the rest
		      nodes are marked by <*inc> + 1.]

  Description        [In the linklist <val>, from position <pos>, the rest
		      nodes are marked by <*inc> + 1.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void absBisect(node_ptr val, int pos, int *inc)
{
  if(*inc > 1048575) *inc = absRearrange1(val);
  ++(*inc);
  for(val=cdr(val);pos;val=cdr(val),pos--);
  for(;val;val=cdr(val))
    val->left.inttype += (*inc);
  (*inc) += (*inc);
}

/**Function********************************************************************

  Synopsis           [Go through the tree rooted at <rn>, for a value
		      in the tree, mark IT!]

  Description        [Go through the tree rooted at <rn>, for a value
		      in the tree, mark IT; this is only used in SETIN.]

  SideEffects        []

  SeeAlso            [absEqual]

******************************************************************************/

static void absEnum(node_ptr val, node_ptr ln, node_ptr rn, int *inc)
{
  int k;
  switch(node_get_type(rn)) {
    case ATOM :
    case NUMBER :
      k = absFindPos(ln, rn);
      if(k != num_val)
        absEqual(val, k, inc);
      return;
    case UNION :
      absEnum(val, ln, car(rn), inc);
      absEnum(val, ln, cdr(rn), inc);
      return;
    default :
      rpterr("Yuan Lu : someThing wrong");
  }
}

/**Function********************************************************************

  Synopsis           [The linklist <val> is divided into |val|/res sets.]

  Description        [The linklist <val> is divided into |val|/res sets.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void absDivide(node_ptr val, int res, int *inc)
{
  int count=0;
  int base = ++(*inc);
  for(;val;val=cdr(val)) {
    val->left.inttype += base;
    if(count==res-1) {
      count=0; base += (*inc);
    } else count++;
  }
  (*inc) = base;
}

/**Function********************************************************************

  Synopsis           [In the linklist <val>, all the nodes whose residue
		      are the same are grouped.]

  Description        [In the linklist <val>, all the nodes whose residue
		      are the same are grouped.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void absModulus(node_ptr val, int res, int *inc)
{
  int count=0;
  int base = ++(*inc);
  for(;val;val=cdr(val)) {
    val->left.inttype += (base+count);
    count = (count==res-1) ? 0 : count+1;
  }
  (*inc) = base + res;
}

/**Function********************************************************************

  Synopsis           [Check if var appears in context.procs.]

  Description        [Check if var appears in context.procs.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int absExistVar(node_ptr var, node_ptr procs, node_ptr context)
{
  node_ptr var1;

  if(!procs) return 0;
  switch(node_get_type(procs)) {
    case ATOM:
      var1 = eval_struct(procs, context);
      if(isamevar(var1, var)) return 1;
      break;
    case TWODOTS:
    case TRUEEXP:
    case FALSEEXP:
    case NUMBER:
    case ARRAY: return 0;
    case CONTEXT:
      return absExistVar(var, cdr(procs), car(context));
    default:
      if(absExistVar(var, car(procs), car(context))) return 1;
      return absExistVar(var, cdr(procs), car(context));
  }
}

/**Function********************************************************************

  Synopsis           [When the values of a variable is not power of 2,
		      a list is generated to represent each path in the ADD.]

  Description        [When the values of a variable is not power of 2,
                      a list is generated to represent each path in the ADD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr absPowerList(node_ptr var, node_ptr abs_val, int num_list, int exponent)
{
  node_ptr var_orig = lookup_symbol_hash(var);
  node_ptr val = cdr(var_orig);
  node_ptr res=Nil;
  add_ptr var_add = (add_ptr) car(var_orig);
  int j;

  if(num_list == power2(exponent)) return abs_val;

  absPowerListRecur(var_add, exponent, &res);

  for(j=val->left.inttype, val=res;val;val=cdr(val)) {
    int i = (int) car(val);
    if(j != i) {
      j = i; abs_val = cdr(abs_val);
    }
    val->left.nodetype = abs_val->left.nodetype;
  }
  return res;
}

/**Function********************************************************************

  Synopsis           [Recursively build the power list of a variable.]

  Description        [Recursively build the power list of a variable.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void absPowerListRecur(add_ptr var_add, int exp, node_ptr *list)
{
  if(exp==0) {
    *list = cons(add_get_leaf(dd_manager, var_add), *list);
  } else {
    add_ptr e, r;
    if(add_isleaf(var_add)) {
      r = add_dup(var_add);
      e = add_dup(var_add);
    } else {
      r = add_then(var_add);
      e = add_else(var_add);
    }
    absPowerListRecur(r, exp-1, list); add_free(dd_manager, r);
    absPowerListRecur(e, exp-1, list); add_free(dd_manager, e);
  }
}

/**Function********************************************************************

  Synopsis           [Check if var appears in context.procs.]

  Description        [Check if var appears in context.procs. It is used when
  aBDD is used for abstraction.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr absTurn2Fun(node_ptr var, node_ptr abs_val, int nv)
{
  node_ptr abs_fun;

  num_val = nv;
  if(isFunIdentity(abs_val)) {
    return new_node(IMPLIES, var, find_node(ATOM, (node_ptr)find_string("unique"), Nil));
  } else if(abs_fun=isFunSymmetry(abs_val)) {
    return new_node(IMPLIES, var, abs_fun);
  } else if(abs_fun=isFunResidue(abs_val)) {
    return new_node(IMPLIES, var, abs_fun);
  } else if(abs_fun=isFunRange(abs_val)) {
    return new_node(IMPLIES, var, abs_fun);
  }
  return Nil;
  /* return new_node(IMPLIES, var, abs_val); */
}

/**Function********************************************************************

  Synopsis           [Check if the abstraction function is identity function.]

  Description        [Check if the abstraction function is identity function.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int isFunIdentity(node_ptr abs_val)
{
  for(;abs_val;abs_val=cdr(abs_val))
    if(car(car(abs_val))) return 0;
  return 1;
}

/**Function********************************************************************

  Synopsis           [Generate a range function.]

  Description        [Generate a range function.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr absRangeFun(node_ptr abs_val, int vp, int stp)
{
  if(!abs_val) return Nil;
  else if(abs_val == st[stp])
    return new_node(TWODOTS, (node_ptr) vp, absRangeFun(cdr(abs_val), vp+1, stp+1));
  else
    return absRangeFun(cdr(abs_val), vp+1, stp);
}

/**Function********************************************************************

  Synopsis           [Check if the abstraction function is range function.]

  Description        [Check if the abstraction function is range function.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr isFunRange(node_ptr abs_val)
{
  int i, pv=-1, stp=0;
  node_ptr absv;

  for(absv=abs_val;absv;absv=cdr(absv)) {
    int av = ((int)car(car(absv)));
    if(av!=pv) {
      for(i=0;i<stp;i++)
        if(st[i]==absv) return Nil;
      st[stp++] = absv;
      pv = av;
    }
  }
  if(stp==num_val) return Nil;
  return absRangeFun(abs_val, 0, 0);
}

/**Function********************************************************************

  Synopsis           [Check if the abstraction function is residue or not.]

  Description        [Check if the abstraction function is residue or not.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr isFunResidue(node_ptr abs_val)
{
  int i, j;
  int val0 = (int) car(car(abs_val));
  node_ptr rl, ql, pl = cdr(cdr(abs_val));

  for(i=2;i<num_val/2;i++,pl=cdr(pl)) {
    if(((int) car(car(pl)))!=val0) continue;
    for(rl=pl,ql=abs_val;rl;rl=cdr(rl)) {
      int val1 = (int) car(car(rl));
      int val2 = (int) car(car(ql));
      if(val1!=val2) break;
      ql = cdr(ql);
      if(ql==pl) ql=abs_val;
    }
    if(!rl) return new_node(EQDEF, find_node(ATOM,
			    (node_ptr) find_string("residue"), 0),
			    find_node(NUMBER, (node_ptr) i, 0));
  }
  return Nil;
}

/**Function********************************************************************

  Synopsis           [Check if the abstraction function is symmetry or not.]

  Description        [Check if the abstraction function is symmetry or not.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr isFunSymmetry(node_ptr abs_val)
{
  return Nil;
}

/**Function********************************************************************

  Synopsis           [Given a list of integers, generate a new list of integers
  in which the values are small integers and are different

  Description        [

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void absRearrange(node_ptr tmp)
{
  int i=0;
  for(;tmp;tmp=cdr(tmp)) {
    unsigned k = (unsigned) car(tmp);
    node_ptr tmp1, tmp2;
    if(k > 1048576) continue;
    tmp2 = tmp->left.nodetype = find_node(NUMBER, (node_ptr)i++, Nil);
    for(tmp1=cdr(tmp);tmp1;tmp1=cdr(tmp1)) {
      if(k==(unsigned) car(tmp1))
	tmp1->left.nodetype = tmp2;
    }
  }
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int absRearrange1(node_ptr tmp)
{
  int i=0;
  node_ptr tmp1;
  for(tmp1=tmp;tmp1;tmp1=cdr(tmp1))
    tmp1->left.inttype += 1048576;
  for(;tmp;tmp=cdr(tmp)) {
    unsigned k = (unsigned) car(tmp);
    if(k < 1048576) continue;
    tmp->left.inttype = i;
    for(tmp1=cdr(tmp);tmp1;tmp1=cdr(tmp1)) {
      if(k==(unsigned) car(tmp1))
        tmp1->left.inttype = i;
    }
    i++;
  }
  return i;
}

