/**CFile***********************************************************************

  FileName    [absAbstract.c]

  PackageName [abs]

  Synopsis    [High-level routines to perform automatic abstraction.]

  Description [This file provides the interface to the abstraction
  routines.]

  Author      [Yuan Lu]

  Copyright   [ Copyright (c) 1998 by ITC-IRST and Carnegie Mellon
  University.  All Rights Reserved.  This software is for educational
  purposes only.  Permission is given to use, copy, modify, and
  distribute this software and its documentation provided that this
  introductory message is not removed and no monies are exchanged. No
  guarantee is expressed or implied by the distribution of this code.
  Send bug-reports and/or questions to: nusmv@irst.itc.it ]

******************************************************************************/

#include "absInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
node_ptr abs_rel_prod = Nil;
bdd_ptr init_bdd_bak;

#ifdef ABS_DEBUG
FILE *err_out;
#endif

static bdd_ptr absAbstractFunCur ARGS((bdd_ptr, node_ptr));
static bdd_ptr absAbstractFunRel2 ARGS((bdd_ptr, node_ptr));
static bdd_ptr absAbstractFunRel ARGS((bdd_ptr, node_ptr));
static bdd_ptr absAbstractFunaBdd ARGS((bdd_ptr, node_ptr));
static void absBuildFunaBDD ARGS((node_ptr));
static node_ptr absAbstractTransPartADD ARGS((node_ptr, node_ptr));
static node_ptr absClusterAbs ARGS((node_ptr, node_ptr));
static int absEvalFormula ARGS((node_ptr, node_ptr, node_ptr *));
static node_ptr absEvalFormulaRecur ARGS((node_ptr, node_ptr, node_ptr *));
/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
void Abs_AbsAbstractInitState()
{
  bdd_ptr f = init_bdd_orig;
  node_ptr abs;

  if(!abs_rel_prod)
    abs_rel_prod = Abs_AbsClusterAbs(abs_expr);

  for(abs=abs_rel_prod;abs;abs=cdr(abs)) {
    bdd_ptr res = absAbstractFunCur(f, car(abs));
    bdd_free(dd_manager, f); f = res;
  }
  init_bdd = f;
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
void Abs_BuildAbstractRelation(node_ptr abs)
{
  static node_ptr ptmv[32], valpt[32];
  int numvar;

  for(;abs;abs=cdr(abs)) {
    node_ptr vf = car(car(abs));
    node_ptr varl = car(vf);
    node_ptr forml = cdr(vf);
    node_ptr vall = car(cdr(car(abs)));
    add_ptr  addcvar, addavar = (add_ptr) car(cdr(cdr(car(abs))));
    add_ptr  res = add_zero(dd_manager);

    for(numvar=0;varl;numvar++,varl=cdr(varl))
      ptmv[numvar] = valpt[numvar] = cdr(lookup_symbol_hash(car(varl)));

    while(ptmv[0]) {
      node_ptr absval, formp;
      int v = 0;
      for(formp=forml;formp;formp=cdr(formp))
	v = 2*v + absEvalFormula(car(formp), car(vf), ptmv);
      for(absval=vall;v;v--,absval=cdr(absval));
      absval = car(absval);

      /* build the BDD for this entry described by ptmv */
      {
	int i;
	add_ptr f, g, h, r;
	add_ptr av, ar = add_one(dd_manager);

	/* relation on concrete variables */
	for(i=0,varl=car(vf);varl;i++,varl=cdr(varl)) {
	  node_ptr cvar = car(varl);
	  addcvar = (add_ptr) car(lookup_symbol_hash(cvar));
	  av = add_leaf(dd_manager, find_atom(car(ptmv[i])));
	  f = add_equal(dd_manager, addcvar, av);
	  g = add_and(dd_manager, ar, f);
	  add_free(dd_manager, av);
	  add_free(dd_manager, f);
	  add_free(dd_manager, ar);
	  ar = g;
	}

	av = add_leaf(dd_manager, find_atom(absval));
	f = add_equal(dd_manager, addavar, av);
	g = add_and(dd_manager, ar, f);
	add_free(dd_manager, av);
	add_free(dd_manager, f);
	add_free(dd_manager, ar);
	ar = add_or(dd_manager, res, g);
	add_free(dd_manager, g);
	add_free(dd_manager, res);
	res = ar;
      }

      /* move ptmv */
      {
	int j;
	for(j=numvar-1;j;j--) {
	  node_ptr *t = ptmv + j;
	  *t = cdr(*t);
	  if(*t) break;
	  *t = *(valpt + j);
	}
	if(j==0) *ptmv = cdr(*ptmv);
      }
    }
    cdr(cdr(car(abs)))->right.bddtype = add_to_bdd(dd_manager, res);
    add_free(dd_manager, res);
  }
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
bdd_ptr Abs_AbsAbstractCur(bdd_ptr f, node_ptr abs)
{
  bdd_ptr g = bdd_dup(f);
  node_ptr abs1, abs2;

  abs2 = reverse(Abs_AbsClusterAbs(abs));
  for(abs1=abs2;abs1;abs1=cdr(abs1)) {
    bdd_ptr res = absAbstractFunCur(g, car(abs1));
    bdd_free(dd_manager, g); g = res;
  }

  for(;abs2;abs2=cdr(abs2)) {
    bdd_free(dd_manager, (bdd_ptr)car(car(abs2)));
    bdd_free(dd_manager, (bdd_ptr)cdr(car(abs2)));
  }
  return g;
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
bdd_ptr Abs_AbsAbstractCurNex(bdd_ptr f, node_ptr abs)
{
  bdd_ptr g = bdd_dup(f);
  node_ptr abs1, abs2;

  abs2 = reverse(Abs_AbsClusterAbs(abs));
  for(abs1=abs2;abs1;abs1=cdr(abs1)) {
    bdd_ptr res = absAbstractFunRel2(g, car(abs1));
    bdd_free(dd_manager, g); g = res;
  }

  for(;abs2;abs2=cdr(abs2)) {
    bdd_free(dd_manager, (bdd_ptr)car(car(abs2)));
    bdd_free(dd_manager, (bdd_ptr)cdr(car(abs2)));
  }
  return g;
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
node_ptr Abs_AbsAbstractTransPartADD(node_ptr trans_expr)
{
  if(!abs_rel_prod)
    abs_rel_prod = Abs_AbsClusterAbs(abs_expr);
  return absAbstractTransPartADD(trans_expr, abs_rel_prod);
}

/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
node_ptr Abs_AbsClusterAbs(node_ptr list)
{
  bdd_ptr one = bdd_one(dd_manager);
  node_ptr last = cons(cons((node_ptr)one, (node_ptr)bdd_dup(one)), Nil);
  return absClusterAbs(list, last);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [

  Description        [

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
node_ptr absAbstractTransPartADD(node_ptr trans, node_ptr abs)
{
  node_ptr backup;
  add_ptr f; bdd_ptr fb;

  if(!trans) return Nil;
  backup = absAbstractTransPartADD(cdr(trans), abs);
  fb = add_to_bdd(dd_manager, (add_ptr)car(trans));
  for(;abs;abs=cdr(abs)) {
    bdd_ptr rb = absAbstractFunRel2(fb, car(abs));
    bdd_free(dd_manager, fb); fb = rb;
  }
  f = bdd_to_add(dd_manager, fb);
  bdd_free(dd_manager, fb);
  return cons((node_ptr)f, backup);
}

/**Function********************************************************************

  Synopsis []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static bdd_ptr absAbstractFunCur(bdd_ptr f, node_ptr abs)
{
  bdd_ptr rel = (bdd_ptr) car(abs);
  bdd_ptr var = (bdd_ptr) cdr(abs);
  return bdd_and_abstract(dd_manager, f, rel, var);

}

/**Function********************************************************************

  Synopsis	     []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static bdd_ptr absAbstractFunRel(bdd_ptr f, node_ptr abs)
{
  bdd_ptr absRC = (bdd_ptr) cdr(cdr(abs));
  bdd_ptr absRN = bdd_shift_forward(dd_manager, absRC);
  node_ptr var = lookup_symbol_hash(car(abs));
  bdd_ptr normVC = add_to_bdd(dd_manager,add_support(dd_manager,(add_ptr)car(var)));
  bdd_ptr normVN = bdd_shift_forward(dd_manager, normVC);
  bdd_ptr g = bdd_and_abstract(dd_manager, f, absRC, normVC);

  f = bdd_and_abstract(dd_manager, g, absRN, normVN);
  bdd_free(dd_manager, g); bdd_free(dd_manager, absRN);
  bdd_free(dd_manager, normVC); bdd_free(dd_manager, normVN);

  return f;
}

/**Function********************************************************************

  Synopsis []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static bdd_ptr absAbstractFunRel2(bdd_ptr f, node_ptr abs)
{
  bdd_ptr rel = (bdd_ptr) car(abs);
  bdd_ptr var = (bdd_ptr) cdr(abs);
  bdd_ptr g = bdd_and_abstract(dd_manager, f, rel, var);

  bdd_ptr reln = bdd_shift_forward(dd_manager, rel);
  bdd_ptr varn = bdd_shift_forward(dd_manager, var);
  bdd_ptr res = bdd_and_abstract(dd_manager, g, reln, varn);

  bdd_free(dd_manager, reln); bdd_free(dd_manager, varn);
  bdd_free(dd_manager, g);

  return res;
}

/**Function********************************************************************

  Synopsis []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static bdd_ptr absAbstractFunaBdd(bdd_ptr f, node_ptr abs)
{
  return f;
}

/**Function********************************************************************

  Synopsis []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr absClusterAbs(node_ptr abs, node_ptr last)
{
  node_ptr t, varl;
  add_ptr var, supp, f, g;
  bdd_ptr suppb, rel;

  if(!abs) {
    bdd_ptr one = bdd_one(dd_manager);
    return cons(cons((node_ptr)one, (node_ptr)bdd_dup(one)), last);
  }
  t = absClusterAbs(cdr(abs), last);

  supp = add_one(dd_manager);
  for(varl=car(car(car(abs)));varl;varl=cdr(varl)) {
    var = (add_ptr) car(lookup_symbol_hash(car(varl)));
    f = add_support(dd_manager, var);
    g = add_and(dd_manager, supp, f);
    add_free(dd_manager, supp); add_free(dd_manager, f);
    supp = g;
  }
  suppb = add_to_bdd(dd_manager, supp);
  add_free(dd_manager, supp);

  rel = (bdd_ptr) cdr(cdr(cdr(car(abs))));
  if(rel == bdd_one(dd_manager)) {
    bdd_ptr f = bdd_and(dd_manager, (bdd_ptr)cdr(car(last)), suppb);
    bdd_free(dd_manager, (bdd_ptr)cdr(car(last)));
    bdd_free(dd_manager, suppb);
    car(last)->right.bddtype = f;
    return t;
  } else {
    bdd_ptr relc = (bdd_ptr) car(car(t));
    if((bdd_size(dd_manager, relc)) > 1000)
      return cons(cons((node_ptr)bdd_dup(rel), (node_ptr)suppb), t);
    else {
      bdd_ptr f = bdd_and(dd_manager, rel, relc);
      bdd_ptr g = bdd_and(dd_manager, suppb, (bdd_ptr)cdr(car(t)));
      bdd_free(dd_manager, relc); bdd_free(dd_manager, (bdd_ptr)cdr(car(t)));
      t->left.nodetype->left.bddtype = f;
      t->left.nodetype->right.bddtype = g;
      return t;
    }
  }
}

/**Function********************************************************************

  Synopsis           [Compute the value of a formula.]

  SideEffects        [Compute the value of a formula.]

  SeeAlso            []

******************************************************************************/
static int absEvalFormula(node_ptr formula, node_ptr vars, node_ptr *values)
{
  node_ptr left, right;

  if(!formula) return 0;
  left = absEvalFormulaRecur(car(formula), vars, values);
  right = absEvalFormulaRecur(cdr(formula), vars, values);
  switch(node_get_type(formula)) {
    case EQUAL: {
      node_ptr lval, rval;
      lval = find_node(node_get_type(left), car(left), cdr(left));
      rval = find_node(node_get_type(right), car(right), cdr(right));
      if(lval == rval) return 1;
      else return 0;
    }
    case LT: {
      int lval, rval;
      lval = (int) car(left);
      rval = (int) car(right);
      if (lval < rval) return 1;
      else return 0;
    }
    case LE: {
      int lval, rval;
      lval = (int) car(left);
      rval = (int) car(right);
      if (lval <= rval) return 1;
      else return 0;
    }
    case GT: {
      int lval, rval;
      lval = (int) car(left);
      rval = (int) car(right);
      if (lval > rval) return 1;
      else return 0;
    }
    case GE: {
      int lval, rval;
      lval = (int) car(left);
      rval = (int) car(right);
      if (lval >= rval) return 1;
      else return 0;
    }
    case SETIN: {
      node_ptr rval;
      for(;right;right = cdr(right)) {
	node_ptr rval = find_node(node_get_type(car(right)), car(car(right)), cdr(car(right)));
	if(left == rval) return 1;
	else return 0;
      }
    }
    default:
      rpterr("sOMeTHING wrOnG");
  }
}

/**Function********************************************************************

  Synopsis           [Recursively compute the value of a sub-formula.]

  SideEffects        [Recursively compute the value of a sub-formula.]

  SeeAlso            []

******************************************************************************/
static node_ptr absEvalFormulaRecur(node_ptr formula, node_ptr vars, node_ptr *values)
{
  if(!formula) return Nil;
  switch(node_get_type(formula)) {
    case DOT :
    case ARRAY: {
      for(;vars;vars=cdr(vars), values++)
        if(formula == car(vars)) break;
#ifdef ABS_DEBUG
      if(!vars) rpterr("Something wRonG");
#endif
      return (car(*values));
    }
    case ATOM :
    case NUMBER : 
      return formula;
    case PLUS : {
      node_ptr left = absEvalFormulaRecur(car(formula), vars, values);
      node_ptr right = absEvalFormulaRecur(cdr(formula), vars, values);
      int lval, rval;
#ifdef ABS_DEBUG
      if(!left | !right | node_get_type(left)!=NUMBER | node_get_type(right)!=NUMBER)
	rpterr("Something wrONg");
#endif
      lval = (int) car(left); rval = (int) car(right);
      return find_node(NUMBER, (node_ptr) (lval+rval), Nil);
    }
    case MINUS : {
      node_ptr left = absEvalFormulaRecur(car(formula), vars, values);
      node_ptr right = absEvalFormulaRecur(cdr(formula), vars, values);
      int lval, rval;
#ifdef ABS_DEBUG
      if(!left | !right | node_get_type(left)!=NUMBER | node_get_type(right)!=NUMBER)
        rpterr("Something wrONg");
#endif
      lval = (int) car(left); rval = (int) car(right);
      return find_node(NUMBER, (node_ptr) (lval-rval), Nil);
    }
    case TIMES : {
      node_ptr left = absEvalFormulaRecur(car(formula), vars, values);
      node_ptr right = absEvalFormulaRecur(cdr(formula), vars, values);
      int lval, rval;
#ifdef ABS_DEBUG
      if(!left | !right | node_get_type(left)!=NUMBER | node_get_type(right)!=NUMBER)
        rpterr("Something wrONg");
#endif
      lval = (int) car(left); rval = (int) car(right);
      return find_node(NUMBER, (node_ptr) (lval*rval), Nil);
    }
    case DIVIDE : {
      node_ptr left = absEvalFormulaRecur(car(formula), vars, values);
      node_ptr right = absEvalFormulaRecur(cdr(formula), vars, values);
      int lval, rval;
#ifdef ABS_DEBUG
      if(!left | !right | node_get_type(left)!=NUMBER | node_get_type(right)!=NUMBER)
        rpterr("Something wrONg");
#endif
      lval = (int) car(left); rval = (int) car(right);
      return find_node(NUMBER, (node_ptr) (lval/rval), Nil);
    }
    case MOD : {
      node_ptr left = absEvalFormulaRecur(car(formula), vars, values);
      node_ptr right = absEvalFormulaRecur(cdr(formula), vars, values);
      int lval, rval;
#ifdef ABS_DEBUG
      if(!left | !right | node_get_type(left)!=NUMBER | node_get_type(right)!=NUMBER)
        rpterr("Something wrONg");
#endif
      lval = (int) car(left); rval = (int) car(right);
      return find_node(NUMBER, (node_ptr) (lval%rval), Nil);
    }
    case UNION : return formula;
    default:
      rpterr("Something WroNg");
  }
}
