(* translate.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: translate.ml,v 1.8 2003/09/17 18:54:07 rflury Exp $ *)

(* translate absyn -> ir *)


module A=Absyn
module T=Ir
module HA=Hashtbl 
module F = (Frame.X86_FRAME : Frame.FrameSig)
module TP = Temp

exception TranslateError of string

let useForeignFuns = ref false

(******************************************************************************)
(* Helpers *)
(******************************************************************************)

(* Translates a type from Absyn.ctype to Ir.pccType *)
let rec ctype2pccType = function
  | A.TyInt -> Pcc.Int
  | A.TyBool -> Pcc.Int
  | A.TyPointer(ctyp) -> Pcc.Pointer(Pcc.Record([Pcc.Pointer(ctype2pccType ctyp); 
						 Pcc.Int;
						 Pcc.Int]))
  | A.TyRecord(s) -> 
      Pcc.Struct(s)
(* let sList = Checker.getStructType s in
   Pcc.Record( List.map (fun (_,t) -> ctype2pccType t) sList ) *)
  | A.NS -> Pcc.Int
	(* raise (TranslateError "Invalid type to function ctype2pccType: NS") *)
  | A.Void -> Pcc.Int   (* Void only used as return type in functions 
		       that do not return *)

(******************************************************************************)
(* Ex Nx Cx *)
(******************************************************************************)
type exp = 
  | Ex of T.exp
  | Nx of T.stmt
  | Cx of ( TP.label * TP.label -> T.stmt )

let unEx = function
  | Ex e -> e
  | Nx s -> T.ESEQ(s, T.CONST(Int32.zero, Pcc.Int))
  | Cx c -> 
      let temp = TP.simpTemp Pcc.Int in
      let tru = TP.newLabel () in
      let fals = TP.newLabel () in
      T.ESEQ(T.SEQ(T.MOVE(T.TEMP(temp), T.CONST(Int32.one, Pcc.Int)),
		   T.SEQ(c(tru, fals), 
			 T.SEQ(T.LABEL(fals),
			       T.SEQ(T.MOVE(T.TEMP(temp), 
					    T.CONST(Int32.zero, Pcc.Int)),
				     T.LABEL(tru))))), 
	     T.TEMP(temp))

let unNx = function
  | Ex e -> T.EXP(e)
  | Nx s -> s
  | Cx c -> raise (TranslateError("Invalid AST: nonsens transformation unNx(Cx) "))

let unCx = function
  | Ex (T.CONST(u, t)) -> 
      if(u = Int32.zero) then
	(function (t,f) -> T.JUMP(f))
      else
	(function (t,f) -> T.JUMP(t))
  | Ex(e) -> 
      (function (t, f) -> T.CJUMP(T.EQ, e, T.CONST(Int32.zero, Pcc.Int), f, t)) 
  | Cx(c) -> c
  | Nx(s) -> raise (TranslateError("Invalid AST: nonsens transformation unCx(Nx)"))


(******************************************************************************)
(* Translate Operands *)
(******************************************************************************)

let trans_op = function
  | A.PLUS -> T.PLUS
  | A.MINUS -> T.MINUS
  | A.TIMES -> T.MUL
  | A.DIVIDE -> T.DIV
  | A.MOD -> T.MOD
  | A.BITAND -> T.AND
  | A.BITOR -> T.OR
  | A.BITNOT -> T.NOT
  | A.BITXOR -> T.XOR
  | A.SHIFTLEFT -> T.SHIFTL
  | A.SHIFTRIGHT -> T.SHIFTR
  | A.PPLUS -> T.PLUS
  | A.PMINUS -> T.MINUS
  | _ -> raise (TranslateError "Invalid op for trans_op")

let trans_rel_op = function 
  | A.EQ -> T.EQ
  | A.NEQ -> T.NE
  | A.LT -> T.LT
  | A.LTE -> T.LE
  | A.GT -> T.GT
  | A.GTE -> T.GE
  | _ -> raise (TranslateError "Invalid op for trans_rel_op")
	

(******************************************************************************)
(* Evaluate Constant Expressions *)
(******************************************************************************)
let eval_const_exp n1 n2 op = 
  let t = Int32.one in
  let f = Int32.zero in
  match op with
  | A.PLUS -> Int32.add n1 n2
  | A.MINUS -> Int32.sub n1 n2
  | A.TIMES -> Int32.mul n1 n2
  | A.DIVIDE -> Int32.div n1 n2
  | A.MOD -> Int32.rem n1 n2
  | A.BITAND -> Int32.logand n1 n2
  | A.BITOR -> Int32.logor n1 n2
  | A.BITXOR -> Int32.logxor n1 n2
  | A.SHIFTLEFT -> Int32.shift_left n1 (Int32.to_int n2)
  | A.SHIFTRIGHT -> Int32.shift_right n1 (Int32.to_int n2)
  | A.EQ -> if(0 = (Pervasives.compare n1 n2)) then t else f
  | A.NEQ -> if(0 = (Pervasives.compare n1 n2)) then f else t
  | A.LT -> if((Pervasives.compare n1 n2) < 0) then t else f
  | A.LTE -> if((Pervasives.compare n1 n2) <= 0) then t else f
  | A.GT -> if((Pervasives.compare n1 n2) > 0) then t else f
  | A.GTE -> if((Pervasives.compare n1 n2) >= 0) then t else f
  | _ -> raise (TranslateError "Invalid op for constant evaluation")


(******************************************************************************)
(* Helpers *)
(******************************************************************************)
	
let isZero n = 	
  (0 = (Pervasives.compare n Int32.zero))

let isNotZero n = 	
  (0 != (Pervasives.compare n Int32.zero))


(* set the used System-functions in sigma 
 * (munch might want to know the return-type) *)
let setSystemFunctionsInSigma () = 
  Checker.setFunType (Helpers.get_external_symbol "_alloc")
    (A.TyPointer(A.TyInt), []);
  Checker.setFunType (Helpers.get_external_symbol "_l3_error")
    ((A.TyInt), []);
  if(!useForeignFuns) then (
    (* useForeignFuns - depreceated: During development, I used 
     * more complex foreign functions that performed pointer operations.
     * This mainly simplified the translation phase, but made also the
     * assembly code much more dense and readable *)
    Checker.setFunType (Helpers.get_external_symbol "_allocFatPointer")
      (A.TyPointer(A.TyInt), []);
    Checker.setFunType (Helpers.get_external_symbol "_pointerAddition")
      (A.TyPointer(A.TyInt), []);
    Checker.setFunType (Helpers.get_external_symbol "_allocHeapWithPointer")
      (A.TyPointer(A.TyInt), []);  
    Checker.setFunType (Helpers.get_external_symbol "_checkFatHeapPointer")
      (A.TyPointer(A.TyInt), [])
   )

(******************************************************************************)
(* Initializing Memory *)
(******************************************************************************)

(* Initializes the location ptr using the given type, 
 * ptr can be any location that can be assigned to *)
let rec ini_mem ptr = function
  | A.TyInt | A.TyBool | A.NS | A.Void -> []
  | A.TyPointer(_) -> 
      if(!useForeignFuns) then
	[ T.MOVE(ptr, F.externalCall "_allocFatPointer" []) ]
      else
	[ T.MOVE(ptr, T.NAME(F.nullPointerLabel ())) ]
  | A.TyRecord(sName) -> 
      let sList = Checker.getStructType sName in
      let len = List.length sList in 

      let get_addr = 
	T.MOVE(ptr,
	       F.externalCall "_alloc" 
		 [T.CONST(Int32.of_int (len * F.wordSize_i), Pcc.Int)] )
      in

      let (_,ini_list) = List.fold_right (fun (_,cty) (c, res) -> 
	(c-1, (ini_base_offi ptr c cty) @ res)
					 ) sList (len-1, []) in
      
      (match ini_list with
      | [] -> [get_addr]
      | _ -> get_addr :: ini_list
      )	  

(* Initializes MEM(base + F.wordSize*offset) 
 * base is the address of the first element of the sturct, 
 * offset the field to be initialized given as an int,
 * and typ the type of the field *)
and ini_base_offi base offset typ = 
  match offset with
  | 0 -> ini_mem (T.MEM(base)) typ
  | _ -> 
      ini_mem (T.MEM(T.BINOP(T.PLUS, 
			     base, 
			     T.CONST(Int32.of_int (offset * F.wordSize_i), 
				     Pcc.Int)))
	      ) typ

(* Initializes MEM(base + F.wordSize*offset) 
 * base is the address of the first element of the sturct, 
 * offset the field to be initialized given as an expression
 * and typ the type of the field *)
and ini_base_off base offset typ = 
  ini_mem (T.MEM(T.BINOP(T.PLUS, 
			 base, 
			 T.BINOP(T.MUL, 
				 offset,
				 T.CONST(F.wordSize, Pcc.Int))))) typ
    
(* Converts a list of statements to 
 * a T.SEQ - list *)
let rec list2SEQ = function
  | [] -> T.nop
  | [s] -> s
  | head :: tail -> T.SEQ(head, list2SEQ(tail))


(******************************************************************************)
(* Translate Expressions *)
(******************************************************************************)

let rec trans_exp = function
  | A.ConstExp(A.IntConst(n,_),ct) -> Ex(T.CONST(n, ctype2pccType ct))
  | A.ConstExp(A.BoolConst(true,_),_) -> Ex(T.CONST(Int32.one, Pcc.Int))
  | A.ConstExp(A.BoolConst(false,_),_) -> Ex(T.CONST(Int32.zero, Pcc.Int))
  | A.ConstExp(A.NULL(_),_) -> 
      if(!useForeignFuns) then (
	Ex(F.externalCall  "_allocFatPointer" [])
      ) else (
	let null_label = TP.stringLabel ".LC_NULL" in
	Ex(T.NAME(F.nullPointerLabel ()))
	  (*
	let tmp = T.TEMP(TP.simpTemp (Pcc.Pointer(Pcc.Int))) in
	Ex(T.ESEQ(
	   list2SEQ
	     [T.MOVE(tmp, 
		     F.externalCall "_alloc" 
		       [T.CONST(Int32.of_int (3 * F.wordSize_i), Pcc.Int)]);
	      T.MOVE(T.MEM(tmp), T.CONST(Int32.zero, Pcc.Int));
	      T.MOVE(T.MEM(T.BINOP(T.PLUS,
				   tmp,
				   T.CONST(F.wordSize, Pcc.Int))),
		     T.CONST(Int32.zero, Pcc.Int));
	      T.MOVE(T.MEM(T.BINOP(T.PLUS,
				   tmp,
				   T.CONST(Int32.of_int (2 * F.wordSize_i), 
					   Pcc.Int))), 
		     T.CONST(Int32.zero, Pcc.Int))
	    ], 
	   tmp
	  ))
	   *)
       )
  | A.OpExp(e1,op,Some(e2),p,cty) -> 
      (match op with
      | A.PLUS | A.MINUS | A.TIMES | 
	A.BITAND | A.BITOR | A.BITXOR |
	A.SHIFTLEFT | A.SHIFTRIGHT -> 
	  let te1 = unEx(trans_exp e1) in
	  let te2 = unEx(trans_exp e2) in
	  (* catch simple constant math *)
	  (match (te1, te2) with
	  | (T.CONST(n1,_), T.CONST(n2,t)) -> 
	      Ex(T.CONST(eval_const_exp n1 n2 op, t))
	  | _ -> Ex(T.BINOP(trans_op op, te1, te2))
	  )
      | A.DIVIDE | A.MOD -> 
	  let te1 = unEx(trans_exp e1) in
	  let te2 = unEx(trans_exp e2) in
	  (* catch simple constant math *)
	  (match (te1, te2) with
	  | (T.CONST(n1,_), T.CONST(n2,t)) when isNotZero(n2) -> 
	      Ex(T.CONST(eval_const_exp n1 n2 op, t))
	  | (_, T.CONST(n2, t)) when isNotZero(n2) -> 
	      Ex(T.BINOP(trans_op op, te1, te2))
	  | _ -> 
	      let (file, line) = Errormsg.pos2info p in
	      let fileLabel = TP.stringLabel file in
	      let zeroDiv = TP.stringLabel "Division by zero" in
	      let zero = TP.newLabel () in
	      let ok = TP.newLabel () in

	      let o1 = T.TEMP(TP.simpTemp (ctype2pccType cty)) in
	      let o2 = T.TEMP(TP.simpTemp (ctype2pccType cty)) in

	      Ex(T.ESEQ(
		 list2SEQ
		   [T.MOVE(o1, te1);
		    T.MOVE(o2, te2);
		    T.CJUMP(T.EQ, o2, T.CONST(Int32.zero, Pcc.Int), zero, ok);
		    T.LABEL(zero);
		    T.EXP(F.externalCall "_l3_error" 
			    [T.NAME(fileLabel);
			     T.CONST(Int32.of_int line, 
				     Pcc.Int);
			     T.NAME(zeroDiv)]);
		    T.JUMP(TP.getReturnLabel ()); (* don't have to preserve regs *)
		    T.LABEL(ok)
		  ],
		 T.BINOP(trans_op op, o1, o2)
		))
	  )
      | (A.PPLUS|A.PMINUS) ->
	  if(!useForeignFuns) then
	    let oper = 
	      if(op = A.PPLUS) then 
		Int32.one 
	      else 
		Int32.zero
	    in
	    Ex(F.externalCall "_pointerAddition" 
		 [unEx(trans_exp e1); unEx(trans_exp e2); 
		  T.CONST(oper , Pcc.Int)] )
	  else (
	    let result = T.TEMP(TP.simpTemp (ctype2pccType cty)) in
	    let from = T.TEMP(TP.simpTemp (ctype2pccType cty)) in
	    let offset = T.TEMP(TP.simpTemp Pcc.Int) in
	    let isNULL = TP.newLabel () in
	    let isNotNULL = TP.newLabel () in
	    let fini = TP.newLabel () in
	    Ex(T.ESEQ(
	       list2SEQ
		 (* NULL remains always NULL *)
		 [T.MOVE(from, unEx(trans_exp e1)); 
		  T.MOVE(offset, unEx(trans_exp e2));
		  T.CJUMP(T.EQ, from, T.NAME(F.nullPointerLabel ()), 
			  isNULL, isNotNULL);
		  T.LABEL(isNotNULL);
		  T.MOVE(result, F.externalCall "_alloc" 
			   [T.CONST(Int32.of_int (3 * F.wordSize_i), Pcc.Int)]);
		  T.MOVE(T.MEM(result), T.MEM(from));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				       result,
				       T.CONST(Int32.of_int 
						 (2 * F.wordSize_i), Pcc.Int))),
			 T.MEM(T.BINOP(T.PLUS, 
				       from,
				       T.CONST(Int32.of_int 
						 (2 * F.wordSize_i), Pcc.Int))));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				       result,
				       T.CONST(F.wordSize, Pcc.Int))),
			 T.BINOP(trans_op op, (* plus or minus *)
				 T.MEM(T.BINOP(T.PLUS, 
					       from,
					       T.CONST(F.wordSize, Pcc.Int))),
				 offset));
		  T.JUMP(fini);
		  T.LABEL(isNULL);
		  T.MOVE(result, T.NAME(F.nullPointerLabel()));
		  T.LABEL(fini);
		], 
	       result
	      ))
	      (*
	    Ex(T.ESEQ(
	       list2SEQ
		 [T.MOVE(result, F.externalCall "_alloc" 
			 [T.CONST(Int32.of_int (3 * F.wordSize_i), Pcc.Int)]);
		  T.MOVE(from, unEx(trans_exp e1)); 
		  T.MOVE(T.MEM(result), T.MEM(from));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				       result,
				       T.CONST(Int32.of_int 
						 (2 * F.wordSize_i), Pcc.Int))),
			 T.MEM(T.BINOP(T.PLUS, 
				       from,
				       T.CONST(Int32.of_int 
						 (2 * F.wordSize_i), Pcc.Int))));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				       result,
				       T.CONST(F.wordSize, Pcc.Int))),
			 T.BINOP(trans_op op, (* plus or minus *)
				 T.MEM(T.BINOP(T.PLUS, 
					       from,
					       T.CONST(F.wordSize, Pcc.Int))),
				 unEx(trans_exp e2)))
		], 
	       result
	      ))
	       *)
	   )
      | A.LOGICAND -> 
	  let l = TP.newLabel () in
	  let te1 = trans_exp e1 in
	  let te2 = trans_exp e2 in
	  (* Detect constants *)
	  (match (te1, te2) with
	  | (Ex(T.CONST(n1,_)), Ex(T.CONST(n2,_))) -> 
	      if(isZero n1 || isZero n2) then
		Cx(fun (t,f) -> T.JUMP(f))
	      else
		Cx(fun (t,f) -> T.JUMP(t))
 	  | (Ex(T.CONST(n,_)), _) -> 
	      if(isZero(n)) then
		Cx(fun (t,f) -> T.JUMP(f))
	      else
		Cx(fun (t,f) -> unCx(te2) (t,f))
	  | (_, Ex(T.CONST(n,_))) -> 
	      if(isZero(n)) then
		Cx(fun (t,f) -> unCx(te1) (f,f)) (* must eval 1st op *)
	      else
		Cx(fun (t,f) -> unCx(te1) (t,f))
	  | _ -> 
	      Cx(fun (t,f) -> 
		T.SEQ(unCx(te1) (l, f), 
		      T.SEQ(T.LABEL(l),
			    unCx(te2) (t, f))))
	  )
      | A.LOGICOR -> 
	  let l = TP.newLabel () in
	  let te1 = trans_exp e1 in
	  let te2 = trans_exp e2 in
	  (* Detect constants *)
	  (match (te1, te2) with
	  | (Ex(T.CONST(n1,_)), Ex(T.CONST(n2,_))) -> 
	      if(isZero n1 && isZero n2) then
		Cx(fun (t,f) -> T.JUMP(f))
	      else
		Cx(fun (t,f) -> T.JUMP(t))
 	  | (Ex(T.CONST(n,_)), _) -> 
	      if(isZero(n)) then
		Cx(fun (t,f) -> unCx(te2) (t,f))
	      else
		Cx(fun (t,f) -> T.JUMP(t))		
	  | (_, Ex(T.CONST(n,_))) -> 
	      if(isZero(n)) then
		Cx(fun (t,f) -> unCx(te1) (t,f))		
	      else
		Cx(fun (t,f) -> unCx(te1) (t,t)) (* MUST EVAL 1st op *)
	  | _ -> 
	      Cx(fun (t,f) -> 
		T.SEQ(unCx(te1) (t, l), 
		      T.SEQ(T.LABEL(l),
			    unCx(te2) (t, f))))
	  )
      | A.LOGICXOR -> 
	  let lt = TP.newLabel () in
	  let lf = TP.newLabel () in
	  let te1 = trans_exp e1 in
	  let te2 = trans_exp e2 in
	  (* Detect constants *)
	  (match (te1, te2) with
	  | (Ex(T.CONST(n1,_)), Ex(T.CONST(n2,_))) -> 
	      if(isZero n1 && isNotZero n2 || isNotZero n1 && isZero n2) then
		Cx(fun (t,f) -> T.JUMP(t))
	      else
		Cx(fun (t,f) -> T.JUMP(f))
 	  | (Ex(T.CONST(n,_)), _) -> 
	      if(isZero(n)) then
		Cx(fun (t,f) -> unCx(te2) (t,f))
	      else
		Cx(fun (t,f) -> unCx(te2) (f,t))
	  | (_, Ex(T.CONST(n,_))) -> 
	      if(isZero(n)) then
		Cx(fun (t,f) -> unCx(te1) (t,f))		
	      else
		Cx(fun (t,f) -> unCx(te1) (f,t))
	  | _ -> 
	      Cx(fun (t,f) -> 
		list2SEQ
		  [unCx(te1) (lt, lf);
		   T.LABEL(lt);
		   unCx(te2) (f, t);
		   T.LABEL(lf);
		   unCx(trans_exp e2) (t, f) (* used 2nd time, must translate again *)
		 ])
	  )
      | A.EQ | A.NEQ | A.LT | A.LTE | A.GT | A.GTE -> 
	  let te1 = trans_exp e1 in
	  let te2 = trans_exp e2 in
	  (match (te1, te2) with
	  | (Ex(T.CONST(n1,_)), Ex(T.CONST(n2,_))) -> 
	      Ex(T.CONST(eval_const_exp n1 n2 op, Pcc.Int))
	  | _ -> 
	      Cx(fun (t,f) -> T.CJUMP(trans_rel_op op, 
				      unEx(te1),
				      unEx(te2),
				      t, f))
	  )
      | A.PEQ -> (* base && offset equal *)
	  let l = TP.newLabel () in
	  let tmp1 = T.TEMP(TP.simpTemp (Pcc.Pointer(Pcc.Int))) in
	  let tmp2 = T.TEMP(TP.simpTemp (Pcc.Pointer(Pcc.Int))) in
	  Cx(fun (t,f) -> 
	    list2SEQ
	      [T.MOVE(tmp1, unEx(trans_exp e1));
	       T.MOVE(tmp2, unEx(trans_exp e2));
	       T.COMMENT("Compare two Pointers",[]);
	       T.CJUMP(T.EQ, 
		       T.MEM(tmp1),
		       T.MEM(tmp2),
		       l, f);
	       T.LABEL(l);
	       T.CJUMP(T.EQ,
		       T.MEM(T.BINOP(T.PLUS, 
				     tmp1,
				     T.CONST(F.wordSize, Pcc.Int))),
		       T.MEM(T.BINOP(T.PLUS, 
				     tmp2,
				     T.CONST(F.wordSize, Pcc.Int))),
		       t, f)
	     ])
      | A.PNEQ -> 
	  Cx(fun (t,f) -> 
	    unCx(trans_exp (A.OpExp(e1, A.PEQ, Some(e2), p, cty))) (f,t))

      | _ -> raise (TranslateError "Invalid op for binary operarnd")
      )
  | A.OpExp(A.ConstExp(A.IntConst(c,_),t), op, None, _, _) -> 
      (match op with
      | A.UMINUS -> 
	  Ex(T.CONST(Int32.neg c, Pcc.Int))
      | A.BITNOT -> 
	  Ex(T.CONST(Int32.lognot c, Pcc.Int))
      | _ -> raise (TranslateError "Invalid op for unary operand on const")
      )
  | A.OpExp(A.ConstExp(A.BoolConst(b,_),t), op, None, _, _) -> 
      (match op with
      | A.LOGICNOT -> 
	  if(b) then
	    Ex(T.CONST(Int32.zero, Pcc.Int))
	  else
	    Ex(T.CONST(Int32.one, Pcc.Int))
      | _ -> raise (TranslateError "Invalid op for unary operand on const")
      )
  | A.OpExp(e1,op,None,_,_) -> 
      (match op with
      | A.UMINUS -> 
	  Ex(T.BINOP(T.MINUS, 
		     T.CONST(Int32.zero, Pcc.Int),
		     unEx(trans_exp e1)))
      | A.LOGICNOT -> 
	  Cx(fun (t,f) -> unCx(trans_exp e1) (f,t))
      | A.BITNOT -> 
	  Ex(T.BINOP(T.NOT, unEx(trans_exp e1), T.CONST(Int32.zero, Pcc.Int)))
      | _ -> raise (TranslateError "Invalid op for unary operand")
      )
  | A.Alloc(e, p, _, _, A.TyPointer(cty)) -> 
      let tmp = T.TEMP(TP.simpTemp Pcc.Int) in
      let lnegative = TP.newLabel () in (* # of elements < 0 *)
      let lpositive = TP.newLabel () in (* # of elements >= 0 *)
      let (file, line) = Errormsg.pos2info p in
      let fileLabel = TP.stringLabel file in
      let allocNeg = TP.stringLabel "Invalid argument to alloc : < 0" in

      let result = T.TEMP(TP.simpTemp 
			    (ctype2pccType (A.TyPointer(cty)))) in
      
      (match cty with
      | A.TyInt | A.TyBool | A.NS | A.Void -> 
	  if(!useForeignFuns) then
	    Ex(T.ESEQ(T.MOVE(result, 
			     F.externalCall "_allocHeapWithPointer"
			       [unEx(trans_exp e)]),
		      result))
	  else (
	    Ex(T.ESEQ(
	       list2SEQ
		 [T.MOVE(tmp, unEx(trans_exp e));
		  T.CJUMP(T.LT, tmp, T.CONST(Int32.zero, Pcc.Int), 
			  lnegative, lpositive);
		  T.LABEL(lnegative);
		  T.EXP(F.externalCall "_l3_error" 
			  [T.NAME(fileLabel);
			   T.CONST(Int32.of_int line, Pcc.Int);
			   T.NAME(allocNeg)]);
		  T.JUMP(TP.getReturnLabel());
		  T.LABEL(lpositive);
		  T.MOVE(result, 
			 F.externalCall "_alloc"
			   [T.BINOP(T.PLUS, 
				    T.BINOP(T.MUL, 
					    tmp, 
					    T.CONST(F.wordSize, Pcc.Int)),
				    T.CONST(Int32.of_int (3 * F.wordSize_i),
					    Pcc.Int))]);
		  T.MOVE(T.MEM(result), 
			 T.BINOP(T.PLUS,
				 result, 
				 T.CONST(Int32.of_int (3 * F.wordSize_i),
					 Pcc.Int)));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS,
				       result,
				       T.CONST(F.wordSize, Pcc.Int))),
			 T.CONST(Int32.zero, Pcc.Int));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS,
				       result,
				       T.CONST(Int32.of_int 
						 (2 * F.wordSize_i), Pcc.Int))),
			 tmp) (* could store # in temp *)
		],
	       result
	      ))
	  )
      | _ -> (* loop through all elements and initialize them *)
	  let num = T.TEMP(TP.simpTemp Pcc.Int) in
	  let lnegative = TP.newLabel () in (* # of elements < 0 *)
	  let lpositive = TP.newLabel () in (* # of elements >= 0 *)
	  let (file, line) = Errormsg.pos2info p in
	  let lentry = TP.newLabel () in (* entry point to the loop *)
	  let lcondition = TP.newLabel () in (* entry point to the loop *)
	  let ldone = TP.newLabel () in (* jump there when done *)
	  
	  (* Used Algo: 
	     num <- e;
	     result <- alloc(num);
	     while(num > 0) {
	       num -= 1;
	       initialize(result[num])
	     } 
	   *)
	  Ex(T.ESEQ( 
	     list2SEQ 
	       [T.COMMENT("Alloc (START)",[]);
		T.MOVE(num, unEx(trans_exp e));
		T.CJUMP(T.LT, num, T.CONST(Int32.zero, Pcc.Int), 
			lnegative, lpositive);
		T.LABEL(lnegative);
		T.EXP(F.externalCall "_l3_error" 
			[T.NAME(fileLabel);
			 T.CONST(Int32.of_int line, Pcc.Int);
			 T.NAME(allocNeg)]);
		T.JUMP(TP.getReturnLabel());
		T.LABEL(lpositive);
		if(!useForeignFuns) then
		  T.MOVE(result,
			 F.externalCall "_allocHeapWithPointer" [num])
		else (
		  list2SEQ
		    [T.MOVE(result, 
			    F.externalCall "_alloc"
			      [T.BINOP(T.PLUS, 
				       T.BINOP(T.MUL, 
					       num,
					       T.CONST(F.wordSize, Pcc.Int)),
				       T.CONST(Int32.of_int (3 * F.wordSize_i),
					       Pcc.Int))]);
		     T.MOVE(T.MEM(result), 
			    T.BINOP(T.PLUS,
				    result, 
				    T.CONST(Int32.of_int (3 * F.wordSize_i),
					    Pcc.Int)));
		     T.MOVE(T.MEM(T.BINOP(T.PLUS,
					  result,
					  T.CONST(F.wordSize, Pcc.Int))),
			    T.CONST(Int32.zero, Pcc.Int));
		     T.MOVE(T.MEM(T.BINOP(T.PLUS,
					  result,
					  T.CONST(Int32.of_int 
						    (2 * F.wordSize_i), Pcc.Int))),
			    num) (* could store # in temp *)
		   ]
		 );
		T.LABEL(lcondition);
		T.CJUMP(T.GT, 
			num,
			T.CONST(Int32.zero, Pcc.Int), 
			lentry, ldone);
		T.LABEL(lentry);
		T.MOVE(num, 
		       T.BINOP(T.MINUS,
			       num,
			       T.CONST(Int32.one, Pcc.Int)));
		list2SEQ (ini_base_off 
			    (T.MEM(result)) num cty);
		T.JUMP(lcondition);
		T.LABEL(ldone);
		T.COMMENT("Alloc (END)",[])
	      ], 
	     result))
      ) (* match on type *)
  | A.Alloc(_) -> 
      raise (TranslateError "Invalid type of alloc: should be TyPointer")
  | A.StackAlloc(e, p, _, _, A.TyPointer(cty)) ->
      let tmp = T.TEMP(TP.simpTemp Pcc.Int) in
      let lnegative = TP.newLabel () in (* # of elements < 0 *)
      let lpositive = TP.newLabel () in (* # of elements >= 0 *)
      let (file, line) = Errormsg.pos2info p in
      let fileLabel = TP.stringLabel file in
      let allocNeg = TP.stringLabel "Invalid argument to alloc : < 0" in

      let result = T.TEMP(TP.simpTemp 
			    (ctype2pccType (A.TyPointer(cty)))) in
      
      (match cty with
      | A.TyInt | A.TyBool | A.NS | A.Void -> 
	  if(!useForeignFuns) then
	    Ex(T.ESEQ(T.MOVE(result, 
			     F.externalCall "_allocHeapWithPointer"
			       [unEx(trans_exp e)]),
		      result))
	  else (
	    Ex(T.ESEQ(
	       list2SEQ
		 [T.COMMENT("STACK_ALLOC(START A)", []);
		  T.MOVE(tmp, unEx(trans_exp e));
		  T.CJUMP(T.LT, tmp, T.CONST(Int32.zero, Pcc.Int), 
			  lnegative, lpositive);
		  T.LABEL(lnegative);
		  T.EXP(F.externalCall "_l3_error" 
			  [T.NAME(fileLabel);
			   T.CONST(Int32.of_int line, Pcc.Int);
			   T.NAME(allocNeg)]);
		  T.JUMP(TP.getReturnLabel());
		  T.LABEL(lpositive);
		  T.MOVE(result, 
			 T.ALLOCA
			   (T.BINOP(T.PLUS, 
				    T.BINOP(T.MUL, 
					    tmp, 
					    T.CONST(F.wordSize, Pcc.Int)),
				    T.CONST(Int32.of_int (3 * F.wordSize_i),
					    Pcc.Int))));
		  T.MOVE(T.MEM(result), 
			 T.BINOP(T.PLUS,
				 result, 
				 T.CONST(Int32.of_int (3 * F.wordSize_i),
					 Pcc.Int)));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS,
				       result,
				       T.CONST(F.wordSize, Pcc.Int))),
			 T.CONST(Int32.zero, Pcc.Int));
		  T.MOVE(T.MEM(T.BINOP(T.PLUS,
				       result,
				       T.CONST(Int32.of_int 
						 (2 * F.wordSize_i), Pcc.Int))),
			 tmp); (* could store # in temp *)
		  T.COMMENT("STACK_ALLOC(END A)",[])
		 ],
	       result
	      ))
	  )
      | _ -> (* loop through all elements and initialize them *)
	  let num = T.TEMP(TP.simpTemp Pcc.Int) in
	  let lnegative = TP.newLabel () in (* # of elements < 0 *)
	  let lpositive = TP.newLabel () in (* # of elements >= 0 *)
	  let (file, line) = Errormsg.pos2info p in
	  let lentry = TP.newLabel () in (* entry point to the loop *)
	  let lcondition = TP.newLabel () in (* entry point to the loop *)
	  let ldone = TP.newLabel () in (* jump there when done *)
	  
	  (* Used Algo: 
	     num <- e;
	     result <- alloc(num);
	     while(num > 0) {
	       num -= 1;
	       initialize(result[num])
	     } 
	   *)
	  Ex(T.ESEQ( 
	     list2SEQ 
	       [T.COMMENT("StackAlloc (START)",[]);
		T.MOVE(num, unEx(trans_exp e));
		T.CJUMP(T.LT, num, T.CONST(Int32.zero, Pcc.Int), 
			lnegative, lpositive);
		T.LABEL(lnegative);
		T.EXP(F.externalCall "_l3_error" 
			[T.NAME(fileLabel);
			 T.CONST(Int32.of_int line, Pcc.Int);
			 T.NAME(allocNeg)]);
		T.JUMP(TP.getReturnLabel());
		T.LABEL(lpositive);
		if(!useForeignFuns) then
		  T.MOVE(result,
			 F.externalCall "_allocHeapWithPointer" [num])
		else (
		  list2SEQ
		    [T.MOVE(result, 
			    T.ALLOCA
			      (T.BINOP(T.PLUS, 
				       T.BINOP(T.MUL, 
					       num,
					       T.CONST(F.wordSize, Pcc.Int)),
				       T.CONST(Int32.of_int (3 * F.wordSize_i),
					       Pcc.Int))));
		     T.MOVE(T.MEM(result), 
			    T.BINOP(T.PLUS,
				    result, 
				    T.CONST(Int32.of_int (3 * F.wordSize_i),
					    Pcc.Int)));
		     T.MOVE(T.MEM(T.BINOP(T.PLUS,
					  result,
					  T.CONST(F.wordSize, Pcc.Int))),
			    T.CONST(Int32.zero, Pcc.Int));
		     T.MOVE(T.MEM(T.BINOP(T.PLUS,
					  result,
					  T.CONST(Int32.of_int 
						    (2 * F.wordSize_i), Pcc.Int))),
			    num) (* could store # in temp *)
		   ]
		 );
		T.LABEL(lcondition);
		T.CJUMP(T.GT, 
			num,
			T.CONST(Int32.zero, Pcc.Int), 
			lentry, ldone);
		T.LABEL(lentry);
		T.MOVE(num, 
		       T.BINOP(T.MINUS,
			       num,
			       T.CONST(Int32.one, Pcc.Int)));
		list2SEQ (ini_base_off 
			    (T.MEM(result)) num cty);
		T.JUMP(lcondition);
		T.LABEL(ldone);
		T.COMMENT("StackAlloc (END)",[])
	      ], 
	     result))
      ) (* match on type *)
  | A.StackAlloc (_, _,_, _, _) ->
      raise (TranslateError"Invalid type of alloc: should be TyPointer")
  | A.Offset(e, _, _) -> 
      Ex(T.MEM(T.BINOP(T.PLUS, 
		       unEx(trans_exp e),
		       T.CONST(F.wordSize, Pcc.Int))))
  | A.Size(e, _, _) -> 
      Ex(T.MEM(T.BINOP(T.PLUS, 
		       unEx(trans_exp e),
		       T.CONST(Int32.of_int (2 * F.wordSize_i), Pcc.Int))))
(*
  | A.Inc(e, _, cty) -> 
      let tmp = TP.simpTemp (ctype2pccType cty) in
      Ex(T.ESEQ(
	 list2SEQ
	   [T.MOVE(T.TEMP(tmp), unEx(trans_exp e));
	    T.MOVE(T.MEM(T.BINOP(T.PLUS,
				 T.TEMP(tmp),
				 T.CONST(F.wordSize, Pcc.Int))), 
		   T.BINOP(T.PLUS, 
			   T.MEM(T.BINOP(T.PLUS,
					 T.TEMP(tmp),
					 T.CONST(F.wordSize, Pcc.Int))), 
			   T.CONST(Int32.one, Pcc.Int)));
	  ],
	 T.TEMP(tmp)))
  | A.Dec(e, _, cty) -> 
      let tmp = TP.simpTemp (ctype2pccType cty) in
      Ex(T.ESEQ(
	 list2SEQ
	   [T.MOVE(T.TEMP(tmp), unEx(trans_exp e));
	    T.MOVE(T.MEM(T.BINOP(T.PLUS,
				 T.TEMP(tmp),
				 T.CONST(F.wordSize, Pcc.Int))), 
		   T.BINOP(T.MINUS, 
			   T.MEM(T.BINOP(T.PLUS,
					 T.TEMP(tmp),
					 T.CONST(F.wordSize, Pcc.Int))), 
			   T.CONST(Int32.one, Pcc.Int)));
	  ],
	 T.TEMP(tmp)))
*)
  | A.Reference(A.Deref(e,_, _), _, cty) -> 
      let tmp = T.TEMP(TP.simpTemp (ctype2pccType cty)) in
      (* let result = T.TEMP(TP.simpTemp (ctype2pccType cty)) in *)
      Ex(T.ESEQ(T.MOVE(tmp, unEx(trans_exp e)), 
		tmp))
      (*
      Ex(T.ESEQ(
	 list2SEQ
	   [T.COMMENT("Reference (&*p)-> allocate and copy Fat Pointer (START)",[]);
	    T.MOVE(result, F.externalCall "_alloc" 
		     [T.CONST(Int32.of_int (3 * F.wordSize_i), Pcc.Int)]);
	    T.MOVE(tmp, unEx(trans_exp e));
	    T.MOVE(T.MEM(result), T.MEM(tmp));
	    T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				 result,
				 T.CONST(F.wordSize, Pcc.Int))),
		   T.MEM(T.BINOP(T.PLUS, 
				 tmp,
				 T.CONST(F.wordSize, Pcc.Int))));
	    T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				 result,
				 T.CONST(Int32.of_int 
					   (2 * F.wordSize_i), Pcc.Int))),
		   T.MEM(T.BINOP(T.PLUS, 
				 tmp,
				 T.CONST(Int32.of_int 
					   (2 * F.wordSize_i), Pcc.Int))));
	    T.COMMENT("Reference -> allocate Fat Pointer (END)",[])
	  ],
	 result))
       *)
  | A.Reference(l, _, cty) -> 
      (match (trans_lval true l, Checker.ctypeOfLval l) with
      | (T.MEM(tl),_) -> 
	  let result = T.TEMP(TP.simpTemp (ctype2pccType cty)) in
	  Ex(T.ESEQ(
	     list2SEQ
	       [T.COMMENT("Reference -> allocate Fat Pointer (START)",[]);
		T.MOVE(result, F.externalCall "_alloc" 
			 [T.CONST(Int32.of_int (3 * F.wordSize_i), Pcc.Int)]);
		T.MOVE(T.MEM(result), tl);
		T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				     result,
				     T.CONST(F.wordSize, Pcc.Int))),
		       T.CONST(Int32.zero, Pcc.Int));
		T.MOVE(T.MEM(T.BINOP(T.PLUS, 
				     result,
				     T.CONST(Int32.of_int 
					       (2 * F.wordSize_i), Pcc.Int))),
		       T.CONST(Int32.one, Pcc.Int));
		T.COMMENT("Reference -> allocate Fat Pointer (END)",[])
	      ],
	     result))
      | _ -> raise (TranslateError "Invalid L-val in Reference")
      )
  | A.Call(A.Ident(fName,_), params, _, _) -> 
      Ex(F.externalCall fName 
	   (List.map (fun (e,_) -> unEx(trans_exp e)) params))
  | A.LVal(l) -> 
     Ex(trans_lval false l)


(******************************************************************************)
(* Translate L-Values *)
(******************************************************************************)

(* must strip off the T.MEM() to get the lval *)
and trans_lval isLval = function
  | A.Var(A.Ident(name,_), cty) -> 
      (match (TP.isEscape name) with
      | true -> 
	  let n = TP.namedTemp name (Pcc.Pointer((ctype2pccType cty))) in
	  T.MEM(T.TEMP(n))
      | false -> 
	  let n = TP.namedTemp name (ctype2pccType cty) in
	  T.TEMP(n) (* This is not an l-val! *)
      )
  | A.Deref(e,p,cty) -> 
      let (file, line) = Errormsg.pos2info p in
      let fileLabel = TP.stringLabel file in
      if(!useForeignFuns) then (
	T.MEM(F.externalCall "_checkFatHeapPointer" [unEx(trans_exp e); 
						     T.NAME(fileLabel);
						     T.CONST(Int32.of_int line, 
							     Pcc.Int)])
       ) else (
	let (file, line) = Errormsg.pos2info p in
        (* for better comments -wjl *)
        let location = file ^ " line " ^ string_of_int line in
	let fileLabel = TP.stringLabel file in
	let negativeOffset = TP.stringLabel "Offset < 0" in
	let bigOffset = TP.stringLabel "Offset >= Size" in
	let res = T.TEMP(TP.simpTemp (Pcc.Pointer(ctype2pccType cty))) in
	let off = T.TEMP(TP.simpTemp (Pcc.Pointer(Pcc.Int))) in
	let negative = TP.newLabel () in (* offset is negative *)
	let positive = TP.newLabel () in (* offset is >= 0 *)
	let big = TP.newLabel () in (* offset is >= size *)
	let ok =  TP.newLabel () in (* if ok *)
 	let exp = T.TEMP(TP.simpTemp (Pcc.Pointer(Pcc.Int))) in

	let ini = 
	  match e with
	  | A.OpExp(pointer, ((A.PPLUS|A.PMINUS) as op), Some(offset), _, _) -> 
	      (* catch the special case *(p + off) 
	       * such that no new pointer is created *)
	      list2SEQ
		[T.COMMENT("Dereference fat pointer *(p + off) [from "
                            ^ location ^ "]",[]);
		 T.MOVE(exp, unEx(trans_exp pointer));
		 T.MOVE(off, 
			T.BINOP(trans_op op, (* calculate final offset *)
				T.MEM(T.BINOP(T.PLUS,  
					      exp,
					      T.CONST(F.wordSize, Pcc.Int))),
				unEx(trans_exp offset)))]
	  | _ -> 
	      list2SEQ
		[T.COMMENT("Dereference fat pointer *p [from "
                            ^ location ^ "]",[]);
		 T.MOVE(exp, unEx(trans_exp e)); (* get the thin-pointer *)
		 T.MOVE(off, T.MEM(T.BINOP(T.PLUS,  (* read the offset once *)
					   exp,
					   T.CONST(F.wordSize, Pcc.Int))))]
	in
	T.ESEQ(
	list2SEQ
	  [ini;
	   T.CJUMP(T.LT, off,
		   T.CONST(Int32.zero, Pcc.Int), 
		   negative, positive);
	   T.LABEL(negative);
	   T.EXP(F.externalCall "_l3_error" 
		   [T.NAME(fileLabel);
		    T.CONST(Int32.of_int line, 
			    Pcc.Int);
		    T.NAME(negativeOffset)]);
	   T.JUMP(TP.getReturnLabel ()); (* don't have to preserve regs *)
	   T.LABEL(positive);
	   T.CJUMP(T.GE, off,
		   T.MEM(T.BINOP(T.PLUS, 
				 exp,
				 T.CONST(Int32.of_int (2 *F.wordSize_i),
					 Pcc.Int))),
		   big, ok);
	   T.LABEL(big);
	   T.EXP(F.externalCall "_l3_error" 
		   [T.NAME(fileLabel);
		    T.CONST(Int32.of_int line, 
			    Pcc.Int);
		    T.NAME(bigOffset)]);
	   T.JUMP(TP.getReturnLabel ()); (* don't have to preserve regs *)
	   T.LABEL(ok);
	   if(isLval) then
	     T.MOVE(res, 
		    T.BINOP(T.PLUS, 
			    T.MEM(exp),
			    T.BINOP(T.MUL, off,
				    T.CONST(F.wordSize, Pcc.Int))))
	   else
	     T.MOVE(res, 
		    T.MEM(T.BINOP(T.PLUS, 
				  T.MEM(exp),
				  T.BINOP(T.MUL, off,
					  T.CONST(F.wordSize, Pcc.Int)))))
	 ], 
	if(isLval) then
	  T.MEM(res) (* might be a dest -> don't dereference before *)
	else
	  res
       )
       )
  | A.Field(l, A.Ident(name,_),_,_) -> 
      let cty = Checker.ctypeOfLval l in
      (match cty with
      | A.TyInt | A.TyBool | A.NS | A.Void | A.TyPointer(_) -> 
	  raise (TranslateError ("Not a struct-type for field access, " ^
				 Checker.ctype2string cty))
      | A.TyRecord(sName) -> 
	  let sList = Checker.getStructType sName in
	  (* must find offset in struct *)
	  let (offset, found) = List.fold_left 
	      (fun (counter, found) (n, _) -> 
		if(found) then 
		  (counter, found)
		else 
		  if(0 = (Pervasives.compare n name)) then 
		    (counter,true)
		  else 
		    (counter+1, false)
	      ) (0,false) sList
	  in
	  if(not found) then
	    raise (TranslateError "Cannot find field in struct");
	  if(offset = 0) then
	    T.MEM(trans_lval false l)
	  else
	    T.MEM(T.BINOP(T.PLUS, 
			  trans_lval false l, 
			  T.CONST(Int32.of_int (F.wordSize_i * offset), 
				  Pcc.Int)))
      )

      
(******************************************************************************)
(* Translate Statements *)
(******************************************************************************)

(* breakTo and continueTo are labels where break and continue jump to *)
let rec trans_stmt breakTo continueTo = function
  | A.Assign(l, _, e, _) -> 
      T.MOVE(trans_lval true l, unEx(trans_exp e)) 
  | A.If(e, _, sl, _) -> 
      let lfalse = TP.newLabel () in 
      let ltrue = TP.newLabel () in 
      list2SEQ
	[unCx(trans_exp e) (ltrue, lfalse);
	 T.LABEL(ltrue);
	 list2SEQ (List.map (fun s -> trans_stmt breakTo continueTo s) sl);
	 T.LABEL(lfalse)
       ]

  | A.IfElse(e, _, sl1, sl2, _) -> 
      let lfalse = TP.newLabel () in 
      let ltrue = TP.newLabel () in 
      let ldone = TP.newLabel () in 
	list2SEQ
	[unCx(trans_exp e) (ltrue, lfalse);
	 T.LABEL(ltrue);
	 list2SEQ (List.map (fun s -> trans_stmt breakTo continueTo s) sl1);
	 T.JUMP(ldone);
	 T.LABEL(lfalse);
	 list2SEQ (List.map (fun s -> trans_stmt breakTo continueTo s) sl2);
	 T.LABEL(ldone)
       ]

  | A.While(e, _, sl) ->
      let lentry = TP.newLabel () in 
      let ldone = TP.newLabel () in 
      let ltest = TP.newLabel () in
	list2SEQ
	  [T.LABEL(ltest);
	   unCx(trans_exp e) (lentry, ldone);
	   T.LABEL(lentry);
	   list2SEQ  (List.map (fun s -> trans_stmt ldone ltest s) sl);
	   T.JUMP(ltest);
	   T.LABEL(ldone)
	  ]
	
  | A.For(s1,_,e,_,s2,_,sl) ->
      let lstartloop = TP.newLabel() in
      let lcondition = TP.newLabel () in
      let lentry = TP.newLabel () in
      let ldone = TP.newLabel () in
      let lcontinue = TP.newLabel () in
      list2SEQ
	[T.LABEL(lstartloop);
	 trans_stmt breakTo continueTo s1;
	 T.LABEL(lcondition);
	 unCx(trans_exp e) (lentry, ldone);
	 T.LABEL(lentry);
	 list2SEQ  (List.map (fun s -> trans_stmt ldone lcontinue s) sl);
	 T.LABEL(lcontinue);
	 trans_stmt ldone lcontinue s2;
	 T.JUMP(lcondition);
	 T.LABEL(ldone)
       ]
  
  | A.StmtExp(e, _) -> 
      (* Easier for Dead-code *)
      T.MOVE(T.TEMP(TP.simpTemp Pcc.Int),
	     unEx(trans_exp e));
      (* T.EXP(unEx(trans_exp e)) *)

  | A.Return(Some(e), _) -> 
      T.SEQ(T.MOVE(T.TEMP(F.getReturnTemp ()), 
		   unEx(trans_exp e)), 
	    T.JUMP(TP.getReturnLabel ()))

  | A.Return(None, _) -> 
      T.JUMP(TP.getReturnLabel ())

  | A.Continue(_) -> 
      T.JUMP(continueTo)

  | A.Break(_) -> 
      T.JUMP(breakTo)

  | A.Blank -> 
      T.nop


(******************************************************************************)
(* Translate Declarations *)
(******************************************************************************)

let rec trans_decl = function
  | A.StructDecl(_) -> T.nop
  | A.VarDecl(A.Ident(name, _), typ, _) -> 
      let ctyp = Checker.typ2ctype typ in
      (match (TP.isEscape name) with
      | true -> 
	  let tmp = T.TEMP(TP.namedTemp name (Pcc.Pointer(ctype2pccType ctyp))) in
	  T.SEQ(T.MOVE(tmp, F.externalCall "_alloc" 
			 [T.CONST(F.wordSize, Pcc.Int)]),
		list2SEQ (ini_mem (T.MEM(tmp)) ctyp))
      | false -> 
	  let tmp = T.TEMP(TP.namedTemp name (ctype2pccType ctyp)) in
	  list2SEQ (ini_mem tmp ctyp)
      )    
  | A.FunDecl(A.Ident(fName,_), rtyp, _, params, body) -> 
      (match body with
      | A.Foreign(_) -> T.nop
      | A.Body(dl, sl) -> 
	  (* Enter proper scope to read escaping analysis *)
	  TP.translateNewFun fName (ctype2pccType (Checker.typ2ctype rtyp)); 
	  (* If parameter escapes, put on heap *)
	  let (paramIni, _) = 
	    List.fold_left (fun (result, counter) (A.Ident(pName,_), typ, _) -> 
	      let ctyp = Checker.typ2ctype typ in 
	      let pccTyp = ctype2pccType ctyp in
	      let ini = 
		match (TP.isEscape pName) with
		| true -> 
		    let res = T.TEMP(TP.namedTemp pName 
				       (Pcc.Pointer(ctype2pccType ctyp))) in
		    [T.SEQ(T.MOVE(res, F.externalCall "_alloc" 
				    [T.CONST(F.wordSize, Pcc.Int)]),
			   T.MOVE(T.MEM(res), F.paramTemp counter pccTyp))]
		| false -> 
		    let res = T.TEMP(TP.namedTemp pName pccTyp) in
		    [T.MOVE(res, F.paramTemp counter pccTyp)]
	      in
	      (result @ ini, counter + 1)
			   ) ([], 1) params 
	  in
	  let varIni = 
	    List.fold_right (fun vdecl res -> 
	      match trans_decl vdecl with
	      | T.COMMENT("",_) -> res
	      | x -> x :: res
			    ) dl []
	  in
	  let body = 
	    List.fold_right (fun stmt res -> 
	      match trans_stmt 0 0 stmt with
	      | T.COMMENT("",_) -> res
	      | x -> x :: res
			    ) sl [T.JUMP(TP.getReturnLabel ())]
	  in
	  (* Store created Look-up tables for this fun *)
	  TP.translatedFun fName;
	  list2SEQ (paramIni @ varIni @ body)
      )

(* returns a list of (fName, Ir.stmt) *)
let trans_program (A.Program(dl)) = 
  setSystemFunctionsInSigma ();
  List.fold_right (fun decl res -> 
    match decl with
    | A.StructDecl(_) | A.VarDecl(_) -> res
    | A.FunDecl(_,_,_,_,A.Foreign(_)) -> res
    | A.FunDecl(A.Ident(fName,_),_,_,_,_) -> 
	(match trans_decl decl with
	| T.COMMENT("",_) -> res 
	| x -> 
	    (fName, x) :: res
	)
		  ) dl []
    
