(* temp.mli *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: temp.mli,v 1.4 2003/08/28 20:25:34 rflury Exp $ *)

exception TempError of string
    
(***********************************************************************)
(* Variables *)
(***********************************************************************)
    
(* Type of a temporary variable *)
type temp = int
	  
(* Type annotations for variables *)
type pcc_type = Pcc.pccType

(* Returns a string representation of a temp *)
val temp2string : temp -> string

(* Return a temp for a named variable of given type *)
val namedTemp : string -> Pcc.pccType -> temp

(* Return the name of the temp used by the user or "#" if it is a compiler-temp *)
val nameOfTemp : temp -> string

(* Return a temp for a new unnamed *)
val simpTemp : Pcc.pccType -> temp

(* Returns PccType of a temporary, 
 * raises Not_found if no type is found for the temp *)
val temp2pccType : temp -> Pcc.pccType

(* Lists all variable-names of the function and their temp int-value *)
val listBindings : unit -> unit

(* Returns the the next temporary that would be handed out *)
val getTempCounter : unit -> temp

(* First temp ID ever assigned -- lower ones are reserved -wjl *)
val firstTemp : temp

(* checks to see that a temp is not a machine reserved temp -wjl *)
val unreserved : temp -> bool
(* or conversely, if it is reserved -wjl *)
val reserved : temp -> bool

(***********************************************************************)
(* Escaping Analysis, var_name -> escape? *)
(***********************************************************************)

(* Add a variable to the escape-list *)
val addEscape : string -> unit
    
(* Check whether a variable escapes *)	
val isEscape : string -> bool

(***********************************************************************)
(* Labels *)
(***********************************************************************)

(* Labels in the assembly language; can refer to a name *)
type label = int

(* Returns a new label *)
val newLabel : unit -> label

(* Returns a label that refers to a name *)
val namedLabel : string -> label

(* Returns a Label pointnig to a string in the .rodata section *)
val stringLabel : string -> label 
	
(* returns the string-representation of a label *)
val label2string : label -> string

(* Jump to this label to return from a function *)
val getReturnLabel : unit -> label

(* Returns a hashtable containing all string labels *)
val getStringLabels : unit -> (label, string) Hashtbl.t

(***********************************************************************)
(* Function scopes *)
(***********************************************************************)

(* The name of the current function *)
val funName : string ref

(* Called before checking a function *)
val checkNewFun : string -> unit

(* Called AFTER checking a function *)
val checkedFun : string -> unit

(* Called before translating a function *)
val translateNewFun : string -> Pcc.pccType -> unit

(* Called after translating a function *)
val translatedFun : string -> unit

(* Sets the different Look-up tables for the given function *)
val openFunLookUp : string -> unit

(* Stores the Look-up tables for the given function *)
val saveFunLookUp : string -> unit
