(* temp.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: temp.ml,v 1.5 2003/09/15 13:25:06 rflury Exp $ *)

module HA=Hashtbl
module A=Absyn

exception TempError of string
    

(* Overview: 
 * Temps are temporary registers that can be used in a program. We will assume
 * that we have an infinite number of registers until we do the register 
 * allocation which is responsible to shrink the number of registers to the
 * available ones. 
 * This file is designed to hand out temps per function - i.e. every function
 * has its own state of how many and which temps have been used. The reason 
 * for this design-decision lies in the back-end. It makes especially 
 * liveness analysis & register allocation easier, where I use the value of
 * the temps as offsets into bitlists. 
 * The variable 'firstTempID' below allows you to specify the smallest 
 * temp ever handed out. All temps < firstTempID can be used as 'real' 
 * machine registers. 
 * The section 'Function scopes' describes a few functions that allow to 
 * handle several functions at the same time. 
 *)


(***********************************************************************)
(* Variables *)
(***********************************************************************)
    
(* Type of a temporary variable *)
type temp = int
      
(* type annotations for variables *)
type pcc_type = Pcc.pccType
      
type ty_name_temp = (string, (temp * pcc_type)) HA.t
type ty_temp_name = (temp, string) HA.t
type ty_temp_ctype = (temp, Pcc.pccType) HA.t
type ty_tmp_forceX = (temp, temp list) HA.t
  
(* Hashtbls to store info about temps *)
let name_temp : ty_name_temp ref = ref (HA.create 371)
let temp_name : ty_temp_name ref = ref (HA.create 317)
let temp_ctype : ty_temp_ctype ref = ref (HA.create 371)
let tmp_forceX : ty_tmp_forceX ref = ref (HA.create 371)

(* this is machine dependent *)
let tmp_counter : temp ref = ref 10 
    
(* roland conveniently forgot this one -wjl *)
let firstTemp = !tmp_counter + 1
let reserved t = t < firstTemp
let unreserved t = t >= firstTemp
    
(* Stores all the bindings by function Name *)
let fName_bindings : 
    (string, (temp * ty_name_temp * ty_temp_name * 
		ty_temp_ctype * ty_tmp_forceX )) HA.t
    =  HA.create 57
    
(* Get a new temporary-number *)
let newTemp () = 
  tmp_counter := !tmp_counter + 1;
  !tmp_counter
    
(* Return a temp for a named variable of given type *)
let namedTemp name typ = 
  if(HA.mem !name_temp name) then
    fst (HA.find !name_temp name)
  else (
    let tmp = newTemp () in
    HA.add !name_temp name (tmp, typ);
    HA.add !temp_name tmp name;
    HA.add !temp_ctype tmp typ;
    tmp
   )
      
(* Return a temp for a new unnamed *)
let simpTemp typ = 
  let tmp = newTemp () in
  HA.add !temp_ctype tmp typ;
  tmp
    
(* Returns PccType of a temporary, 
 * raises Not_found if no type is found for the temp *)
let temp2pccType temp = 
  HA.find !temp_ctype temp
    
(* Returns a string representation of a temp *)
let temp2string t = 
  Pervasives.string_of_int t
    
(* Lists all variable-names of the function and their temp int-value *)
let listBindings () =
  HA.iter (fun s (i,_) -> 
    print_string (s ^ ":");
    print_int i;
    print_string "  ") !name_temp
    
(* Returns the the next temporary that would be handed out *)
let getTempCounter () = 
  !tmp_counter + 1


(* Return the name of the temp used by the user or "#" if it is a compiler-temp *)
let nameOfTemp temp =
  (try
    HA.find !temp_name temp 
  with Not_found -> "#"
  )

(***********************************************************************)
(* Escaping Analysis, var_name -> escape? *)
(***********************************************************************)

(* Escape-Analysis maps var-name to true if *)
module EscapeListM =
  struct
    type t = string
    let compare s1 s2 = Pervasives.compare s1 s2
  end
module EscapeList = Set.Make (EscapeListM)

(* The set containing all variable-names that escape *)
let escapes : EscapeList.t ref = ref EscapeList.empty
    
(* Add a variable to the escape-list *)
let addEscape name = 
  if(not (EscapeList.mem name !escapes)) then 
    escapes := EscapeList.add name !escapes
	
(* Check whether a variable escapes *)	
let isEscape name = 
  EscapeList.mem name !escapes
   
(* History of escaping-lists by function-name *)
let fName_escapes : (string, EscapeList.t) HA.t = HA.create 57
    
(***********************************************************************)
(* Labels *)
(***********************************************************************)
    
(* Labels in the assembly language; can refer to a name *)
type label = int
      
let label_counter = ref 0
    
(* For fast indexing, always in both directions *)
let name_label : (string, int) HA.t = HA.create 57
let label_name : (int, string) HA.t = HA.create 57
let label_string : (int, string) HA.t = HA.create 57
let string_label : (string, int) HA.t = HA.create 57
    
(* Returns a new label *)
let newLabel () = 
  label_counter := !label_counter + 1;
  !label_counter
    
(* Returns a label that refers to a name *)
let namedLabel name = 
  try 
    HA.find name_label name
  with Not_found -> 
    let tmp = newLabel () in
    let symbol = Helpers.get_external_symbol name in
    HA.add name_label symbol tmp;
    HA.add label_name tmp symbol;
    tmp
      
(* Returns a Label pointnig to a string in the .rodata section *)
let stringLabel s = 
  try 
    HA.find string_label s
  with Not_found -> 
    let tmp = newLabel () in
    HA.add string_label s tmp;
    HA.add label_string tmp s;
    tmp
      
(* returns the string-representation of a label *)
let label2string label = 
  (try
    HA.find label_name label (* don't change this! *)
  with Not_found -> 
    if(HA.mem label_string label) then
      ".LC" ^ (Pervasives.string_of_int label)
    else
      ".L" ^ (Pervasives.string_of_int label)
  )

(* Jump to this label to return from a function *)
let returnLabel = ref (newLabel ())
    
(* Jump to this label to return from a function *)
let getReturnLabel () =
  !returnLabel

(* Returns a hashtable containing all string labels *)
let getStringLabels () = 
  label_string
    
(***********************************************************************)
(* Function scopes *)
(***********************************************************************)

(* The name of the current function *)
let funName = ref ""
    
(* Called before checking a function *)
let checkNewFun fName = 
  funName := fName;
  escapes := EscapeList.empty
      
(* Called AFTER checking a function *)
let checkedFun fName = 
  HA.add fName_escapes fName !escapes
    
(* Called before translating a function with given return type *)
let translateNewFun fName retType =
  funName := fName;
  (try
    tmp_counter := 10; (* 1..8 reserved - this is machine dependent *)
    escapes := HA.find fName_escapes fName;
    name_temp := HA.create 371;
    temp_name := HA.create 371;
    temp_ctype := HA.create 371;
    tmp_forceX := HA.create 371;
  with Not_found -> 
    raise (TempError ("lost escaping info for fun " ^ fName))
  )
    
(* Called after translating a function *)
let translatedFun fName = 
  HA.replace fName_bindings fName (!tmp_counter, 
				   !name_temp, 
				   !temp_name,
				   !temp_ctype, 
				   !tmp_forceX)
    
(* Sets the different Look-up tables for the given function *)
let openFunLookUp fName = 
  funName := fName;
  (try
    escapes := HA.find fName_escapes fName;
    let (c, nt, tn, tc, tf) = HA.find fName_bindings fName in
    tmp_counter := c;
    name_temp := nt;
    temp_name := tn;
    temp_ctype := tc;
    tmp_forceX := tf;
  with Not_found -> 
    raise (TempError ("Lost escaping or name->temp info for fun " ^ fName))
  )
    
(* Stores the Look-up tables for the given function *)
let saveFunLookUp fName = 
  HA.replace fName_bindings fName (!tmp_counter, 
				   !name_temp, 
				   !temp_name,
				   !temp_ctype, 
				   !tmp_forceX)



