(* regalloc.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: regalloc.ml,v 1.4 2003/09/26 16:54:49 rflury Exp $ *)

module HA = Hashtbl
module TP = Temp
module A = Assem
module BL = Bitlist
module IG = Igraph
module F = (Frame.X86_FRAME : Frame.FrameSig)
module T = Ir
module E = Errormsg
module AR = Array

exception RegallocError of string

type node_t = {simplifyWL: BL.bitList_t;
	       freezeWL: BL.bitList_t;
	       spillWL: BL.bitList_t;
	       spilled: BL.bitList_t;
	       coalesced: BL.bitList_t;
	       colored: BL.bitList_t;
	       selectStack: BL.bitList_t;
	       stack: TP.temp Stack.t;
	     }

(* Store tuples (int * int) in a set, used for Move-lists *)
module MoveListM = 
  struct
    type t = (TP.temp * TP.temp)
    let compare m1 m2 = 
      if(fst m1 = fst m2) then
	if(snd m1 = snd m2) then
	  0
	else if(snd m1 > snd m2) then
	  1
	else
	  -1
      else if(fst m1 > fst m2) then
	1 
      else
	-1
  end
module MoveList = Set.Make (MoveListM)

type move_t = {mutable coalescedMoves: MoveList.t;
	       mutable constrainedMoves: MoveList.t;
	       mutable frozenMoves: MoveList.t;
	       mutable worklistMoves: MoveList.t;
	       mutable activeMoves: MoveList.t;
	     }

(* Constant pool *)
(* # of 'usable' registers *)
let k = 6
(* # of registers *)
let regs = 8

let adjSet: IG.igraph_t = ref (AR.make_matrix 1 1 0)
let adjList: (TP.temp, TP.temp list) HA.t ref = ref (HA.create 1009)
let degree: (TP.temp, int) HA.t ref = ref (HA.create 1009)
let moveList: (TP.temp, MoveList.t) HA.t ref = ref (HA.create 1009)
let iniBL () = BL.newBitList (TP.getTempCounter ()) 
let alias: (TP.temp, TP.temp) HA.t ref = ref (HA.create 1009)
let color: (TP.temp, TP.temp) HA.t ref = ref (HA.create 1009)
let node_info: node_t ref = ref {simplifyWL = BL.newBitList 1;
				 freezeWL = BL.newBitList 1;
				 spillWL = BL.newBitList 1;
				 spilled = BL.newBitList 1;
				 coalesced = BL.newBitList 1;
				 colored = BL.newBitList 1;
				 selectStack = BL.newBitList 1;
				 stack = Stack.create ();
			       } 
    
let move_info : move_t ref = ref {coalescedMoves = MoveList.empty;
				  constrainedMoves = MoveList.empty;
				  frozenMoves = MoveList.empty;
				  worklistMoves = MoveList.empty;
				  activeMoves = MoveList.empty;
				} 

(***********************************************************************)
(* Helpers for debugging *)
(***********************************************************************)
let print_node_info node_info = 
  print_string "\nsimplifyWL: "; 
  BL.iterTrue (fun x -> Printf.printf "%i, " x) (!node_info).simplifyWL;
  print_string "\nfreezeWL: "; 
  BL.iterTrue (fun x -> Printf.printf "%i, " x) (!node_info).freezeWL;
  print_string "\nspillWL: "; 
  BL.iterTrue (fun x -> Printf.printf "%i, " x) (!node_info).spillWL;
  print_string "\nspilled: "; 
  BL.iterTrue (fun x -> Printf.printf "%i, " x) (!node_info).spilled;
  print_string "\ncoalesced: "; 
  BL.iterTrue (fun x -> Printf.printf "%i, " x) (!node_info).coalesced;
  print_string "\ncolored: "; 
  BL.iterTrue (fun x -> Printf.printf "%i, " x) (!node_info).colored;
  print_string "\nselectStack: "; 
  BL.iterTrue (fun x -> Printf.printf "%i, " x) (!node_info).selectStack;
  print_newline ()
  
let precolored n = 
  n <= regs

let notPrecolored n = 
  n > regs

let nodeMoves n =
  (try
    MoveList.inter (HA.find (!moveList) n) 
      (MoveList.union ((!move_info).activeMoves) ((!move_info).worklistMoves))
  with Not_found -> 
    MoveList.empty
  )

let moveRelated n = 
  not (MoveList.is_empty (nodeMoves n))

let makeWorklist () = 
  HA.iter (fun n _ -> 
    (try
      if((HA.find (!degree) n) >= k) then
	BL.setBit (!node_info).spillWL n
      else if(moveRelated n) then
	BL.setBit (!node_info).freezeWL n
      else
	BL.setBit (!node_info).simplifyWL n
    with Not_found -> 
      raise (RegallocError 
	       (Printf.sprintf "makeWorklist: temp %i not found in degree" n))
    )
	  ) (!adjList)
    
(* Returns a bit-list of all adjascent temps of temp n *)
let adjacent n = 
  let tmp = BL.orBLnew (!node_info).selectStack (!node_info).coalesced in
  let al = BL.newBitList (TP.getTempCounter ()) in
  List.iter (fun x -> BL.setBit al x) (HA.find (!adjList) n);
  BL.andBLnew al (BL.notBLnew tmp)

let enableMoves nodeBL = 
  BL.iterTrue (fun n -> 
    MoveList.iter (fun m -> 
      if(MoveList.mem m (!move_info).activeMoves) then (
	(!move_info).activeMoves <- 
	  MoveList.remove m (!move_info).activeMoves;
	(!move_info).worklistMoves <- 
	  MoveList.add m (!move_info).worklistMoves;
       )
		  ) (nodeMoves n)
	      ) nodeBL

let enableOneMove n = 
  MoveList.iter (fun m -> 
    if(MoveList.mem m (!move_info).activeMoves) then (
      (!move_info).activeMoves <- 
	MoveList.remove m (!move_info).activeMoves;
      (!move_info).worklistMoves <- 
	MoveList.add m (!move_info).worklistMoves;
     )
		) (nodeMoves n)
    
let decrementDegree m = 
  let d = 
    (try
      HA.find (!degree) m 
    with Not_found -> 
      raise (RegallocError 
	       (Printf.sprintf "decrementDegree: temp %i not found in degree" m))
    ) in
  HA.replace (!degree) m (d-1);
  if(d = k) then (
    let a = adjacent m in
    BL.setBit a m;
    enableMoves a;
    BL.resetBit (!node_info).spillWL m;
    if(moveRelated m) then
      BL.setBit (!node_info).freezeWL m
    else
      BL.setBit (!node_info).simplifyWL m
   )
	

(* returns true if there was a temp to simplify *)
let simplify () = 
  let n = BL.findSet (!node_info).simplifyWL in
  if(n < 0) then
    false
  else (
    BL.resetBit (!node_info).simplifyWL n;
    Stack.push n (!node_info).stack;
    BL.setBit (!node_info).selectStack n;
    let a = adjacent n in
    BL.iterTrue (fun m -> 
      decrementDegree m
		) a;
    E.showProgress "s";
    if(!E.debug) then
      print_node_info node_info;
    true
   )


let addWorkList u =
  (try
    if(notPrecolored u && 
       (not (moveRelated u)) && 
       ((HA.find (!degree) u) < k)) then (

      (* DEBUG (2 lines) *)
      if(not (BL.getBit (!node_info).freezeWL u)) then (
	print_node_info node_info;	
	Printf.printf "addWorkList, temp %i not in freezeWL\n" u;
	flush stdout;
       );
      
      BL.resetBit (!node_info).freezeWL u;
      BL.setBit (!node_info).simplifyWL u;
     )
  with Not_found -> 
    raise (RegallocError 
	     (Printf.sprintf "addWorkList: temp %i not found in degree" u))
  )

let ok_fun t r = 
  (try
    precolored t || (HA.find (!degree) t) < k || IG.getBit t r adjSet
  with Not_found -> 
    raise (RegallocError 
	     (Printf.sprintf "ok_fun: temp %i not found in degree" t))
  )    

let conservative nodes = 
  let result = BL.fold (fun res n -> 
    (try
      if(HA.find (!degree) n >= k) then
	res + 1
      else 
	res
    with Not_found -> 
      raise (RegallocError 
	       (Printf.sprintf "conservative: temp %i not found in degree" n))
    )    
		       ) 0 nodes in
  result < k

let rec getAlias n = 
  if(BL.getBit (!node_info).coalesced n) then
    (try
      getAlias (HA.find (!alias) n)
    with Not_found -> 
      raise (RegallocError 
	       (Printf.sprintf "getAlias: temp %i not found in alias" n))
    )
  else n


let addEdge u v =
  if((not (IG.getBit u v adjSet)) && u != v) then (
    IG.setBit u v adjSet;
    IG.setBit v u adjSet;
    if(notPrecolored u) then (
      (try 
	HA.replace (!adjList) u (v :: (HA.find (!adjList) u));
	HA.replace (!degree) u (1 + (HA.find (!degree) u));
      with Not_found -> 
	raise (RegallocError
		 (Printf.sprintf 
		    "addEdge: temp %i not found in adjList or degree (1)" u))
      )
     );
    if(notPrecolored v) then (
      (try
	HA.replace (!adjList) v (u :: (HA.find (!adjList) v));
	HA.replace (!degree) v (1 + (HA.find (!degree) v));
      with Not_found -> 
	raise (RegallocError
		 (Printf.sprintf 
		    "addEdge: temp %i not found in adjList or degree (2)" v))
      )
     )
   )
	
(* Use u instead of v (replace v with u) *)
let combine u v = 
  E.print_debug (Printf.sprintf "Combine: for %i use now %i\n" v u);
  (* IG.printIGraph adjSet; *)
  if(IG.getBit u v adjSet) then
    raise (RegallocError 
	     (Printf.sprintf 
		"combine: try to combine interfereing temps %i and %i"
		u v));
  if(BL.getBit (!node_info).freezeWL v) then
    BL.resetBit (!node_info).freezeWL v
  else (
    (* Sanity check *)
    if(not (BL.getBit (!node_info).spillWL v)) then (
      print_node_info node_info;
      raise (RegallocError (
	     Printf.sprintf 
	       "combine: temp %i neither in spill nor freeze" v));
     );
    BL.resetBit (!node_info).spillWL v;
   );
  BL.setBit (!node_info).coalesced v;
  HA.add (!alias) v u;

  if(HA.mem (!moveList) u) then (
    if(HA.mem (!moveList) v) then
      HA.replace (!moveList) u 
	(MoveList.union (HA.find (!moveList) u) (HA.find (!moveList) v));
      (* else
	 HA.replace moveList u (HA.find moveList u) *)
   ) else (
    if(HA.mem (!moveList) v) then
      HA.replace (!moveList) u (HA.find (!moveList) v)
    else
      HA.replace (!moveList) u MoveList.empty;
   );

  enableOneMove v;
  let adj = adjacent v in
  BL.iterTrue (fun t -> 
    if(!E.debug) then
      Printf.printf "Combine, add edge %i <-> %i\n" t u;
    addEdge t u;
    decrementDegree t;
	      ) adj;
  
  (* Must add edges also for temps already on the stack - they could be 
   * precolored *)
  List.iter (fun t -> 
    if(not (BL.getBit adj t)) then (
      if(!E.debug) then
	Printf.printf "Combine, add edge (to pushed temp) %i <-> %i\n" t u;
      addEdge t u;
     )
	    ) (HA.find (!adjList) v);
  
  (try
    if((HA.find (!degree) u >= k) && BL.getBit (!node_info).freezeWL u) then (
      BL.resetBit (!node_info).freezeWL u;
      BL.setBit (!node_info).spillWL u;
     )
  with Not_found -> 
    raise (RegallocError
	     (Printf.sprintf "combine: temp %i not found in degree" u))
  )
	
      
let freezeMoves u = 
  MoveList.iter (fun m -> 
    let (x,y) = m in
    let v = 
      if(getAlias y = getAlias u) then
	getAlias x
      else
	getAlias y
    in
    (!move_info).activeMoves <- 
      MoveList.remove m (!move_info).activeMoves;
    (!move_info).frozenMoves <- 
      MoveList.add m (!move_info).frozenMoves;
    if(not (moveRelated v) && HA.find (!degree) v < k) then (
      BL.resetBit (!node_info).freezeWL v;
      BL.setBit (!node_info).simplifyWL v
     )
		) (nodeMoves u)

let freeze () =
  let u = BL.findSet (!node_info).freezeWL in
  if(u < 0) then
    false
  else (
    (* find a temp with min degree in freezeWL *)
    let deg = ref 0 in
    let tmp = ref u in
    BL.iterTrue (fun t -> 
      if(HA.mem !degree t) then
	let d = HA.find !degree t in
	if(d >= !deg) then (
	  tmp := t;
	  deg := d;
	 )
		) (!node_info).freezeWL;
    let u = !tmp in  
    BL.resetBit (!node_info).freezeWL u;
    BL.setBit (!node_info).simplifyWL u;
    freezeMoves u;
    E.showProgress "f";
    true;
   )
      

(* Returns true if the worklistMove was not empty, else false *)
let coalesce () = 
  if(MoveList.is_empty (!move_info).worklistMoves) then
    false
  else (
    let m = MoveList.choose (!move_info).worklistMoves in
    (!move_info).worklistMoves <- 
      MoveList.remove m (!move_info).worklistMoves;
    let (x,y) = m in
    let x = getAlias x in
    let y = getAlias y in
    let (u,v) = 
      if(precolored y) then
	(y,x)
      else
	(x,y)
    in
    if(u=v) then (
      (!move_info).coalescedMoves <- 
	MoveList.add m (!move_info).coalescedMoves;
      addWorkList u;
     ) else (
      if(precolored v || IG.getBit u v adjSet) then (
	(!move_info).constrainedMoves <- 
	  MoveList.add m (!move_info).constrainedMoves;
	addWorkList u;
	addWorkList v;
       ) else (
	if(precolored u && 
	   BL.fold (fun res t -> res && ok_fun t u) true (adjacent v) 
	 || notPrecolored u && 
	   conservative (BL.orBLnew (adjacent u) (adjacent v))) then (
	  (!move_info).coalescedMoves <- 
	    MoveList.add m (!move_info).coalescedMoves;
	  combine u v;
	  addWorkList u
	 )
	else (
	  (!move_info).activeMoves <- MoveList.add m (!move_info).activeMoves;
	 )
       )
     );

    E.showProgress "c";
    if(!E.debug) then
      print_node_info node_info;
    true
   )

let selectSpill () = 
  (* find temp with max. degree *)
  let (_, m) = BL.fold (fun (d, m) t -> 
    let tmp =
      (try
	HA.find (!degree) t
      with Not_found -> 
	raise (RegallocError
		 (Printf.sprintf "selectSpill: temp %i not found in degree" t))
      ) in
    
    if(tmp > d) then
      (tmp, t)
    else
      (d, m)
		       ) (-1,-1) (!node_info).spillWL
  in
  if(m < 0) then
    false
  else (
    BL.resetBit (!node_info).spillWL m;
    BL.setBit (!node_info).simplifyWL m;
    freezeMoves m;
    E.showProgress "p";
    true
   )
    
let assignColors () =
  let spilled = ref false in
  HA.clear (!color);
  for i = 1 to regs do
    HA.add (!color) i i;
  done;

  (* DEBUG *)
  if(!E.debug) then (
    Printf.printf "Assign Colors:\n";
    IG.printIGraph adjSet;
    Printf.printf "Stack:";
    Stack.iter (fun x -> Printf.printf " %i" x) (!node_info).stack;
   );

  while(not (Stack.is_empty (!node_info).stack)) do
    let n = Stack.pop (!node_info).stack in
    let okColors = ref [1;2;3;4;5;6] in
    let list = 
      (try
	HA.find (!adjList) n
      with Not_found -> 
	raise (RegallocError
		 (Printf.sprintf "assignColors: temp %i not found in adjList" n))
      ) in

    List.iter (fun w -> 
      let ali = getAlias w in
      if(BL.getBit (!node_info).colored ali || precolored ali) then
	(try
	  okColors := Helpers.list_minus !okColors [HA.find (!color) ali];
	  (* Printf.printf "[%i(%i):%i]" w ali (HA.find (!color) ali); *)
	with Not_found -> 
	  raise (RegallocError
		   (Printf.sprintf "assignColors: temp %i not found in color (1)" 
		      ali))
	);
	      ) list;
    (* Is this really necessary? FIXME *)
    if(notPrecolored n) then (
      match !okColors with
      | [] -> 
	  if(!E.debug) then
	    Printf.printf "Spill temp %i\n" n;
	  
	  spilled := true;
	  BL.setBit (!node_info).spilled n
      | c :: tail -> 
	  if(!E.debug) then
	    Printf.printf "Choosen color for temp %i is %i\n" n c; 
	  
	  BL.setBit (!node_info).colored n;
	  HA.add (!color) n c
     )
  done;

  (* DEBUG *)
  if(!E.debug) then (
    BL.iterTrue (fun n -> 
      let c = 
	(try
	  HA.find (!color) (getAlias n)
	with Not_found -> 0
	) in
      Printf.printf "Choosen color for (aliased) temp %i is %i\n" n c;     
		) (!node_info).coalesced;
   );
  
  if(not !spilled) then (
    BL.iterTrue (fun n -> 
      let c = 
	(try
	  HA.find (!color) (getAlias n)
	with Not_found -> 
	  raise (RegallocError
		   (Printf.sprintf 
		      "assignColors: temp %i not found in color (for alias %i)" 
		      (getAlias n) n))
	) in
      HA.add (!color) n c;
		) (!node_info).coalesced
   )



(* T.TEMP(tmp) -> T.MEM(T.BINOP(T.MINUS, T.TEMP(F.bp ()), offset)) *)
let rewriteProgram tmp offset (fName, bb) =
  if(!E.debug) then
    Printf.printf "rewriteProgram, repalce temp %i with offset %i\n" tmp offset;
  let rec spill_exp exp = 
    match exp with
    | T.CONST(_) | T.NAME(_) | T.PHI(_) -> exp 
    | T.TEMP(t) -> 
	if(t = tmp) then
	  T.MEM(T.BINOP(T.MINUS, T.TEMP(F.bp ()), 
			T.CONST(Int32.of_int offset,Pcc.Int)))
	else
	  exp
    | T.BINOP(b, e1, e2) -> T.BINOP(b, spill_exp e1, spill_exp e2)
    | T.MEM(e) -> T.MEM(spill_exp e)
    | T.CALL(l,el) -> T.CALL(l, List.map (fun e -> spill_exp e) el)
    | T.ESEQ(s,e) -> T.ESEQ(spill_stmt s, spill_exp e)

  and spill_stmt stmt = 
    match stmt with
    | T.MOVE(e1, e2) -> T.MOVE(spill_exp e1, spill_exp e2)
    | T.EXP(e) -> T.EXP(spill_exp e)
    | T.LABEL(_) | T.JUMP(_) | T.COMMENT(_) -> stmt
    | T.CJUMP(r, e1, e2, l1, l2) -> 
	T.CJUMP(r, spill_exp e1, spill_exp e2, l1, l2)
    | T.SEQ(s1, s2) -> T.SEQ(spill_stmt s1, spill_stmt s2)
    | T.INVARIANT(_) -> stmt
  in
  (fName, List.map (fun li -> (List.map (fun s -> spill_stmt s) li)) bb)


let build live_info = 
  HA.iter (fun lab bb_rec -> 
    let live = 
      (try
	BL.copyBitListNew (HA.find (live_info.Liveness.lab_OUT) lab) 
      with Not_found -> 
	raise (RegallocError
		 (Printf.sprintf "build: label %i not found in live_info" lab))
      ) in
    let _ = List.fold_right (fun i r -> 
      (match i with
      | Assem.MOVE(_, src, dst) -> 
	  (* BL.resetBit live src; *)
	  let m = if(src < dst) then (src, dst) else (dst, src) in
	  if(HA.mem (!moveList) src) then
	    HA.replace (!moveList) src (MoveList.add m 
				       (HA.find (!moveList) src))
	  else
	    HA.add (!moveList) src (MoveList.singleton m);
	  
	  if(HA.mem (!moveList) dst) then
	    HA.replace (!moveList) dst (MoveList.add m 
				       (HA.find (!moveList) dst))
	  else
	    HA.add (!moveList) dst (MoveList.singleton m);
	  
	  (!move_info).worklistMoves <-
	    MoveList.add m (!move_info).worklistMoves;
      | _ -> ()
      );    
      let def = Assem.getDst i in
      let src = Assem.getSrc i in
      List.iter (fun d -> BL.setBit live d) def;
      List.iter (fun d -> 
	BL.iterTrue (fun l -> addEdge l d ) live;
		) def;
      List.iter (fun d -> BL.resetBit live d ) def;
      List.iter (fun u -> BL.setBit live u) src;
      r
			    ) (!bb_rec).Liveness.bb 0 in
    if(not (BL.isEqual live (HA.find (live_info.Liveness.lab_IN) lab))) then
      raise (RegallocError
	       (Printf.sprintf "build: Inset does not match iterative IN"));
    ()
      	  ) live_info.Liveness.lab_bb


(* Handles spills by inserting store and load instructions just before 
 * the uses or defines - as Appel does *)
let rewriteProgram2 renameHash (fName, lili) = 
  (fName, 
   List.map (fun (asmList, labList) -> 
     let res = ref [] in
     let emit i = res := i :: !res in
     let delayed = ref [] in
     let delayed_emit i = delayed := i :: !delayed in
     let emitDelayed () = (* in reverse order - no problem *)
       List.iter (fun i -> emit i) !delayed;
       delayed := []
     in
     List.iter (fun i -> 
       match i with
       | A.LABEL(_) -> emit i
       | A.MOVE(s, src, dst) -> 
	   if(HA.mem renameHash src) then (
	     let soff = HA.find renameHash src in
	     if(HA.mem renameHash dst) then (
	       let doff = HA.find renameHash dst in
	       let tmp = TP.simpTemp (TP.temp2pccType dst) in
	       emit (A.OPER("movl\t " ^ string_of_int (- soff) ^ "('s0), 'd0", 
			    [F.bp ()], [tmp], []));
	       emit (A.OPER("movl\t 's0, " ^ string_of_int (- doff) ^ "('s1)",
			    [tmp; F.bp ()], [], []));
	      ) else
	       emit (A.OPER("movl\t " ^ string_of_int (- soff) ^ "('s0), 'd0", 
			    [F.bp ()], [dst], []));
	    ) else (
	     if(HA.mem renameHash dst) then (
	       let doff = HA.find renameHash dst in
	       emit (A.OPER("movl\t 's0, " ^ string_of_int (- doff) ^ "('s1)",
			    [src; F.bp ()], [], []));
	      ) else
	       emit i
	    )
       | A.OPER(s, src, dst, lab) -> 
	   let rdst = ref dst in
	   let newSrc = List.map (fun s -> 
	     if(HA.mem renameHash s) then (
	       let soff = HA.find renameHash s in
	       let stmp = TP.simpTemp (TP.temp2pccType s) in
	       emit (A.OPER("movl\t " ^ string_of_int (- soff) ^ "('s0), 'd0", 
			    [F.bp ()], [stmp], []));
	       (* Temp used as src and dst - laod & store *)
	       if(List.mem s !rdst) then (
		 rdst := Helpers.list_replace_entry !rdst s stmp;
		 delayed_emit 
		   (A.OPER("movl\t 's0, " ^ string_of_int (- soff) ^ "('s1)",
			   [stmp; F.bp ()], [], []));
		);
	       
	       stmp
	      ) else s
				 ) src in
	   let newDst = List.map (fun d -> 
	     if(not (List.mem d src)) then (
	       if(HA.mem renameHash d) then (
		 let doff = HA.find renameHash d in
		 let dtmp = TP.simpTemp (TP.temp2pccType d) in
		 delayed_emit 
		   (A.OPER("movl\t 's0, " ^ string_of_int (- doff) ^ "('s1)",
			   [dtmp; F.bp ()], [], []));
		 dtmp
		) else d
	      ) else d
				 ) !rdst in
	   emit (A.OPER(s, newSrc, newDst, lab));
	   emitDelayed ();
	       ) asmList;
     (List.rev !res, labList)
	    ) lili
  )


exception FINISHED_REGALLOC

(* Does not call munch anymore - used if temps from munch have been spilled *)
let rec main2 spillOffset spilledList lab_invar munched = 
  let (fName, _) = munched in
  let live_info = Liveness.liveness_fun munched in
  Liveness.insertPCCinvar munched live_info lab_invar;

  adjSet := !(IG.iniIGraph (TP.getTempCounter ()));
  adjList := HA.create 1009;
  degree := HA.create 1009;
  moveList := HA.create 1009;
  alias := HA.create 1009;
  color := HA.create 1009;

  (* initialize degree and adjList such that they contain 0 or [], 
   * if not changed *)
  for i=1 to (TP.getTempCounter () - 1) do
    HA.add (!degree) i 0;
    HA.add (!adjList) i [];
  done;

  (* Precolored Temps have 'infinite' degree *)
  for i=1 to 8 do
    HA.replace (!degree) i 268000000;
  done;

  let iniBL () = BL.newBitList (TP.getTempCounter ()) in
  node_info := {simplifyWL = iniBL ();
		freezeWL = iniBL ();
		spillWL = iniBL ();
		spilled = iniBL ();
		coalesced = iniBL ();
		colored = iniBL ();
		selectStack = iniBL ();
		stack = Stack.create ();
	      };
  move_info := {coalescedMoves = MoveList.empty;
		constrainedMoves = MoveList.empty;
		frozenMoves = MoveList.empty;
		worklistMoves = MoveList.empty;
		activeMoves = MoveList.empty;
	      };

  build live_info;
  makeWorklist ();
  
  (try
    while(true) do
      if(not (simplify ())) then
	if(not (coalesce())) then
	  if(not (freeze ())) then
	    if(not (selectSpill ())) then
	      raise FINISHED_REGALLOC;
    done;
  with FINISHED_REGALLOC -> ()
  );
  
  assignColors ();
  
  let spilled = ref false in
  let renameHash = HA.create 57 in
  BL.iterTrue (fun s -> 
    spilled := true;
    spillOffset := 4 + !spillOffset;
    spilledList := (s, !spillOffset) :: !spilledList;
    HA.add renameHash s !spillOffset;
	      ) (!node_info).spilled;
  
  if(!spilled) then (
    main2 spillOffset spilledList lab_invar 
      (rewriteProgram2 renameHash munched)
   ) else (
    (* find all used colors *)
    (* Assume that %eax, %ecx, %edx, %esp and %ebp are always used *)
    let used_colors = [ref 1; ref 2; ref 3; ref 0; 
		       ref 0; ref 0; ref 7; ref 8 ] in
    (try
      HA.iter (fun t c -> 
	if(notPrecolored t) then
	  List.nth used_colors (c-1) := c
	      ) (!color);
    with Failure("nth") -> 
      raise (RegallocError ("Invalid Register choosen in regalloc"))
    );
    let used_colors = List.fold_left (fun res x -> 
      if(0 = !x) then res else !x :: res) [] used_colors in
    (* END find all used colors *)
    
    TP.saveFunLookUp fName;
    
    (munched, (!color), !spillOffset, !spilledList, used_colors, lab_invar)
   )

let remunch_style_regalloc = ref false 

(* The main loop - calling munch again for spills *)
(* I think that it might be better to translate the code via maximal munch when
 * there are spills. There is however the restriction, that none of the spills
 * was introduced in the translation-phase. This is not very common, as their 
 * live-range is very small, they though interfere with only a few other temps.
 * If Regalloc decides to spill a temp introduced by munch, I switch to the 
 * Appel - style and insert load and store instructions *)
let rec main spillOffset spilledList lab_invar (fName, slili) =
  (* DEBUG *)
  if(!E.debug) then
    Printf.eprintf "Processing function %s\n" fName;

  (* DEBUG *)
  if(!E.debug) then (
    List.iter (fun sli -> 
      List.iter (fun s -> Ir.pstmt s) sli 
	      ) slili;
   );
  

  (* Open the function TENV - don't close, don't want changes if spilled *)
  TP.openFunLookUp fName; 
  
  (* max. Temp that might get spilled, temps with a 
   * higher number are produced in munch and cannot be spilled *)
  let noMunchSpillTemps = TP.getTempCounter () - 1 in
  
  (* BUILD *)
  let munched = Munch.munch_fun (fName, slili) in 
  
  E.showProgress (Printf.sprintf 
		    "\nRegalloc %s, %i (max) Temps, %i before munch\n" 
		    fName (TP.getTempCounter ()) noMunchSpillTemps);
  (* DEBUG *)
  if(!E.debug) then
    Munch.print_fun (fun i s -> string_of_int i) munched; 
  
  let live_info = Liveness.liveness_fun munched in
  Liveness.insertPCCinvar munched live_info lab_invar;
  
  adjSet := !(IG.iniIGraph (TP.getTempCounter ()));
  adjList := HA.create 1009;
  degree := HA.create 1009;
  moveList := HA.create 1009;
  alias := HA.create 1009;
  color := HA.create 1009;
  
  (* initialize degree and adjList such that they contain 0 or [], 
   * if not changed *)
  for i=1 to (TP.getTempCounter () - 1) do
    HA.add (!degree) i 0;
      HA.add (!adjList) i [];
  done;
  
  (* Precolored Temps have 'infinite' degree *)
  for i=1 to 8 do
    HA.replace (!degree) i 268000000;
  done;
  
  let iniBL () = BL.newBitList (TP.getTempCounter ()) in
  node_info := {simplifyWL = iniBL ();
		freezeWL = iniBL ();
		spillWL = iniBL ();
		spilled = iniBL ();
		coalesced = iniBL ();
		colored = iniBL ();
		selectStack = iniBL ();
		stack = Stack.create ();
	      };
  move_info := {coalescedMoves = MoveList.empty;
		constrainedMoves = MoveList.empty;
		frozenMoves = MoveList.empty;
		worklistMoves = MoveList.empty;
		activeMoves = MoveList.empty;
	      };
  
  build live_info;
  makeWorklist ();
  
  (try
    while(true) do
      if(not (simplify ())) then
	if(not (coalesce())) then
	  if(not (freeze ())) then
	    if(not (selectSpill ())) then
	      raise FINISHED_REGALLOC;
    done;
  with FINISHED_REGALLOC -> ()
  );
  
  assignColors ();
  
  let munchTempSpilled = 
    BL.fold (fun res s -> 
      res or (s > noMunchSpillTemps)
	    ) false (!node_info).spilled in
  if(munchTempSpilled) then (
    (* At least one temp from munch was spilled -> cannot call munch again *)
    let renameHash = HA.create 57 in
    BL.iterTrue (fun s -> 
      spillOffset := 4 + !spillOffset;
      spilledList := (s, !spillOffset) :: !spilledList;
      HA.add renameHash s !spillOffset;
		) (!node_info).spilled;
    
    E.print_debug "\n\t\tSwitch to APPEL-style spilling\n";
    Printf.printf "\n\t\tSwitch to APPEL-style spilling\n";
    
    main2 spillOffset spilledList lab_invar 
      (rewriteProgram2 renameHash munched)
      
   ) else (
    (* No temp from munch was spilled -> can call munch again *)
    (* Spill all temps in the IR-tree - only possible if no
     * spills on temps created by munch *)
    let bb = ref slili in
    let spilled = BL.fold (fun res s -> 
      E.print_debug (Printf.sprintf "Spill temp %i\n" s);
      spillOffset := 4 + !spillOffset;
      spilledList := (s, !spillOffset) :: !spilledList;
      let (_, tmp) = rewriteProgram s !spillOffset (fName, !bb) in
      bb := tmp;
      true
			  ) false (!node_info).spilled
    in
    
    if(spilled) then (
      main spillOffset spilledList lab_invar (fName, !bb) 
     )
    else (
      (* find all used colors *)
      (* Assume that %eax, %ecx, %edx, %esp and %ebp are always used *)
      let used_colors = [ref 1; ref 2; ref 3; ref 0; 
			 ref 0; ref 0; ref 7; ref 8 ] in
      (try
	HA.iter (fun t c -> 
	  if(notPrecolored t) then
	    List.nth used_colors (c-1) := c
		) (!color);
      with Failure("nth") -> 
	raise (RegallocError ("Invalid Register choosen in regalloc"))
      );
      let used_colors = List.fold_left (fun res x -> 
	if(0 = !x) then res else !x :: res) [] used_colors in
      (* END find all used colors *)
      
      Temp.saveFunLookUp fName;
      
      (munched, (!color), !spillOffset, !spilledList, used_colors, lab_invar)
     )
   ) (* if(munchTempSpilled) -> else *)
      
let regalloc_program prog = 
  let lab_invar = HA.create 57 in
  List.map (fun (fName, slili) -> 
    let spillOffset = ref 0 in
    let spilledList = ref [] in
    if(!remunch_style_regalloc) then (
      main spillOffset spilledList lab_invar (fName, slili)      
     ) else (
      TP.openFunLookUp fName; 
      main2 spillOffset spilledList lab_invar (Munch.munch_fun (fName, slili))
     )
	   ) prog
