(* munch.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: munch.ml,v 1.8 2003/09/17 18:54:07 rflury Exp $ *)

module TP = Temp
module FR = (Frame.X86_FRAME : Frame.FrameSig)
module A = Assem
module T = Ir

exception MunchError of string

(* If true, %ecx and %edx are stored on the stack before calling a function
 * and restored on return *)
let storeCallerSave = ref false

(* Don't increment %esp after each fun-call immediately, do it at the end 
 * of the BB *)
let inc_esp_bb = ref 0 
(* but let grow %esp at most max_inc_esp_bb in each bb *)
let max_inc_esp_bb = 1000 (* bytes *)

(* Precolored Temps: the registers *)
let eax = 1
let ecx = 2
let edx = 3
let ebx = 4
let esi = 5
let edi = 6
let esp = 7
let ebp = 8

type munchAcc_t = {code: string;        (* the code to be inserted *)
		   par: TP.temp list;   (* the params used *)
		   lab: TP.label list;  (* the labels used *)
		   nn: int;             (* the new counter for the params *)
		   mun: bool;           (* true if munch was called *)
		   memA: bool;          (* true if memory was accessed *)
		   tmp: bool;           (* true if only a tmp is accessed in code *)
		   succ: bool           (* true if successful, else false *)
		 }

(* stores the 'munched' list of instructions *)
let result = ref [] 

(* Add a new instruction to the list *)
let emit i = 
  result := i :: !result

(* Called before jumping to another BB to reset %esp, which is not always
 * done straight after the fun-call (little optimization) *)
let restore_esp () =
  if(!inc_esp_bb != 0) then (
    emit(A.OPER("addl\t $" ^ string_of_int (!inc_esp_bb) ^ ", %esp",
		[], [], []));
    inc_esp_bb := 0
   )

(* Returns the PccType of an expression *)
let rec exp2PccType = function
  | T.ALLOCA e -> Pcc.Bogus
  | T.CONST(_,t) -> t
  | T.NAME(_) -> Pcc.Pointer(Pcc.Int) (* FIXME, should be char* *)
  | T.TEMP(t) -> 
      (try
	TP.temp2pccType t
      with Not_found -> Pcc.Int (* FIXME : is this always true? *)
      )
  | T.BINOP(op, e1, e2) -> (* This is somewhat an approximation *)
      let t1 = exp2PccType e1 in
      let t2 = exp2PccType e2 in
      (match (t1, t2) with
      | (Pcc.Int, Pcc.Int) -> Pcc.Int
      | (Pcc.Int, x) -> x
      | (x, _) -> x
      )
  | T.MEM(e) -> Pcc.Pointer(exp2PccType e)
  | T.CALL(l, _) -> 
      let (rt, _) = Checker.getFunType (TP.label2string l) in 
      Translate.ctype2pccType rt
  | T.ESEQ(_,e) -> exp2PccType e
  | T.PHI(li) -> Pcc.Int (* PCC FIXME - what's the type? *)


let rec munchStmt = function
  | T.MOVE(T.TEMP(t1), T.TEMP(t2)) -> 
      emit(A.MOVE("movl\t 's0, 'd0",
		  t2, t1))
	
  | T.MOVE(T.TEMP(t1), e2) -> 
      let src = munchAcc "s" 0 true false false (T.MEM(e2)) in
      if(src.succ) then (
	emit(A.OPER("leal\t " ^ src.code ^ ", 'd0",
		    src.par, [t1], src.lab))
       ) else (
	let src = munchAcc "s" 0 true true false e2 in
	if(src.tmp) then
	  emit(A.MOVE("movl\t " ^ src.code ^ ", 'd0",
		      List.hd src.par, t1))
	else
	  emit(A.OPER("movl\t " ^ src.code ^ ", 'd0",
		      src.par, [t1], src.lab));
       )

  | T.MOVE(e1, e2) -> 
      let dst = munchAcc "d" 0 true true true e1 in
      let src = munchAcc "s" 0 (not dst.memA) true false e2 in
      if(src.tmp && dst.tmp) then
	emit(A.MOVE("movl\t " ^ src.code ^ ", " ^ dst.code,
		    List.hd src.par, List.hd dst.par))
      else
	emit(A.OPER("movl\t " ^ src.code ^ ", " ^ dst.code,
		    src.par @ dst.par, dst.par, src.lab @ dst.lab))
  | T.EXP(e) -> 
      let _ = munchExp e in () (* not always dead code (side-effects) *)
	
  | T.LABEL(l) -> 
      emit(A.LABEL("'l0:",
		   l))

  | T.JUMP(l) -> 
      restore_esp ();
      emit(A.OPER("jmp\t 'l0",
		  [], [], [l]))
	  
  | T.CJUMP(op, e1, e2, t, f) -> (* fall through on false *)
      let op2asm = function
	| T.EQ -> "je"
	| T.NE -> "jne"
	| T.LT -> "jl"
	| T.LE -> "jle"
	| T.GT -> "jg"
	| T.GE -> "jge"
      in
      let negOp = function
	| T.EQ -> T.EQ
	| T.NE -> T.NE
	| T.LT -> T.GT
	| T.LE -> T.GE
	| T.GT -> T.LT
	| T.GE -> T.LE
      in
      (* e1 - e2 *)
      (* only first operand is allowed to be an immediate value *)
      let co1 = munchAcc "s" 0 true true false e1 in
      let co2 = munchAcc "s" co1.nn (not co1.memA) true false e2 in
      if(not (co1.tmp || co1.memA)) then ( (* snd operand is immedaite *)
	if(not (co2.tmp || co2.memA)) then ( (* fst operand is also imm. *)
	  let co1 = munchAcc "s" 0 true true true e1 in
	  restore_esp();
	  emit(A.OPER("cmpl\t " ^ co2.code ^ ", " ^ co1.code, 
		      co1.par @ co2.par, [], co1.lab @ co2.lab));
	  emit(A.OPER(op2asm op ^ "\t 'l0",
		      [], [], [t;f]))
	 ) else (
	  restore_esp();
	  emit(A.OPER("cmpl\t " ^ co1.code ^ ", " ^ co2.code, 
		      co1.par @ co2.par, [], co1.lab @ co2.lab));
	  emit(A.OPER(op2asm (negOp op) ^ "\t 'l0",
		      [], [], [t;f]))
	 )
       ) else (
	restore_esp();
	emit(A.OPER("cmpl\t " ^ co2.code ^ ", " ^ co1.code, 
		    co1.par @ co2.par, [], co1.lab @ co2.lab));
	emit(A.OPER(op2asm op ^ "\t 'l0",
		    [], [], [t;f]))
       )
  | T.SEQ(s1, s2) -> 
      munchStmt s1;
      munchStmt s2

  | T.COMMENT(s,li) -> 
      emit(A.OPER("/*" ^ s ^ "*/", 
		  li, [], []))

  | T.INVARIANT(li) -> 
      (* PCC FIXME 
       * (will probably be a new label with the things hooked up) *)
      emit(A.OPER("/* INVARIANT - PCC FIXME - tranlation " ^
		  "from IR-invariant to asm not implemented */",
		  [], [], []))
	
(* Returns the string for a binary operator *)
and bop2asm = function
  | T.PLUS -> "addl"
  | T.MINUS -> "subl"
  | T.DIV -> "idivl"
  | T.MUL -> "imul"
  | T.MOD -> "idivl"
  | T.AND -> "andl"
  | T.OR -> "orl"
  | T.XOR -> "xorl"
  | T.NOT -> "notl"
  | T.SHIFTL -> "sall"
  | T.SHIFTR -> "sarl"

and munchExp = function

  | T.CONST(i, pccT) -> 
      let tmp = TP.simpTemp pccT in
      emit(A.OPER("movl\t $" ^ Int32.to_string i ^ ", 'd0",
		  [], [tmp], []));
      tmp
  | T.NAME(l) -> 
      (* PCC FIXME , a pointer to a fat-pointer *)
      let tmp = TP.simpTemp (Pcc.Pointer(Pcc.Record(
					 [Pcc.Int;Pcc.Int;Pcc.Int]))) in
      emit(A.OPER("movl\t $'l0, 'd0",
		  [], [tmp], [l]));
      tmp
  | T.TEMP(t) -> t

  | T.BINOP((T.DIV|T.MOD) as op, e1, e2) -> 
      let pccT = exp2PccType e1 in
      let tmp = TP.simpTemp pccT in 
      munchStmt(T.MOVE(T.TEMP(tmp), e1)); 

      (* get return register *)
      let res = match op with
      | T.DIV -> eax
      | T.MOD -> edx
      | _ -> raise (MunchError "Div/Mod got other op")
      in

      let acc = munchAcc "s" 0 true true true e2 in      

      munchStmt(T.MOVE(T.TEMP(eax), T.TEMP(tmp))); 
      emit(A.OPER("cltd",
		  [eax], [eax; edx], []));
      emit(A.OPER("idivl\t " ^ acc.code,
		  acc.par @ [eax; edx], [res], acc.lab));
      let ret = TP.simpTemp pccT in
      emit(A.MOVE("movl\t 's0, 'd0",
		  res, ret));
      ret
  | T.BINOP((T.PLUS|T.MINUS|T.MUL|T.AND|T.OR|T.XOR) as op, e1, e2) -> 
      let pccT = exp2PccType e1 in      
      let tmp = TP.simpTemp pccT in 

      (* try to use leal *)
      let src = munchAcc "s" 0 true false false (T.MEM(T.BINOP(op, e1, e2))) in
      if(src.succ) then (
	emit(A.OPER("leal\t " ^ src.code ^ ", 'd0",
		    src.par, [tmp], src.lab))
       ) else (
	munchStmt(T.MOVE(T.TEMP(tmp), e1));
	(* try again, now with e1 == temp *)
	let src = munchAcc "s" 0 true false false 
	    (T.MEM(T.BINOP(op, T.TEMP(tmp), e2))) in
	if(src.succ) then (
	  emit(A.OPER("leal\t " ^ src.code ^ ", 'd0",
		      src.par, [tmp], src.lab))
	 ) else (
	  (* did not work with leal *)
	  let src = munchAcc "s" 0 true true false e2 in
	  
	  emit(A.OPER(bop2asm op ^ "\t " ^ src.code ^ ", 'd0", 
		      src.par @ [tmp], [tmp], src.lab));
	 )
       );
      tmp
  | T.ALLOCA e ->
      let tmp = TP.simpTemp Pcc.Bogus in
      munchStmt (T.MOVE(T.TEMP tmp, e));
      emit(A.OPER("subl\t 's0, %esp",
		  [tmp], [], []));
      emit(A.OPER("movl\t %esp, 'd0",
		  [], [tmp], []));
      tmp
  | T.BINOP(T.NOT, e1, e2) -> 
      let pccT = exp2PccType e1 in      
      let tmp = TP.simpTemp pccT in 
      munchStmt(T.MOVE(T.TEMP(tmp), e1));
      emit(A.OPER("notl\t 's0",
		  [tmp], [tmp], []));
      tmp
  | T.BINOP((T.SHIFTL|T.SHIFTR) as op, e1, e2) -> 
      let pccT = exp2PccType e1 in
      let tmp = TP.simpTemp pccT in 
      munchStmt(T.MOVE(T.TEMP(tmp), e1));

      let src =
	match e2 with
	| T.CONST(c,_) -> 
	    (* Only take LSB 8 bits *)
	    let cut = Int32.logand c (Int32.of_int 255) in 
	      {code = "$" ^ Printf.sprintf "%ld" cut;
	       par = []; 
	       lab = []; 
	       nn = 0; 
	       mun = false; 
	       memA = false;
	       tmp = false;
	       succ = true }
	| _ -> 
	    munchAcc "s" 0 true true false e2 in (* mem-Acc ok, move to %ecx *)

      (* If src contains a temp, it must be in ecx *)
      if(src.tmp) then (
	emit(A.MOVE("movl\t " ^ src.code ^ ", 'd0", 
		    List.hd src.par, ecx));
	emit(A.OPER(bop2asm op ^ "\t 's0, 'd0",
		    [ecx; tmp], [tmp], []))
       ) else if(src.memA) then (
	 emit(A.OPER("movl\t " ^ src.code ^ ", 'd0",
		     src.par, [ecx], src.lab));
	 emit(A.OPER(bop2asm op ^ "\t 's0, 'd0",
		     [ecx; tmp], [tmp], []))
	) else
	 emit(A.OPER(bop2asm op ^ "\t " ^ src.code ^ ", 'd0",
		     src.par @ [tmp], [tmp], src.lab));
      
      tmp
  | T.MEM(e) -> 
      let pccT = exp2PccType e in
      let tmp = TP.simpTemp pccT in 
      let src = munchAcc "s" 0 true false false (T.MEM(e)) in
      if(src.succ) then (
	emit(A.OPER("movl\t " ^ src.code ^ ", 'd0",
		    src.par, [tmp], src.lab))
       ) else (
	let src = munchAcc "s" 0 true true false e in
	if(src.tmp) then (
	  emit(A.MOVE("movl\t " ^ src.code ^ ", 'd0",
		      List.hd src.par, tmp))
	) else (
	  emit(A.OPER("movl\t " ^ src.code ^ ", 'd0",
		      src.par, [tmp], src.lab))
	 );
	emit(A.OPER("movl\t ('s0), 'd0",
		    [tmp], [tmp], []));
       );
      tmp
  | T.CALL(l, el) -> 
      if(!storeCallerSave) then (
	(* This version saves the caller-save regs on the stack before calling *)
	let len = List.length el in
	let ws = FR.wordSize_i in
	emit(A.OPER("subl\t $" ^ string_of_int (ws * (len+2)) ^ ", %esp",
		    [], [], []));
	let _ = List.fold_left (fun c exp -> 
	  (match c with
	  | 0 -> 
	      let src = munchAcc "s" 0 false true false exp in
	      emit(A.OPER("movl\t " ^ src.code ^ ", ('d0)",
			  src.par, [esp], src.lab))
	  | _ -> 
	      let src = munchAcc "s" 0 false true false exp in
	      emit(A.OPER("movl\t " ^ src.code ^ ", " ^ 
			  string_of_int (ws * c) ^ "('d0)",
			  src.par, [esp], src.lab))
	  );
	  c + 1) 0 el in
	(* store %ecx, %edx *)
	emit(A.OPER("movl\t %ecx, " ^ Printf.sprintf "%d" (ws * (len + 1)) ^ 
		    "(%esp)",
		    [], [], []));
	emit(A.OPER("movl\t %edx, " ^ Printf.sprintf "%d" (ws * len) ^ "(%esp)",
		    [], [], []));
	(* call the fun *)
	emit(A.OPER("call\t 'l0",
		    [], [eax], [l])); (* ecx, edx not in dst: saved on stack *)
	
	if(len > 0) then
	  emit(A.OPER("addl\t $" ^ string_of_int (ws * len) ^ ", %esp",
		      [], [], []));
	(* restore %exc, %edx *)
	emit(A.OPER("popl\t %edx",
		    [], [], []));
	emit(A.OPER("popl\t %ecx",
		    [], [], []));
	eax (* The return-value is stored in %eax *)

       ) else (
	(* This version does not store the caller-save regs -> 
	 * might give more spills *)
	let len = List.length el in
	let ws = FR.wordSize_i in
	let revli = List.rev el in
	
	List.iter (fun exp -> 
	  let src = munchAcc "s" 0 true true false exp in
	  emit(A.OPER("pushl\t " ^ src.code,
		      src.par, [], src.lab))
		  ) revli;
	(* call the fun *)
	emit(A.OPER("call\t 'l0",
		    [], [eax;ecx;edx], [l])); 
	if(len > 0) then (
	  inc_esp_bb := !inc_esp_bb + ws * len;
	  if(!inc_esp_bb >= max_inc_esp_bb) then (
	    emit(A.OPER("addl\t $" ^ string_of_int (!inc_esp_bb) ^ ", %esp",
			[], [], []));
	    inc_esp_bb := 0;
	   );
	 );

	eax (* The return-value is stored in %eax *)
       )

  | T.ESEQ(_) -> raise (MunchError "Found illegal ESEQ in munch")

  | T.PHI(el) -> raise (MunchError "Not yet implemented : PHI nodes in munch")

(* Tries to evaluate an expression as good as possible
 * tok is "s" if this exp is a source, "d" if its a destination
 * n is the number with which the temp-tokens start
 * mem is a bool and set to true when the exp might access memory
 * munch is set to true if calls to munch are ok, else to false
 * noCnst is set to true if no immediate constants are allowed
 * exp is the expression to translate
 * Returns a record of type munchAcc_t
 * where code      the string to be inserted, 
 *       par       a list of the temps used
 *       lab       a list of the labels used
 *       nn        the next number to be used for the temp-tokens
 *       mun       set to true if munchExp is called
 *       memA      set to true if the expression accesses mem
 *       succ      set to true if the call was successful 
 *                 (only set if munch is false)
 *)

and munchAcc tok n mem munch noCnst exp = 
  let token tok n = 
    "'" ^ tok ^ string_of_int n
  in
  match exp with
  | T.CONST(c,t) -> 
      if(noCnst) then ( (* destination, must move const in temp *)
	let tmp = TP.simpTemp Pcc.Int in 
	emit(A.OPER("movl\t $" ^ Printf.sprintf "%ld" c ^ ", 'd0", 
		    [], [tmp], []));
	{code = token tok n;
	 par = [tmp]; 
	 lab = []; 
	 nn = n+1; 
	 mun = false; 
	 memA = false;
	 tmp = true;
	 succ = true }
       ) else 
	{code = "$" ^ Printf.sprintf "%ld" c;
	 par = []; 
	 lab = []; 
	 nn = n; 
	 mun = false; 
	 memA = false;
	 tmp = false;
	 succ = true }
	  
  | T.NAME(l) -> 
      (* destination, must move const in temp *) 
      let tmp = TP.simpTemp (Pcc.Pointer(Pcc.Record(
					 [Pcc.Int;Pcc.Int;Pcc.Int]))) in 
      emit(A.OPER("movl\t $'l0, 'd0", 
		  [], [tmp], [l]));
      {code = token tok n;
       par = [tmp]; 
       lab = []; 
       nn = n+1; 
       mun = false; 
       memA = false;
       tmp = true;
       succ = true }
	  
  | T.TEMP(t) -> {code = token tok n;
		  par = [t];
		  lab = [];
		  nn = n+1;
		  mun = false;
		  memA = false;
		  tmp = true;
		  succ = true }
  | _ -> 
      if(mem) then (
	(* test for valid scaling factors *)
	let isScale c = 
	  match Int32.to_int c with
	  | 1 | 2 | 4 | 8 -> true
	  | _ -> false 
	in
	let isScalePlus1 c = 
	  match Int32.to_int c with
	  | 3 | 5 | 9 -> true
	  | _ -> false 
	in

	match exp with
	(* MEM(R) *)
	| T.MEM(T.TEMP(t)) -> 
	    {code = "(" ^ token tok n ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R + Imm) *)
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t), T.CONST(c,_))) 
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_), T.TEMP(t))) -> 
	    {code = Printf.sprintf "%ld" c ^ "(" ^ token tok n ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R - Imm) *)
	| T.MEM(T.BINOP(T.MINUS, T.TEMP(t), T.CONST(c,_))) -> 
	    {code = Printf.sprintf "%ld" (Int32.neg c) ^ "(" ^ token tok n ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R1 + R2) *)
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t1), T.TEMP(t2))) -> 
	    {code = "(" ^ token tok n ^ ", " ^ token tok (n+1) ^ ")";
	     par = [t1; t2];
	     lab = [];
	     nn = n+2;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R1 + R2 + Imm) *)
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_), 
			T.BINOP(T.PLUS, T.TEMP(t1), T.TEMP(t2))))
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t1), 
			T.BINOP(T.PLUS, T.CONST(c,_), T.TEMP(t2))))
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t1), 
			T.BINOP(T.PLUS, T.TEMP(t2), T.CONST(c,_))))
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.PLUS, T.TEMP(t1), T.TEMP(t2)), 
			T.CONST(c,_)))
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.PLUS, T.CONST(c,_), T.TEMP(t1)), 
			T.TEMP(t2)))
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.PLUS, T.TEMP(t1), T.CONST(c,_)), 
			T.TEMP(t2))) -> 
	    {code = Printf.sprintf "%ld" c ^ "(" ^ 
	     token tok n ^ ", " ^ token tok (n+1) ^ ")";
	     par = [t1; t2];
	     lab = [];
	     nn = n + 2;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }
			    
	(* MEM(R * s)  where s = 1 | 2 | 4 | 8 *)
	| T.MEM(T.BINOP(T.MUL, T.TEMP(t), T.CONST(c,_))) 
	| T.MEM(T.BINOP(T.MUL, T.CONST(c,_), T.TEMP(t))) when (isScale c) -> 
	    {code = "(, " ^ token tok n ^ ", " ^ Printf.sprintf "%ld" c ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R * s) where s = 3 | 5 | 9 *)
	| T.MEM(T.BINOP(T.MUL, T.TEMP(t), T.CONST(c,_))) 
	| T.MEM(T.BINOP(T.MUL, T.CONST(c,_), T.TEMP(t))) when (isScalePlus1 c) -> 
	    {code = "(" ^ token tok n ^ ", " ^ token tok n ^ ", " 
	     ^ Printf.sprintf "%ld" (Int32.sub c Int32.one) ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R*s + Imm) where s = 3 | 5 | 9 *)
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_),
			T.BINOP(T.MUL, T.CONST(s,_), 
				T.TEMP(t))))
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_),
			T.BINOP(T.MUL, T.TEMP(t), 
				T.CONST(s,_)))) 
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t)), 
			T.CONST(c,_))) 
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.MUL, T.TEMP(t), T.CONST(s,_)), 
			T.CONST(c,_))) when (isScalePlus1 s) -> 
	    {code = Printf.sprintf "%ld" c ^ "(" ^ token tok n ^ ", " ^ 
	     token tok n ^ ", " ^ Printf.sprintf "%ld" (Int32.sub s Int32.one) ^ 
	     ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R*s - Imm) where s = 3 | 5 | 9 *)
	| T.MEM(T.BINOP(T.MINUS, T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t)), 
			T.CONST(c,_))) 
	| T.MEM(T.BINOP(T.MINUS, T.BINOP(T.MUL, T.TEMP(t), T.CONST(s,_)), 
			T.CONST(c,_))) when (isScalePlus1 s) -> 
	    {code = Printf.sprintf "%ld" (Int32.neg c) ^ "(" ^ token tok n ^
	     ", " ^ token tok n ^ ", " ^ Printf.sprintf "%ld" 
					   (Int32.sub s Int32.one) ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }


	(* MEM(R*s + Imm) where s = 1 | 2 | 4 | 8 *)
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_),
			T.BINOP(T.MUL, T.CONST(s,_), 
				T.TEMP(t))))
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_),
			T.BINOP(T.MUL, T.TEMP(t), 
				T.CONST(s,_)))) 
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t)), 
			T.CONST(c,_))) 
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.MUL, T.TEMP(t), T.CONST(s,_)), 
			T.CONST(c,_))) when (isScale s) -> 
	    {code = Printf.sprintf "%ld" c ^ "(, " ^ 
	     token tok n ^ ", " ^ Printf.sprintf "%ld" s ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }

	(* MEM(R*s - Imm) where s = 1 | 2 | 4 | 8 *)
	| T.MEM(T.BINOP(T.MINUS, T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t)), 
			T.CONST(c,_))) 
	| T.MEM(T.BINOP(T.MINUS, T.BINOP(T.MUL, T.TEMP(t), T.CONST(s,_)), 
			T.CONST(c,_))) when (isScale s) -> 
	    {code = Printf.sprintf "%ld" (Int32.neg c) ^ "(, " ^ 
	     token tok n ^ ", " ^ Printf.sprintf "%ld" s ^ ")";
	     par = [t];
	     lab = [];
	     nn = n+1;
	     mun = false;
	     memA = true;
	     tmp = false;
	     succ = true }


	(* MEM(R1 + R2 * s) *)
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t1), 
			T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))))
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t1), 
			T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))))
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_)),
			T.TEMP(t1)))
	| T.MEM(T.BINOP(T.PLUS, T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2)),
			T.TEMP(t1))) when (isScale s) -> 
	   {code = "(" ^ token tok n ^ ", " ^ token tok (n+1) ^ 
	    ", " ^ Printf.sprintf "%ld" s ^ ")";
	    par = [t1; t2];
	    lab = [];
	    nn = n + 2;
	    mun = false;
	    memA = true;
	    tmp = false;
	    succ = true }

        (* MEM(R1 + R2 * s + Imm) *)
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t1), 
			T.BINOP(T.PLUS,
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))),
				T.CONST(c,_))))
	| T.MEM(T.BINOP(T.PLUS, T.TEMP(t1), 
			T.BINOP(T.PLUS,
				T.CONST(c,_),
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))))))
	| T.MEM(T.BINOP(T.PLUS, 
			T.BINOP(T.PLUS,
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				 T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))),
				T.CONST(c,_)), 
			T.TEMP(t1)))
	| T.MEM(T.BINOP(T.PLUS, 
			T.BINOP(T.PLUS,
				T.CONST(c,_),
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2)))), 
			T.TEMP(t1))) 
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_),
			T.BINOP(T.PLUS,
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))),
				T.TEMP(t1))))
	| T.MEM(T.BINOP(T.PLUS, T.CONST(c,_),
			T.BINOP(T.PLUS,
				T.TEMP(t1), 
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))))))
	| T.MEM(T.BINOP(T.PLUS, 
			T.BINOP(T.PLUS,
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))),
				T.TEMP(t1)), 
			T.CONST(c,_)))
	| T.MEM(T.BINOP(T.PLUS, 
			(T.BINOP(T.PLUS,T.TEMP(t1),T.CONST(c,_))|
			T.BINOP(T.PLUS,	T.CONST(c,_),T.TEMP(t1))),
			(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
			T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2))))) 
	| T.MEM(T.BINOP(T.PLUS, 
			T.BINOP(T.PLUS,
				T.TEMP(t1),
				(T.BINOP(T.MUL, T.TEMP(t2), T.CONST(s,_))|
				T.BINOP(T.MUL, T.CONST(s,_), T.TEMP(t2)))), 
			T.CONST(c,_))) when (isScale s) -> 

	      {code = Printf.sprintf "%ld" c ^ "(" ^ 
	       token tok n ^ ", " ^ token tok (n+1) ^ ", " ^ 
	       Printf.sprintf "%ld" s ^ ")";	       
	       par = [t1; t2];
	       lab = [];
	       nn = n+2;
	       mun = false;
	       memA = true;
	       tmp = false;
	       succ = true }

	(* Special case often introduced by translate *)
	| T.MEM(T.BINOP(T.PLUS, T.MEM(e1), e2)) ->
	    if(munch) then (
	      let tmp = TP.simpTemp (exp2PccType (T.MEM(e1))) in 
	      (* evaluate the memory access *)
	      munchStmt(T.MOVE(T.TEMP(tmp), T.MEM(e1)));
	      munchAcc tok n mem munch noCnst (T.MEM(T.BINOP(T.PLUS, 
							     T.TEMP(tmp),
							     e2)))
		
	     ) else
	      {code = "";
	       par = [];
	       lab = [];
	       nn = n;
	       mun = false;
	       memA = false;
	       tmp = false;
	       succ = false }

	    
	| T.MEM(x) -> (* keep the dereference here *)
	    if(munch) then 
	      let tmp = TP.simpTemp (exp2PccType x) in 
	      (* call munchAcc *)
	      let src = munchAcc "s" 0 true true false x in
	      if(src.tmp) then
		emit(A.MOVE("movl\t " ^ src.code ^ ", 'd0",
			    List.hd src.par, tmp))
	      else
		emit(A.OPER("movl\t " ^ src.code ^ ", 'd0",
			    src.par, [tmp], src.lab));
	      {code = "(" ^ token tok n ^ ")";
	       par = [tmp];
	       lab = [];
	       nn = n+1;
	       mun = true;
	       memA = true;
	       tmp = false;
	       succ = true }
	    else
	      {code = "";
	       par = [];
	       lab = [];
	       nn = n;
	       mun = false;
	       memA = false;
	       tmp = false;
	       succ = false }
	| _ -> (* Must not be a destination! *)
	    if(tok = "d") then raise (MunchError "Invalid Destination in munchAcc");
	    if(munch) then 
(*	      munchAccExpand *)

	      {code = token tok n;
	       par = [munchExp exp];
	       lab = [];
	       nn = n+1;
	       mun = true;
	       memA = false;
	       tmp = true;
	       succ = true }

	    else
	      {code = "";
	       par = [];
	       lab = [];
	       nn = n;
	       mun = false;
	       memA = false;
	       tmp = false;
	       succ = false }

       ) else (* not allowed to access memory *)
	if(munch) then
	  {code = token tok n;
	   par = [munchExp exp];
	   lab = [];
	   nn = n+1;
	   mun = true;
	   memA = false;
	   tmp = true;
	   succ = true }
	    
	else
	  {code = "";
	   par = [];
	   lab = [];
	   nn = n;
	   mun = false;
	   memA = false;
	   tmp = false;
	   succ = false }




(* Does not open the fun-environment *)
let munch_fun (fName, lili) = 
  (* TP.clearForced (); *)
  (fName, 
   List.map (fun li -> 
     result := [];
     List.iter (fun s -> munchStmt s) li;
     (* Iter over stmts in Basic Block *)
     inc_esp_bb := 0;
     (List.rev !result, A.getLabels (List.hd !result))
	    ) lili) 


let munch_program prog = 
  List.map (fun (fName, lili) -> 
    TP.openFunLookUp fName;
    let res = munch_fun (fName, lili) in
    TP.saveFunLookUp fName;
    res
	   ) prog


let print_fun map (fName, lili) = 
  let invar2string label = "" in
  List.iter (fun (li,lab) -> 
    List.iter (fun i -> print_string (A.format map invar2string i)) li;
    List.iter (fun l -> Printf.printf "Label %i, " l) lab;
    print_newline();
	    ) lili
  
let print map prog = 
  List.iter (fun (fName, lili) -> 
    TP.openFunLookUp fName; 
    print_fun map (fName, lili)
	    ) prog
  


