(* main.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: main.ml,v 1.14 2003/09/26 20:44:59 rflury Exp $ *)

let pretty_print_AST = ref false in
let dump_AST = ref false in
let dump_c_AST = ref false in
let evaluate = ref false in
let print_alloca = ref false in
let print_ir = ref false in
let print_lin_ir = ref false in
let print_triple_ir = ref false in
let print_feld_ir = ref false in
let print_inst_ir = ref false in
let print_cse_ir = ref false in
let print_bb_ir = ref false in
let print_traced_ir = ref false in
let print_traced_ir_ssa = ref false in
let print_munch = ref false in
let print_liveness = ref false in
let print_igraph = ref false in
let print_ssa = ref false in
let no_ssa = ref false in
let use_my_regalloc = ref false in

let only_type_check = ref false in
let noCPP = ref false in

let infile = ref "" in
let setinfile = ref false in
let arg_name s = infile := s; setinfile := true in

let outfile = ref "" in
let setoutfile = ref false in
let readOutFile s = outfile := s; setoutfile := true in

let executable = ref false in
let outfile_exe = ref "" in
let setoutfile_exe = ref false in
let exeName n = outfile_exe := n; setoutfile_exe := true; executable := true in

let cppFlagList = ref [] in
let cppFlag f = cppFlagList := f :: !cppFlagList in

let gccFlagList = ref [] in
let gccFlag f = gccFlagList := f :: !gccFlagList in

let set_optimizationLevel level = 
  Errormsg.optimizationLevel := level in

let main argc argv =
  (try
    let optimizationLevels = "
\t0 : no optimizations
\t1 : Dead code elimination (requires SSA)
\t2 : -O 1 & Constant propagation & folding (requires SSA)
\t3 : -O 2 & Common subexpression (requires SSA) 
\t4 : -O 3 & Bound-check elimination (requires SSA) (default)"
    in
    
    let speclist = 
      [("-ppa",Arg.Set(pretty_print_AST),"     Pretty print abstract syntax tree");
       ("-da",Arg.Set(dump_AST),"      Dump abstract syntax tree");
       ("-dca",Arg.Set(dump_c_AST),"     Dump checked abstract syntax tree");
       ("-t",Arg.Set(only_type_check),"       Only type-check program");

       ("-dalloca",Arg.Set(print_alloca),"      Dump alloca'd abstract syntax");
       ("-IR",Arg.Set(print_ir),"      Print the IR-tree (without SEQ)");
       ("-IRlin",Arg.Set(print_lin_ir),"   Print the linearized IR-tree");
       ("-IRtrip", Arg.Set print_triple_ir, "  Print the tripleized IR-tree");
       ("-IRbb", Arg.Set(print_bb_ir), "    Print the IR-tree with basic blocks");
       ("-IRfeld", Arg.Set print_feld_ir, "  Print the constant-feld IR-tree");
       ("-IRinst", Arg.Set print_inst_ir, "  Print the instrumented IR-tree");
       ("-IRtrace", Arg.Set(print_traced_ir)," Print the traced IR-tree");
       ("-IRcse", Arg.Set print_cse_ir, " Print the CSE's IR-tree");
       (* ("-IRtrssa", Arg.Set(print_traced_ir_ssa)," Print the traced IR-tree after SSA"); *)

       ("-munch", Arg.Set(print_munch),"   Print the munched ASM");
       ("-live", Arg.Set(print_liveness),"    Print liveness analysis");
       ("-igraph", Arg.Set(print_igraph),"  Print interference graphs");

(*       ("-noSSA", Arg.Set(no_ssa),"   Disables SSA-convertion");
       ("-showSSA", Arg.Set(print_ssa)," Print info about SSA-convertion");
       ("-myRAlloc", Arg.Set(use_my_regalloc),"Invokes my own register allocation (which is slower)");
       ("-SpillMUN", Arg.Set(Regalloc.remunch_style_regalloc), 
	"Try to call Munch again if some temps have been spilled");
       ("-O", Arg.Int(set_optimizationLevel), ("       Set the optimization level" ^ optimizationLevels));
*)
       ("-strClrSv", Arg.Set(Munch.storeCallerSave), "Store %ecx and %edx before calling a fun");
       (* ("-UF", Arg.Set(Translate.useForeignFuns), "      Use foreign functions for complexer tasks"); *)

       ("-w",Arg.Set(Errormsg.warningFlag),"       Enable warnings");
       ("-progress", Arg.Set(Errormsg.showProgressFlag), "Shows the progress in the compilation process");
       ("-summary", Arg.Set(Summary.showSummaryFlag), " Shows a short summary of the compilation process");
       ("-e",Arg.Set(evaluate),"       Evaluate program in soft");
       ("-o",Arg.String(readOutFile),"       Write the assembly-code in the given file");
       ("-EXE",Arg.Set(executable),"     Create executable file");
       ("-EXEo",Arg.String(exeName),"    Create executable file with given name");
       ("-cpp", Arg.String(cppFlag),"     Pass the following flag to cpp");
       ("-gcc", Arg.String(gccFlag),"     Pass the following flag to gcc");
       ("-windows", Arg.Set(Helpers.isWindows), " Compile for Windows");
       ("-nocpp", Arg.Set(noCPP), "   Disables preprocessing the source with cpp");
       ("-debug",Arg.Set(Errormsg.debug),"   Enable Debug printing")] in
    let descript = 
      Sys.executable_name ^ " [options]* <file> [options]*" in
    
    (* parse the arguments *)
    let args = Array.fold_left (fun res x -> res ^ " " ^ x) "" argv in 
    Arg.parse speclist arg_name args;

    if(not !setinfile) then (
      (* no file given *)
      Arg.usage speclist descript;
      exit(1)
     );

    (* Check *.s output-file *)
    if(not !setoutfile) then (
      (try
	let pos = Str.search_backward 
	    (Str.regexp_string ".") 
	    !infile
	    (String.length !infile)
	in
	outfile := (String.sub !infile 0 pos) ^ ".s";
      with Not_found -> 
	outfile := !infile ^ ".s";
      )
     );

    (* Check output-file *)
    if(not !setoutfile_exe) then (
      (try
	let pos = Str.search_backward 
	    (Str.regexp_string ".") 
	    !infile
	    (String.length !infile)
	in
	outfile_exe := String.sub !infile 0 pos ^ ".exe";
      with Not_found -> 
	outfile_exe := !outfile ^ ".exe";
      )
     );

    (* Preprocess the code with CPP *)
    let src_file = 
      if(!noCPP) then 
	!infile
      else (
        (* (* annoying -wjl *)
	if(Sys.file_exists(!infile ^ ".cpp")) then (
	  Printf.eprintf ("File '%s.cpp' already exists. Overwrite? (y|n) ") 
	    !infile;
	  Pervasives.flush Pervasives.stderr;
	  let answer = String.lowercase (Pervasives.read_line ()) in
	  if(String.contains answer 'n') then 
	    exit(1)
	 );
         *)
	(* Call CPP *)
	let commandLine = Command_line.cpp_commandLine infile cppFlagList in
	Errormsg.print_debug 
	  (Printf.sprintf "Command line to cpp is:  %s \n" commandLine);
	let cppOk = Sys.command commandLine in ();
	if(cppOk != 0) then (
	  Printf.eprintf "cpp failed\n";
	  exit(1);
	 );
	(!infile ^ ".cpp")
       ) in

    (* open the preprocessed src-file *)
    let ic = open_in src_file in

    (* fun called before exiting compiler *)
    let close_files () = 
      close_in ic;
      if(not !noCPP) then 
	let _ = Sys.command ("rm " ^ src_file) in ();
    in

    (try 
      let _ = Errormsg.startFile !infile in
      let lexbuf = Lexing.from_channel ic in
      let absyntree= 
	(try 
	  Grammar.program Lexer.token lexbuf
	with
	  Parsing.Parse_error -> 
	    Errormsg.error (Lexing.lexeme_start lexbuf,
			    Lexing.lexeme_end lexbuf)
	      "syntax error"; close_files (); exit(-1)
	) in
      if(!pretty_print_AST) then Absyn.pprint_ast absyntree;
      if(!dump_AST) then Absyn.dump_ast absyntree;

      let checked = Checker.check absyntree in 
      if(!dump_c_AST) then Absyn.dump_c_ast checked;

      if(!only_type_check) then (close_files (); exit(0) );

      if(!evaluate) then Evaluator.eval_program checked; 

      (* stack allocation! -wjl *)
      (*
      let alloca_optimized = Alloca.optimize_program checked in
      if(!print_alloca) then Absyn.dump_c_ast alloca_optimized;
      *)
      let alloca_optimized = checked in

      let ir = Translate.trans_program alloca_optimized in
      if(!print_ir) then Ir.print_ir ir;

      let linear_ir = Canon.linearize ir in
      if(!print_lin_ir) then Ir.print_lin_ir' linear_ir; 

      let triple_ir = Canon.tripleize linear_ir in
      if !print_triple_ir then Ir.print_triple_ir' triple_ir;

      let bb_ir = Canon.basicBlocks triple_ir in
	if(!print_bb_ir) then Ir.print_bb_ir' bb_ir;

      (* optimization passes -wjl *)

      (* path profiling instrumentality project *)
      let instrument_function (fname, blocks) =
      Canon.with_fname fname
      (fun () ->
        let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
        let cfg = CFG.make sane_blocks in
         (*
         let () = print_string ("DEBUG: " ^ fname ^ "\n") in
         let () = CFG.print (fname ^ "-cfg.dot") cfg in
         let () = print_string "printed cfg.dot... [enter]" in
         let _ = read_line () in
         *) (* DEBUG *)
        let cfg = Path.instrument fname cfg in
        let inst_blocks : (Ir.stmt list) list =
            List.rev (* bblock_fold goes in forward order -wjl *)
            (CFG.bblock_fold
                (fun ~name ~targets stmts nbs ->
                    (* NB: targets = [] ==> name = Temp.getReturnLabel ()
                       -- don't cons on the phantom .L1 block -wjl *)
                    if targets = [] then nbs else (stmts) :: nbs)
                []
                cfg)
        in
        let insane_blocks =
            List.map (fun stmts -> (stmts, ref false)) inst_blocks in
          (fname, insane_blocks)
        )

      in

      (*
      (* (just print the results of reaching defs analysis for now) *)
      let process_function (fname, blocks) =
        let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
        let cfg = CFG.make sane_blocks in
        let () = print_endline ("*** " ^ fname) in
        let (in_sets, out_sets) = Reaching.DFA.analyze cfg in
        (* XXX snarfed mostly from ir.ml -- refactor -wjl *)
        Temp.openFunLookUp fname;
        Ir.ps "bindings: "; Temp.listBindings ();
        Ir. pnl ();
        CFG.bblock_fold
            (fun ~name ~targets stmts () ->
                List.iter Ir.pstmt' stmts;
                Ir.pnl ();
                Ir.ps "reaching in-set:\n  {\n";
                let inset = CFG.NodeMap.find name in_sets in
                Reaching.DefinitionSet.iter
                            (fun def ->
                                Ir.ps "    ";
                                Ir.ps (Reaching.string_of_definition def);
                                Ir.pnl ())
                            inset;
                Ir.ps "  }";
                Ir.pnl ();
                Ir.pnl ();
                Ir.pnl ())
            ()
            cfg
      in
      let () = List.iter process_function bb_ir in
      *)

      (*
      (* (print the results of liveness analysis for now) *)
      let process_function (fname, blocks) =
        let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
        let cfg = CFG.make sane_blocks in
        let () = print_endline ("*** " ^ fname) in
        let (in_sets, out_sets) = Live_vars.DFA.analyze cfg in
        (* XXX snarfed mostly from ir.ml -- refactor -wjl *)
        Temp.openFunLookUp fname;
        Ir.ps "bindings: "; Temp.listBindings ();
        Ir.pnl ();
        CFG.bblock_fold
            (fun ~name ~targets stmts () ->
                Util.iteri
                  (fun i stmt ->
                      Ir.pstmt' stmt;
                      let in_set = Live_vars.InstrMap.find (name,i) in_sets in
                      let out_set = Live_vars.InstrMap.find (name,i) out_sets in
                      (*
                      Ir.ps "   {{ live in:";
                      Util.TempSet.iter
                        (fun t -> Ir.ps (" t" ^ Temp.temp2string t))
                        in_set;
                      Ir.ps " ; live out:";
                      *)
                      Ir.ps "    {{ live out: ";
                      Util.TempSet.iter
                        (fun t -> Ir.ps (" t" ^ (Temp.temp2string t)))
                        out_set;
                      Ir.ps " }}";
                      Ir.pnl ())
                  stmts;
                  Ir.pnl ())
            ()
            cfg
      in
      let () = List.iter process_function bb_ir in
      *)

      (*
      let process_dominators (fname, blocks) =
        let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
        let cfg = CFG.make sane_blocks in
	let dom = Dominators.dominators cfg in
	let loops = Dominators.natural_loops dom cfg in
	let (reachin, reachout) = Reaching.DFA.analyze cfg in
        let (livein, liveout) = Live_vars.DFA.analyze cfg in
	let () = print_endline "did flow analyses" in 
	let hoist_tgts = Hoist.hoist_targets cfg reachin liveout in
	let cfg1 = Hoist.cfg_update_preheaders cfg hoist_tgts in
	let cfg2 = Hoist.cfg_delete cfg1 hoist_tgts in
	let dom_blocks =
	  List.rev
	    (CFG.bblock_fold (fun ~name ~targets b bs -> b :: bs) [] cfg2)
	in
        let () = print_endline ("*** " ^ fname) in
        (* XXX snarfed mostly from ir.ml -- refactor -wjl *)
        Temp.openFunLookUp fname;
        Ir.ps "bindings: "; Temp.listBindings ();
        Ir.pnl ();
	Ir.ps "dominators: ";
	Ir.pnl();
	Dominators.print_dominators Format.std_formatter dom;
	Ir.pnl();
	Ir.ps "natural loops:";
	Ir.pnl();
	Dominators.print_loops Format.std_formatter loops;
	Ir.pnl();
	Ir.ps "hoist targets:";
	Ir.pnl();
	Hoist.print_hoist_targets Format.std_formatter hoist_tgts;
        (*
        let insane_blocks =
            List.map (fun stmts -> (stmts, ref false)) dom_blocks in
        (fname, insane_blocks)
        *)
      in
      let () = List.iter process_dominators bb_ir in
      *)

    (*****
     * commenting out optimizations for path profiling
     *

      let loop_hoisting (fname, blocks) =
        let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
        let cfg = CFG.make sane_blocks in
	let dom = Dominators.dominators cfg in
	let loops = Dominators.natural_loops dom cfg in
	let (reachin, reachout) = Reaching.DFA.analyze cfg in
        let (livein, liveout) = Live_vars.DFA.analyze cfg in
	let cfg = Hoist.hoist cfg reachin liveout in 
	let dom_blocks =
	  List.rev
	    (CFG.bblock_fold
                (fun ~name ~targets b bs ->
                    if targets = [] then bs else b :: bs)
                []
                cfg)
	in
        let insane_blocks =
            List.map (fun stmts -> (stmts, ref false)) dom_blocks in
          (fname, insane_blocks)
      in 


     (* constant propagation, folding *)
     let cprop_function (fname, blocks) =
       let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
       let cfg = CFG.make sane_blocks in
        let (in_sets, out_sets) = Reaching.DFA.analyze cfg in
        let nodeblocks : (CFG.node * Ir.stmt list) list =
            List.rev (* bblock_fold goes in forward order -wjl *)
            (CFG.bblock_fold
                (fun ~name ~targets stmts nbs ->
                    (* NB: targets = [] ==> name = Temp.getReturnLabel ()
                       -- don't cons on the phantom .L1 block -wjl *)
                    if targets = [] then nbs else (name, stmts) :: nbs)
                []
                cfg)
        in
        let propped_blocks = Cprop.propagate nodeblocks in_sets in
        let feld_blocks = Constant_fold.constant_fold propped_blocks in
        (* dual to sane_ blocks -wjl *)
        let insane_blocks =
            List.map (fun stmts -> (stmts, ref false)) feld_blocks in
        (fname, insane_blocks)
      in

      let cse_function (fname, blocks) = 
        let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
        let cfg = CFG.make sane_blocks in
        let (in_sets, out_sets) = Aexp.DFA.analyze cfg in
        let nodeblocks : (CFG.node * Ir.stmt list) list =
            List.rev (* bblock_fold goes in forward order -wjl *)
            (CFG.bblock_fold
                (fun ~name ~targets stmts nbs ->
                    (* NB: targets = [] ==> name = Temp.getReturnLabel ()
                       -- don't cons on the phantom .L1 block -wjl *)
                    if targets = [] then nbs else (name, stmts) :: nbs)
                []
                cfg)
        in
        let cse_blocks = Cse.cse nodeblocks in_sets in
        (* dual to sane_ blocks -wjl *)
        let insane_blocks =
            List.map (fun stmts -> (stmts, ref false)) cse_blocks in
        (fname, insane_blocks)
      in

      (* XXX XXX XXX XXX this *really* should be a higher-order function *)
      let deadcode_function (fname, blocks) = 
        let sane_blocks = List.map (fun (stmts, _boolref) -> stmts) blocks in
        let cfg = CFG.make sane_blocks in
        let (inflow, outflow) = Live_vars.DFA.analyze cfg in
        let nodeblocks : (CFG.node * Ir.stmt list) list =
            List.rev (* bblock_fold goes in forward order -wjl *)
            (CFG.bblock_fold
                (fun ~name ~targets stmts nbs ->
                    (* NB: targets = [] ==> name = Temp.getReturnLabel ()
                       -- don't cons on the phantom .L1 block -wjl *)
                    if targets = [] then nbs else (name, stmts) :: nbs)
                []
                cfg)
        in
        let deaded_blocks = Deadcode.eliminate nodeblocks ~outflow in
        (* dual to sane_ blocks -wjl *)
        let insane_blocks =
            List.map (fun stmts -> (stmts, ref false)) deaded_blocks in
        (fname, insane_blocks)
      in

      let rec loop ir n =
        (* let () = Printf.printf "iteration %d...\n" n in *)
        (* let () = flush stdout in *)
        let ir' = List.map cprop_function ir in
        let ir' = (* List.map cse_function *) ir' in
        let ir' = List.map deadcode_function ir' in
        (* let () = print_endline "BEFORE:" in *)
        (* let () = Ir.print_feld_ir' ir' in *)
        let ir' = List.map loop_hoisting ir' in
        (* let () = print_endline "AFTER:" in *)
        (* let () = Ir.print_feld_ir' ir' in *)
        if ir = ir' (* || n = 1000 *) then ir' else loop ir' (n + 1)
      in

      let feld_ir = loop bb_ir 0 in
      if !print_feld_ir then Ir.print_feld_ir' feld_ir;

    *
    * end of optimizations removed for profiling
    *****)

      let inst_ir = bb_ir in (* comment below to turn off instrumentation *)
      (*
      let inst_ir = List.map instrument_function bb_ir in
      if (!print_inst_ir) then Ir.print_inst_ir' inst_ir;
      *)

      let traced_ir = Canon.traceSchedule inst_ir in 
      if(!print_traced_ir) then Ir.print_traced_ir' traced_ir;

      (* Perform SSA-Convertion && Optimizations *)
      (*
      let traced_ir =
	if(not !no_ssa) then (
	  if(!print_ssa) then
	    Ssa.calc_n_print traced_ir
	  else
	    Ssa.calc_prog traced_ir
	 ) else
	  traced_ir
      in
      if(!print_traced_ir_ssa) then Ir.print_traced_ir traced_ir;
       *)

      (* If print munch | liveness | igraph, exit after printing *)
      if(!print_munch || !print_liveness || !print_igraph) then (
	let munched = Munch.munch_program traced_ir in
	if(!print_munch) then Munch.print (fun i s -> string_of_int i) munched; 
	let live_info = Liveness.liveness_program munched in
	if(!print_liveness) then Liveness.print_program live_info;
	if(!print_igraph) then Liveness.print_igraph live_info;
	close_files(); exit(0);
       );
      
      let allocated = 
	(*if(!use_my_regalloc) then
	  MyRegalloc.regalloc_program traced_ir
	else
	 *)
	  Regalloc.regalloc_program traced_ir
      in
      let oc = open_out (!outfile) in
      Finalize.finalize_prog allocated !infile oc;
      close_out oc;
      close_files ();
      
      Summary.print_summary ();

      if(!executable) then ( (* call gcc *)
	(* Call gcc *)
	let commandLine = 
	  Command_line.gcc_commandLine outfile outfile_exe gccFlagList in
	Errormsg.print_debug 
	  (Printf.sprintf "command line to gcc is %s \n" commandLine);
	let gccOk = Sys.command commandLine in ();
	if(gccOk != 0) then (
	  Printf.eprintf "gcc failed\n";
	  exit(1);
	 );
       )
      
    with
    | Lexer.EXIT
    | Errormsg.EXIT
    | Checker.EXIT
    | Evaluator.EXIT -> 
	close_files (); exit(-1)
    | Translate.TranslateError(s) -> 
	Printf.eprintf "TranslateError: %s" s; 
	close_files (); exit(-1)
    )
  with 
  | Sys_error(s) -> 
      Printf.eprintf "I/O error: %s\n" s; exit(-1) 
  );
  exit(0)
in	

main (Array.length Sys.argv) Sys.argv;;



