(* liveness.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: liveness.ml,v 1.2 2003/08/11 13:04:43 rflury Exp $ *)

module HA = Hashtbl
module TP = Temp
module A = Assem
module BL = Bitlist
module IG = Igraph
module F = (Frame.X86_FRAME : Frame.FrameSig)
module E = Errormsg

exception LivenessError of string

(* Record that stores info about a basic block *)
type bb_rec = {mutable bb: Assem.instr list;
	       mutable pred: TP.label list;
	       mutable succ: TP.label list;
	       mutable mark: bool; (* for DFS *)
	     }

(* Stores the Def-Use of a BB *)
type bb_du = {def: BL.bitList_t;
	      use: BL.bitList_t }


(* Returns a htbl mapping labels to BB-records, 
 * Input is a translated function from munch 
 * The Temp-environment must be adjusted for the fun by calling 
 * TP.openFunLookUp fName; previous to this fun-call *)
let read_fun (fName, lili) = 
  (* Htbl, maps label to BB-record *)
  let lab_bb : (TP.label, bb_rec ref) HA.t = HA.create 107 in
  
  (* The return-label - not implemented *)
  HA.add lab_bb (TP.getReturnLabel ()) 
    (ref {bb = []; pred = []; succ = []; mark = false});
  
  (* Add the code and the 'next' list of each BB *)
  List.iter (fun (code, next) -> 
    match List.hd code with
    | A.LABEL(_, l) -> 
	HA.add lab_bb l (ref {bb = code; pred = []; succ = next; mark = false});
    | _ -> raise (LivenessError "First stmt of BB not label")
	    ) lili;
  
  (* Generate the predecessor list for each BB *)
  List.iter (fun (code, next) -> 
    List.iter (fun toLab -> 
      match List.hd code with
      | A.LABEL(_, fromLab) -> 
	  (try
	    let bbrec = HA.find lab_bb toLab in
	    (!bbrec).pred <- fromLab :: (!bbrec).pred
	  with Not_found -> 
	    raise (LivenessError ("Lost BB with label " ^ string_of_int toLab))
	  )
      | _ -> raise (LivenessError "First stmt of BB not label")
	      ) next; 
	    ) lili;
  lab_bb
    

(* This fun assumes that all temps have been created by Temp with
 * the property that a consecutive range from 10 to x was used
 * The Temp-environment must be adjusted for the fun by calling 
 * TP.openFunLookUp fName; previous to this fun-call 
 * Returns a Hash-tbl maping label to define-use record *)
(* ALGO for def and use of BB: 
   ---------------------------
   DEF = {}; USE = {};
   For all commands c in BB (in the order of execution)
       USE = USE U (use(c) - DEF);
       DEF = DEF U def(c);
*)
let makeBitLists lab_bb = 
  let lab_bit : (TP.label, BL.bitList_t) HA.t = HA.create 107 in
  let maxTemp = TP.getTempCounter () in

  let lab_du : (TP.label, bb_du) HA.t = HA.create 107 in

  (try
    HA.iter (fun label bbrec -> 
      let use = BL.newBitList maxTemp in
      let def = BL.newBitList maxTemp in
      
      List.iter (fun i -> 
	let src = A.getSrc i in  (* uses *)
	let dst = A.getDst i in (* defs *)
	List.iter (fun p -> 
	  if(not(BL.getBit def p)) then BL.setBit use p
		  ) src;
	List.iter (fun p -> BL.setBit def p) dst;
		) (!bbrec).bb; (* iter over the basic block *)
      
      HA.add lab_du label {def = def; use = use};
	    ) lab_bb;
    lab_du
  with _ -> 
    raise (LivenessError "Uncaught Exceptoin in makeBitList")
  )

(* Calculates the IN and OUT set of BB. 
 * The Temp-environment must be adjusted for the fun by calling 
 * TP.openFunLookUp fName; previous to this fun-call 
 * Returns two hash-tables mapping labels to IN and OUT BitLists *)
(*ALGO: (Appel, p. 214, bug-fixed)
  for each BB:
      IN = {};
      OUT = {};
  repeat
    for each BB:
      OUT = IN(succBB); 
      IN = USE U (OUT - DEF);
  until OUT and IN do not change anymore. 
*)	
let livenessBB lab_bb lab_du =  
  (try
    (* the work-list of all labels *)
    let wList = ref [] in
    (* perform backwards DFS on pred-nodes *)
    let rec dfs label = 
      let bbrec = 
      (try
	HA.find lab_bb label
      with _ -> 
	raise (LivenessError ("Lost info about label " ^ (string_of_int label)));
      ) in
      if(not (!bbrec).mark) then (
	(!bbrec).mark <- true;
	List.iter (fun l -> dfs l) (!bbrec).pred;
	wList := label :: !wList;
	(* necessary for infinite loops *)
	List.iter (fun l -> dfs l) (!bbrec).succ;
       )
    in
    (* Reorder BB in a backward flow; Appel p. 217 / p. 380 *)
    dfs (TP.getReturnLabel ());

    (* Because I do some optimizations and recognize true & false, 
     * I don't get labels in the label-list that are never jumped to
     * but the control-flow could reach -> add all non-marked *)
    HA.iter (fun lab bbrec -> 
      if(not (!bbrec).mark) then (
	dfs lab
       )
	    ) lab_bb;


    let lab_IN : (TP.label , BL.bitList_t) HA.t = HA.create 107 in
    let lab_OUT : (TP.label , BL.bitList_t) HA.t = HA.create 107 in

    List.iter (fun l -> 
      HA.add lab_IN l (BL.newBitList (TP.getTempCounter ()));
      HA.add lab_OUT l (BL.newBitList (TP.getTempCounter ()));
	      ) !wList;

    let altered = ref true in
    let update a = altered := !altered || a in
    while(!altered) do
      altered := false;
      List.iter (fun lab-> 
	(* OUT = IN(succBB);  *)
	let outBL = HA.find lab_OUT lab in
	let bb = HA.find lab_bb lab in
	let succList = (!bb).succ in
	List.iter (fun l -> 
	  update(BL.orBLto2alter (HA.find lab_IN l) outBL)
		  ) succList;
	(* IN = USE U (OUT - DEF); *)      
	let inBL = HA.find lab_IN lab in
	let outBL = HA.find lab_OUT lab in
	let du = HA.find lab_du lab in
	update(BL.orBLtoR_ABCalter inBL du.use outBL du.def)
		) !wList
    done; (* while *)

    (lab_IN, lab_OUT, !wList)
      
  with 
  | LivenessError(x) -> raise (LivenessError(x));
  | _ -> raise (LivenessError "Uncaught Exceptoin in livenessBB")
  )    
    

type liveness_info = {fName: string;
		      lab_bb: (TP.label, bb_rec ref) HA.t;
		      lab_du: (TP.label, bb_du) HA.t;
		      lab_IN: (TP.label, BL.bitList_t) HA.t;
		      lab_OUT: (TP.label, BL.bitList_t) HA.t;
		      lab_List: TP.label list; (* a list of all labels *)
		    }


(* Inserts the Invariants for PCC in every label that has > 1 predecessor *)
let insertPCCinvar (fName, bblist) info lab_invar = 
  let getList label = 
    let tmp = 
      (try
	let bb_rec = HA.find info.lab_bb label in
	if(List.length ((!bb_rec).pred) > 1) then
	  BL.getTempList (HA.find info.lab_IN label) (fun x -> x)
	else
	  []
      with Not_found -> 
	raise (LivenessError ("No IN-set found for BB with label " ^
			      string_of_int label))
      )
    in
    List.map (fun x -> Ir.TypeInvar(x)) tmp
  in
  List.iter (fun (code, next) -> 
    match code with
    | A.LABEL(s, l) :: tail -> 
	HA.replace lab_invar l (getList l);
    | _ -> 
	raise (LivenessError ("BB does not start with a label"))
	    ) bblist
    
    
(* Performs the liveness analysis and returns the result stored
 * in a liveness_info record. 
 * The parameter is expected to a single function translated by munch *)
let liveness_fun (fName, bblist) = 
  E.showProgress (Printf.sprintf "liveness on %s " fName);
  (* TP.openFunLookUp fName; *)
  let lab_bb = read_fun (fName, bblist) in
  let lab_du = makeBitLists lab_bb in
  let (lab_IN, lab_OUT, lab_List) = livenessBB lab_bb lab_du in

  {fName = fName; 
   lab_bb = lab_bb;
   lab_du = lab_du;
   lab_IN = lab_IN;
   lab_OUT = lab_OUT;
   lab_List = lab_List}
    
(* Performs the liveness analysis and returns the result stored
 * in a liveness_info record. 
 * The parameter is expected to be the result of the munch-algo *)
let liveness_program program = 
  List.map (fun (fName, bblist) -> 
    TP.openFunLookUp fName;
    liveness_fun (fName, bblist)
	   ) program

(* Prints the liveness information of a function *)
let print_fun i = 
  let ps = print_string in
  let pnl = print_newline in
  let pi = print_int in
  ps "-------------------------------------------------\n"; 
  ps i.fName; pnl ();
  ps "-------------------------------------------------\n"; 
  HA.iter (fun l bb_rec ->  
    ps (TP.label2string l); ps ":\n";
    ps "  Pred: "; List.iter (fun l -> pi l; ps ", ") (!bb_rec).pred;
    pnl();
    ps "  Succ: "; List.iter (fun l -> pi l; ps ", ") (!bb_rec).succ;
    pnl();
    let du = HA.find (i.lab_du) l in
    ps "  Def:  "; List.iter (fun l -> pi l; ps ", ") 
      (BL.getTempList du.def (fun x -> x));
    pnl();
    ps "  Use:  "; List.iter (fun l -> pi l; ps ", ") 
      (BL.getTempList du.use (fun x -> x));
    pnl();      
    let in_li = HA.find (i.lab_IN) l in
    ps "  IN:   "; List.iter (fun l -> pi l; ps ", ") 
      (BL.getTempList in_li (fun x -> x));
    pnl();
    let out_li = HA.find (i.lab_OUT) l in
    ps "  OUT:  "; List.iter (fun l -> pi l; ps ", ") 
      (BL.getTempList out_li (fun x -> x));
    pnl();
	  ) i.lab_bb
    
(* Prints the liveness information of a program *)
let print_program live_info = 
  List.iter (fun i -> print_fun i) live_info
      

(************************************************************************)
(* Interference Graph - not Appel, his one is in regalloc (build) *)
(************************************************************************)

let makeIGraph info = 
(* TP.openFunLookUp info.fName; *)
  let ig = IG.iniIGraph (TP.getTempCounter ()) in
  let move_temps : (TP.temp, TP.temp) HA.t = HA.create 317 in

  HA.iter (fun label bbrec -> 
    (* A list of all out-temps *)
    let out = ref (BL.getTempList (HA.find info.lab_OUT label) (fun x -> x)) in

    List.iter (fun instr -> 
      let def = Assem.getDst instr in
      let src = Assem.getSrc instr in
      (match instr with
      | Assem.MOVE(_, src, dst) -> 
	  List.iter (fun t -> 
	    if(t != src) then
	      IG.makeInterfere t dst ig;
		    ) !out;
	  HA.add move_temps src dst;
	  HA.add move_temps dst src;
      | _ -> 
	  List.iter (fun d -> 
	    List.iter (fun t -> 
	      IG.makeInterfere t d ig;
		      ) !out;
		    ) def;
      );
      (* uptdate out *)
      (* all def. temps are not anymore in the out-set of the prev. instr. *)
      out := Helpers.list_minus !out def;
      (* all used temps must be in the out-set of the prvious instruction *)
      out := Helpers.list_union !out src;

	      ) (List.rev (!bbrec).bb);

    (* sanity check *)
    (* Iterative calculated out should be equal to IN (test) *)
    let inli = BL.getTempList (HA.find info.lab_IN label) (fun x -> x) in
    List.iter (fun t -> 
      if(not (List.mem t inli)) then 
	raise (LivenessError (TP.label2string label ^
			      " Iterative calculated OUT does not match IN"));
	      ) !out;
    List.iter (fun t -> 
      if(not (List.mem t !out)) then 
	raise (LivenessError (TP.label2string label ^
			      " Iterative calculated OUT does not match IN"));
	      ) inli;
    (* end sanity check *)
       
    	  ) info.lab_bb;
  
  (* Add interferences for fixed temps *)
  (* Not used, shift only works with %cl
     let forced = TP.getForcedHtbl () in
     HA.iter (fun temp flist -> 
     List.iter (fun t -> 
     IG.makeInterfere temp t ig
     ) (F.complementReg flist)
     ) !forced;
   *)

  (* All Registers interfere mutually *)
  for i=1 to 8 do
    for j=1 to 8 do
      IG.makeInterfere i j ig;
    done;
  done;
  
  (* All temps interfere with %esp and %ebp *) 
  for i=11 to (TP.getTempCounter () - 1) do
    IG.makeInterfere i (F.sp ()) ig;
    IG.makeInterfere i (F.bp ()) ig;
  done;

  (* no temp interferes with itself *)
  for i=0 to (TP.getTempCounter () - 1) do
    IG.resetBit i i ig; 
  done;

  (* Return the Interference Graph *)
  (ig, move_temps)


let print_igraph info = 
  List.iter (fun i -> 
    TP.openFunLookUp i.fName;
    let (ig,_) = makeIGraph i in
    Printf.printf "*************** IGraph for %s *****************\n" i.fName;
    IG.printIGraph (ig)
	    ) info
