(* ir.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: ir.ml,v 1.4 2003/09/17 18:54:07 rflury Exp $ *)

(* The Intermediate Representation Tree Language *)

module TP = Temp
module A = Absyn

type pcc_type = Pcc.pccType

(* Expressions *)
type exp = 
  | CONST of int32 * Pcc.pccType
  | NAME of TP.label
  | TEMP of TP.temp
  | BINOP of binop * exp * exp
  | MEM of exp
  | CALL of TP.label * exp list
  | ALLOCA of exp
  | ESEQ of stmt * exp 
  | PHI of exp list

(* Statements *)	
and stmt =
  | MOVE of exp * exp
  | EXP of exp 
  | LABEL of TP.label
  | JUMP of TP.label
  | CJUMP of relop * exp * exp * TP.label * TP.label
  | SEQ of stmt * stmt
  | COMMENT of string * TP.temp list
  | INVARIANT of pccInvariant list

(* Binary operators *)
and binop =
  | PLUS | MINUS | DIV | MUL | MOD 
  | AND | OR | XOR | NOT (* bitwise *)
  | SHIFTL | SHIFTR 

(* Relational operators *)
and relop =
  | EQ | NE | LT | LE | GT | GE    

(* Loop invariants for PCC *)
and pccInvariant = 
    (* State the type of a temporary *)
  | TypeInvar of TP.temp
    (* Give a relation for a temporary *)
  | RelInvar of TP.temp * relop * int32

let nop = EXP(CONST(Int32.zero, Pcc.Int))

(* negates a relational operator *)
let notRelop = function
  | EQ -> NE
  | NE -> EQ
  | LT -> GE
  | LE -> GT
  | GT -> LE
  | GE -> LT

(* print ir-tree *)

(* Allow to print to a string - not to stdout *)
let ir_print_string = ref ""
let print_2_string = ref false
let no_newline = ref false

let ps s = 
  if(!print_2_string) then
    ir_print_string := !ir_print_string ^ s
  else
    print_string s

let pnl () = 
  if(not !no_newline) then (
    if(!print_2_string) then
      ir_print_string := !ir_print_string ^ "\n"
    else
      print_newline ()
   )

let pp_level = ref 0 

(* print new-line with indentation on following line *)
let pnls () = 
  if(not !no_newline) then (
    let rec iter = function
      | 0 -> ()
      | x -> ps " "; iter (x-1)
    in
    pnl ();
    iter !pp_level
   )

(* modify the level of indentation *)
let inc x = 
  pp_level := !pp_level + x
let dec x = 
  pp_level := !pp_level - x


let pop = function
  | PLUS -> ps "PLUS"
  | MINUS -> ps "MINUS"
  | DIV -> ps "DIV"
  | MUL -> ps "MUL"
  | MOD -> ps "MOD"
  | AND -> ps "AND"
  | OR -> ps "OR"
  | XOR -> ps "XOR"
  | NOT -> ps "NOT"
  | SHIFTL -> ps "SHIFTL"
  | SHIFTR -> ps "SHIFTR"


let prelop = function
  | EQ -> ps "EQ"
  | NE -> ps "NE"
  | LT -> ps "LT"
  | LE -> ps "LE"
  | GT -> ps "GT"
  | GE -> ps "GE"

let rec pexp = function
  | ALLOCA(e) -> ps "ALLOCA("; pexp e; ps ")"
  | CONST(i,pt) -> ps "CONST("; ps (Int32.to_string i); ps ")"
  | TEMP(t) -> ps "TEMP("; ps (TP.temp2string t); ps ")"
  | BINOP(o,e1,e2) -> 
      ps "BINOP("; inc 6; pop o; ps ", "; pnls ();
      pexp e1; ps ", "; pnls ();
      pexp e2; ps ")"; dec 6
  | MEM(e) -> ps "MEM("; inc 4; pexp e; ps")"; dec 4
  | CALL(l, e1 :: eli) -> 
      ps "CALL("; inc 5; ps (TP.label2string l); ps ", "; pnls ();
      pexp e1; 
      List.iter (fun e -> ps ", "; pnls (); pexp e) eli;
      ps ")"; dec 5
  | CALL(l, []) -> 
      ps "CALL("; ps (TP.label2string l); ps ")"
  | ESEQ(s, e) -> ps "ESEQ("; inc 5; pstmt s; pexp e; ps ")"; dec 5
  | NAME(l) -> ps "NAME("; ps (TP.label2string l); ps ")"
  | PHI(t :: tl) -> 
      ps "PHI("; inc 4; 
      pexp t; 
      List.iter (fun t -> ps ", "; pexp t) tl;
      ps ")"; dec 4
  | PHI([]) -> 
      ps "PHI()";

and pstmt = function
  | COMMENT(s,_) -> ps "COMMENT("; ps s; ps ")"; pnls ()
  | MOVE(e1,e2) -> ps "MOVE("; inc 5; pexp e1; ps ", "; pnls ();
      pexp e2; ps ")"; dec 5; pnls ()
  | EXP(e) -> ps "EXP("; inc 4; pexp e; ps ")"; dec 4; pnls ()
  | LABEL(l) -> ps "LABEL("; ps (TP.label2string l); ps ")"; pnls ()
  | JUMP(l) -> ps "JUMP("; ps (TP.label2string l); ps ")"; pnls ()
  | CJUMP(relop, e1, e2, lt, lf) -> 
      ps "CJUMP("; inc 6; prelop relop; ps ", "; pnls ();
      pexp e1; ps ", "; pnls ();
      pexp e2; ps ", "; pnls ();
      ps (TP.label2string lt); ps ", ";
      ps (TP.label2string lf); ps ")"; dec 6; pnls ()
  | SEQ(s1, s2) -> pstmt s1; pstmt s2
  | INVARIANT(li) -> ps "INVARIANTS( TODO )"

let pop' = function
  | PLUS -> ps "+"
  | MINUS -> ps "-"
  | DIV -> ps "/"
  | MUL -> ps "*"
  | MOD -> ps "%"
  | AND -> ps "AND"
  | OR -> ps "OR"
  | XOR -> ps "XOR"
  | NOT -> ps "NOT"
  | SHIFTL -> ps "SHIFTL"
  | SHIFTR -> ps "SHIFTR"

let prelop' = function
  | EQ -> ps "=="
  | NE -> ps "!="
  | LT -> ps "<"
  | LE -> ps "<="
  | GT -> ps ">"
  | GE -> ps ">="

let rec pexp' = function
  | ALLOCA(e) -> ps "ALLOCA("; pexp' e; ps ")"
  | CONST(i,pt) -> ps "CONST("; ps (Int32.to_string i); ps ")"
  | TEMP(t) -> ps "t"; ps (TP.temp2string t); 
  | BINOP(op, e1,e2) ->
      ps "(";
      pexp' e1;
      ps " ";
      pop' op;
      ps " ";
      pexp' e2;
      ps ")";
(*      
      ps "("; inc 6; pop o; ps ", "; pnls ();
      pexp' e1; ps ", "; pnls ();
      pexp' e2; ps ")"; dec 6
*)
  | MEM(e) ->
      ps "["; pexp' e; ps "]";
      (* ps "MEM("; inc 4; pexp' e; ps")"; dec 4 *)
  | CALL(l, e1 :: eli) ->
      ps (TP.label2string l);
      ps "(";
      pexp' e1; 
      List.iter (fun e -> ps ", "; pexp' e) eli;
      ps ")"
  | CALL(l, []) -> 
      ps (TP.label2string l); ps "()"
  | ESEQ(s, e) -> ps "ESEQ("; inc 5; pstmt' s; pexp' e; ps ")"; dec 5
  | NAME(l) -> ps "NAME("; ps (TP.label2string l); ps ")"
  | PHI(t :: tl) -> 
      ps "PHI("; inc 4; 
      pexp' t; 
      List.iter (fun t -> ps ", "; pexp' t) tl;
      ps ")"; dec 4
  | PHI([]) -> 
      ps "PHI()";
and pstmt' = function
  | COMMENT(s,_) -> ps "// "; ps s; pnls ()
  | MOVE(e1,e2) ->
      pexp' e1;
      ps " := ";
      pexp' e2;
      pnls();
  | EXP(e) -> ps "EXP("; inc 4; pexp' e; ps ")"; dec 4; pnls ()
  | LABEL(l) -> ps "LABEL("; ps (TP.label2string l); ps "): "; pnls ()
  | JUMP(l) -> ps "goto "; ps (TP.label2string l); pnls ()
  | CJUMP(relop, e1, e2, lt, lf) -> 
      ps "if ("; pexp' e1; ps " "; prelop' relop; ps " "; pexp' e2; ps ")";
      ps " then goto "; ps (TP.label2string lt);
      ps " else goto "; ps (TP.label2string lf);
      pnls ()
  | SEQ(s1, s2) -> pstmt' s1; pstmt' s2
  | INVARIANT(li) -> ps "INVARIANTS( TODO )"



(* Print an IR-expression to a string *)
let ir_pexp2str e =
  print_2_string := true;
  no_newline := true;
  ir_print_string := "";
  let tmp = !pp_level in
  pp_level := 0;
  pexp e;
  print_2_string := false;
  pp_level := tmp;
  no_newline := false;
  !ir_print_string

(* Print an IR-statement to a string *)
let ir_pstmt2str s =
  print_2_string := true;
  no_newline := true;
  ir_print_string := "";
  let tmp = !pp_level in
  pp_level := 0;
  pstmt' s; (* XXX prime  -wjl *)
  print_2_string := false;
  pp_level := tmp;
  no_newline := false;
  !ir_print_string


(* Prints the IR-tree from a list of functions *)
let print_ir fun_list =
  ps "================= IR ====================\n";
  List.iter (fun (fName, s) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    pnl ();
    pstmt s; 
    pnl (); pnl()
	    ) fun_list;
  ps "=============== END IR ==================\n"


let print_ir' fun_list =
  ps "================= IR ====================\n";
  List.iter (fun (fName, s) -> 
	       TP.openFunLookUp fName;
	       ps "bindings: "; TP.listBindings ();
	       pnl ();
	       pstmt' s; 
	       pnl (); pnl()
	    )
    fun_list;
  ps "=============== END IR ==================\n"


(* Prints the linearized IR-tree from a list of functions *)
let print_lin_ir_aux what fun_list =
  ps ("============== " ^ what ^ " IR ================\n");
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    pnl ();
    List.iter (fun s -> pstmt s) sli;
    pnl (); pnl()
	    ) fun_list;
  ps ("============ END " ^ what ^ " IR ==============\n")

let print_lin_ir_aux' what fun_list =
  ps ("============== " ^ what ^ " IR ================\n");
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    pnl ();
    List.iter (fun s -> pstmt' s) sli;
    pnl (); pnl()
	    ) fun_list;
  ps ("============ END " ^ what ^ " IR ==============\n")

let print_lin_ir fun_list = print_lin_ir_aux "LINEAR" fun_list
let print_triple_ir fun_list = print_lin_ir_aux "TRIPLE" fun_list

let print_lin_ir' fun_list = print_lin_ir_aux' "LINEAR" fun_list
let print_triple_ir' fun_list = print_lin_ir_aux' "TRIPLE" fun_list


(* Prints the Basic Block IR-tree from a list of functions *)
let print_bb_ir fun_list =
  ps "=========== BASIC BLOCK IR =============\n";
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    pnl ();
    List.iter (fun (l, _) -> 
      List.iter (fun s -> pstmt s) l;
      pnl ()
	      ) sli;
    
    pnl (); pnl()
	    ) fun_list;
  ps "========= END BASIC BLOCK IR ===========\n"


(* Prints the Basic Block IR-tree from a list of functions *)
let print_bb_ir' fun_list =
  ps "=========== BASIC BLOCK IR =============\n";
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    pnl ();
    List.iter (fun (l, _) -> 
      List.iter (fun s -> pstmt' s) l;
      pnl ()
	      ) sli;
    
    pnl (); pnl()
	    ) fun_list;
  ps "========= END BASIC BLOCK IR ===========\n"

(* Prints the Basic Block IR-tree from a list of functions *)
let print_feld_ir' fun_list =
  ps "=========== CONSTANT-FELD IR =============\n";
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    pnl ();
    List.iter (fun (l, _) -> 
      List.iter (fun s -> pstmt' s) l;
      pnl ()
	      ) sli;
    
    pnl (); pnl()
	    ) fun_list;
  ps "========= END CONSTANT-FELD IR ===========\n"


(* Prints the Basic Block IR-tree from a list of functions *)
let print_inst_ir' fun_list =
  ps "=========== INSTRUMENTED IR =============\n";
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    pnl ();
    List.iter (fun (l, _) -> 
      List.iter (fun s -> pstmt' s) l;
      pnl ()
	      ) sli;
    
    pnl (); pnl()
	    ) fun_list;
  ps "========= END INSTRUMENTED IR ===========\n"




(* Prints the Traced IR-tree from a list of functions *)
let print_traced_ir fun_list =
  ps "============= TRACED IR ===============\n";
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    inc 1;
    pnl ();
    List.iter (fun l -> 
      List.iter (fun s -> pstmt s) l;
      pnl ()
	      ) sli;
    dec 1;
    pnl (); pnl()
	    ) fun_list;
  ps "=========== END TRACED IR =============\n"


(* Prints the Traced IR-tree from a list of functions *)
let print_traced_ir' fun_list =
  ps "============= TRACED IR ===============\n";
  List.iter (fun (fName, sli) -> 
    TP.openFunLookUp fName;
    ps "bindings: "; TP.listBindings ();
    inc 1;
    pnl ();
    List.iter (fun l -> 
      List.iter (fun s -> pstmt' s) l;
      pnl ()
	      ) sli;
    dec 1;
    pnl (); pnl()
	    ) fun_list;
  ps "=========== END TRACED IR =============\n"



