(* igraph.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: igraph.ml,v 1.2 2003/08/11 13:04:43 rflury Exp $ *)

(* description:
   Implements a module that provides a interference Graph and functions
   to access it. 
   
   Functions:
   ----------
   - create an iGraph for at least n variables
   - add a varialbe (mapping from vID to position in matrix.. is done here)
   - make interfere two vars
   - add two vars to a coalescing-list (all moves)
   - check if two vars interfere

   - make IList (InterferenceList)
   - get all variables that interfere with a given var
   - coalesce two vars

*)

module BL = Bitlist
module AR = Array
module HA = Hashtbl
      
type igraph_t = Temp.temp array array ref

(* Create an IGraph matrix that can hold at least n temps *)
let iniIGraph n =
  let len = n / 31 + 2 in 
  ref (AR.make_matrix (n + 62 - (n mod 31)) len 0)
  
    
(* BIT-LEVEL ************************************************************)
(* returns true if the bit at OFFSET x, y is one, else false *)
let getBit x y ig =
  let n = (!ig).(x).(y / 31) in (* get int *)
  let mask = n lsr (y mod 31) in
  if((mask land 1) <> 0) then true else false

(* sets the bit at OFFSET x, y in the matrix ig to one *)
let setBit x y ig =
  let n = (!ig).(x).(y / 31) in (* get int *)
  let mask = 1 lsl (y mod 31) in
  (!ig).(x).(y / 31) <- (n lor mask)

(* sets the bit at OFFSET x, y in the matrix ig to zero *)
let resetBit x y ig =
  let n = (!ig).(x).(y / 31) in (* get int *)
  let mask = lnot (1 lsl (y mod 31)) in
  (!ig).(x).(y / 31) <- (n land mask)
      
(* makes interfere temps 'a' and 'b' *)
let makeInterfere a b ig =
  setBit a b ig;
  setBit b a ig 

(* returns true if the temps 'a' and 'b' interfere.
 * I return the OR of the two places such that only one
 * must be set -> used for coalescing such that I do not have
 * to update also the columns (which would be very expensive) *)
let qInterfere a b ig =
  getBit a b ig || getBit b a ig 
    
(* Returns a list of temps that interfere with x *)
let getTempList x ig = 
  BL.getTempList (ref ((!ig).(x))) (fun x -> x)

(* Print the I-Graph *)
let printIGraph ig = 
  let ps = print_string in
  let pi = print_int in 
  let n = Array.length !ig in
  for i = 0 to (n-1) do 
    match getTempList i ig with
    | [] -> ()
    | list -> pi i; ps " : "; List.iter (fun t -> pi t; ps ", ") list; ps "\n"
  done;
  
(************************************************************************)
(* Not implemented yet *)
(************************************************************************)

(* coalesces vIDs a and b by oring their bit-lists to b
 * and by changing the vid_bitPos and bitPos_vid such that
 * a points also to b.
 *)
(*
let coalesce a b sR =
  let offa = getOff a sR in
  let offb = getOff b sR in
  (try
    (!((!sR).ig)).(offb) <- 
      Array.mapi (fun i a -> ((!((!sR).ig)).(offb).(i)) lor a) (!((!sR).ig)).(offa);
  with Invalid_argument(_) -> Errormsg.error "" (1,1) "Fatal Error: igraph.ml:coalesce list not of same size";
    exit(1)
  );   
  HA.replace ((!sR).oVh) offa b;
  HA.replace ((!sR).vOh) a offb
*)

(* vid_cnt: vid -> #of interferences at the moment
 * vid_if: vid -> list of interfering vids
 * pushable is a stack which contains all vids that have less than 6
 * interferences
 *)
(* let initializeColoring sR vid_if vid_cnt pushable = *)
  (* reset all self-interferences *)
(*  for i = 0 to (!sR).vOhcount do
    resetBit i i (!sR).ig;
  done;
  for i = 0 to (!sR).vOhcount  do
    let listID = BL.getVIDfromBL (ref ((!((!sR).ig)).(i)) ) ((!sR).oVh) in 
    let len = List.length listID in
    let vID = HA.find ((!sR).oVh) i in 
    HA.add vid_if vID listID;
    if (vID > 8) then
      if(len < 6) then begin
	Stack.push vID pushable;
	HA.add vid_cnt vID (-1);
      end
      else
	HA.add vid_cnt vID len
    else
      HA.add vid_cnt vID (-1);
  done;
*)
