%{
(* grammar *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: grammar.mly,v 1.5 2003/09/17 18:54:07 rflury Exp $ *)

module A = Absyn

let context l r :Errormsg.pos = ( rhs_start l , rhs_end r )

let make_lval pos = function
  | A.LVal(l) -> l
  | _ -> Errormsg.error pos "Not a valid L-value"; raise Errormsg.EXIT

%}

%token <string> ID
%token <int32> INT
%token LBRACE RBRACE LPAREN RPAREN LBRACKET RBRACKET
%token PLUS MINUS TIMES DIVIDE MOD PPLUS PMINUS
%token LOGICAND LOGICOR LOGICXOR LOGICNOT
%token BITOR BITAND BITXOR SHIFTLEFT SHIFTRIGHT
%token UMINUS AMPERSAND BITNOT
%token LOGICEQ LOGICNOTEQ LOGICLESS LOGICLESSEQ LOGICMORE LOGICMOREEQ
%token PLOGICEQ PLOGICNOTEQ
%token ASSIGN PLUSASSIGN MINUSASSIGN TIMESASSIGN DIVIDEASSIGN MODASSIGN
%token COMMA SEMICOL RARROW DOT COLON
%token INTEGER BOOL
%token ALLOC OFFSET SIZE IF ELSE FOR WHILE RETURN TRUE FALSE NULL 
%token VAR STRUCT VOID FOREIGN CONTINUE BREAK
%token EOF 

%start program

%type <Absyn.program> program

%left LOGICOR 
%left LOGICXOR
%left LOGICAND
%left LOGICEQ LOGICNOTEQ PLOGICEQ PLOGICNOTEQ
%left LOGICLESS LOGICLESSEQ LOGICMORE LOGICMOREEQ
%left PPLUS PMINUS
%left BITOR
%left BITXOR
%left BITAND
%left SHIFTLEFT SHIFTRIGHT
%left PLUS MINUS
%left TIMES DIVIDE MOD
%left UMINUS AMPERSAND UTIMES BITNOT LOGICNOT
%left RARROW DOT LBRACKET

%%


program:
| EOF
    { A.Program( [] ) }
| decllist
    { let tmp = $1 in A.Program( (fst tmp) @ (snd tmp)) }

/* want all struct declarations first */
decllist:
| fundecl
    { ([],[$1]) }
| structdecl
    { ([$1],[]) }
| fundecl decllist
    { let tmp1 = $1 in 
      let tmp2 = $2 in 
      ((fst tmp2) ,tmp1 :: (snd tmp2)) }
| structdecl decllist
    { let tmp1 = $1 in 
      let tmp2 = $2 in 
      (tmp1 :: (fst tmp2) ,(snd tmp2)) }

structdecl:
| STRUCT ID LBRACE structbody RBRACE SEMICOL
    { A.StructDecl( A.Ident($2, context 2 2), $4) }


fundecl:
| VOID ID LPAREN arglist RPAREN body
    { A.FunDecl(A.Ident($2, context 2 2), A.VOID, context 1 1, $4, $6) }
| VOID ID LPAREN RPAREN body
    { A.FunDecl(A.Ident($2, context 2 2), A.VOID, context 1 1, [], $5) }
| types ID LPAREN arglist RPAREN body
    { A.FunDecl(A.Ident($2, context 2 2), $1, context 1 1, $4, $6) }
| types ID LPAREN RPAREN body
    { A.FunDecl(A.Ident($2, context 2 2), $1, context 1 1, [], $5) }
    
body:
| LBRACE vdecllist stmtseq RBRACE
    { A.Body($2, $3) }
| LBRACE  stmtseq RBRACE
    { A.Body([], $2) }
| LBRACE vdecllist RBRACE
    { A.Body($2, []) }
| LBRACE RBRACE
    { A.Body([],[]) }
| FOREIGN
    { A.Foreign(context 1 1) }

/********************** Declarations **********************/

vdecllist:
| vdecl
    { $1 }
| vdecl vdecllist
    { $1 @ $2 }
    
vdecl:
| VAR idents COLON types SEMICOL
    { List.map (fun i -> A.VarDecl(i, $4, context 4 4 )) $2 }

idents:
| ID
    { [ A.Ident($1, context 1 1) ] }
| ID COMMA idents
    { A.Ident($1, context 1 1) :: $3 }

structbody:
| ID COLON types
    { [ (A.Ident($1, context 1 1), $3, context 3 3) ] }
| ID COLON types SEMICOL
    { [ (A.Ident($1, context 1 1), $3, context 3 3) ] }
| ID COLON types SEMICOL structbody
    { (A.Ident($1, context 1 1), $3, context 3 3) :: $5 }

arglist:
| ID COLON types
    { [ (A.Ident($1, context 1 1), $3, context 3 3) ] }
| ID COLON types COMMA arglist
    { (A.Ident($1, context 1 1), $3, context 3 3) :: $5 }

types:
| INTEGER
    { A.Int }
| BOOL
    { A.Bool }
| ID
    { A.User( A.Ident($1, context 1 1) ) }
| types TIMES
    { A.Pointer( $1 ) }

/********************** Statements *************************/

stmtseq:
| stmt
    { [$1] }
| stmt stmtseq
    { $1 :: $2 }
    
stmt:
| simp SEMICOL
    { $1 }
| control
    { $1 } 
| SEMICOL        /* Empty simp is allowed! */
    { A.Blank }

simp:
| simpassign
    { $1 }
| simprest
    { $1 }

simpassign:
| exp PLUSASSIGN exp 
    { A.Assign(make_lval (context 1 1) $1, context 1 1, 
	       A.OpExp($1, A.PLUS, Some($3), context 3 3, A.Void), 
	       context 1 3) }
| exp MINUSASSIGN exp
    { A.Assign(make_lval (context 1 1) $1, context 1 1, 
	       A.OpExp($1, A.MINUS, Some($3), context 3 3, A.Void), 
	       context 1 3) }
| exp TIMESASSIGN exp
    { A.Assign(make_lval (context 1 1) $1, context 1 1, 
	       A.OpExp($1, A.TIMES, Some($3), context 3 3, A.Void), 
	       context 1 3) }
| exp DIVIDEASSIGN exp
    { A.Assign(make_lval (context 1 1) $1, context 1 1, 
	       A.OpExp($1, A.DIVIDE, Some($3), context 3 3, A.Void), 
	       context 1 3) }
| exp MODASSIGN exp
    { A.Assign(make_lval (context 1 1) $1, context 1 1, 
	       A.OpExp($1, A.MOD, Some($3), context 3 3, A.Void), 
	       context 1 3) }
| exp ASSIGN exp
    { A.Assign(make_lval (context 1 1) $1, context 1 1, $3 , context 1 3) }

simprest:
| RETURN exp 
    { A.Return(Some($2), context 1 2) }
| RETURN
    { A.Return(None, context 1 2) }
| exp
    { A.StmtExp($1, context 1 2) }
| CONTINUE
    { A.Continue(context 1 1) }
| BREAK
    { A.Break(context 1 1) }

/********************** Control ***************************/

control:
| IF LPAREN exp RPAREN block ELSE block    /* default is shifting->ok */
    { A.IfElse( $3, context 3 3, $5, $7, context 1 7)}
| IF LPAREN exp RPAREN block
    { A.If( $3, context 3 3, $5, context 1 5) }
| FOR LPAREN simpassign SEMICOL exp SEMICOL simp RPAREN block
    { A.For( $3, context 3 3, $5, context 5 5, $7, context 7 7, $9) }
| FOR LPAREN SEMICOL exp SEMICOL simp RPAREN block
    { A.For( A.Blank, context 2 3, $4, context 4 4, $6, context 6 6, $8) }
| FOR LPAREN simpassign SEMICOL exp SEMICOL RPAREN block
    { A.For( $3, context 3 3, $5, context 5 5, A.Blank, context 6 7, $8) }
| FOR LPAREN SEMICOL exp SEMICOL RPAREN block
    { A.For( A.Blank, context 2 3, $4, context 4 4, A.Blank, context 5 6, $7) }
| WHILE LPAREN exp RPAREN block
    { A.While($3, context 3 3, $5) }

block:
| stmt
    { [$1] }
| LBRACE RBRACE
    { [] }
| LBRACE stmtseq RBRACE
    { $2 }

/********************** Constants *************************/

constexp:
| INT
    { A.IntConst( $1, context 1 1 ) }
| TRUE
    { A.BoolConst( true, context 1 1 ) }
| FALSE
    { A.BoolConst( false, context 1 1 ) }
| NULL
    { A.NULL( context 1 1) }

/********************** Expressions ***********************/

exp:
| exp LOGICOR exp                                                  /* || */
    { A.OpExp($1, A.LOGICOR, Some($3), context 1 3, A.Void) }
| exp LOGICXOR exp                                                 /* ^^ */
    { A.OpExp($1, A.LOGICXOR, Some($3), context 1 3, A.Void) }
| exp LOGICAND exp                                                 /* && */
    { A.OpExp($1, A.LOGICAND, Some($3), context 1 3, A.Void) }
| exp LOGICEQ exp                                                  /* == */
    { A.OpExp($1, A.EQ, Some($3), context 1 3, A.Void) }
| exp LOGICNOTEQ exp                                               /* != */
    { A.OpExp($1, A.NEQ, Some($3), context 1 3, A.Void) }
| exp LOGICLESS exp                                                 /* < */
    { A.OpExp($1, A.LT, Some($3), context 1 3, A.Void) }
| exp LOGICLESSEQ exp                                              /* <= */
    { A.OpExp($1, A.LTE, Some($3), context 1 3, A.Void) }
| exp LOGICMORE exp                                                 /* > */
    { A.OpExp($1, A.GT, Some($3), context 1 3, A.Void) }
| exp LOGICMOREEQ exp                                              /* >= */
    { A.OpExp($1, A.GTE, Some($3), context 1 3, A.Void) }
| exp PLOGICEQ exp                                                /* *== */
    { A.OpExp($1, A.PEQ, Some($3), context 1 3, A.Void) }
| exp PLOGICNOTEQ exp                                             /* *!= */
    { A.OpExp($1, A.PNEQ, Some($3), context 1 3, A.Void) }
| exp  PLUS exp                                                     /* + */
    { A.OpExp($1, A.PLUS, Some($3), context 1 3, A.Void) }
| exp  MINUS exp                                                    /* - */
    { A.OpExp($1, A.MINUS, Some($3), context 1 3, A.Void) }
| exp TIMES exp                                                     /* * */
    { A.OpExp($1, A.TIMES, Some($3), context 1 3, A.Void) } 
| exp DIVIDE exp                                                    /* / */
    { A.OpExp($1, A.DIVIDE, Some($3), context 1 3, A.Void) }
| exp MOD exp                                                       /* % */
    { A.OpExp($1, A.MOD, Some($3), context 1 3, A.Void) }
| exp BITAND exp                                                 /* & */
    { A.OpExp($1, A.BITAND, Some($3), context 1 3, A.Void) }
| exp BITOR exp                                                     /* | */
    { A.OpExp($1, A.BITOR, Some($3), context 1 3, A.Void) }
| exp BITXOR exp                                                    /* ^ */
    { A.OpExp($1, A.BITXOR, Some($3), context 1 3, A.Void) }
| BITNOT exp                                                        /* ~ */
  { A.OpExp($2, A.BITNOT, None , context 1 2, A.Void) }
| exp SHIFTLEFT exp                                                /* << */
    { A.OpExp($1, A.SHIFTLEFT, Some($3), context 1 3, A.Void) }
| exp SHIFTRIGHT exp                                               /* << */
    { A.OpExp($1, A.SHIFTRIGHT, Some($3), context 1 3, A.Void) }
| exp  PPLUS exp                                                   /* p+ */
    { A.OpExp($1, A.PPLUS, Some($3), context 1 3, A.Void) }
| exp  PMINUS exp                                                  /* p- */
    { A.OpExp($1, A.PMINUS, Some($3), context 1 3, A.Void) }
| MINUS exp      %prec UMINUS                         /* - (Unary minus) */
  { A.OpExp($2, A.UMINUS, None, context 1 2, A.Void) }
| LOGICNOT exp                                                      /* ! */
    { A.OpExp($2, A.LOGICNOT, None, context 1 2, A.Void) }
| BITAND exp  %prec AMPERSAND                           /* & (reference) */
    { A.Reference(make_lval (context 2 2) $2, context 1 2, A.Void) }
| constexp                                                   /* constant */
    { A.ConstExp($1, A.Void) }
| LPAREN exp RPAREN                                             /* (exp) */
    { $2 }
| ALLOC LPAREN exp COMMA types RPAREN                           /* alloc */
    { A.Alloc($3, context 3 3, $5, context 5 5, A.Void) } 
| OFFSET LPAREN exp RPAREN                                   /* offset() */
    { A.Offset($3, context 3 3, A.Void) }
| SIZE LPAREN exp RPAREN                                       /* size() */
    { A.Size($3, context 3 3, A.Void) }
| ID LPAREN args RPAREN
    { A.Call(A.Ident($1,context 1 1), $3, context 3 3, A.Void) }
| ID LPAREN RPAREN
    { A.Call(A.Ident($1,context 1 1), [], context 2 3, A.Void) }
| ID                                                       /* identifier */
    { A.LVal(A.Var(A.Ident($1, context 1 1), A.Void)) }        
| TIMES exp                  %prec UTIMES             /* * (dereference) */
    { A.LVal(A.Deref($2, context 1 2, A.Void)) } 
| exp LBRACKET exp RBRACKET                       /* a[b] (array access) */
    { A.LVal(A.Deref(A.OpExp($1, A.PPLUS, Some($3), context 1 4, A.Void), 
		    context 1 4, A.Void)) }
| exp RARROW ID                                  /* a->b (struct access) */
    { A.LVal(A.Field(A.Deref($1, context 1 1, A.Void), 
		     A.Ident($3,context 3 3),  context 1 3, A.Void)) }
| exp DOT ID                                      /* a.b (struct access) */
    { A.LVal(A.Field(make_lval (context 1 1) $1, A.Ident($3, context 3 3) , 
		     context 1 3, A.Void)) }

   
    
args:
| exp 
    { [($1, context 1 1)] }
| exp COMMA args
    { ($1, context 1 1) :: $3 }

%%
