(* frame.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: frame.ml,v 1.2 2003/08/11 13:04:43 rflury Exp $ *)

module T = Ir
module HA = Hashtbl
module A = Absyn
module TP = Temp

module type FrameSig = 
  sig

    (* the word-size of the machine *)
    val wordSize_i : int
    val wordSize : int32

    (* The function call *)
    val externalCall : string -> Ir.exp list -> Ir.exp 

    (* Return code to access the n-th parameter, n=1,2,3... of a fun *)
    val paramTemp : int -> Pcc.pccType -> Ir.exp


    (* returns the temporary for the stack-pointer *)
    val sp : unit -> TP.temp

    (* returns the temporary for the base-pointer *)
    val bp : unit -> TP.temp

    (* Store return value in this temp *)
    val getReturnTemp : unit -> TP.temp

    (* Returns the complementary list of registers *)
    val complementReg : TP.temp list -> TP.temp list

    (* Returns the label for the NULL-pointer *)
    val nullPointerLabel : unit -> TP.label

    (* the first valid temporary (not precolored ones) *)
    val firstTemp : unit -> TP.temp

    (* returns true if the passed temp is a precolorerd one *)
    val precolored : TP.temp -> bool

  end


module X86_FRAME = 
  struct

    (* Temp -> Register Mapping
     * 1 -> %eax
     * 2 -> %ecx
     * 3 -> %edx
     * 4 -> %ebx    (callee-save)
     * 5 -> %esi    (callee-save)
     * 6 -> %edi    (callee-save)
     * 7 -> %esp    stack
     * 8 -> %ebp    base (frame)
     *)

    (* returns the temporary for the given register *)
    let sp () = 7
    let bp () = 8
	
    (* Store return value in this temp *)
    let getReturnTemp () = 1

    (* the word-size of the machine *)
    let wordSize_i = 4
    let wordSize = Int32.of_int wordSize_i

    (* The function call *)	
    let externalCall s args = 
      T.CALL(TP.namedLabel s, args)

    (* Return IR-code to access the n-th parameter, n=1,2,3... of a fun *)
    let paramTemp n typ = 
      T.MEM(T.BINOP(T.PLUS, 
		    T.TEMP(bp ()),
		    T.CONST(Int32.of_int (4 * (n+1)), Pcc.Int))) 

    (* Returns the complementary list of registers *)
    let complementReg list = 
      List.fold_left (fun res t -> 
	if(List.mem t list) then
	  res 
	else
	  t :: res
		     ) [] [1;2;3;4;5;6;7;8]

    let nullPointerLab = TP.newLabel () 
	
    (* Returns the label for the NULL-pointer *)
    let nullPointerLabel () = 
      nullPointerLab

    (* the first valid temporary (not precolored ones *)
    let firstTemp () = 10

    (* returns true if the passed temp is a precolorerd one *)
    let precolored t =
      t > 0 && t <= 8

  end






